package com.openexchange.office.tools;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * An enumeration to provide specific error information from the server to
 * the client side. The enumeration provides method to convert to JSON object
 * or to be created from a JSON object.
 *
 * {@link ErrorCode}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public enum ErrorCode {

    NO_ERROR(0, "NO_ERROR", "No error"),
    GENERAL_ARGUMENTS_ERROR(10, "GENERAL_ARGUMENTS_ERROR", "The request have been called with missing or wrong arguments"),
    FILTER_NOBACKUP_ERROR(100, "FILTER_NOBACKUP_ERROR", "The document is inconsistent and no backup file could be created!"),
    FILTER_OPERATION_ERROR(101, "FILTER_OPERATION_ERROR", "The document is inconsistent and is saved as doc_ox file!"),
    FILTER_QUOTA_REACHED_ERROR(102, "FILTER_QUOTA_REACHED_ERROR", "The document could not be written due to quota limitations!"),
    CREATEDOCUMENT_CONVERSION_FAILED_ERROR(200, "CREATEDOCUMENT_CONVERSION_FAILED_ERROR", "The document couldn't be created because of an conversion error!"),
    CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR(201, "CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR", "The document couldn't be created because the template file couldn't be read!"),
    CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR(202, "CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR", "The document coulnd't be created because the default template file couldn't be read!"),
    CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR(203, "CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR", "The document couldn't be created due to missing permissions!"),
    CREATEDOCUMENT_QUOTA_REACHED_ERROR(204, "CREATEDOCUMENT_QUOTA_REACHED_ERROR", "The document couldn't be created because the quota has been reached!"),
    RENAMEDOCUMENT_FAILED_ERROR(300, "RENAMEDOCUMENT_FAILED_ERROR", "Renaming the document failed!"),
    RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR(301, "RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR", "The document couldn't be renamed as the validation of the new file name failed. File name contains invalid characters!"),
    LOADDOCUMENT_FAILED_ERROR(400, "LOADDOCUMENT_FAILED_ERROR", "Loading the document failed!"),
    LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR(401, "LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR", "Loading the document failed because operations couldn't be generated!"),
    LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR(402, "LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR", "Loading the document failed because it's password protected!"),
    LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR(403, "LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR", "Loading the document failed because the complexity of the content is too high!"),
    LOADDOCUMENT_FILE_NOT_FOUND_ERROR(404, "LOADDOCUMENT_FILE_NOT_FOUND_ERROR", "Loading the document failed because the file couldn't be found!"),
    LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR(405, "LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR", "Loading the document failed because the server-side engine used for calculating formulas is not available."),
    HANGUP_INVALID_OPERATIONS_SENT_ERROR(500, "HANGUP_INVALID_OPERATIONS_SENT_ERROR", "Client sent invalid operations and therefore lost edit rights!"),
    HANGUP_NO_EDIT_RIGHTS_ERROR(501, "HANGUP_NO_EDIT_RIGHTS_ERROR", "Client sent operations without having edit rights. Operations are ignored and client must switch to read-only mode!"),
    HANGUP_INVALID_OSN_DETECTED_ERROR(502, "HANGUP_INVALID_OSN_DETECTED_ERROR", "Client has invalid OSN after losing edit rights. Must switch to read-only mode."),
    HANGUP_CALCENGINE_NOT_RESPONDING_ERROR(503, "HANGUP_CALCENGINE_NOT_RESPONDING_ERROR", "Calcengine is not responding. Clients must be set to read-only to resync again."),
    SAVEDOCUMENT_FILE_NOT_FOUND_ERROR(600, "SAVEDOCUMENT_FILE_NOT_FOUND_ERROR", "Saving document failed because the file couldn't be found."),
    COPYDOCUMENT_FAILED_ERROR(700, "COPYDOCUMENT_FAILED_ERROR", "Copying the document failed due to unknown reasons."),
    COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR(701, "COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR", "Copying the document failed, because the user's folder is unknown."),
    COPYDOCUMENT_FILENAME_UNKNOWN_ERROR(702, "COPYDOCUMENT_FILENAME_UNKNOWN_ERROR", "Copying the document failed, because the file name is unknown"),
    COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR(703, "COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR", "Copying the document failed, because the template format couldn't be determined."),
    GETTEMPLATELIST_UNKNOWN_ERROR(800, "GETTEMPLATELIST_UNKNOWN_ERROR", "Retrieving the template list failed."),
    GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR(801, "GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR", "The user folder is unknown, therefore the user template list cannot be retrieved."),
    GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR(802, "GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR", "There was an IO exception accessing the local file system."),
    LOCALFILE_FILE_NOT_FOUND_ERROR(900, "LOCALFILE_FILE_NOT_FOUND_ERROR", "The local file cannot be found."),
    LOCALFILE_CANNOT_READ_FILE_ERROR(901, "LOCALFILE_CANNOT_READ_FILE_ERROR", "The local file cannot be read."),
    LOCALFILE_INVALID_ID_ERROR(902, "LOCALFILE_INVALID_ID_ERROR", "The file id provided is invalid.");

    private final int m_code;
    private final String m_codeAsStringConstant;
    private final String m_description;

    private ErrorCode(int code, String codeAsString, String description) {
      this.m_code = code;
      this.m_codeAsStringConstant = codeAsString;
      this.m_description = description;
    }

    public String getDescription() {
       return m_description;
    }

    public int getCode() {
       return m_code;
    }

    public String getCodeAsStringConstant() {
        return m_codeAsStringConstant;
    }

    @Override
    public String toString() {
      return m_code + ": " + m_description;
    }

    public JSONObject getAsJSON() {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("errorCode", this.getCode());
            jsonObject.put("error", this.getCodeAsStringConstant());
            jsonObject.put("errorDescription", this.getDescription());
        } catch (final JSONException e) {
            //
        }
        return jsonObject;
    }

    static public int getErrorCodeFromJSON(JSONObject jsonObject, int defaultValue) {
        int result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optInt("errorCode", defaultValue);
        }
        return result;
    }

    static public String getErrorConstantFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("error", defaultValue);
        }
        return result;
    }

    static public String getErrorDescriptionFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("errorDescription", defaultValue);
        }
        return result;
    }
 }
