/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageExceptionCodes;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.groupware.ldap.User;
import com.openexchange.log.Log;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link OXDocument}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
class OXDocument {

    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     *  The session of the client that requested a service which needs documents
     *  access.
     *
     * @param fileHelper
     *  The FileHelper instance that is responsible for the file access regarding
     *  the document.
     *
     * @param resourceManager
     *  The ResourceManager instance which stores resources addressed by the document
     *  to be available via REST-API and other documents.
     */
    OXDocument(Session session, FileHelper fileHelper, ResourceManager resourceManager) {
        super();

        m_fileHelper = fileHelper;
        m_resourceManager = resourceManager;

        if (null != m_fileHelper) {
            final InputStream documentStm = m_fileHelper.getDocumentStream(session);

            if (null != documentStm) {
                try {
                    m_documentBuffer = IOUtils.toByteArray(documentStm);
                } catch (IOException e) {
                    LOG.error("RT connection: unable to retrieve document stream, exception catched", e);
                } finally {
                    IOUtils.closeQuietly(documentStm);
                }
            }

            if (null != m_documentBuffer) {
                final ZipInputStream zipInputStm = new ZipInputStream(new ByteArrayInputStream(m_documentBuffer));
                ZipEntry entry = null;
                byte[] documentBuffer = null;
                byte[] operationsBuffer = null;
                byte[] extendedDataBuffer = null;

                try {
                    // create a dummy resource manager, if no one is given
                    if (null == m_resourceManager) {
                        m_resourceManager = new ResourceManager(m_fileHelper.getServiceLookup());
                    }

                    while ((entry = zipInputStm.getNextEntry()) != null) {
                        final String entryName = entry.getName();

                        if (entryName.equals(FileHelper.OX_DOCUMENT_EXTENDEDDATA)) {
                            try {
                                // read document identifier
                                extendedDataBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve extended data stream, exception catched", e);
                            }
                        } else if (entryName.equals(FileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                            try {
                                // read original document
                                documentBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve original document stream, exception catched", e);
                            }
                        } else if (entryName.equals(FileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME)) {
                            try {
                                // read original document
                                final String op = IOUtils.toString(zipInputStm);
                                // non continuous osn numbers lead to a load error in the frontend, so we have to remove them as some operations are filtered by the calcengine
                                operationsBuffer = op.replaceAll("\\\"osn\\\":[0-9]+,?", "").getBytes();
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve operations stream, exception catched", e);
                            }
                        } else if (entryName.startsWith(FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX)) {
                            // read resource and put into the resource manager, if not already contained
                            final String resourceName = entryName.substring(FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX.length());

                            if (!m_resourceManager.containsResource(Resource.getUIDFromResourceName(resourceName))) {
                                byte[] resourceBuffer = null;

                                try {
                                    resourceBuffer = IOUtils.toByteArray(zipInputStm);
                                } catch (IOException e) {
                                    LOG.error("RT connection: unable to retrieve resource document stream, exception catched", e);
                                }

                                m_resourceManager.addResource(resourceBuffer);
                            }
                        }

                        zipInputStm.closeEntry();
                    }
                } catch (IOException e) {
                    LOG.error("RT connection: Exception catched while reading the document stream", e);
                } finally {
                    IOUtils.closeQuietly(zipInputStm);

                    // set content of original document
                    if (null != documentBuffer) {
                        m_documentBuffer = documentBuffer;
                    }
                    if (null != extendedDataBuffer) {
                        try {
                            m_uniqueDocumentId = new JSONObject(new String(extendedDataBuffer));
                        } catch (JSONException e) {
                            LOG.error("RT connection: unable to setup json object for unique document identifier", e);
                        }
                    }

                    // set content of already available operations
                    if (null != operationsBuffer) {
                        try {
                            m_documentOperations = new JSONObject(new String(operationsBuffer));
                        } catch (JSONException e) {
                            LOG.error("RT connection: unable to setup json object for document operations", e);
                        }
                    }
                }
            }

            if (null == m_documentBuffer) {
                // We were not able to retrieve the document buffer. Try to find out
                // some information about the file itself retrieving the meta data.
                // It's possible that the file has been removed in the mean time and
                // we get an exception.
                try {
                    m_fileHelper.getMetaDataFromFile(session, null);
                } catch (OXException e) {
                    m_lastErrorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                    if (e.getPrefix() == "IFO") {
                        if (e.getCode() == InfostoreExceptionCodes.NOT_EXIST.getNumber()) {
                            m_lastErrorCode = ErrorCode.LOADDOCUMENT_FILE_NOT_FOUND_ERROR;
                        }
                    } else if (e.getPrefix() == "FILE_STORAGE") {
                        if (e.getCode() == FileStorageExceptionCodes.FILE_NOT_FOUND.getNumber()) {
                            m_lastErrorCode = ErrorCode.LOADDOCUMENT_FILE_NOT_FOUND_ERROR;
                        }
                    }
                }
                LOG.warn("OXDocument Ctor: could not retrieve a valid document buffer from given session.");
            }
        }
    }

    /**
     * Retrieves the operations from the document and optional additional operations.
     *
     * @param session
     *  The session of the client requesting the document operations.
     *
     * @param importer
     *  The importer instance to be used to read the document stream and provide the
     *  operations.
     *
     * @param appendIOperations
     *  Optional operations which should be appended to the document operations. Normally
     *  used if there are queued operations which are not part of the last document stream.
     *
     * @return
     *  The JSONObject containing all operations from the document, the saved document operations and the given operations.
     */
    JSONObject getOperations(Session session, IImporter importer, JSONObject appendOperations) throws FilterException {
        JSONObject jsonResult = null;

        if ((null != session) && (null != m_documentBuffer) && (null != importer)) {
            final InputStream documentStm = new ByteArrayInputStream(m_documentBuffer);
            try {
                String userLangCode = null;

                if (session instanceof ServerSession) {
                    final User user = ((ServerSession) session).getUser();

                    if (null != user) {
                        userLangCode = FileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                    }
                }
                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                jsonResult = importer.createOperations(session, documentStm, docProps);

                String uniqueDocumentId = docProps.optString(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, null);
                if (null != uniqueDocumentId) {
                    try {
                        this.setUniqueDocumentId(Integer.parseInt(uniqueDocumentId));
                    } catch (NumberFormatException e) {
                        // nothing to do as the unique document identifier
                        // is only an optional value
                    }
                }

                if (LOG.isInfoEnabled()) {
                    LOG.info("TIME getOperations: " + (System.currentTimeMillis() - startTime));
                }

            } finally {
                IOUtils.closeQuietly(documentStm);
            }
        }

        // getting operations from the original document is mandatory
        if (null != jsonResult) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                ConnectionHelper.appendJSON(jsonResult, m_documentOperations);
            }

            // append given operations
            if (null != appendOperations) {
                ConnectionHelper.appendJSON(jsonResult, appendOperations);
            }
        }

        return jsonResult;
    }

    /**
     * Retrieves the last error code set by the latest method.
     *
     * @return
     *  The last error code set by the last method called.
     */
    ErrorCode getLastError() {
        return m_lastErrorCode;
    }

    /**
     * Retrieves the resource manager used by this document.
     *
     * @return the ResourceManager used to read the document
     */
    ResourceManager getResourceManager() {
        return m_resourceManager;
    }

    /**
     * Provides the unique document id stored in the
     * document.
     *
     * @return
     *  Returns the unique document id stored in the
     *  document or -1 if no id is available.
     */
    public int getUniqueDocumentId() {
        int id = -1;

        if (null != m_uniqueDocumentId) {
            id = m_uniqueDocumentId.optInt("uniqueDocumentIdentifier", -1);
        }

        return id;
    }

    /**
     * Sets a new unique document id to be stored
     * in the document on the next successful
     * flushDocument.
     *
     * @param uniqueId
     *  The new unique document id to be stored.
     */
    public void setUniqueDocumentId(int uniqueId) {
        if (null == m_uniqueDocumentId) {
            m_uniqueDocumentId = new JSONObject();
        }

        try {
            m_uniqueDocumentId.put("uniqueDocumentIdentifier", uniqueId);
        } catch (JSONException e) {
            // this is not a fatal error as we can work without this unique
            // id. just the speed up of the local cache won't work
            LOG.error("RT connection: Couldn't set unique document id to JSONObject", e);
        }
    }

    /**
     * Provides the new document stream that can be used to store
     * the document to the info store.
     *
     * @param session
     *  The session of the client requesting the document operations.
     *
     * @param exporter
     *  The exporter instance used to create the new document stream.
     *
     * @param resourceManager
     * The resource manager used by this document.
     *
     * @param messageChunkList
     *  The message chunk list containing the pending operations to be stored
     *  in the new document.
     *
     * @param version
     *  The version of the document used for document properties.
     *
     * @param errorCode
     *  The array containing a possible error code.
     *
     * @param finalSave
     *  Specifies if the new document stream should be handled as the
     *  final save. This can be used by the exporter to optimize the
     *  document, e.g. finally remove non-referenced resources.
     *
     * @return
     *  The stream of the new document containing the message chunk operations.
     *  Can be null if the creation of the stream failed or contains a debug
     *  document stream. The result depends on the errorCode.
     */
    InputStream getResolvedDocumentStream(Session session, IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList,
    		final String version, int[] errorCode, boolean finalSave, boolean saveAsTemplate) {

    	InputStream resolvedDocumentStream = (null != m_documentBuffer) ? new ByteArrayInputStream(m_documentBuffer) : null;
        final JSONObject combinedOperations = new JSONObject();

        errorCode[0] = MessageData.SAVE_ERROR_NONE;
        m_lastErrorCode = ErrorCode.NO_ERROR;

        // get combined operations
        if ((null != m_documentOperations) || !messageChunkList.isEmpty()) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                ConnectionHelper.appendJSON(combinedOperations, m_documentOperations);
            }

            // apply collected operations
            if (null != messageChunkList) {
                ConnectionHelper.appendOperationChunks(combinedOperations, messageChunkList, true);
            }
        }

        // if there are any operations, try to export them with the document exporter first
        if ((null != resolvedDocumentStream) && (combinedOperations.has("operations")||saveAsTemplate) && (null != exporter) && (null != resourceManager)) {
            InputStream combinedStm = null;

            // set error code to worst case first
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;
            m_lastErrorCode = ErrorCode.FILTER_NOBACKUP_ERROR;

            try {
                String userLangCode = null;
                String userName = "";

                if (session instanceof ServerSession) {
                    final User user = ((ServerSession) session).getUser();

                    if (null != user) {
                        userLangCode = FileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                        userName = user.getDisplayName();
                    }
                }

                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                docProps.put(DocumentProperties.PROP_LAST_MODIFIED_BY, userName);
                docProps.put(DocumentProperties.PROP_SAVE_TEMPLATE_DOCUMENT, saveAsTemplate);
                int documentOSN = this.getUniqueDocumentId();
                if (documentOSN != -1) {
                    docProps.put(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, String.valueOf(documentOSN));
                }
                if (null != version) {
                    // only use a valid version string
                    docProps.put(DocumentProperties.PROP_VERSION, version);
                }
                
                final String mergeOperations = combinedOperations.has("operations") ? combinedOperations.getJSONArray("operations").toString() : null;

                combinedStm = exporter.createDocument(
                    session,
                    resolvedDocumentStream,
                    mergeOperations,
                    resourceManager,
                    docProps,
                    finalSave);

                if (LOG.isInfoEnabled()) {
                    LOG.info("TIME createDocument: " + (System.currentTimeMillis() - startTime));
                }

            } catch (FilterException e) {
                LOG.error("Document export filter threw a filter exception while exporting the document", e);
            } catch (Exception e) {
                LOG.error("Document export filter threw an exception while exporting the document", e);
            } finally {
                IOUtils.closeQuietly(resolvedDocumentStream);

                if (null != (resolvedDocumentStream = combinedStm)) {
                    errorCode[0] = MessageData.SAVE_ERROR_NONE;
                    m_lastErrorCode = ErrorCode.NO_ERROR;
                }
            }
        } else if (combinedOperations.has("operations")) {
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;
            m_lastErrorCode = ErrorCode.FILTER_NOBACKUP_ERROR;

            if (LOG.isErrorEnabled()) {
                if (null == exporter) {
                    LOG.error("Document export filter not available - not possible to provide resolved stream");
                }
                if (null == resourceManager) {
                    LOG.error("Resource manager not available - not possible to provide resolved stream");
                }
                if (null == resolvedDocumentStream) {
                    LOG.error("Document stream not available - not possible to provide resolved stream");
                }
            }
        }

        /*
        {
            // !!! DEBUG code to create OX rescue documents by setting the document name to the    !!!
            // !!! name of the env var 'OX_RESCUEDOCUMENT_NAME'; may be commented for final release!!!
            final String oxRescueDocumentName_debug = System.getenv("OX_RESCUEDOCUMENT_NAME");

            if ((null != oxRescueDocumentName_debug) && (null != m_fileHelper)) {
                try {
                    final String filename = m_fileHelper.getFilenameFromFile(session);

                    if ((null != filename) && filename.startsWith(oxRescueDocumentName_debug)) {
                        errorCode[0] = MessageData.SAVE_ERROR_OPERATIONS;
                        m_lastErrorCode = ErrorCode.FILTER_OPERATION_ERROR;
                    }
                } catch (OXException e) {
                    LOG.error("Could not retrieve file name from file for saving OX rescue document");
                }
            }
        }
         */

        // if the there was an error before, we write a zipped OXDocument with
        // the original document and the operations content in separate streams
        if (MessageData.SAVE_ERROR_NONE != errorCode[0]) {
            InputStream inputStm = null;
            final ByteArrayOutputStream byteArrayStm = new ByteArrayOutputStream(8192);
            final ZipOutputStream zipOutputStm = new ZipOutputStream(byteArrayStm);

            // set error code to worst case
            errorCode[0] = MessageData.SAVE_ERROR_NOBACKUP;
            m_lastErrorCode = ErrorCode.FILTER_NOBACKUP_ERROR;

            // write original document as well as collected operations
            // and collected resources into different zip entry streams
            try {
                // write original document stream
                if (null != m_documentBuffer) {
                    inputStm = new ByteArrayInputStream(m_documentBuffer);

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(FileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        LOG.error("Exception while writing original document stream to backup document", e);
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected operations, that were added
                // by editing users, into separate zip entry stream
                if (!combinedOperations.isEmpty()) {
                    inputStm = new ByteArrayInputStream(combinedOperations.toString().getBytes("UTF-8"));

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(FileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        if (LOG.isErrorEnabled()) {
                            LOG.error("Exception while writing operations stream to backup document", e);
                        }
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected resources, that were added by
                // editing users, into separate zip entry streams
                if (null != resourceManager) {
                    final Hashtable<Long, byte[]> resourceBuffers = resourceManager.getByteArrayTable();

                    for (final Long curKey : resourceBuffers.keySet()) {
                        final byte[] curResourceBuffer = resourceBuffers.get(curKey);

                        if (null != curResourceBuffer) {
                            final String curResourceName = FileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX + Resource.getResourceNameFromUID(curKey.longValue());

                            inputStm = new ByteArrayInputStream(curResourceBuffer);

                            try {
                                zipOutputStm.putNextEntry(new ZipEntry(curResourceName));
                                IOUtils.copy(inputStm, zipOutputStm);
                                zipOutputStm.flush();
                            } catch (IOException e) {
                                if (LOG.isErrorEnabled()) {
                                    LOG.error("Exception while writing collected resources to backup document", e);
                                }
                            } finally {
                                zipOutputStm.closeEntry();
                                IOUtils.closeQuietly(inputStm);
                                inputStm = null;
                            }
                        }
                    }
                }

                zipOutputStm.flush();
            } catch (IOException e) {
                LOG.error("Exception while writing backup document", e);
            } finally {
                IOUtils.closeQuietly(zipOutputStm);
                final byte[] byteArray = byteArrayStm.toByteArray();

                if (byteArray.length > 0) {
                    resolvedDocumentStream = new ByteArrayInputStream(byteArray);
                    errorCode[0] = MessageData.SAVE_ERROR_OPERATIONS;
                    m_lastErrorCode = ErrorCode.FILTER_OPERATION_ERROR;
                }
            }
        }

        return resolvedDocumentStream;
    }

    /**
     * Saves the new document to the info store.
     *
     * @param session
     *  The session of the client requesting the document operations.
     *
     * @param exporter
     *  The exporter instance used to create the new document stream.
     *
     * @param resourceManager
     * The resource manager used by this document.
     *
     * @param messageChunkList
     *  The message chunk list containing the pending operations to be stored
     *  in the new document.
     *
     * @param errorCode
     *  The array containing a possible error code.
     *
     * @param revisionless
     *  Specifies if the new document should be saved as a new version or
     *  using the latest
     *
     * @param finalSave
     *  Specifies if the new document stream should be handled as the
     *  final save. This can be used by the exporter to optimize the
     *  document, e.g. finally remove non-referenced resources.
     *
     * @return
     *  The new version of the file, which can be the same as before for a
     *  revisionless save. Can be null if saving was not successful and
     *  the errorCode array contains the errorCode which describe what failed.
     */
    String save(Session session, IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList, int[] errorCode, boolean revisionless, boolean finalSave) {
        String currentVersion = null;
        String fileVersion = null;

        try {
            currentVersion = m_fileHelper.getVersionFromFile(session);
        } catch (OXException e) {
            if (e.getPrefix() == "IFO") {
                if (e.getCode() == InfostoreExceptionCodes.NOT_EXIST.getNumber()) {
                    errorCode[0] = MessageData.SAVE_ERROR_FILE_NOT_FOUND;
                    m_lastErrorCode = ErrorCode.SAVEDOCUMENT_FILE_NOT_FOUND_ERROR;
                    return fileVersion;
                }
            }
        }

        // revisions are incremented by one - determine current version and
        // provide an incremented version.
        if ((null != currentVersion) && !revisionless) {
            try {
                int currentRevision = Integer.parseInt(currentVersion);
                currentRevision++;
                currentVersion = Integer.toString(currentRevision);
            } catch (final NumberFormatException e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Exception while determining the new version", e);
                }
            }
        }

        InputStream documentStm = getResolvedDocumentStream(
            session,
            exporter,
            resourceManager,
            messageChunkList,
            currentVersion,
            errorCode,
            finalSave,
            false);

        if ((MessageData.SAVE_ERROR_NOBACKUP != errorCode[0]) && (null != documentStm)) {
            final ErrorCode noError = ErrorCode.NO_ERROR;
            final ErrorCode[] writeDocErrorCode = { noError };

            fileVersion = m_fileHelper.writeDocumentStream(
                session,
                documentStm,
                Properties.OX_RESCUEDOCUMENT_EXTENSION_APPENDIX,
                MessageData.SAVE_ERROR_OPERATIONS == errorCode[0],
                writeDocErrorCode,
                revisionless);

            if (writeDocErrorCode[0].getCode() != 0) {
                // error evaluation
                if (writeDocErrorCode[0] == ErrorCode.FILTER_QUOTA_REACHED_ERROR) {
                    errorCode[0] = MessageData.SAVE_ERROR_QUOTA_REACHED;
                    m_lastErrorCode = ErrorCode.FILTER_QUOTA_REACHED_ERROR;
                }
            }
        }

        IOUtils.closeQuietly(documentStm);

        return fileVersion;
    }

    // - Members ---------------------------------------------------------------

    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(OXDocument.class);

    private FileHelper m_fileHelper = null;

    private ErrorCode m_lastErrorCode = ErrorCode.NO_ERROR;

    private byte[] m_documentBuffer = null;

    private JSONObject m_documentOperations = null;

    private ResourceManager m_resourceManager = null;

    private JSONObject m_uniqueDocumentId = null;
}
