/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.base.Optional;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.capabilities.CapabilitySet;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.groupware.ldap.User;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineClipBoard;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.DocumentFormat;
import com.openexchange.office.tools.DocumentFormatHelper;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.monitoring.CloseEvent;
import com.openexchange.office.tools.monitoring.CloseType;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.OpenEvent;
import com.openexchange.office.tools.monitoring.OpenType;
import com.openexchange.office.tools.monitoring.OperationsEvent;
import com.openexchange.office.tools.monitoring.OperationsType;
import com.openexchange.office.tools.monitoring.SaveEvent;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.realtime.Asynchronous;
import com.openexchange.realtime.group.GroupDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.Duration;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.file.external.QuotaFileStorageExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tools.session.ServerSessionAdapter;
import com.openexchange.tx.TransactionAwares;

/**
 * The Connection class implements the necessary specialties of a OX Document instance. This includes loading/saving documents, apply
 * changes, switching the edit rights between clients and send update messages to the connected clients. {@link Connection}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
final public @NotThreadSafe
class Connection extends GroupDispatcher {

    /**
     * A PrepareLosingEditRightsTimeoutRunnable provides a time-out mechanism which is needed while switching edit rights from one client to
     * another client.
     *
     * @author Carsten Driesner
     */
    private static class PrepareLosingEditRightsTimeoutRunnable implements Runnable {

        private final Connection m_connection;

        private final Session m_session;

        private final ID m_fromId;

        /**
         * Initializes a new instance of a PrepareLosingEditRightsTimeoutRunnable.
         *
         * @param connection The reference to the Connection instance that needs the time-out mechanism.
         * @param session The session of the client that has to release the edit rights.
         * @param fromId The ID of the client that has to release the edit rights.
         */
        public PrepareLosingEditRightsTimeoutRunnable(final Connection connection, Session session, ID fromId) {
            super();
            m_connection = connection;
            m_session = session;
            m_fromId = fromId;
        }

        /**
         * The method that gets called by the timer service. It calls an internal method
         * of the Connection instance to handle a timeout case.
         */
        @Override
        public void run() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: RT handling waiting for answer [canloseeditrights]. Timeout reached, no answer from client: " + m_fromId.toString());
            }

            // call the canLoseEditRights method setting the timeout parameter to true
            m_connection.canLoseEditRights(m_session, m_fromId, 0, true);
        }
    }

    /**
     * JSONDebugWriter implements java.io.Writer to limit the log output of received operations.
     *
     * @author Carsten Driesner
     */
    private static class JSONDebugWriter extends java.io.Writer {

        private final StringBuffer m_output;

        private static final int MAX_LENGTH = 1024;

        private int lastWrittenLength = -1;

        public JSONDebugWriter() {
            super();

            m_output = new StringBuffer(512);
        }

        public String getData() {
            return m_output.toString();
        }

        @Override
        public void close() throws IOException {
            // nothing
        }

        @Override
        public void flush() throws IOException {
            // nothing
        }

        /**
         * Writes the content of the provided buffer using a predefined limit where the buffer content is cut of.
         *
         * @param cbuf A character buffer to be written.
         * @param off A offset within the buffer.
         * @param len The number of character to output beginning from off.
         */
        @Override
        public void write(char[] cbuf, int off, int len) throws IOException {
            int correctLen = len;

            // Limit the amount of data written for an operation.
            // We assume that image data is written in chunks which exceeds
            // our own MAX_LENGTH
            if (lastWrittenLength < MAX_LENGTH) {
                m_output.append(cbuf, off, correctLen = Math.min(len, MAX_LENGTH));
            } else if (len < MAX_LENGTH) {
                m_output.append(cbuf, off, correctLen);
            }

            lastWrittenLength = correctLen;
        }
    }

    /**
     * Initializes a new {@link Connection}.
     *
     * @param services
     *  A service lookup instance which provides access to other services needed by the OX Documents GroupDispatcher.
     *
     * @param id
     *  The ID which references the resource used for this OX Documents GroupDispatcher. Normally references a file in the
     *  Infostore.
     */
    public Connection(ServiceLookup serviceLookup, ID id) {
        super(id, m_handler);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Connection created: " + id.toString());
        }

        // initialize ServiceLookup and ResourceManager
        m_services = serviceLookup;
        m_resourceManager = new ResourceManager(m_services);
        m_lastSaveTime = new Date();
        // No need to support a user-based max time for sending last actions.
        m_nMaxTimeForSendingLastActions = getConfigurationService().getIntProperty(
            "io.ox/office//module/switchingEditRightsTimeout",
            DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS);

        if (m_nMaxTimeForSendingLastActions != 0) {
            // set time in milli seconds
            m_nMaxTimeForSendingLastActions *= 1000;
        }
        setCreateVersion(true);

        // resource layout is: folderId.fileId
        final String resource = id.getResource();

        if (null != resource) {
            final int fileIdDelimiter = resource.indexOf('.');
            String folderId = null;
            String fileId = null;

            if ((-1 < fileIdDelimiter) && (fileIdDelimiter < (resource.length() - 1))) {
                folderId = resource.substring(0, fileIdDelimiter);
                fileId = resource.substring(fileIdDelimiter + 1);
            }

            if ((null != folderId) && (folderId.length() > 0) && (null != fileId) && (fileId.length() > 0)) {
                m_fileHelper = new FileHelper(m_services, folderId, fileId);
            }

            m_syncStatus = new Sync(folderId, fileId);
        }

        if (LOG.isDebugEnabled()) {
            if (null != m_fileHelper) {
                LOG.debug("RT connection: Established for resource: " + id.toString());
            } else {
                LOG.debug("RT connection: NOT established for resource: " + id.toString());
            }
        }
    }

    /**
     * Provides the configuration service.
     *
     * @return
     *  A configuration service instance.
     */
    private ConfigurationService getConfigurationService() {
        return m_services.getService(com.openexchange.config.ConfigurationService.class);
    }

    /**
     * Handles the "firstjoined" notification provided by the Realtime-Framework. This method is called
     * when the first client joins/creates the document.
     *
     * @param id
     *  The ID of the first client that joins/creates to the document (chat-room). (non-Javadoc)
     *
     * @see com.openexchange.realtime.group.GroupDispatcher#firstJoined(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void firstJoined(ID id) {
        m_threadId = Thread.currentThread().getId();
        setCreateVersion(true);

        if (null != m_fileHelper) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: First client joined RT connection: " + ((null != id) ? id.toString() : "n/a"));
                LOG.debug("RT connection: Thread Id = " + String.valueOf(m_threadId));
            }
        } else if (LOG.isErrorEnabled()) {
            LOG.error("RT connection: Import of operations not possible due to invalid file parameters: " + ((null != id) ? id.toString() : ""));
        }

        m_resourceManager.lockResources(true);
    }

    /**
     * Handles the "join" notification provided by the Realtime-Framework. This method is called when a
     * client joins the document.
     *
     * @param id
     *  The ID of the client that want to join the document (chat-room).
     *
     * @see com.openexchange.realtime.group.GroupDispatcher#onJoin(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onJoin(ID id) {
        super.onJoin(id);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onJoin called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Client joined: " + ((null != id) ? id.toString() : ""));
        }

        int curClientCount = 0;
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: onJoin called from different thread " + String.valueOf(threadId));
        }

        synchronized (m_connectionStatus) {
            if (id != null) {
                try {
                    ServerSession currentSession = id.toSession();
                    m_connectionStatus.addActiveUser(id.toString(), currentSession.getUser().getDisplayName(), new JSONObject());
                } catch (OXException e) {
                    LOG.error("RT connection: onJoin catches exception while retrieving session from joining user ID. ", e);
                }
            }
            curClientCount = m_connectionStatus.getActiveClients();
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                debugPrintActiveClients(m_connectionStatus);
            }
        }

        boolean joiningClientReceivesEditRights = false;

        synchronized(m_connectionStatus) {
            String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            String currentEditUser = m_connectionStatus.getCurrentEditingUserId();

            if ((1 == curClientCount) && (null != id)) {
                // first client always receives the edit rights
                joiningClientReceivesEditRights = true;
            } else if (((null != currentEditUser) && (currentEditUser.length() == 0)) &&
                       ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() == 0))) {
                // Fix for 30583 & better edit rights handling
                // Currently no one has the edit rights and no one wants to
                // receive the edit rights => next joining client receives the edit rights
                joiningClientReceivesEditRights = true;
            }
        }

        if (joiningClientReceivesEditRights) {
            try {
                // No need that acquireEdit sends any update message
                // therefore set silent to true. As stated above
                // the current state is transferred via getWelcomeMessage.
                acquireEdit(null, id, true);
            } catch (OXException e) {
                LOG.error("RT connection: onJoin calling acquireEdit failed with exception", e);
            }
        }

        if (curClientCount > 1) {
            // send update to other clients to update their active clients count etc.
            updateClients(new MessageData(null, syncGetConnectionStatus(), null), id);
        }
    }

    /**
     * Handles the "leave" notification provided by the Realtime-Framework.
     * This method is called when a client leaves the document group.
     * Attention: If getSignOffMessage is marked with the annotation
     * "Asynchronous" onLeave is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id
     *  The ID of the last client that wants to leave the document (chat-room).
     *
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id) {
        super.onLeave(id);

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onLeave called from different thread " + String.valueOf(threadId));
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onLeave called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Client " + id.toString() + " leaving => Flushing operations");
        }

        Session session = Connection.getServerSession(null, id);

        // local variables storing
        int curClientCount = 0;
        boolean editorLeaves = false;
        ConnectionStatus connectionStatus = null;
        String lastModifyingUserId = "";
        String wantsEditRightsUser = "";
        String currentEditUser = "";

        synchronized (m_connectionStatus) {

            lastModifyingUserId = m_connectionStatus.getLastModifyingUserId();
            curClientCount = m_connectionStatus.getActiveClients();
            if (curClientCount > 0) {
                try {
                    boolean removeUserStatus = m_connectionStatus.removeActiveUser(id.toString());
                    if (LOG.isDebugEnabled() && removeUserStatus) {
                        LOG.debug("RT connection: onLeave removed this active user:" + id.toString());
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: onLeave catches JSON exception while removing active user", e);
                }

                curClientCount = m_connectionStatus.getActiveClients();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                    debugPrintActiveClients(m_connectionStatus);
                }
            }

            // Check if the leaving user wants the edit rights. If yes we have
            // to remove the user to enable others to acquire them.
            wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            currentEditUser = m_connectionStatus.getCurrentEditingUserId();
            if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                ID wantsEditRightsUserId = new ID(wantsEditRightsUser);
                if (wantsEditRightsUserId.equals(id)) {
                    // Reset wantsEditRightsUser from connectionStatus
                    m_connectionStatus.setWantsEditRightsUser("", "");
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Wants to edit client: " + wantsEditRightsUser + " leaving, reset wantsEditRightsUser");
                    }
                }
            }

            // Handle if the leaving user has edit rights
            if ((null != currentEditUser) && (currentEditUser.length() > 0)) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(id)) {
                    // Reset current editing user from connectionStatus
                    m_connectionStatus.setCurrentEditingUser("", "");
                    editorLeaves = true;

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Editor " + currentEditUser + " leaving, reset editor");
                    }

                    if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Editor " + currentEditUser + " left, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                        }
                        // Set wantsEditRightsUser as new editor if known
                        m_connectionStatus.setCurrentEditingUser(wantsEditRightsUser, m_connectionStatus.getWantsEditRightsUserName());
                        // Reset wantsEditRightsUser so others can acquire the edit rights
                        m_connectionStatus.setWantsEditRightsUser("", "");
                        // cancel timer as we already set the new editor
                        this.cancelPrepareLosingEditRightsTimer();
                    }
                }
            }
            connectionStatus = (ConnectionStatus) m_connectionStatus.clone();
        }

        if (curClientCount > 0) {
            updateClientsExceptSender(new MessageData(null, connectionStatus), id);
        }

        // this call normally doesn't do anything as the getSignOffMessage
        // already made a flushDocument.
        flushDocument(session, false, (curClientCount == 0));

        if (!editorLeaves && (null != currentEditUser) && (currentEditUser.length() > 0)) {
            // We want to use the session of the current editor to create
            // temporary versions. Otherwise this can confuse users if they
            // look at the version history.
            ID currentEditUserId = new ID(currentEditUser);
            setCreatorSynchronized(currentEditUserId);
        } else if (editorLeaves && (id.toString().equals(lastModifyingUserId))) {
            // Check if Editor leaves that she also made the latest changes to the document.
            // Otherwise we would change the creator to a virtual editor that just acquired
            // edit rights without any modification.
            setCreatorSynchronized(session);
            // if editor leaves the next save must create a new version
            setCreateVersion(true);
        }

        // update statistics with standard CLOSE type
        if (null != m_fileHelper) {
            Statistics.handleDocumentEvent(new CloseEvent(m_fileHelper.getDocumentType(), CloseType.CLOSE));
        }
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework. This method is called when the last client left the document
     * and the Realtime-Framework wants to dispose the GroupDispatcher instance. Attention: If getSignOffMessage or onLeave is marked with
     * the annotation "Asynchronous" onDispose is also called from the same background thread. Therefore this method must be thread-safe,
     * too.
     *
     * @param id
     *  The ID of the last client that left the document (chat-room).
     *
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        super.onDispose(id);

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onDispose called for document: " + getDocumentFolderAndFileId());
            LOG.debug("RT connection: Last client leaving => Flushing operations");
        }

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isDebugEnabled()) {
            LOG.debug("RT connection: onDispose called from different thread " + String.valueOf(threadId));
        }

        if (m_calcClient != null) {
            try {
                m_calcClient.destroyDocument(m_calcDocumentHandle);
            } catch (Exception e) {
                LOG.error("RT connection: Error deleting spreadsheet document", e);
            }
        }

        // The connection object will be disposed. therefore we don't have
        // any client which can be informed. we have to ignore the error
        // for this flushDocument call. the next client will get this
        // info loading the document.
        flushDocument(getServerSession(null, id), true, true);

        m_resourceManager.lockResources(false);
        synchronized (m_connectionStatus) {
            cancelPrepareLosingEditRightsTimer();
            m_connectionStatus.setWantsEditRightsUser("", "");

            // update statistics with TIMEOUT close type, if there are still active clients by number
            if ((m_connectionStatus.getActiveClients() > 0) && (null != m_fileHelper)) {
                Statistics.handleDocumentEvent(new CloseEvent(m_fileHelper.getDocumentType(), CloseType.TIMEOUT));
            }
        }
    }

    /**
     * Provides a stanza for the first message sent to the client joining a document (chat-room). The OX Documents client expects to find the
     * document operation within the stanza provided by this method.
     *
     * @param onBehalfOf The ID of the client which wants to join the document.
     * @return
     *  The stanza sent to the client which wants to join the document. The stanza includes the document operations and
     *  operations which are currently pending for the next save. The stanza also has the update state including the editor ID.
     *
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message welcomeMessage = null;
        JSONObject lastOperation = null;
        JSONObject firstOperation = null;
        boolean opsFromDocument = true;
        DocumentEvent documentEvent = null;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            final ServerSession serverSession = getServerSession(null, onBehalfOf);
            final OXDocument oxDocument = new OXDocument(serverSession, m_fileHelper, m_resourceManager);
            final JSONObject jsonResult = new JSONObject();
            final JSONObject operations = new JSONObject();
            int messageChunkListSize = 0;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [getWelcomeMessge] called for document: " + getDocumentFolderAndFileId());
            }

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            ErrorCode errorCode = oxDocument.getLastError();

            // Initialize result object with operations from persistent document first.
            // Check for FilterExceptions that can occur due to problems with the file.
            try {
                ConnectionHelper.appendJSON(operations, oxDocument.getOperations(serverSession, getImporter(), null));
            } catch (final FilterException e) {
                errorCode = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: getWelcomeMessage, Exception catched", e);
                }

                if (e.getErrorcode() == FilterException.ErrorCode.WRONG_PASSWORD || e.getErrorcode() == FilterException.ErrorCode.UNSUPPORTED_ENCRYPTION_USED) {
                    errorCode = ErrorCode.LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR;
                }
                else if(e.getErrorcode() == FilterException.ErrorCode.COMPLEXITY_TOO_HIGH) {
                    errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR;
                }
            }

            if (errorCode.getCode() != ErrorCode.NO_ERROR.getCode()) {
                // In case of an error we have to leave here and send a message with the correct error information.
                JSONObject jsonExtraData = new JSONObject();
                try {
                    jsonExtraData.put(MessageData.KEY_CLIENT_ID, onBehalfOf.toString());
                } catch (JSONException jsonEx) {
                    LOG.error("RT connection: getWelcomeMessage, Exception catched while setting client id to message.", jsonEx);
                }
                final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), finalizeJSONResult(
                    errorCode.getAsJSON(),
                    null,
                    jsonExtraData));

                // create error welcome message
                welcomeMessage = new Message();
                welcomeMessage.setFrom(fromId);
                welcomeMessage.setTo(onBehalfOf);
                welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

                return welcomeMessage;
            }

            if (m_fileHelper != null && (m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.XLSX || m_fileHelper.getDocumentFormat() == com.openexchange.office.tools.DocumentFormat.ODS) && m_calcDocumentHandle == null && operations.has("operations")) {
                final CalcEngineClientFactory calcFactory = m_services.getService(CalcEngineClientFactory.class);
                try {
                    m_calcClient = calcFactory.get();
                    m_calcDocumentHandle = m_calcClient.createDocument();
                    LOG.info("context[file:"+getDocumentFolderAndFileId()+",doc-handle:"+m_calcDocumentHandle+"] calc document created");

                    final int MAX_OPERATIONS = 1000;

                    JSONArray opArray = operations.getJSONArray("operations");
                    m_uiOperations = new JSONObject();
                    if (opArray.length() > MAX_OPERATIONS) {
                        int opCount = opArray.length();
                        int copyPosition = 0;
                        JSONArray uiOperations = new JSONArray();

                        while (copyPosition < opCount) {
                            JSONObject opPart = new JSONObject();
                            JSONArray partArray = new JSONArray();
                            for (int op = 0; op < MAX_OPERATIONS && copyPosition < opCount; ++op) {
                                partArray.put(op, opArray.get(copyPosition));
                                ++copyPosition;
                            }
                            opPart.put("operations", partArray);
                            JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, opPart.toString(false));
                            if (resultObject.has("uiOperations")) {
                                try {
                                    JSONArray newUIOps = resultObject.getJSONArray("uiOperations");
                                    for (int i = 0; i < newUIOps.length(); i++) {
                                        uiOperations.put(newUIOps.get(i));
                                    }
                                } catch (JSONException e) {
                                    LOG.error("RT connection: Error adding ui operations to JSON object", e);
                                }
                            }
                        }
                        try {
                            m_uiOperations.put("operations", uiOperations);
                        } catch (JSONException e) {
                            LOG.error("RT connection: Error adding operations to JSON object", e);
                        }

                    } else {
                        JSONObject resultObject = executeSpreadsheetOperationFailsafe(fromId, operations.toString(false));
                        if (resultObject.has("uiOperations")) {
                            try {
                                m_uiOperations.put("operations", resultObject.getJSONArray("uiOperations"));
                            } catch (JSONException e) {
                                LOG.error("RT connection: Error adding ui operations to JSON object", e);
                            }
                        }
                        LOG.debug(resultObject.toString());
                    }
                    executeSpreadsheetOperationFailsafe(fromId, "{\"operations\":[{\"name\":\"loadFinished\"}]}");
                } catch (NoClassDefFoundError e) {
                    LOG.error("RT connection: Error executing spreadsheet operations, NoClassDefFoundError catched!", e);
                    return impl_createErrorMessage4WelcomeMessage (onBehalfOf, ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR);
                } catch (UnsatisfiedLinkError e) {
                    LOG.error("RT connection: Error executing spreadsheet operations, UnsatisfiedLinkError catched!", e);
                    return impl_createErrorMessage4WelcomeMessage (onBehalfOf, ErrorCode.LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR);
                } catch (Exception e) {
                    LOG.error("RT connection: Error executing spreadsheet operations", e);
                    return impl_createErrorMessage4WelcomeMessage (onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR);
                }
            }

            // Determine the operation state number and append additional operations
            // applied to the document since the last save action.
            synchronized (m_messageChunkList) {
                int documentOSN = -1;
                messageChunkListSize = m_messageChunkList.size();

                JSONObject opsObject = (m_uiOperations != null) ? m_uiOperations : operations;
                try {
                    documentOSN = oxDocument.getUniqueDocumentId();

                    // set sync object with the current document OSN
                    int version = -1;
                    String versionString = m_fileHelper.getVersionFromFile(serverSession);
                    if (null != versionString) {
                        version = Integer.parseInt(versionString);
                    }
                    m_syncStatus.updateSyncInfo(version, documentOSN);
                } catch (OXException e){
                    // not essential for OX Document therefore no error handling needed
                    LOG.error("RT connection: Exception thrown while creating JSON sync info object", e);
                }

                if (m_messageChunkList.size() == 0) {
                    // If we don't have additional operations we have to send
                    // the current operation state number for the document. Therefore
                    // calculate or provide the current operation state number.
                    // Otherwise we don't provide the number from here but let
                    // the additional operations send it.
                    try {
                        int operationCount = opsObject.getJSONArray("operations").length();
                        lastOperation = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                        if (!lastOperation.has("osn")) {

                            if (m_connectionStatus.getOperationStateNumber() > 0) {
                                lastOperation.put("osn", m_connectionStatus.getOperationStateNumber() - 1);
                            } else {
                                // determine initial osn. we prefer the document OSN but if not available
                                // fallback to the operation count
                                int initialOSN = (documentOSN != -1) ? documentOSN : operationCount;
                                lastOperation.put("osn", initialOSN - 1);
                                // Set initial operation state number
                                m_connectionStatus.setOperationStateNumber(initialOSN);
                            }
                            lastOperation.put("opl", 1);
                        }

                    } catch (JSONException e) {
                        LOG.error("RT connection: Error setting operation state number to last operation", e);
                        return impl_createErrorMessage4WelcomeMessage (onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR);
                    }
                } else if (m_messageChunkList.size() > 0) {
                    opsFromDocument = false;
                    // Adding operations applied to the document since the last save action.
                    // These operations will provide the current operation state number, too.
                    ConnectionHelper.appendOperationChunks(operations, m_messageChunkList, false);

                    if (LOG.isDebugEnabled()) {
                        // Additional analyzing code to find root cause of missing operations
                        firstOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                        lastOperation = debugGetLastOperationFromMessageChunkList(m_messageChunkList);
                    }
                }
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Connection.getWelcomeMessage, message chunk list size = " + Integer.toString(messageChunkListSize));
                if (null != firstOperation) {
                    LOG.debug("RT connection: Connection.getWelcomeMessage, first Operation in messsage chunk list: " + operationToString(firstOperation));
                }
                if (null != lastOperation) {
                    if (opsFromDocument) {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in document: " + operationToString(lastOperation));
                    } else {
                        LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in messsage chunk list: " + operationToString(lastOperation));
                    }
                }
            }

            // check m_fileHelper which can be zero if ctor was called with
            // a invalid ID.
            if (null != m_fileHelper) {
                synchronized (m_connectionStatus) {
                    // set writeProtected status
                    m_connectionStatus.setWriteProtected(m_fileHelper.isWriteProtected());
                    // set lock state
                    m_connectionStatus.setLockState(m_fileHelper.lockedUntil() != null);
                    // set locked by user
                    m_connectionStatus.setLockedByUser(m_fileHelper.getLockedByUser());
                }
            }

            // add additional client startup data
            final JSONObject jsonExtraData = new JSONObject();

            // add client id
            try {
                jsonExtraData.put(MessageData.KEY_CLIENT_ID, onBehalfOf.toString());
                jsonExtraData.put(Sync.SYNC_INFO, m_syncStatus.toJSON());
            } catch (JSONException e) {
                LOG.error("RT connection: Error adding client ID to JSON object.", e);
            }

            // putting all operations into an 'actions' object
            final JSONObject actionData = new JSONObject();
            try {
                if (m_uiOperations != null) {
                    actionData.put("actions", (new JSONArray()).put(m_uiOperations));
                } else if (!operations.isEmpty()) {
                    actionData.put("actions", (new JSONArray()).put(operations));
                } else {
                    jsonExtraData.put("hasErrors", true);

                    // update statistics with load error
                    if (null != m_fileHelper) {
                        documentEvent = new DocumentEvent(m_fileHelper.getDocumentType(), ErrorType.LOAD);
                    }
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Error adding operations to JSON", e);
                return impl_createErrorMessage4WelcomeMessage (onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR);
            }

            final MessageData messageData = new MessageData(
                new MessageChunk(actionData, getId()),
                syncGetConnectionStatus(),
                finalizeJSONResult(jsonResult, null, jsonExtraData));

            // set operations chunk of collected JSON objects as payload
            welcomeMessage = new Message();
            welcomeMessage.setFrom(fromId);
            welcomeMessage.setTo(onBehalfOf);
            welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [welcome message] to: " + onBehalfOf.toString());
            }
        } else {
            // update statistics with load error
            if (null != m_fileHelper) {
                documentEvent = new DocumentEvent(m_fileHelper.getDocumentType(), ErrorType.LOAD);
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
            }
        }

        // update statistics: if no (error) documentEvent has been
        // created by now, a new document has been successfully opened
        Statistics.handleDocumentEvent((null != documentEvent) ? documentEvent : getOpenEvent());

        return welcomeMessage;
    }

    /**
     * Provides a message which can be used within the getWelcomeMessage() method to provide
     * error states.
     *
     * @param onBehalfOf
     *  The ID of the client that triggered the welcomeMessage means that just have joined the
     *  Connection. Must be set otherwise the message won't be sent by the Realtime Framework.
     *
     * @param errorCode
     *  The initialized ErrorCode to be sent as the result of the getWelcomeMessage() method.
     *
     * Note : This generates an error message usable as an answer for getWelcomeMessage() only.
     *        If you want to make it more generic you have to make sure the payload
     *        has to be more flexible extended ...
     */
    private Message impl_createErrorMessage4WelcomeMessage (ID onBehalfOf, ErrorCode errorCode)
    {
        JSONObject jsonExtraData = new JSONObject();
        try {
            jsonExtraData.put(MessageData.KEY_CLIENT_ID, onBehalfOf.toString());
        } catch (JSONException jsonEx) {
            LOG.error("RT connection: Error adding client ID to JSON object for error message", jsonEx);
        }
        final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), finalizeJSONResult(
            errorCode.getAsJSON(),
            null,
            jsonExtraData));

        // create error welcome message
        Message welcomeMessage = new Message();
        welcomeMessage.setFrom(getId());
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), "office", "getactions")).build()));

        return welcomeMessage;
    }

    /**
     * Sends a "hangup" message to clients to switch to read-only mode and show an error
     * message.
     *
     * @param fromId
     *  The ID of the client that sends the "hangup" message. Must not be null to ensure
     *  that the message is send.
     * @param statusToSend
     *  The connection status to be sent via the hangup message.
     * @param errorCode
     *  The error code to be sent via the hangup message. This should specify why the hangup
     *  has been sent to the client. Can be null if no error can be specified which triggers
     *  a generic error message on the client.
     * @param toAll
     *  Must be set to send the "hangup" message to all clients sharing the document otherwise
     *  the message is sent to the client with ID fromId.
     * @param logMessage
     *  The message to be added to the logging entry in debug mode only.
     */
    private void impl_sendHangUp(ID fromId, ConnectionStatus statusToSend, ErrorCode errorCode, final boolean toAll, final String logMessage) throws OXException
    {
    	if (fromId == null)
    		throw new RuntimeException ("Without fromId no hang up !");

    	if (statusToSend == null)
    		statusToSend = syncGetConnectionStatus();

        final MessageData messageData = new MessageData(
            null,
            statusToSend,
            (errorCode == null) ? null : finalizeJSONResult(
                errorCode.getAsJSON(),
                null,
                null));

	    final Message returnMessage = new Message();
    	returnMessage.setFrom(fromId);
        returnMessage.addPayload(
        		new PayloadTree(PayloadTreeNode.builder().withPayload(
        				new PayloadElement(messageData, MessageData.class.getName(), "office", "hangup")).build()));

        if (LOG.isDebugEnabled())
            LOG.debug("RT connection: "+ (logMessage == null ? "" : logMessage));

        if (toAll)
        	relayToAll(returnMessage);
        else
        	relayToID(returnMessage, fromId);
    }

    /**
     * Resets the edit client to ensure that the client is not able to make
     * any changes on the document anymore. This method is normally used if
     * a "hangup" message is sent to the specific client because it triggered
     * an invalid state (e.g. sent incorrect operations, failed to sent all
     * operations in time while switching edit rights)
     *
     * @param fromId
     *  The ID of the editor to be reseted. It must be the current editor ID of
     *  this Connection instance otherwise the method won't do anything.
     *
     * @return
     *  TRUE if the editor has been reseted or FALSE if not.
     */
    private boolean impl_resetEditor(ID fromId) {

        if (null == fromId) {
            return false;
        }

        boolean result = false;
        synchronized(m_connectionStatus) {
            String currentEditUser = m_connectionStatus.getCurrentEditingUserId();

            if (null != currentEditUser) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(fromId)) {
                    String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
                    if ((null != wantsEditRightsUser) && (wantsEditRightsUser.length() > 0)) {
                        // Set wantsToEditUser as new editor to speed up switching process in this case
                        m_connectionStatus.setCurrentEditingUser(wantsEditRightsUser.toString(), m_connectionStatus.getWantsEditRightsUserName());
                        // cancel a possible switching timeout timer
                        this.cancelPrepareLosingEditRightsTimer();
                        LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                    } else {
                        // Reset edit user
                        m_connectionStatus.setCurrentEditingUser("", "");
                        LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor.");
                    }
                    result = true;
                }
            }
        }

        return result;
    }

    /**
     * Provides a stanza for the last message sent to the client leaving a document (chat-room).
     * ATTENTION: This method is called by a background thread from the Realtime-Framework.
     * Therefore this method must be implemented thread-safe.
     *
     * @param onBehalfOf
     *  The ID of the client which wants to leave the document.
     *
     * @return
     *  The stanza sent to the client which wants to leave the document. (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#getSignOffMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    @Asynchronous
    public Stanza getSignOffMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message signoffMessage = null;

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: [signoff message] called for document: " + getDocumentFolderAndFileId());
        }

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            int activeClients = 0;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [signoff message] for: " + onBehalfOf.toString());
            }

            synchronized (m_connectionStatus) {
                activeClients = m_connectionStatus.getActiveClients();
            }

            final ServerSession serverSession = getServerSession(null, onBehalfOf);
            final JSONObject jsonResult = flushDocument(serverSession, false, (activeClients == 1));
            this.impl_sendFlushInfoOnError(jsonResult, onBehalfOf, true);

            // send back the JSON result object returned by flushDocument
            signoffMessage = new Message();
            signoffMessage.setFrom(fromId);
            signoffMessage.setTo(onBehalfOf);
            signoffMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, null, null), "json", "office", "closedocument")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [signoff message] to: " + onBehalfOf.toString());
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't send [signoff message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return signoffMessage;
    }

    /**
     * Handles the inactivity notice of the real-time framework for a
     * client that is a member of this connection instance. Currently the
     * handler disables the remote selection of inactive clients.
     *
     * @param stanza
     *  The Stanza containing the inactive client identified by ElementPath
     *  'com.openexchange.realtime.client' and the Duration of inactivity
     *  identified by 'com.openexchange.realtime.client.inactivity'.
     */
    @Override
    public void handleInactivityNotice(Stanza stanza) {
        Optional<ID> inactiveClient = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime", "client"), ID.class);
        Optional<Duration> inactivityDuration = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime.client", "inactivity"), Duration.class);

        if (inactiveClient.isPresent() && inactivityDuration.isPresent()) {
            String id = inactiveClient.get().toString();
            Duration timeOfDuration = inactivityDuration.get();
            boolean enableActiveUser = false;

            LOG.info("RT connection: User=" + inactiveClient.get() + " was inactive for " + inactivityDuration.get());

            if (timeOfDuration.equals(Duration.NONE)) {
                enableActiveUser = true;
            } else if (timeOfDuration.getValueInS() > 0) {
                enableActiveUser = false;
            }

            ConnectionStatus statusToSend = null;
            synchronized(m_connectionStatus) {

                boolean isUserEnabled = m_connectionStatus.isActiveUserEnabled(id);
                if (isUserEnabled != enableActiveUser) {
                    if (enableActiveUser) {
                        m_connectionStatus.enableActiveUser(id);
                    } else {
                        m_connectionStatus.disableActiveUser(id);
                    }
                    LOG.info("RT connection: User=" + id + " enabled=" +String.valueOf(enableActiveUser));
                    statusToSend = (ConnectionStatus)m_connectionStatus.clone();
                }
            }

            if (null != statusToSend) {
                updateClients(new MessageData(null, statusToSend, null), inactiveClient.get());
            }
        }
    }
    /**
     * merge the content of the 'contents' element of a local result to the global result
     * @param globalChanged changed cells information of previous operations
     * @param localChanged changed cells information of current operations
     * @return information of all cells, older information is overwritten by new cells
     */
    JSONArray mergeContents( JSONArray globalChanged, JSONArray localChanged )  {
        
        try {
            Set<Long> localHashes = new HashSet<Long>();

            Iterator<Object> localIt = localChanged.iterator();
            while(localIt.hasNext()) {
                JSONObject cell = (JSONObject)localIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                localHashes.add(new Long(posValue));
            }
            
            Iterator<Object> globalIt = globalChanged.iterator();
            while(globalIt.hasNext()) {
                JSONObject cell = (JSONObject)globalIt.next();
                JSONArray start = cell.getJSONArray("start");
                long posValue = start.getLong(0) + (start.getLong(1) << 16);
                if(!localHashes.contains(new Long(posValue))) {
                    localChanged.put(cell);
                }
            }
        } catch (JSONException e) {
            //should never occur
        }
        return localChanged;
    }

    /**
     * merge the content of the 'changed' element of a local result to the global result
     * @param jsonResult global result object
     * @param localJsonResult local result object
     */
    void mergeChanged( JSONObject jsonResult, JSONObject localJsonResult) {
        try {
            if(localJsonResult.has("changed")) {
                if( !jsonResult.has("changed"))
                    jsonResult.put("changed",  new JSONObject());
                JSONObject globalChanged = jsonResult.getJSONObject("changed");
                if(!globalChanged.has("type") || !globalChanged.getString("type").equals("all")) {
                    JSONObject localChanged = localJsonResult.getJSONObject("changed");
                    globalChanged.put("type",  localChanged.getString("type"));
                    if(!globalChanged.getString("type").equals("all")) {
                        //if still not 'all' is changed merge the cell/column/row ranges
                        if(localChanged.has("sheets"))
                        {
                            if(!globalChanged.has("sheets"))
                                globalChanged.put("sheets", localChanged.getJSONObject("sheets"));
                            else {
                                JSONObject globalSheets = globalChanged.getJSONObject("sheets");
                                JSONObject localSheets = localChanged.getJSONObject("sheets");
                                Set<String> localSheetSet = localSheets.keySet();

                                Iterator<String> localSheetIt = localSheetSet.iterator();
                                while(localSheetIt.hasNext()) {
                                    String localSheetKey = localSheetIt.next();
                                    JSONObject localSheet = localSheets.getJSONObject(localSheetKey);
                                    if(!globalSheets.has(localSheetKey)) {
                                        globalSheets.put(localSheetKey, localSheet);
                                    }
                                    else {
                                        JSONObject globalSheet = globalSheets.getJSONObject(localSheetKey);
                                        //
                                        if(localSheet.has("contents"))
                                        {
                                            if(globalSheet.has("contents")) {
                                                globalSheet.put("contents", mergeContents( globalSheet.getJSONArray("contents"), localSheet.getJSONArray("contents")) );
                                            }
                                        	else
                                        		globalSheet.put("contents", localSheet.getJSONArray("contents"));
                                        }

                                        if(localSheet.has("cells")) {
                                            if(!globalSheet.has("cells"))
                                                globalSheet.put("cells", localSheet.getJSONArray("cells"));
                                            else {
                                                //merge cells "cells:[{start:[1,1],end:[3,3]}, {start:[5,7]}]"
                                                JSONArray globalCells = globalSheet.getJSONArray("cells");
                                                JSONArray localCells = localSheet.getJSONArray("cells");
                                                CellsMerger cellsMerger = new CellsMerger(globalCells);
                                                cellsMerger.add(localCells);
                                                globalSheet.put("cells", cellsMerger.getCells());
                                            }
                                        }
                                        if(localSheet.has("rows")) {
                                            if(!globalSheet.has("rows"))
                                                globalSheet.put("rows", localSheet.getJSONArray("rows"));
                                            else {
                                                //merge rows "rows:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalRows = globalSheet.getJSONArray("rows");
                                                JSONArray localRows = localSheet.getJSONArray("rows");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalRows);
                                                rowColsMerger.add(localRows);
                                                globalSheet.put("rows", rowColsMerger.getRowCols());
                                            }
                                        }
                                        if(localSheet.has("columns")) {
                                            if(!globalSheet.has("columns"))
                                                globalSheet.put("columns", localSheet.getJSONArray("columns"));
                                            else {
                                                //merge columns "columns:[{first:5, last:9}, {first:12}]"
                                                JSONArray globalColumns = globalSheet.getJSONArray("columns");
                                                JSONArray localColumns = localSheet.getJSONArray("columns");
                                                RowColsMerger rowColsMerger = new RowColsMerger(globalColumns);
                                                rowColsMerger.add(localColumns);
                                                globalSheet.put("cols", rowColsMerger.getRowCols());
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            } catch (JSONException e) {
                LOG.debug("Exception in RT connection: mergeChanged" + jsonResult.toString() + " / " + localJsonResult.toString());
        }


    }

    /**
     * Handles the request to apply new actions to the document. The method controls
     * that the sender has the edit rights and it makes some basic operation checking
     * to protected the document consistency.
     *
     * @notice Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "applyactions" }, { namespace: "office", element: "actions", data:
     * "[{operations: [...]}, ...]" }] }
     *
     * @param stanza
     *  A Stanza which contain the operations to be applied to the current document.
     *
     * @return
     *  The message
     *
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();
        final String editor = m_connectionStatus.getCurrentEditingUserId();
        final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
        boolean stanzaFromEditor = (null != editor) && (null != fromId) && (fromId.equals(editorId));
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleApplyActions called from different thread " + String.valueOf(threadId));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: [applyActions] received for document: " + getDocumentFolderAndFileId());
        }

        // Check that we have a ID for the sender stanza and that the sender
        // has the edit rights.
        final Message returnMessage = new Message();
        if ((null != fromId) && isMember(fromId) && stanzaFromEditor) {
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "actions"), true);
            MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
            MessageChunk uiActionChunk = null;
            JSONObject jsonResult = new JSONObject();
            boolean emptyActionChunk = false;

            // increase number of received actions
            m_nActionsReceived++;

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: [applyActions] received, applied from: " + fromId.toString());
            }

            // Server must check the consistency of the operations sent by the current editor client.
            // If check fails the client must receive a 'hangup' message and be excluded from the
            // document group.
            if (!hasValidOperations(actionChunk)) {
                // send hangup message and ignore operations
                impl_sendHangUp(fromId, null, ErrorCode.HANGUP_INVALID_OPERATIONS_SENT_ERROR, HANG_UP_ID, "sent hangup due to inconsistent operations in message chunk.");
                if (impl_resetEditor(fromId)) {
                    // also send a update message with the new/reseted editor to all clients
                    updateClients(new MessageData(null, syncGetConnectionStatus()), fromId);
                }
                return returnMessage;
            }

            boolean sendToAll = false;
            boolean isPaste = false;
            // update spreadsheet document
            if (m_calcClient != null && actionChunk.getOperations().has("actions")) {
                try {
                    JSONObject firstSourceOp = new JSONObject();
                    int sourceOperationLength = 0;
                    JSONArray chunkActions = actionChunk.getOperations().getJSONArray("actions");
                    for( int chunkAction = 0; chunkAction< chunkActions.length(); ++chunkAction) {
                        JSONObject localJsonResult = new JSONObject();
                        JSONObject sheetOperations = chunkActions.getJSONObject(chunkAction);
                        JSONArray operations = sheetOperations.getJSONArray("operations");
                        sourceOperationLength += operations.length();
                        JSONObject firstChunkOperation = operations.getJSONObject(0);
                        //put locale into the first operation of _all_ chunks
                        if(!firstChunkOperation.has("parse"))
                        {
                            String userLanguage = getServerSession(null, fromId).getUser().getPreferredLanguage();
                            firstChunkOperation.put("parse", userLanguage);
                        }

                        if(chunkAction == 0)
                            firstSourceOp = firstChunkOperation;
                        isPaste = operations.length() == 1 && operations.getJSONObject(0).getString("name").equals( "paste" );
                        if(isPaste) {
                            JSONObject pasteOp = operations.getJSONObject(0);
                            CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                            event.sUserSession = pasteOp.getString("UserSession");
                            event.sTargetSelectionJSON = pasteOp.getString("TargetSelection");
                            event.sClipboardHandle = pasteOp.getString("ClipboardHandle");
                            if (ECalcEngineError.E_NONE == m_calcClient.paste(m_calcDocumentHandle, event)) {
                                localJsonResult = new JSONObject(event.sPasteOperationsJSON);
                            }
                        }
                        else
                            localJsonResult = executeSpreadsheetOperationFailsafe(fromId, sheetOperations.toString());
                        if( chunkAction == 0)
                            jsonResult = localJsonResult;
                        else {
                            //add result elements to jsonResult
                            if(localJsonResult.has("undoCount"))
                                jsonResult.put("undoCount", localJsonResult.get("undoCount"));
                            if(localJsonResult.has("redoCount"))
                                jsonResult.put("redoCount", localJsonResult.get("redoCount"));
                            if(localJsonResult.has("changedOperations")) {
                                JSONArray lclChangedOp = localJsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                                JSONArray resultChangedOp = jsonResult.getJSONObject("changedOperations").getJSONArray("operations");
                                for(int op = 0; op < lclChangedOp.length(); ++op)
                                    resultChangedOp.put( lclChangedOp.get(op));
                            }
                            mergeChanged(jsonResult, localJsonResult);
                        }
                    }
                  sendToAll = true;
                  LOG.info(jsonResult.toString());
                      if (jsonResult.has("changedOperations")) {
                          JSONObject actions = new JSONObject();
                          JSONObject changedOperations = jsonResult.getJSONObject("changedOperations");
                          JSONArray changedOpsArray = new JSONArray();
                          JSONArray ops = changedOperations.optJSONArray("operations");

                          emptyActionChunk = (ops == null || ops.length() == 0);
                          changedOpsArray.put(0, changedOperations);
                          actions.put("actions", changedOpsArray);
                          actionChunk = new MessageChunk(actions, actionChunk.getSender());
                          jsonResult.remove("changedOperations");
                      }
                      JSONObject actions = new JSONObject();
                      JSONArray uiActionsArray = new JSONArray();
                      JSONObject uiOperations = new JSONObject();
                      if (jsonResult.has("uiOperations")) {
                          JSONArray uiOperationsArray = jsonResult.getJSONArray("uiOperations");
                          int osnDiff = sourceOperationLength - uiOperationsArray.length();
                          int opState = m_connectionStatus.getOperationStateNumber();
                          for (int op = 0; op < uiOperationsArray.length(); ++op) {
                              uiOperationsArray.getJSONObject(op).put("opl", 1).put("osn", opState + op);
                          }
                          if (osnDiff > 0) {
                              uiOperationsArray.getJSONObject(uiOperationsArray.length() - 1).put("opl", osnDiff + 1);
                          }
                          uiOperations.put("operations", uiOperationsArray);
                      } else {
                          // create a dummy operation to align osl/opn
                          int osn = firstSourceOp.getInt("osn");
                          JSONObject dummyOp = new JSONObject();
                          dummyOp.put("name", "noOp");
                          dummyOp.put("osn", osn);
                          dummyOp.put("opl", sourceOperationLength);
                          JSONArray operationsArray = new JSONArray();
                          operationsArray.put(0, dummyOp);
                          uiOperations.put("operations", operationsArray);
                      }
                      uiActionsArray.put(0, uiOperations);

                      actions.put("actions", uiActionsArray);
                      uiActionChunk = new MessageChunk(actions, fromId);

                } catch (JSONException e) {
                    LOG.debug("Error executing spreadsheet operation", e);
                } catch (Exception e) {
                    LOG.error("RT connection: Error executing spreadsheet operation", e);
                    throw new OXException (e);
                }
            }

            // handling operation state number for operations from clients
            try {
                MessageChunk osnActionChunk = uiActionChunk != null ? uiActionChunk : actionChunk;
                if (osnActionChunk.getOperations().hasAndNotNull("actions")) {
                    JSONArray allActions = osnActionChunk.getOperations().getJSONArray("actions");
                    JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                    if (lastAction.has("operations")) {
                        JSONArray allOps = lastAction.getJSONArray("operations");
                        JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                        // update statistics with incoming operations requests
                        if (null != m_fileHelper) {
                            Statistics.handleDocumentEvent(new OperationsEvent(m_fileHelper.getDocumentType(), OperationsType.INCOMING));
                        }

                        if (lastOp.has("osn") && lastOp.has("opl")) {
                            int osn = -1;
                            synchronized (m_messageChunkList) {
                                osn = lastOp.getInt("osn") + lastOp.getInt("opl");
                                m_connectionStatus.setOperationStateNumber(osn);
                            }
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: handleApplyActions, lastOp received = " + operationToString(lastOp));
                                LOG.debug("RT connection: handleApplyActions, current server osn = " + osn);
                            }
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.debug("RT connection: handleApplyActions: Error setting operation state number to operation", e);
            }

            // update the shared state
            int msgChunkListSize = 0;
            synchronized (m_messageChunkList) {
                if (!emptyActionChunk) {
                    m_messageChunkList.add(actionChunk);
                }
                if (uiActionChunk != null) {
                    ConnectionHelper.appendJSON(m_uiOperations, uiActionChunk.toJSON());
                }
                msgChunkListSize = m_messageChunkList.size();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));
            }

            final MessageData messageData = new MessageData(
                uiActionChunk != null ? uiActionChunk : actionChunk,
                syncGetConnectionStatus(),
                finalizeJSONResult(jsonResult, jsonRequest, null));

            // Send the message to all participants (excluding the one who sent it originally)
            returnMessage.setFrom(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
            }

            if( isPaste )
                relayToAll(returnMessage, stanza);
            else
                relayToAllExceptSender(returnMessage, stanza);

            // update statistics with distributed operations count
            int otherClientCount = 0;

            synchronized (m_connectionStatus) {
                otherClientCount = m_connectionStatus.getActiveClients() - 1;
                m_connectionStatus.setLastModifyingUserId(fromId.toString());
            }

            if ((otherClientCount > 0) && (null != m_fileHelper)) {
                Statistics.handleDocumentEvent(new OperationsEvent(
                    m_fileHelper.getDocumentType(),
                    OperationsType.DISTRIBUTED,
                    otherClientCount));
            }

            // update further clients
            if (sendToAll) {
                // update all Clients with the result, too
                updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), fromId);
                m_nActionsReceived = 0;
            } else if (m_nActionsReceived >= ACTIONS_UNTIL_FORCE_UPDATE) {
                // send update to sender, too
                m_nActionsReceived = 0;

                final Message senderMessage = new Message();
                final MessageData senderMessageData = new MessageData(null, syncGetConnectionStatus(), null);
                senderMessage.setFrom(fromId);
                senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(senderMessageData, MessageData.class.getName(), "office", "update")).build()));
                if (stanza.getTracer() != null) {
                    senderMessage.setTracer(stanza.getTracer());
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [update] to editor for keep alive, applied from: " + fromId.toString());
                }
                relayToID(senderMessage, fromId);
            }

        } else {
            if (null != fromId && !stanzaFromEditor) {
                // The sender has no edit rights, therefore we send a hang up message
                // so the client will be notified and can reload the document.
                impl_sendHangUp(fromId, null, ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR, HANG_UP_ID, "Relaying [hangup] due to missing edit rights, applied from: " + fromId.toString());
            }

            if (LOG.isDebugEnabled()) {
                if (!isMember(fromId)) {
                    LOG.debug("RT connection: Didn't relay [applyActions] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
                }
                if (!stanzaFromEditor) {
                    LOG.debug("RT connection: Didn't relay [applyActions] because the requester doesn't have the edit rights");
                }
            }
        }
        return returnMessage;
    }

    /**
     * Stanzas layout: AcquireEdit { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "acquireedit" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAcquireEdit(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAcquireEdit called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = getServerSession(null, fromId);
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleAcquireEdit error retrieving osn from JSON object. ", e);
            }

            if (LOG.isDebugEnabled()) {
                String editor = syncGetConnectionStatus().getCurrentEditingUserId();
                LOG.debug("RT connection: Handling [acquireEdit], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
                LOG.debug("RT connection: Editor: " + ((editor != null) ? editor : "unknown"));
            }

            acquireEdit(serverSession, fromId, false);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [acquireEdit] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: CanLoseEditRights { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "canloseeditrights" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCanLoseEditRights(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleCanLoseEditRights called from different thread " + String.valueOf(threadId));
        }

        if ((null != fromId) && isMember(fromId)) {
            final Session serverSession = getServerSession(null, fromId);
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleCanLoseEditRights cannot retrieve osn from JSON request. ", e);
            }

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [canLoseEditRights], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
            }

            canLoseEditRights(serverSession, fromId, requesterOSN, false);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Didn't handle [canLoseEditRights] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: AddResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "addresource" }, { namespace: "office", element: "resource", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAddResource(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleAddResource called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "resource"), false);
            final ServerSession serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = addResource(serverSession, jsonRequest);

            // send back the JSON result object returned by addResource
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "resource")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [addResource], called from: " + fromId.toString());
            }

            // send the message to the one who requested to add the resource
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [addResource] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: FlushDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "flushdocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleFlushDocument(Stanza stanza) throws OXException {
        ID fromId = getIDFromStanza(stanza);

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleFlushDocument called from different thread " + String.valueOf(threadId));
        }

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = flushDocument(serverSession, false, false);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [flushDocument], called from: " + fromId.toString());
            }

            // send back the JSON result object returned by flushDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "flushdocument")).build()));
            send(returnMessage);
            // send possible flushDocument() error code to other clients
            impl_sendFlushInfo(jsonResult, fromId, true);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [flushDocument] since the requester was no member");
        }
    }

    /**
     *
     * @param stanza
     * @param isUndo
     * @throws OXException
     */
    private void processUndoRedo(Stanza stanza, boolean isUndo) throws OXException {
        ID fromId = getIDFromStanza(stanza);

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            // final Session serverSession = getServerSession(jsonRequest, fromId);
            Stanza applyStanza = new Message();
            applyStanza.setFrom(fromId);
            JSONObject undoAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject undo = new JSONObject();

            if (LOG.isDebugEnabled()) {
                if (isUndo) {
                    LOG.debug("RT connection: Handling [undo], called from: " + fromId.toString());
                } else {
                    LOG.debug("RT connection: Handling [redo], called from: " + fromId.toString());
                }
            }

            try {
                undo.put("name", isUndo ? "undo" : "redo");
                undo.put("osn", m_connectionStatus.getOperationStateNumber());
                undo.put("opl", 1);
                operations.put(0, undo);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                undoAction.put("actions", actions);
            } catch (JSONException e) {
                LOG.debug("RT connection: processUndoRedo, error creating undoActions JSON object. ", e);
            }
            applyStanza.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(undoAction, jsonRequest, null), "json", "office", "actions")).build()));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path = new ElementPath("office", "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "undo")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "undo" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleUndo(Stanza stanza) throws OXException {
        processUndoRedo(stanza, true);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "redo" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRedo(Stanza stanza) throws OXException {
        processUndoRedo(stanza, false);
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "replace" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleReplaceAll(Stanza stanza) throws OXException {
        final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        try {
            final JSONArray operationsArray = new JSONArray();
            JSONObject operations = new JSONObject();
            try {
                jsonRequest.put("name", "replace");
                LOG.info(jsonRequest.toString());
                operationsArray.put(jsonRequest);
                operations.put("operations", operationsArray);
            } catch (JSONException e) {
                LOG.error("JSONException on handleReplaceAll", e);
            }
            // return operations;
            JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
            jsonResult = checkJSONResult(jsonResult);
            if (jsonResult.has("changedOperations")) {
                JSONObject actions = new JSONObject();
                JSONArray changedOperations = new JSONArray();
                changedOperations.put(0, jsonResult.getJSONObject("changedOperations"));
                actions.put("actions", changedOperations);
                MessageChunk actionChunk = new MessageChunk(actions, fromId);
                jsonResult.remove("changedOperations");

                // update the shared state
                synchronized (m_messageChunkList) {
                    m_messageChunkList.add(actionChunk);
                }
            }
            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [replaceAll] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [replaceAll] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (JSONException je) {
            LOG.error("Exception on handleReplaceAll", je);
        } catch (Exception e) {
            LOG.error("Exception on handleReplaceAll", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "query" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleQuery(Stanza stanza) throws OXException {
        final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        try {
            final JSONArray operationsArray = new JSONArray();
            JSONObject operations = new JSONObject();
            jsonRequest.put("name", "query");
            LOG.debug(jsonRequest.toString());
            operationsArray.put(jsonRequest);
            operations.put("operations", operationsArray);

            // return operations;
            JSONObject jsonResult = executeSpreadsheetOperationFailsafe(fromId, operations.toString());
            jsonResult = checkJSONResult(jsonResult);
            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [query] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [query] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (Exception e) {
        	LOG.error("Exception on handleQuery", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "copy" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopy(Stanza stanza) throws OXException {
        final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
        final ID fromId = stanza.getFrom();
        try {
            JSONObject jsonResult = new JSONObject();
            if (m_calcClient != null) {
                try {
                    CalcEngineClipBoardEvent event = CalcEngineClipBoardEvent.NEW();
                    event.sClipboardHandle = CalcEngineClipBoard.newHandle();
                    event.sUserSession = jsonRequest.getJSONObject("session").getString("resource");
                    event.sSourceSelectionJSON = jsonRequest.getString("selection");
                    if (ECalcEngineError.E_NONE == m_calcClient.copy(m_calcDocumentHandle, event)) {
                        jsonResult.put("clipboardHandle", event.sClipboardHandle);
                    }
                } catch (Exception e) {
                    LOG.error("Error creating/executing copy action", e);
                }
            }

            final Message returnMessage = new Message();
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection sending [query] result to: " + stanza.getFrom().toString());
            }

            // send the message to the one who requested to rename the document
            send(returnMessage);

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection relaying [query] result, called from: " + stanza.getFrom().toString());
            }

            // update all Clients with the result, too
            updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
        } catch (Exception e) {
        	LOG.error("Exception on handleQuery()", e);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "paste" }, { namespace: "office", element: "document", data: "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handlePaste(Stanza stanza) throws OXException {
        ID fromId = getIDFromStanza(stanza);
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            // final Session serverSession = getServerSession(jsonRequest, fromId);
            Stanza applyStanza = new Message();
            final String editor = m_connectionStatus.getCurrentEditingUserId();
            final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
            applyStanza.setFrom(editorId);
            JSONObject pasteAction = new JSONObject();
            JSONArray actions = new JSONArray();
            JSONObject operationObj = new JSONObject();
            JSONArray operations = new JSONArray();
            JSONObject paste = new JSONObject();

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [paste], called from: " + fromId.toString());
            }

            try {
                paste.put("name", "paste");
                paste.put("UserSession", jsonRequest.getJSONObject("session").getString("resource"));
                paste.put("TargetSelection", jsonRequest.getString("selection").toString());
                paste.put("ClipboardHandle", jsonRequest.getString("clipboardHandle").toString());
                paste.put("osn", m_connectionStatus.getOperationStateNumber());
                paste.put("opl", 1);
                operations.put(0, paste);
                operationObj.put("operations", operations);

                actions.put(0, operationObj);
                pasteAction.put("actions", actions);
            } catch (JSONException e) {
                LOG.error("RT connection: handlePaste catches exception while creating JSON actions object. ", e);
            }

            applyStanza.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(pasteAction, jsonRequest, null), "json", "office", "actions")).build()));
            Message applyMessage = handleApplyActions(applyStanza);
            JSONObject jsonResult = new JSONObject();

            ElementPath path = new ElementPath("office", "update");
            for (PayloadTree loadIter : applyMessage.getPayloadTrees(path)) {
                loadIter.getNumberOfNodes();
                Object data = loadIter.getRoot().getData();
                if (data instanceof MessageData) {
                    jsonResult = ((MessageData) data).toJSON();
                }
            }

            // send back the JSON result object returned by handleApplyActions
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));
            send(returnMessage);
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "getdocument" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleGetDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleGetDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final ServerSession serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getDocument(serverSession, jsonRequest);

            // send back the JSON result object returned by getDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Handling [getDocument], sending to: " + fromId.toString());
            }

            // send the message to the one who requested to get the document
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [getDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "gethtmlcode" }, { namespace: "office", element: "document", data: "{... : ...,}" }]
     * }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleGetHtmlCode(Stanza stanza) throws OXException {

        // Performance test code: html code is sent to the server and unchanged returned to the client.

        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleGetHtmlCode called from different thread " + String.valueOf(threadId));
        }

        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final JSONObject jsonResult = new JSONObject();

            try {
                jsonResult.put("htmlcode", jsonRequest.getString("htmlcode"));
            } catch (final JSONException e) {
                LOG.error("RT connection: handleGetHtmlCode catches exception creating JSON result. ", e);
            }

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [getHtmlCode] result to: " + stanza.getFrom().toString());
            }

            // always send the message to the one who requested the html code
            send(returnMessage);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [getHtmlCode] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }

    }

    /**
     * Stanzas layout: RenameDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "renamedocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRenameDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleRenameDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = renameDocument(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [renameDocument] result to: " + stanza.getFrom().toString());
            }

            // always send the message to the one who requested to rename the document
            send(returnMessage);

            // if rename was successful update all clients with the result, too
            try {
                final JSONObject errorCode = jsonResult.getJSONObject(MessageData.KEY_FILE_CHANGERESULT);
                if ((null != errorCode) && (ErrorCode.getErrorCodeFromJSON(errorCode, -1) == 0)) {
                    jsonResult.remove(MessageData.KEY_FILE_CHANGERESULT);

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [renameDocument] result, called from: " + stanza.getFrom().toString());
                    }

                    // update all Clients with the result, too
                    updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
                }
            } catch (final JSONException e) {
                LOG.debug("RT connection: handleRenameDocument catches exception while creating JSON result. ", e);
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [renameDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: CopyDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "copydocument" }, { namespace: "office", element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopyDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleCopyDocument called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was sent as an
        // internal backend call
        if (null != fromId) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "document"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = copyDocument(serverSession, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "document")).build()));

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Sending [handleCopyDocument] result to: " + stanza.getFrom().toString());
            }

            // send the return message to the one who requested to copy document
            send(returnMessage);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [handleCopyDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: UpdateView { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "updateview" }, { namespace: "office", element: "spreadsheet", data: "{... : ...,}"
     * }] }
     *
     * @param stanza
     * @throws OXException
     */
    public Boolean handleUpdateView(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        Message returnMessage = null;
        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleUpdateView called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "spreadsheet"), false);
            final Session serverSession = getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getSpreadsheetViewUpdate(fromId, serverSession, jsonRequest);
            if (jsonResult != null && !jsonResult.has("error")) {
                returnMessage = new Message();
                returnMessage.setFrom(getId());
                returnMessage.setTo(fromId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(finalizeJSONResult(jsonResult, jsonRequest, null), "json", "office", "spreadsheet")).build()));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Sending [updateView] result to: " + stanza.getFrom().toString());
                }

                // send the message to the one who requested to rename the document
                send(returnMessage);

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Relaying [updateView] result, called from: " + stanza.getFrom().toString());
                }

                // update all Clients with the result, too
                updateClients(new MessageData(null, syncGetConnectionStatus(), jsonResult), null);
            }
        }
        return new Boolean(returnMessage != null);
    }

    /**
     * Stanzas layout: UpdateUserData { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "updateuserdata" }, { namespace: "office", element: "state", data: "{... : ...,}"
     * }] }
     *
     * @param stanza
     * @throws OXException
     * @throws JSONException
     */
    public void handleUpdateUserData(Stanza stanza) {
        final ID fromId = stanza.getFrom();

        long threadId = Thread.currentThread().getId();
        if ((threadId != m_threadId) && LOG.isErrorEnabled()) {
            LOG.error("RT connection: handleUpdateUserData called from different thread " + String.valueOf(threadId));
        }

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (null != fromId) {
            LOG.debug("RT connection: [updateUserData] received by " + fromId.toString());
            final JSONObject jsonRequest = getJSONRequest(stanza, new ElementPath("office", "state"), false);

            // update user data
            m_connectionStatus.updateUserData(fromId.toString(), jsonRequest);

            JSONObject jsonResult = m_connectionStatus.toJSON();
            LOG.debug("RT connection: relaying [updateUserData]");
            //updateClients(new MessageData(null, null, jsonResult), null);
            updateClientsExceptSender(new MessageData(null, syncGetConnectionStatus(), jsonResult), fromId);

        } else if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Didn't handle [updateUserData] since the requester was no member");
        }
    }

    /**
     * Checks if a saving operation for this connection is currently in progress.
     *
     * @return TRUE if there is a saving in progress otherwise FALSE.
     */
    public boolean savingDocumentInProgress() {
        return (this.m_saveDocumentInProgress.get() > 0);
    }

    /**
     * Provides the time stamp of the latest save operation.
     *
     * @return The date of the latest save operation.
     */
    public Date getSaveTimeStamp() {
        return this.m_lastSaveTime;
    }

    /**
     * Provides the number of operations not applied to a saved revision of the document.
     *
     * @return The number of pending operations (not saved in the document file yet).
     */
    public long getNumberOfPendingOperations() {
        long num = 0;
        synchronized (m_messageChunkList) {
            num = this.m_messageChunkList.size();
        }
        return num;
    }

    /**
     * Tries to flush a modified document due to a dispose notification
     * from the Realtime framework.
     *
     * @return
     *  TRUE if the saving the document was successful otherwise
     *  FALSE. The method also provides TRUE if there was no need to save
     *  the document as there are no pending operations.
     */
    public boolean saveDocumentOnDispose() {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        boolean modified = false;
        boolean result = false;

        synchronized (m_messageChunkList) {
            modified = !m_messageChunkList.isEmpty();
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("RT connection: Save document on dispose initiated");
        }
        // Try to use the session information from the edit user.
        // If a client leaves the document it will be saved, therefore
        // there must be an editing user if we have operations in the
        // message chunk list.
        if (modified && (null != editingUserId)) {
            ID userId = null;
            try {
                userId = new ID(editingUserId);
            } catch (Exception e) {
                LOG.error("RT connection: Save document on dispose => no user for saving found!", e);
            }

            if (null != userId) {
                try {
                    Session session = userId.toSession();
                    // No need to check result - we cannot reach any client here
                    flushDocument(session, true, true);
                } catch (OXException e) {
                    LOG.error("RT connection: Flushing document on dispose resulted in exception!", e);
                }
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Save document on dispose - no need to flush document");
            }
        }

        return result;
    }

    /**
     * Make a fail safe revision of the current document. This method is called by a timer thread and
     * therefore must be implemented thread-safe.
     *
     * @return
     *  TRUE if the document has been saved successfully, otherwise FALSE.
     */
    public boolean failSafeSaveDocument() {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        boolean result = false;

        if (editingUserId != null) {
            ID userId = new ID(editingUserId);
            Session session = null;

            // retrieve session information from the edit user
            // if there is no editing user we don't need to make a
            // fail safe save
            try {
                session = userId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: failSafeSaveDocument catches exception while retrieving session from user ID. ", e);
            }

            if (null != session) {

                // make sure we are the only method to trigger the saving process
                if (this.m_saveDocumentInProgress.getAndIncrement() > 0) {
                    // there is a saving in progress so just leave
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Fail safe save triggered: saving document in progress => nothing to do");
                    }
                    this.m_saveDocumentInProgress.decrementAndGet();
                    return false;
                }

                final long startTime = System.currentTimeMillis();
                final int[] errorCode = { MessageData.SAVE_ERROR_NONE };

                // close document in every case to write left over operations,
                // if nothing needs to be written, newFileVersion will be the empty string
                try {
                    int osn = -1;
                    int messageChunkListSize = 0;
                    JSONObject outputOperation = null;

                    ArrayList<MessageChunk> currentMessages = null;
                    synchronized (m_messageChunkList) {
                        currentMessages = (ArrayList<MessageChunk>) m_messageChunkList.clone();
                        osn = m_connectionStatus.getOperationStateNumber();
                    }

                    String newFileVersion = null;

                    if (!currentMessages.isEmpty()) {
                        final boolean createVersion = !this.getCreateVersion();
                        final String revisionless = createVersion ? "with new version" : "revisionless";

                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Fail safe save triggered: Trying to save document " + revisionless);
                        }

                        if (LOG.isDebugEnabled()) {
                            // Additional analyzing code to find root cause of missing operations
                            LOG.debug("RT connection: Fail safe save (before save): Current message chunk list size = " + currentMessages.size());
                            outputOperation = debugGetLastOperationFromMessageChunkList(currentMessages);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Fail safe save (before save): Last Operation in messsage chunk list = " + operationToString(outputOperation));
                                outputOperation = null;
                            }
                        }

                        final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);

                        oxDocument.setUniqueDocumentId(osn);
                        newFileVersion = oxDocument.save(
                            session,
                            getExporter(),
                            m_resourceManager,
                            currentMessages,
                            errorCode,
                            !createVersion,
                            false);

                        // update Statistics with saved document count (either OPS100 or 15MINS type)
                        if (null != m_fileHelper) {
                            Statistics.handleDocumentEvent(new SaveEvent(
                                m_fileHelper.getDocumentType(),
                                (!createVersion) ? SaveType.OPS_100 : SaveType.OPS_15MINS));
                        }

                        // clean messageChunkList in case the messageChunkList has been flushed
                        if ((null != newFileVersion) && (newFileVersion.length() > 0)) {

                            if (createVersion) {
                                // after saving a new version we save revisionless
                                setCreateVersion(false);
                            }

                            int currSize = 0;
                            synchronized (m_messageChunkList) {
                                currSize = m_messageChunkList.size();

                                // Remove all operations which have been saved recently
                                // leaving just the new ones added during the save process.
                                int index = currentMessages.size() - 1;

                                if (index == m_messageChunkList.size() - 1) {
                                    m_messageChunkList.clear(); // optimization
                                } else if ((index > 0) && ((index + 1) < m_messageChunkList.size())) {
                                    // New operation list must only hold the new operations
                                    // therefore we need to remove the old ones.
                                    ArrayList<MessageChunk> copy = (ArrayList<MessageChunk>) m_messageChunkList.clone();
                                    List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                    m_messageChunkList.clear();
                                    m_messageChunkList.addAll(newPart);
                                }
                                if (LOG.isDebugEnabled()) {
                                    messageChunkListSize = m_messageChunkList.size();
                                    outputOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                                }
                            }
                            m_lastSaveTime = new Date();
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: Fail safe save (after save): Before adjustment, message chunk list size = " + currSize);
                                LOG.debug("RT connection: Fail safe save (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                                if (null != outputOperation) {
                                    LOG.debug("RT connection: Fail safe save (after save): First operation in messsage chunk list = " + operationToString(outputOperation));
                                }
                            }
                        } else {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("RT connection: Fail safe save, documentsave reports no new file version => Possible loss of document data!");
                            }
                        }
                    }

                    if (LOG.isWarnEnabled()) {
                        if ((MessageData.SAVE_ERROR_NOBACKUP == errorCode[0])) {
                            LOG.warn("RT connection: Fail safe save: Document operations failed completely!");
                        } else if ((MessageData.SAVE_ERROR_OPERATIONS == errorCode[0])) {
                            LOG.warn("RT connection: Fail safe save: Save to native document format failed => OX document replacement archive has been written!");
                        }
                    }

                    if (LOG.isDebugEnabled()) {
                        if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                            LOG.debug("RT connection: Fail safe  save: Flushing operations to native document succeeded");
                        } else {
                            LOG.debug("RT connection: Fail safe save: No need to save document");
                        }
                    }

                    result = true;
                } catch (Exception e) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Exception detected while doing a fail safe save", e);
                    }
                } finally {
                    m_saveDocumentInProgress.decrementAndGet();
                    long saveDocumentTime = (System.currentTimeMillis() - startTime);
                    if (LOG.isInfoEnabled()) {
                        LOG.info("RT connection: Fail safe save, TIME to flushDocument: " + saveDocumentTime + "ms");
                    }

                    // in case of an error we send a flush info error to all
                    // clients
                    if (MessageData.SAVE_ERROR_NONE != errorCode[0]) {
                        this.impl_sendFlushInfoOnError(errorCode[0], userId, false);
                    }
                }
            }
        }

        if (result) {
            // Trigger update message so clients can update files view
            m_connectionStatus.setFailSafeSaveDone(true);
            updateClients(new MessageData(null, syncGetConnectionStatus()), null);
            m_connectionStatus.setFailSafeSaveDone(false);
        }

        return result;
    }

    /**
     * Sends a update message to all clients of the document except
     * the sender. This method can be used if a update should only
     * be sent to all other clients, because the sender already have
     * all information available.
     *
     * @param messageData
     *  The message data to be sent to all clients.
     * @param senderId
     *  The client who sent a request which resulted in this update
     *  message. In the normal case this is the editor client. This
     *  client won't be part of the clients that will receive the
     *  update message.
     */
    private void updateClientsExceptSender(MessageData messageData, ID senderId) {
        impl_updateClients(messageData, senderId, true);
    }

    /**
     * Sends a update message to all clients of the document.
     *
     * @param messageData
     *  The message data to be sent to all clients.
     * @param fromId
     *  The client who sent a request which resulted in this update message.
     *  In the normal case this is the editor client.
     */
    private void updateClients(MessageData messageData, ID fromId) {
        impl_updateClients(messageData, fromId, false);
    }

    /**
     * Internal method to sends a update message to a set of clients of
     * the document.
     *
     * @param messageData
     *  The message data to be sent to all clients.
     * @param fromId
     *  The client who sent a request which resulted in this update message.
     *  In the normal case this is the editor client.
     * @param exceptSender
     *  If set to TRUE the sender/fromId won't be part of the clients
     *  that will received the update message.
     */
    private void impl_updateClients(MessageData messageData, ID fromId, boolean exceptSender) {
        final Message updateMessage = new Message();
        final ID senderId = (null != fromId) ? fromId : getId();

        if (null != senderId) {
            updateMessage.setFrom(senderId);
            updateMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "update")).build()));

            if (LOG.isDebugEnabled()) {
                String updateClients = exceptSender ? "updateClientsExceptSender" : "updateClients";
                LOG.debug("RT connection: Handling [" + updateClients + "], originated by: " + senderId.toString());
            }

            try {
                if (exceptSender) {
                    relayToAllExceptSender(updateMessage);
                } else {
                    relayToAll(updateMessage);
                }
            } catch (OXException e) {
                if (exceptSender) {
                    LOG.error("RT connection: [updateClients] catches exception for relayToAllExceptSender.", e);
                } else {
                    LOG.error("RT connection: [updateClients] catches exception for relayToAll.", e);
                }
            }
        }
    }

    /**
     * Checks and sends a flushErrorInfo message to all clients if the result
     * JSON object contains an error, which describes problems on flushing
     * a document to OX Drive.
     *
     * @param result
     *  The JSONObject directly returned by the flushDocument() call.
     *
     * @param fromId
     *  The ID of the client that triggered the flushDocument. Must not
     *  be null, otherwise the flushErrorInfo message cannot be sent.
     *
     * @param exceptSender
     *  Specifies if the client with fromId should get the flushErrorInfo
     *  message or not.
     */
    private void impl_sendFlushInfoOnError(JSONObject result, ID fromId, boolean exceptSender) {
        final ID senderId = (null != fromId) ? fromId : getId();

        if ((null != result) && (null != senderId) && (result.optInt(MessageData.KEY_ERROR_CODE) != 0)) {
            impl_sendFlushInfo(result, senderId, exceptSender);
        }
    }

    /**
     * Checks and sends a flushErrorInfo message to all clients if the result
     * JSON object contains an error, which describes problems on flushing
     * a document to OX Drive.
     *
     * @param errorCode
     *  The error code of the flushDocument operation, which could be zero or
     *  non-zero (that is the error code).
     *
     * @param fromId
     *  The ID of the client that triggered the flushDocument. Must not
     *  be null, otherwise the flushErrorInfo message cannot be sent.
     *
     * @param exceptSender
     *  Specifies if the client with fromId should get the flushErrorInfo
     *  message or not.
     */
    private void impl_sendFlushInfoOnError(int errorCode, ID fromId, boolean exceptSender) {
        final ID senderId = (null != fromId) ? fromId : getId();

        if ((null != senderId) && (errorCode != 0)) {
            JSONObject result = new JSONObject();
            try {
                result.put(MessageData.KEY_ERROR_CODE, errorCode);
                impl_sendFlushInfo(result, senderId, exceptSender);
            } catch (JSONException e) {
                LOG.error("Exception while setting up result for flushErrorInfo notification", e);
            }
        }
    }

    /**
     * Sends a flushErrorInfo message to all clients using the provided JSON
     * object that must contain the error code.
     *
     * @param result
     *  The JSONObject directly returned by the flushDocument() call containing
     *  the error code.
     *
     * @param fromId
     *  The ID of the client that triggered the flushDocument. Must not
     *  be null, otherwise the flushErrorInfo message cannot be sent.
     *
     * @param exceptSender
     *  Specifies if the client with fromId should get the flushErrorInfo
     *  message or not.
     */
    private void impl_sendFlushInfo(JSONObject result, ID senderId, boolean exceptSender) {
        try {
            final MessageData messageData = new MessageData(null, null, result);
            final Message flushErrorInfoMessage = new Message();

            flushErrorInfoMessage.setFrom(senderId);
            flushErrorInfoMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), "office", "flusherrorinfo")).build()));

            if (exceptSender) {
                relayToAllExceptSender(flushErrorInfoMessage);
            } else {
                relayToAll(flushErrorInfoMessage);
            }
        } catch (OXException e) {
            LOG.error("RT connection: [flushErrorInfo] catches exception for relayToAll.", e);
        }
    }

    /**
     * Implementation method to process a "acquireedit" request from a client. Sending the
     * required "preparelosingeditrights" message to the current editor and setup a timeout
     * thread to handle non-responsive clients.
     *
     * @param session
     *  The session of the user who wants to receive edit rights.
     * @param fromId
     *  The ID of the user who wants to receive edit rights.
     * @param silent
     *  If set to TRUE the method won't send any update nor preparelosingeditrights
     *  notification. Used by methods which already know that switching edit rights can be
     *  done immediately.
     */
    private void acquireEdit(Session session, ID fromId, boolean silent) throws OXException {
        Session curSession = session;
        String displayUserName = "";

        if ((null == curSession) && (null != fromId)) {
            try {
                curSession = fromId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: acquireEdit catches exception while retrieving session from sender ID. ", e);
            }
        }

        if ((null != curSession) && (curSession instanceof ServerSession)) {
            final User curUser = ((ServerSession) curSession).getUser();

            if (null != curUser) {
                displayUserName = curUser.getDisplayName();
            }
        }

        if (null != fromId) {

            String wantsEditRightsUserId = "";
            String hasEditRightsUserId = "";

            synchronized (m_connectionStatus) {
                wantsEditRightsUserId = m_connectionStatus.getWantsEditRightsUserId();
                hasEditRightsUserId = m_connectionStatus.getCurrentEditingUserId();
            }

            // Check if user tries to acquire edit rights although she has it already.
            // Can be related to a slow connection and the user tries again and again
            // to acquire the rights.
            if ((null != hasEditRightsUserId) && (hasEditRightsUserId.length() > 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                if (fromId.equals(currentEditorId)) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Ignore [acquireedit] from editor");
                    }
                    return;
                }
            }

            if ((null == hasEditRightsUserId) || (hasEditRightsUserId.length() == 0)) {
                String wantsToEdit = "";
                ConnectionStatus connectionStatus = null;

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: acquireedit no current editor found, editor now: " + fromId.toString() + ", wants to edit:" + wantsEditRightsUserId);
                }

                synchronized (m_connectionStatus) {
                    // first acquire edit rights on a non-acquired document can be directly approved
                    m_connectionStatus.setCurrentEditingUser(fromId.toString(), displayUserName);
                    if (wantsEditRightsUserId.equals(fromId.toString())) {
                        // Special case: If user1 acquires edit rights and the editor leaves and afterwards
                        // the user1 again acquires edit rights, user1 will be editor and also the user who
                        // wants to get these rights. So we have to reset wantsToEditRightsUser!
                        m_connectionStatus.setWantsEditRightsUser("", "");
                    }
                    wantsToEdit = m_connectionStatus.getWantsEditRightsUserId();
                    connectionStatus = (ConnectionStatus) m_connectionStatus.clone();
                }
                if (!silent) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [updateClients] to all clients, editor:" + fromId.toString() + ", wants to edit:" + wantsToEdit);
                    }
                    updateClients(new MessageData(null, connectionStatus), null);
                }
            } else if ((null == wantsEditRightsUserId) || (wantsEditRightsUserId.length() == 0)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                Message returnMessageLose = null;
                Message returnMessageGet = null;
                MessageData messageData = null;

                synchronized (m_connectionStatus) {
                    // Start "preparelosingeditrights" timeout timer to prevent that we have a document
                    // where the edit rights are pending forever (e.g. the current editor loses connection
                    // while we are switching the edit rights - waiting for canLoseEditRights)
                    cancelPrepareLosingEditRightsTimer();
                    startPrepareLosingEditRightsTimer(session, currentEditorId);

                    // store user who wants to receive the edit rights
                    m_connectionStatus.setWantsEditRightsUser(fromId.toString(), displayUserName);

                    if (!silent) {
                        // Send the message to the client who will lose the edit rights
                        returnMessageLose = new Message();
                        returnMessageGet = new Message();
                        messageData = new MessageData(null, (ConnectionStatus) m_connectionStatus.clone(), null);
                    }
                }

                if (!silent) {
                    if (null == returnMessageGet) {
                        returnMessageGet = new Message();
                    }

                    // send info notification to the client which will receive the edit rights
                    returnMessageGet.setFrom(fromId);
                    returnMessageGet.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "acceptedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [acceptedacquireeditrights] to client requested edit rights, applied from: " + fromId.toString());
                    }

                    relayToID(returnMessageGet, fromId);

                    if (null == returnMessageLose) {
                        returnMessageLose = new Message();
                    }

                    // send message to the client which will lose the edit rights for preparation
                    returnMessageLose.setFrom(currentEditorId);
                    returnMessageLose.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "preparelosingeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [preparelosingeditrights] to client with edit rights, to: " + currentEditorId.toString());
                    }

                    relayToID(returnMessageLose, currentEditorId);
                }
            } else {
                ID wantsEditorRightsId = new ID(wantsEditRightsUserId);

                if (!fromId.equals(wantsEditorRightsId) && !silent) {
                    // Someone else already wants to acquire the edit rights. Therefore we
                    // have to decline this request until the edit rights changed.

                    final Message returnMessage = new Message();
                    final MessageData messageData = new MessageData(null, syncGetConnectionStatus(), null);

                    returnMessage.setFrom(fromId);
                    returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), "office", "declinedacquireeditrights")).build()));

                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Relaying [declinedacquireeditrights] to client requested edit rights while there is a switch in progress, applied from: " + fromId.toString());
                        LOG.debug("RT connection: Relaying [declinedacquireeditrights], wantsEditorRights=" + wantsEditRightsUserId + ", editor=" + hasEditRightsUserId);
                    }

                    relayToID(returnMessage, fromId);
                }
            }
        }
    }

    /**
     * Implementation method to handle the response from the current editor client to give up the edit rights.
     * This method is called by the "preparelosingeditrights" timer thread and the RT core. Therefore it must
     * be thread-safe implemented.
     *
     * @param session
     *  The session of the user who was notified via "preparelosingeditrights"
     * @param fromId
     *  The id who was notified via "preparelosingeditrights".
     * @param clientOSN
     *  The osn of the client responding to the "preparelosingeditrights" notification.
     * @param timeout
     *  Specifies that the method is called from the timeout callback.
     */
    protected void canLoseEditRights(@SuppressWarnings("unused") Session session, ID fromId, int clientOSN, boolean timeout) {

        ConnectionStatus statusToSend = null;
        int serverOSN = clientOSN;
        ID currentEditorId = null;

        synchronized (m_connectionStatus) {

            currentEditorId = new ID(this.m_connectionStatus.getCurrentEditingUserId());
            if (timeout || currentEditorId.equals(fromId)) {
                String newEditorId = m_connectionStatus.getWantsEditRightsUserId();

                this.cancelPrepareLosingEditRightsTimer();

                // Current editor completed to send her latest changes. It's now safe
                // to switch the edit rights.
                m_connectionStatus.setCurrentEditingUser(newEditorId, m_connectionStatus.getWantsEditRightsUserName());
                // Remove the requester so others can acquire the edit rights
                m_connectionStatus.setWantsEditRightsUser("", "");
                // Create copy of the connection status we can use updateClients
                // outside our synchronized and have a consistent state
                if (timeout) {
                    m_connectionStatus.disableActiveUser(fromId.toString());
                }
                statusToSend = (ConnectionStatus) m_connectionStatus.clone();
                serverOSN = m_connectionStatus.getOperationStateNumber();

                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Switching edit rights to client: " + newEditorId);
                }
            }
        }

        if (statusToSend != null) {
            updateClients(new MessageData(null, statusToSend), null);
        }

        // Check the current server osn and determine if the client losing
        // edit rights is out of sync. Do this only if we don't have a timeout
        // scenario.
        // Additional information:
        // Checking both client and server OSN with a less rigid
        // scheme, as it's possible that a client is behind the last edit client
        // and this should not block transferring edit rights. As the "real",
        // UI related client, edit rights are processed synchronized with the
        // operations this won't result in bad scenarios. Currently (7.4.2) this
        // can result in strange behaviour where the client is switching fast
        // between edit rights bubble messages (if the client catched up with
        // operation processing).
        // This could be fixed using snooping the "message queue" on the client side
        // and prevent to switch between "old" update states but just use the latest
        // one from the queue.
        // Bug fix: 30656 use less rigid check for "canloseeditrights" and client OSN
        if (!timeout && (clientOSN > serverOSN) && (null != currentEditorId)) {
            try {
                impl_sendHangUp(fromId, statusToSend, ErrorCode.HANGUP_INVALID_OSN_DETECTED_ERROR, HANG_UP_ID, "Relaying [hangup] to client that has recently lost edit rights due to inconsitent osn, called from: " + fromId.toString() + " with client-osn: " + String.valueOf(clientOSN) + " server-osn: " + String.valueOf(serverOSN));
            } catch (OXException e) {
                LOG.error("RT connection: Exception while sending [hangup] to editor client due to inconsistent OSN", e);
            }
        }
    }

    /**
     * Adds a resource to the distributed managed file resource, so it can be
     * used by the client via a media url.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The addResource request which includes the resourceid
     * @return A json object containing the result of the addResource request.
     */
    private JSONObject addResource(Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String managedResourceId = jsonRequest.optString("resourceid", "");

            // we got a file id to create a distributed managed file resource from
            if (managedResourceId.length() > 0) {
                final Resource managedResource = m_resourceManager.createManagedResource(managedResourceId);

                if (null != managedResource) {
                    m_resourceManager.addResource(managedResource);
                    LOG.debug("RT connection: addResource added resource " + managedResourceId + "to resourceManager.");

                    try {
                        jsonResult.put("resourceid", managedResource.getManagedId());
                    } catch (JSONException e) {
                        LOG.error("RT connection: addResource - could not retrieve managed resource id");
                    }
                } else {
                    LOG.error("RT connection: addResource - managed resource could not be retrieved from id: " + managedResourceId);
                }
            } else {
                // try to create the resource from the given data
                final String data64 = jsonRequest.optString("dataurl", "");
                byte[] data = null;

                if (null != data64) {
                    final String searchPattern = "base64,";
                    int pos = data64.indexOf(searchPattern);

                    if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                        data = Base64.decode(data64.substring(pos));
                    }
                }

                if ((data != null) && (data.length > 0)) {
                    final String fileName = jsonRequest.optString("filename", "");
                    final String fileExtension = FileHelper.getExtension(fileName);

                    String newExtension = jsonRequest.optString("add_ext", "");

                    // extension to add to resource fileName
                    if (newExtension.length() < 1) {
                        newExtension = fileExtension;
                    }

                    // crc32 of resource data
                    int crc32 = jsonRequest.optInt("add_crc32", 0);

                    if (0 == crc32) {
                        crc32 = Resource.getCRC32(data);
                    }

                    // resource id
                    final long uid = m_resourceManager.addResource(crc32, data);
                    LOG.debug("RT connection: addResource added resource " + Long.toString(uid) + "to resourceManager.");

                    try {
                        String resourceName = null;

                        switch (m_fileHelper.getDocumentFormat()) {
                            case DOCX: {
                                resourceName = "word/media/uid" + Long.toHexString(uid) + "." + newExtension;
                                break;
                            }
                            case PPTX: {
                                resourceName = "ppt/media/uid" + Long.toHexString(uid) + "." + newExtension;
                                break;
                            }
                            case XLSX: {
                                resourceName = "xl/media/uid" + Long.toHexString(uid) + "." + newExtension;
                                break;
                            }
                            case ODT: {
                                resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                                break;
                            }
                            case ODS: {
                                resourceName = "Pictures/uid" + Long.toHexString(uid) + "." + newExtension;
                                break;
                            }

                            case NONE:
                            case ODG:
                            case ODP:
                            default: {
                                break;
                            }
                        }

                        if (null != resourceName) {
                            jsonResult.put("added_filename", resourceName);
                        }
                    } catch (Exception e) {
                        LOG.error("RT connection: addResource catches exception adding filename to JSON result. ", e);
                    }
                }
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * Provides the stream and mimetype of a requested document.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The request containing the folder/file id and the
     *                    target format.
     * @return A json object containing the stream and mimetype.
     */
    private JSONObject getDocument(ServerSession session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String documentFormat = jsonRequest.optString("documentformat", "native"); // ["native", "pdf"]
            final String returnType = jsonRequest.optString("returntype", "dataurl"); // ["dataurl", "resourceid"]
            final String source = jsonRequest.optString("source", "file"); //
            final String folder = jsonRequest.optString("folder");
            final String id = jsonRequest.optString("id");
            final String module = jsonRequest.optString("module");
            final String attachment = jsonRequest.optString("attachment");
            final int[] errorCode = { MessageData.SAVE_ERROR_NONE };
            ArrayList<MessageChunk> currentMessages = null;
            MailServletInterface mailInterface = null;
            InputStream documentStm = null;
            InputStream pdfDocumentStm = null;

            synchronized (m_messageChunkList) {
                currentMessages = (ArrayList<MessageChunk>) m_messageChunkList.clone();
            }

            try {
                if (source.equals("mail")) {
                    try {
                        mailInterface = MailServletInterface.getInstance(session);
                        MailPart part = mailInterface.getMessageAttachment(folder, id, attachment, false);

                        if (part != null) {
                            documentStm = part.getInputStream();
                        }
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from mail part. ", e);
                    }
                } else if (source.equals("task")) {
                    try {
                        documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folder),
                            Integer.parseInt(attachment),
                            Integer.parseInt(module),
                            Integer.parseInt(id),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from task attachment. ", e);
                    }
                } else {
                    documentStm = (new OXDocument(session, m_fileHelper, m_resourceManager)).getResolvedDocumentStream(
                        session,
                        getExporter(),
                        m_resourceManager,
                        currentMessages,
                        null,
                        errorCode,
                        false,
                        false);
                }

                // convert document stream content to PDF, if requested
                if (null != documentStm) {
                    String fileName = jsonRequest.optString("filename", "document");
                    String mimeType = jsonRequest.optString("mimetype", "");

                    if (documentFormat.equals("pdf") && (MessageData.SAVE_ERROR_NONE == errorCode[0])) {
                        final IManager dcManager = m_services.getService(IManager.class);

                        // convert content of document stream to PDF
                        if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                            final HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                            final HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);

                            jobProperties.put(Properties.PROP_INPUT_STREAM, documentStm);
                            jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);

                			try {
                    			final CapabilityService capService = m_services.getService(CapabilityService.class);
                				final CapabilitySet capSet = (null != capService) ? capService.getCapabilities(session) : null;

                				// enable sharepoint conversion, depending on the user capability set
                				if ((null != capSet) && capSet.contains("sharepointconversion")) {
                                    jobProperties.put(Properties.PROP_FEATURES_ID, Integer.valueOf(Feature.SHAREPOINT_PDFCONVERSION.id()));
                				}
                			} catch (OXException e ) {
                				if (LOG.isErrorEnabled()) {
                					LOG.error("RT connection: getDocument catches exception trying to retrieve 'sharepointconversion' capability.", e);
                				}
                			}

                            pdfDocumentStm = dcManager.convert("pdf", jobProperties, resultProperties);
                            IOUtils.closeQuietly(documentStm);
                            documentStm = pdfDocumentStm;
                            pdfDocumentStm = null;

                            fileName = fileName + ".pdf";
                            mimeType = "application/pdf";
                        }
                    }

                    if (mimeType.length() < 1) {
                        mimeType = FileHelper.getMimeType(fileName);
                    }

                    // set resulting data at JSONObject; document
                    // stream might be null after a conversion error
                    if (null != documentStm) {
                        // add resourceid or dataurl to the result object, depending on the value of returnType
                        ConnectionHelper.appendData(m_resourceManager, jsonResult, documentStm, returnType, mimeType);

                        // add filename and mimetype properties to the valid result object
                        if (!jsonResult.isEmpty()) {
                            try {
                                jsonResult.put("resourcename", fileName);
                                jsonResult.put("mimetype", mimeType);
                            } catch (JSONException e) {
                                LOG.error("RT connection: getDocument catches exception setting resource name/mimetype to JSON result. ", e);
                            }
                        }

                        IOUtils.closeQuietly(documentStm);
                    }
                }
            } finally {
                IOUtils.closeQuietly(mailInterface);
                IOUtils.closeQuietly(documentStm);
                IOUtils.closeQuietly(pdfDocumentStm);
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * Renames the name of the document. This doesn't influence the folder&file id.
     *
     * @param session The session of the user who wants to rename a document.
     * @param jsonRequest The request containing the folder/file id and the new name.
     *
     * @return A json object containing the result of the rename request.
     */
    private JSONObject renameDocument(Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();
        String requestedNewFileName = "";

        try {
            String newFileName = null;
            final String[] newFileVersion = { "" };
            final ErrorCode errorCode[] = { ErrorCode.NO_ERROR };

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper) && jsonRequest.has(MessageData.KEY_FILE_NAME)) {
                requestedNewFileName = jsonRequest.getString(MessageData.KEY_FILE_NAME);
                newFileName = m_fileHelper.renameDocument(session, requestedNewFileName, newFileVersion, errorCode);
            }

            if (null != newFileName) {
                jsonResult.put(MessageData.KEY_FILE_NAME, newFileName);
                jsonResult.put(MessageData.KEY_FILE_VERSION, newFileVersion[0]);
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, errorCode[0].getAsJSON());
            } else {
                // In case of an error set at least the change result entry
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, errorCode[0].getAsJSON());
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: renameDocument - failed to rename document to " + requestedNewFileName, e);

            // Fix for bug 29261: Don't return an empty jsonResult which results in a response
            // where the property "hasErrors" is set to "true". This forces the client to immediately
            // show an error bubble and stop accepting server messages.
            try {
                // provide change result to the client
                jsonResult.put(MessageData.KEY_FILE_CHANGERESULT, ErrorCode.RENAMEDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: renameDocument catches exception trying to set rename result to JSON object. ", je);
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * Saves the current document including the latest changes in the message
     * chunk list to a specified folder. The current state of the connection
     * including the message chunk list, editor etc. are not changed.
     *
     * @param session
     *  The session of the user that requested to copy the current document to
     *  a sepcified folder.
     *
     * @param jsonRequest
     *  The request containing the folder/file id, the target folder id, the
     *  new file name. The method uses the folder of the document if the target
     *  folder id is missing, if this is not possible, the user folder is used.
     *  If the new file name is missing the original file name is used.
     *
     * @return
     *   A json object containing the result of the copy document request.
     */
    private JSONObject copyDocument(final Session session, JSONObject jsonRequest) {
        JSONObject jsonResult = new JSONObject();
        String targetFolder = "";
        boolean asTemplate = false;

        try {
            String newFileName = null;

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper)) {
                boolean bCreateDocumentCopy = true;
                targetFolder = jsonRequest.optString(MessageData.KEY_TARGET_FOLDER);
                newFileName = jsonRequest.optString(MessageData.KEY_FILE_NAME);
                asTemplate = jsonRequest.optBoolean(MessageData.KEY_AS_TEMPLATE);

                if ((null == targetFolder) || (targetFolder.length() == 0)) {
                    // There is no target folder
                    boolean bCanUseFolder = false;
                    if (asTemplate) {
                        // templates are always created in the user folder
                    } else {
                        // First we try to use the source folder and check the
                        // permissions. If we cannot write, we take the users
                        // default folder as fallback.
                        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;

                        targetFolder = m_fileHelper.getFolderId();
                        int[] permissions = FileHelper.getFolderPermissions(fileAccess, targetFolder);
                        if (null != permissions) {
                            // Check folder permissions that we can create a copy
                            // of the document file and write to it.
                            bCanUseFolder = (permissions[FileHelper.FOLDER_PERMISSIONS_FOLDER] >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER) &
                                            (permissions[FileHelper.FOLDER_PERMISSIONS_WRITE] > FileStoragePermission.NO_PERMISSIONS);
                        }
                    }

                    // Fallback use the user folder to create the document
                    // For template this is the default case!
                    if (!bCanUseFolder) {
                        targetFolder = FileHelper.getUserFolderId(session, m_services);
                        if (null == targetFolder) {
                            jsonResult.put(MessageData.KEY_DOC_COPYRESULT, ErrorCode.COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR.getAsJSON());
                            bCreateDocumentCopy = false;
                        }
                    }
                }

                if ((null == newFileName) || (newFileName.length() == 0)) {
                    // We don't have a new file name for the copy of the
                    // document. Therefore we retrieve the original name.
                    try {
                        newFileName = m_fileHelper.getFilenameFromFile(session);
                    } catch (OXException e) {
                        newFileName = null;
                    }
                    if (null == newFileName) {
                        jsonResult.put(MessageData.KEY_DOC_COPYRESULT, ErrorCode.COPYDOCUMENT_FILENAME_UNKNOWN_ERROR.getAsJSON());
                        bCreateDocumentCopy = false;
                    }
                }

                if (bCreateDocumentCopy) {
                    ErrorCode errorCode = impl_copyDocument(session, targetFolder, newFileName, asTemplate, jsonResult);
                    jsonResult.put(MessageData.KEY_DOC_COPYRESULT, errorCode.getAsJSON());
                }
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: copyDocument - failed to copy document.", e);

            try {
                // provide change result to the client
                jsonResult.put(MessageData.KEY_DOC_COPYRESULT, ErrorCode.COPYDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: copyDocument catches exception trying to set result to JSON object. ", je);
            }
        }

        return checkJSONResult(jsonResult);
    }

    /**
     * Creates a copy of the current document including changes within the
     * message chunk list.
     *
     * @param session
     *  The session of the user that requested to create a copy of the current
     *  document.
     *
     * @param targetFolder
     *  The target folder where the document should be stored to. Must not be
     *  null or an empty string.
     *
     * @param fileName
     *  The file name of the new document without extension. Must not be null
     *  or an empty string.
     *
     * @param asTemplate
     *  Specifies if the new document should be a template.
     *
     * @param jsonResult
     *  Contains the json part of the copy document method. If the result
     *  of the method is no error, these values can be used to retrieve
     *  the new file id, folder id, version, mime type and file name.
     *
     * @return
     *  The error code of the request. Must be ErrorCode.NO_ERROR if no error
     *  occurred, otherwise it describes the problem that prevented to store
     *  the document in the target folder.
     */
    private ErrorCode impl_copyDocument(final Session session, String targetFolder, String fileName, boolean asTemplate, JSONObject jsonResult) {
        ErrorCode errorCode = ErrorCode.COPYDOCUMENT_FAILED_ERROR;

        if ((null != session) && (null != targetFolder) && (null != fileName)) {

            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
            final int[] resolveErrorCode = { MessageData.SAVE_ERROR_NONE };
            InputStream documentStm = null;
            boolean rollback = false;
            boolean giveUp = false;

            if (null != fileAccess) {

                try {
                    ArrayList<MessageChunk> currentMessages = null;
                    int osn = -1;

                    // make sure that no save operation interrupts our copy document processing
                    this.m_saveDocumentInProgress.getAndIncrement();
                    synchronized (m_messageChunkList) {
                        currentMessages = (ArrayList<MessageChunk>)m_messageChunkList.clone();
                        osn = m_connectionStatus.getOperationStateNumber();
                    }

                    // set up the new file object used to create a new infostore item
                    final File targetFile = new DefaultFile();
                    final File sourceFile = m_fileHelper.getMetaDataFromFile(session, FileStorageFileAccess.CURRENT_VERSION);
                    String extension = FileHelper.getExtension(sourceFile.getFileName());
                    String mimeType = sourceFile.getFileMIMEType();

                    // We want to copy the document and store it as a template
                    // document. Make sure that the document is not already a
                    // docment template.
                    if (asTemplate && !DocumentFormatHelper.isSupportedTemplateFormat(mimeType, extension)) {
                        // the document should be copied and "transformed" into
                        // a document template. We have to find out what format
                        // must be used.
                        HashMap<String, String> templateFormatInfo = DocumentFormatHelper.getTemplateFormatInfoForDocument(mimeType, extension);
                        if (null != templateFormatInfo) {
                            mimeType = templateFormatInfo.get(Properties.PROP_MIME_TYPE);
                            extension = templateFormatInfo.get(Properties.PROP_INPUT_TYPE);
                        } else {
                            // We don't know that template format to "transform"
                            // the source document to. Therefore we have to give
                            // up here and provide an error to the requesting
                            // client.
                            errorCode = ErrorCode.COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR;
                            giveUp = true;
                        }
                    }

                    if (!giveUp) {
                        // use provided values to initialize file object
                        targetFile.setId(FileStorageFileAccess.NEW);
                        targetFile.setFileName(fileName + "." + extension);
                        targetFile.setFolderId(targetFolder);
                        targetFile.setFileMIMEType(mimeType);
                        targetFile.setDescription(sourceFile.getDescription());

                        // Create the new document stream containing the latest changes
                        // from the message chunk list.
                        final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);
                        oxDocument.setUniqueDocumentId(osn);

                        documentStm = oxDocument.getResolvedDocumentStream(
                            session,
                            getExporter(),
                            m_resourceManager,
                            currentMessages,
                            null,
                            resolveErrorCode,
                            false,
                            asTemplate);

                        if (null != documentStm) {
                            fileAccess.startTransaction();
                            rollback = true;
                            fileAccess.saveDocument(targetFile, documentStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                            fileAccess.commit();
                            rollback = false;

                            // return actual parameters of new file
                            jsonResult.put("id", targetFile.getId());
                            jsonResult.put("folder_id", targetFile.getFolderId());
                            jsonResult.put("version", targetFile.getVersion());
                            jsonResult.put("filename", targetFile.getFileName());
                            jsonResult.put("file_mimetype", targetFile.getFileMIMEType());
                            errorCode = ErrorCode.NO_ERROR;
                        }
                    }
                } catch (OXException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to copy document.", e);
                    errorCode = ErrorCode.CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR;

                    if (e instanceof OXException) {
                        // special handling for OXExceptions
                        OXException ox = e;

                        // special handling for file storage exceptions
                        if (ox.getPrefix().equalsIgnoreCase("FLS")) {
                            if (ox.getCode() == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                                // set specific error code if quota reached
                                errorCode = ErrorCode.CREATEDOCUMENT_QUOTA_REACHED_ERROR;
                            }
                        }
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to setup result json object");
                } finally {
                    // Roll-back (if needed) and finish
                    if (rollback) {
                        TransactionAwares.rollbackSafe(fileAccess);
                    }

                    TransactionAwares.finishSafe(fileAccess);
                    IOUtils.closeQuietly(documentStm);

                    // decrement counter to enable saving again
                    this.m_saveDocumentInProgress.decrementAndGet();
                }
            }
        }

        return errorCode;
    }

    /**
     * @return
     */
    private JSONObject getSpreadsheetViewUpdate(final ID fromId, Session session, JSONObject jsonRequest) {
        JSONObject updateViewOperation = createUpdateViewOperation(jsonRequest);
        JSONObject resultObject = null;
        try {
            resultObject = executeSpreadsheetOperationFailsafe(fromId, updateViewOperation.toString());
        } catch (Exception e) {
            LOG.debug("Exception on getSpreadsheetViewUpdate()", e);
        }

        return checkJSONResult(resultObject);
    }

    /**
     *
     * @param fromId
     * @param operation
     * @return
     * @throws Exception
     */
    private JSONObject executeSpreadsheetOperationFailsafe(final ID fromId, final String operation) throws Exception {

    	final ContextAwareLogHelp aLog = mem_LogHelp ();
    	aLog.enterContext("doc-handle", m_calcDocumentHandle);

    	LOG.info(mem_LogHelp ().forLevel(ELogLevel.E_INFO)
    			               .toLog("execute operation"));

    	final StringBuffer result = new StringBuffer(256);
        ECalcEngineError error = ECalcEngineError.E_NONE;
        boolean sendHangUp = false;

        try {
        error = m_calcClient.executeOperation(m_calcDocumentHandle, operation, result);
        } catch (Throwable ex) {
        	LOG.error(mem_LogHelp ().forLevel(ELogLevel.E_ERROR)
 	               .toLog("... got exception"), ex);
        	sendHangUp = true;
        }

        LOG.info(mem_LogHelp ().forLevel(ELogLevel.E_INFO)
	               .toLog("... got state : "+error));

        if (error == ECalcEngineError.E_DOCUMENT_LOST)
        	sendHangUp = true;

        if (sendHangUp) {
        	LOG.error(mem_LogHelp ().forLevel(ELogLevel.E_ERROR)
 	               .toLog("... hang up all clients"));
            impl_sendHangUp (fromId, null, ErrorCode.HANGUP_CALCENGINE_NOT_RESPONDING_ERROR, HANG_UP_ALL, null);
        	throw new Exception("error [" + error + "] on execute operation '" + operation + "'.");
        }

        if (error != ECalcEngineError.E_NONE) {
            throw new Exception("error [" + error + "] on execute operation '" + operation + "'.");
        }

        final String resultStr = result.toString ();
    	LOG.debug(mem_LogHelp ().forLevel(ELogLevel.E_DEBUG)
	               .toLog("... result : '"+resultStr+"'"));
        JSONObject resultObject = new JSONObject(resultStr);
        return resultObject;
    }

//    /* disabled !
//     * restore not possible by non well designed core ...
//     * refactor the underlying design and reactivate this method
//     */
//    private void restoreSpreadsheetFromOperationCache(final String lastFailedOperation,
//    		final StringBuffer resultOfLastOperation) throws Exception {
//
//        // let exception out !
//        // outside code rely on that .-)
//
//    	if (m_messageChunkList != null) {
//    		final int c = m_messageChunkList.size();
//    		int i = 0;
//	    	final String[] restoreOperations = new String[c];
//
//	    	for (MessageChunk chunk : m_messageChunkList) {
//	    		final JSONObject jsonObj = chunk.getOperations();
//	    		final String jsonStr = jsonObj.toString();
//	    		restoreOperations[i++] = jsonStr;
//	    	}
//
//	    	LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] start restore of document");
//	        final ECalcEngineError error = m_calcClient.restoreDocument (m_calcDocumentHandle, restoreOperations);
//	        if (error != ECalcEngineError.E_NONE)
//	        	throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] restore failed.");
//    	}
//
//    	LOG.info("context[doc-handle:"+m_calcDocumentHandle+"] apply last operation (after restore)");
//    	resultOfLastOperation.setLength(0); // reset !
//    	final ECalcEngineError error = m_calcClient.executeOperation (m_calcDocumentHandle, lastFailedOperation, resultOfLastOperation);
//        if (error != ECalcEngineError.E_NONE)
//        	throw new Exception ("context[doc-handle:"+m_calcDocumentHandle+"] apply of last operation (after restore) - failed.");
//    }

    /**
     *
     * @param requestData
     * @return
     */
    private static JSONObject createUpdateViewOperation(JSONObject requestData) {
        final JSONArray operationsArray = new JSONArray();
        JSONObject operations = new JSONObject();
        try {
            requestData.put("name", "updateView");
            LOG.debug(requestData.toString());
            operationsArray.put(requestData);
            operations.put("operations", operationsArray);
        } catch (JSONException e) {
            LOG.error("Exception on createUpdateViewOperation", e);
        }
        return operations;
    }

    /**
     * Provides the importer for the current document.
     *
     * @return The Importer for the current file helper object
     */
    private IImporter getImporter() {
        IImporter importer = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentFormat()) {
                case DOCX: {
                    importer = m_services.getService(com.openexchange.office.ooxml.docx.Importer.class);
                    break;
                }

                case PPTX: {
                    importer = m_services.getService(com.openexchange.office.ooxml.pptx.Importer.class);
                    break;
                }

                case XLSX: {
                    importer = m_services.getService(com.openexchange.office.ooxml.xlsx.Importer.class);
                    break;
                }

                case ODT:
                case ODS: {
                    importer = m_services.getService(com.openexchange.office.odf.Importer.class);
                    break;
                }

                case NONE:
                case ODG:
                case ODP:
                default: {
                    break;
                }
            }
        }

        return importer;
    }

    /**
     * Provides the exporter for the current document depending on
     * the format.
     *
     * @return The Exporter for the current file helper object
     */
    private IExporter getExporter() {
        IExporter exporter = null;

        if (null != m_fileHelper) {
            switch (m_fileHelper.getDocumentFormat()) {

                case DOCX: {
                    exporter = m_services.getService(com.openexchange.office.ooxml.docx.Exporter.class);
                    break;
                }

                case PPTX: {
                    exporter = m_services.getService(com.openexchange.office.ooxml.pptx.Exporter.class);
                    break;
                }

                case XLSX: {
                    exporter = m_services.getService(com.openexchange.office.ooxml.xlsx.Exporter.class);
                    break;
                }

                case ODT:
                case ODS: {
                    exporter = m_services.getService(com.openexchange.office.odf.Exporter.class);
                    break;
                }

                case NONE:
                case ODG:
                case ODP:
                default: {
                    break;
                }
            }
        }

        return exporter;
    }

    /**
     * Provides thread-safe the current connection status as a shallow clone.
     *
     * @return The current connection status as a shallow clone.
     */
    private ConnectionStatus syncGetConnectionStatus() {
        synchronized (m_connectionStatus) {
            return (ConnectionStatus) m_connectionStatus.clone();
        }
    }

    /**
     * @param stanza
     * @param elementPath
     * @param keepActions
     * @return
     */
    static private JSONObject getJSONRequest(Stanza stanza, ElementPath elementPath, boolean keepActions) {
        final JSONObject jsonRequest = new JSONObject();

        // We're iterating over all messages that are constructed with the office.operations element path
        for (final PayloadTree payload : stanza.getPayloadTrees(elementPath)) {
            final Object data = payload.getRoot().getData();

            if (data instanceof JSONArray) {
                final JSONArray jsonArray = (JSONArray) data;
                final JSONObject jsonObject = new JSONObject();

                try {
                    jsonObject.put("actions", jsonArray);
                } catch (JSONException e) {
                    LOG.error("RT connection: getJSONRequest catches exception trying to to put actions to the JSON object. ", e);
                }

                ConnectionHelper.appendJSON(jsonRequest, jsonObject, keepActions);
            } else if (data instanceof JSONObject) {
                ConnectionHelper.appendJSON(jsonRequest, (JSONObject) data, keepActions);
            }
        }

        return jsonRequest;
    }

    /**
     * @param jsonResult
     * @param jsonRequest
     * @param jsonExtra
     * @return
     */
    static private JSONObject finalizeJSONResult(JSONObject jsonResult, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != jsonResult) {
            // add unique id element from JSON requests to JSON result
            if (null != jsonRequest) {
                try {
                    // add unique message id, if given in request
                    if (jsonRequest.has(MessageData.KEY_UNIQUE_ID)) {
                        jsonResult.put(MessageData.KEY_UNIQUE_ID, jsonRequest.get(MessageData.KEY_UNIQUE_ID));
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: getJSfinalizeJSONResult catches exception trying to to put unique id to the JSON object. ", e);
                }
            }

            // add extra data, if set
            if (null != jsonExtra) {
                ConnectionHelper.appendJSON(jsonResult, jsonExtra);
            }
        }

        return jsonResult;
    }

    /**
     * Checks the json result object is empty and sets the "hasErrors"
     * property to indicate an errornous state.
     *
     * @param jsonResult
     *  A json object containing the result of a request which should be
     *  checked.
     *
     * @return
     *  The jsonResult which contains a "hasError" property if it doesn't
     *  contain any properties.
     */
    static private JSONObject checkJSONResult(JSONObject jsonResult) {
        if ((null != jsonResult) && jsonResult.isEmpty()) {
            try {
                jsonResult.put("hasErrors", true);
            } catch (JSONException e) {
                LOG.error("RT connection: checkJSONResult catches exception trying to to put hasErrors to the JSON object. ", e);
            }
        }

        return jsonResult;
    }

    /**
     * Retrieves the sender ID from stanza.
     *
     * @param stanza The stanza provided by the Realtime framework that should
     * be used to retrieve the sender ID from.
     *
     * @return
     *  The id of the user who sent the request. If the request was done synchronously this is
     *  the "real" ID and not the synthetic.
     *  ATTENTION: In that case the fromID must be used for the answer message.setTo() method!
     */
    static private ID getIDFromStanza(Stanza stanza) {
        ID id = null;
        if (null != stanza) {
            id = stanza.getOnBehalfOf();
            if (null == id) {
                id = stanza.getFrom();
            }
        }

        return id;
    }

    /**
     * Retrieves the session object from a JSON request or ID. The method
     * tries to use the JSON request first and if not successful uses the
     * provided ID to retrieve the session. Either of these
     *
     * @param jsonRequest The json object containing the request data or null if not available.
     * @param sessionId The ID of the client requesting a service or null if not available.
     * @return The server session object or null if no session could be
     * retrieved from the jsonRequest nor from the ID.
     */
    static private ServerSession getServerSession(JSONObject jsonRequest, ID sessionId) {
        ServerSession serverSession = null;

        // try to get session from json request
        if (null != jsonRequest) {
            try {
                if (jsonRequest.has("session")) {
                    final JSONObject jsonSession = jsonRequest.getJSONObject("session");
                    if (null != jsonSession) {
                        final Session session = ServerServiceRegistry.getInstance().getService(SessiondService.class, true).getSession(
                            jsonSession.getString("resource"));
                        serverSession = ServerSessionAdapter.valueOf(session);
                    }
                }
            } catch (Exception e) {
                LOG.error("RT connection: Message handler not able to retrieve server session from json request!", e);
            }
        }

        // try to get session from ID
        if ((null == serverSession)) {
            try {
                serverSession = (null != sessionId) ? sessionId.toSession() : null;
            } catch (Exception e) {
                LOG.error("RT connection: Message handler not able to get server session!", e);
            }
        }

        return serverSession;
    }

    /**
     *  Save the latest changes of the document to the infostore.
     *
     * @param session
     *  The session of the client that requested to store the lastest changes.
     *
     * @param forceSave
     *  Forces a save even if a concurrent save is in progress. This should
     *  only be used in special situations, like a shutdown and an emergency save is needed.
     *
     * @param finalSave
     *  Specifies if this is the last flush so the filter can do cleanup work.
     *
     * @return
     *  A json object containing the result of the save process.
     */
    private JSONObject flushDocument(Session session, boolean forceSave, boolean finalSave) {
        final JSONObject jsonResult = new JSONObject();

        if (null != session) {
            long startTime = System.currentTimeMillis();

            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: flushDocument called");
            }
            // We only want to have just ONE save document process with
            // only ONE exception: if forceSave is set (only if the connection
            // is going to be disposed) we have to write the latest state.
            int saveCount = this.m_saveDocumentInProgress.getAndIncrement();
            if ((saveCount > 0) && !forceSave) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: flushDocument/failSafeSave in progress, no need to flush");
                }
                this.m_saveDocumentInProgress.getAndDecrement();
                return jsonResult;
            }

            // close document in every case to write left over operations,
            // if nothing needs to be written, newFileVersion will be the empty string
            try {
                final int[] errorCode = { MessageData.SAVE_ERROR_NONE };
                String newFileVersion = null;
                String oldFileVersion = null;
                ArrayList<MessageChunk> currentMessages = null;
                int osn = -1;
                int currSize = 0;

                synchronized (m_messageChunkList) {
                    if (!m_messageChunkList.isEmpty()) {
                        currentMessages = (ArrayList<MessageChunk>) m_messageChunkList.clone();
                    }
                    osn = m_connectionStatus.getOperationStateNumber();
                }

                if (null != currentMessages && !currentMessages.isEmpty()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("RT connection: Flushing operations, current message chunk list size = " + currentMessages.size());
                        LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                        JSONObject outputOperation = debugGetLastOperationFromMessageChunkList(currentMessages);
                        if (null != outputOperation) {
                            LOG.debug("RT connection: Flushing operations (before save): last operation in current message chunk list = " + operationToString(outputOperation));
                        }
                    }

                    try {
                        oldFileVersion = m_fileHelper.getVersionFromFile(session);
                    } catch (OXException e) {
                        if (e.getPrefix() == "IFO") {
                            // check possible problems reported by infostore service
                            if (e.getCode() == InfostoreExceptionCodes.NOT_EXIST.getNumber()) {
                                try {
                                    // set JSON result error code
                                    if (LOG.isWarnEnabled()) {
                                        LOG.warn("RT connection: flushing document failed, original document couldn't be found!");
                                    }
                                    jsonResult.put(MessageData.KEY_ERROR_CODE, MessageData.SAVE_ERROR_FILE_NOT_FOUND);
                                } catch (JSONException je) {
                                    LOG.error("RT connection: Couldn't add error code to JSON result. Flushing document failed, original document couldn't be found!");
                                }
                            }
                        }
                    }

                    final boolean createVersion = this.getCreateVersion();
                    final OXDocument oxDocument = new OXDocument(session, m_fileHelper, m_resourceManager);
                    oxDocument.setUniqueDocumentId(osn);

                    newFileVersion = oxDocument.save(
                        session,
                        getExporter(),
                        m_resourceManager,
                        currentMessages,
                        errorCode,
                        !createVersion,
                        finalSave);

                    // clean messageChunkList in case the messageChunkList has been flushed
                    if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                        int messageChunkListSize = 0;
                        JSONObject outputOperation = null;

                        if (createVersion) {
                            // After a new version we want to save revisionless
                            // until an editor leaves the document.
                            setCreateVersion(false);
                        }

                        synchronized (m_messageChunkList) {
                            currSize = m_messageChunkList.size();

                            // update sync status to the latest saved document
                            int version = Integer.parseInt(newFileVersion);
                            this.m_syncStatus.updateSyncInfo(version, osn);

                            // Remove all operations which have been saved recently
                            // leaving just the new ones added during the save process.
                            int index = currentMessages.size() - 1;

                            if (index == m_messageChunkList.size() - 1) {
                                m_messageChunkList.clear(); // optimization
                            } else if ((index >= 0) && ((index + 1) < m_messageChunkList.size())) {
                                // New operation list must only hold the new operations
                                // therefore we need to remove the old ones.
                                ArrayList<MessageChunk> copy = (ArrayList<MessageChunk>) m_messageChunkList.clone();
                                List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                m_messageChunkList.clear();
                                m_messageChunkList.addAll(newPart);
                            }
                            if (LOG.isDebugEnabled()) {
                                messageChunkListSize = m_messageChunkList.size();
                                outputOperation = debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                            }
                            m_lastSaveTime = new Date();
                            osn = m_connectionStatus.getOperationStateNumber();
                        }

                        if (LOG.isDebugEnabled()) {
                            LOG.debug("RT connection: Flushing operations completed.");
                            LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                            LOG.debug("RT connection: Flushing operations (after save): Before adjustment, message chunk list size = " + currSize);
                            LOG.debug("RT connection: Flushing operations (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Flushing operation (after save): First operation in messsage chunk list = " + operationToString(outputOperation));
                            }
                        }

                        if ((null != oldFileVersion) && (oldFileVersion.equals(newFileVersion))) {
                            // logical check: a successful revision save must create a new file version
                            if (LOG.isWarnEnabled()) {
                                LOG.warn("RT connection: Flushing operations successful but file version didn't change => Possible loss of document data!");
                            }
                        }

                        // update Statistics with saved document count (standard CLOSE type)
                        if (null != m_fileHelper) {
                            Statistics.handleDocumentEvent(new SaveEvent(m_fileHelper.getDocumentType(), SaveType.CLOSE));
                        }

                    } else {
                        // Handle error cases if office code is able to do it
                        // the error code is sent to the client, see below
                        if (LOG.isWarnEnabled()) {
                            LOG.warn("RT connection: Flushing operations reports no new file version => Possible loss of document data!");
                        }
                    }
                } else if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: No changes in message chunk list found => no need to flush document");
                }

                try {
                    // set JSON result error code and new file version
                    jsonResult.put(MessageData.KEY_ERROR_CODE, errorCode[0]);
                    jsonResult.put(Sync.SYNC_INFO, m_syncStatus.toJSON());

                    if (null != newFileVersion) {
                        jsonResult.put(MessageData.KEY_FILE_VERSION, newFileVersion);
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: Flush operations catches exception while trying to put error code/version to JSON result. ", e);
                }

                if (LOG.isWarnEnabled()) {
                    if ((MessageData.SAVE_ERROR_NOBACKUP == errorCode[0])) {
                        LOG.warn("RT connection: Flushing operations failed completely => Possible loss of document data!");
                    } else if ((MessageData.SAVE_ERROR_OPERATIONS == errorCode[0])) {
                        LOG.warn("RT connection: Flushing operations to native document format failed => OX document replacement archive has been written!");
                    }
                }

                if (LOG.isDebugEnabled()) {
                    if ((null != newFileVersion) && (newFileVersion.length() > 0)) {
                        LOG.info("RT connection: Flushing operations to native document succeeded");
                    } else {
                        LOG.info("RT connection: Closing document succeeded without needing to flush operations");
                    }
                }
            } catch (Exception e) {
                LOG.error("RT connection: Exception while flushing operations to document!", e);
            } finally {
                m_saveDocumentInProgress.decrementAndGet();
                // logging time for flushing the document
                long flushDocumentTime = (System.currentTimeMillis() - startTime);
                if ((flushDocumentTime > MAX_TIME_FOR_FLUSHDOCUMENT) && LOG.isWarnEnabled()) {
                    LOG.warn("RT connection: TIME to flushDocument exceeded client timeout: " + flushDocumentTime + "ms");
                } else if (LOG.isInfoEnabled()) {
                    LOG.info("RT connection: TIME to flushDocument: " + flushDocumentTime + "ms");
                }
            }
        }

        return jsonResult;
    }

    /**
     * Starts a new "preparelosingeditrights" timer thread which ensures that
     * a non-responsive client won't block switchting edit rights.
     *
     * @param session The session of the current editor client
     * @param id The ID of the current editor client.
     */
    private void startPrepareLosingEditRightsTimer(Session session, ID id) {
        TimerService timerService = m_services.getService(TimerService.class);

        if ((timerService != null) && (m_nMaxTimeForSendingLastActions > 0)) {
            // zero or negative values mean there is no time out at all
            m_prepareLosingEditRightsTimer = timerService.schedule(new PrepareLosingEditRightsTimeoutRunnable(this, session, id), m_nMaxTimeForSendingLastActions);
        }
    }

    /**
     * Cancels a running "preparelosingeditrights" timer. Won't do anything
     * if there is no preparelosingeditrights timer running.
     */
    private void cancelPrepareLosingEditRightsTimer() {
        // Cancel possible timeout timer
        if (m_prepareLosingEditRightsTimer != null) {
            m_prepareLosingEditRightsTimer.cancel();
            m_prepareLosingEditRightsTimer = null;
        }
    }

    /**
     * Provides the json data of the first operation of the first chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the first operation of the first chunk.
     *
     * @return The json data of the first operation of the first chunk or null if no chunk/operation is
     * available.
     */
    private static JSONObject debugGetFirstOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(0);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray("actions");
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(0);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(0);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Provides the json data of the last operation of the last chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the last operation of the last chunk.
     *
     * @return The json data of the last operation of the last chunk or null if no chunk/operation is
     * available.
     */
    private static JSONObject debugGetLastOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(chunkList.size() - 1);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray("actions");
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(actions.length() - 1);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Print all client user ids for debug purpose.
     *
     * @param connectionStatus
     *  The connection status where the active clients are stored within.
     */
    private static void debugPrintActiveClients(ConnectionStatus connectionStatus) {
        if (null != connectionStatus) {
            JSONArray activeUsers = connectionStatus.getActiveUsers();
            for (int i = 0; i < activeUsers.length(); i++) {
                JSONObject activeUser = activeUsers.optJSONObject(i);
                if (null != activeUser) {
                    String userId = activeUser.optString("userId", null);
                    LOG.debug("RT connection: Member = " + ((null != userId) ? userId : "unknown"));
                }
            }
        }
    }

    /**
     * Transforms an operation JSONObject to a string using a special writer to ensure
     * that the output won't be bigger that a maximum number of characters.
     *
     * @param operation The json object containing the data of an operation.
     *
     * @return A string representing the operation.
     */
    private static String operationToString(JSONObject operation) {
        String result = "";

        if (null != operation) {
            JSONDebugWriter myWriter = null;

            try {
                myWriter = new JSONDebugWriter();
                operation.write(myWriter);
                result = myWriter.getData();
            } catch (final Exception e) {
                LOG.warn("RT connection: operationToString catches exception writing operation to string. ", e);
            } finally {
                IOUtils.closeQuietly(myWriter);
            }
        }

        return result;
    }

    /**
     * Provides the folder and file id that this connection is associated with.
     *
     * @return A string which contains the folder and file id separated by a colon.
     *  If an id is unknown "n/a" is used instead.
     */
    private String getDocumentFolderAndFileId() {
        String folderFileId = "n/a.n/a";

        if (null != m_fileHelper) {
            final String folderId = m_fileHelper.getFolderId();
            final String fileId = m_fileHelper.getFileId();

            folderFileId = (folderId == null) ? "n/a" : folderId;
            folderFileId += ".";
            folderFileId += (fileId == null) ? "n/a" : fileId;
        }

        return folderFileId;
    }

    /**
     * Checks that the operations of a message chunk comply to
     * our minimal requirements:
     * 1. always ascending
     * 2. without any gap in osn/opl by consecutive operations
     *
     * @param chunk
     *  The MessageChunk which should be checked.
     *
     * @return
     *  TRUE if the operations inside the message chunk comply
     *  to the requirements or FALSE if not.
     */
    private boolean hasValidOperations(MessageChunk chunk) {
        boolean result = false;
        int serverOSN = syncGetConnectionStatus().getOperationStateNumber();
        JSONObject chunkOpsObject = chunk.getOperations();

        try {
            JSONArray actionsArray = chunkOpsObject.optJSONArray("actions");
            if (null != actionsArray && actionsArray.length() > 0) {
                int actions = actionsArray.length();

                for (int i=0; i < actions; i++) {
                    JSONObject actionsObject = actionsArray.getJSONObject(i);
                    JSONArray operationsArray = actionsObject.optJSONArray("operations");

                    if ((null != operationsArray) && (operationsArray.length() > 0)) {
                        int operations = operationsArray.length();
                        for (int j=0; j < operations; j++) {
                            JSONObject opsObject = operationsArray.getJSONObject(j);
                            int opsOPL = opsObject.optInt("opl");
                            int opsOSN = opsObject.optInt("osn");

                            if (opsOSN != serverOSN) {
                                // found inconsistent osn in chunk
                                return false;
                            }
                            serverOSN = opsOSN + opsOPL;
                        }
                    }
                }
            }

            result = true;
        } catch (final Exception e) {
            LOG.warn("RT connection: hasValidOperations catches exception trying to retrieve operations from chunkList. ", e);
        }

        return result;
    }

    /**
     * Provides an OpenEvent instance to be used for administrative purposes like
     * VM surveillance.
     *
     * @return An OpenEvent instance describing the current document.
     */
    private OpenEvent getOpenEvent() {
        OpenEvent openEvent = null;

        if (null != m_fileHelper) {
            final DocumentFormat documentFormat = m_fileHelper.getDocumentFormat();

            if (null != documentFormat) {
                OpenType documentOpenType;

                switch (documentFormat) {
                    case DOCX:
                    case XLSX:
                    case PPTX: {
                        documentOpenType = OpenType.MSX;
                        break;
                    }

                    case ODT:
                    case ODS:
                    case ODP: {
                        documentOpenType = OpenType.ODF;
                        break;
                    }

                    case NONE:
                    case ODG:
                    default: {
                        documentOpenType = OpenType.NONE;
                        break;
                    }
                }

                if (OpenType.NONE != documentOpenType) {
                    openEvent = new OpenEvent(m_fileHelper.getDocumentType(), documentOpenType, m_fileHelper.getDocumentSize());
                }
            }
        }

        return openEvent;
    }

    /**
     * Provides a context aware logging instance to used for logging messages.
     *
     * @return The context aware logging instance.
     */
    private ContextAwareLogHelp mem_LogHelp () {
    	try {
	    	if (m_logHelp == null)
	    		m_logHelp = new ContextAwareLogHelp (LOG);
	    	return m_logHelp;
    	} catch (Exception ex) {
    		throw new RuntimeException (ex);
    	}
    }

    /**
     * Sets the creator of a the current document file using in a synchronized
     * way with concurrent save operations.
     *
     * @param newCreator
     *  The ID of the user that should be set as the new creator. This won't
     *  work if the ID represents a synthetic user or is null.
     *
     * @return
     *  The method returns true if the creator could be set successfully,
     *  otherwise false.
     */
    private boolean setCreatorSynchronized(ID newCreator) {
        boolean result = false;

        if (null != newCreator) {
            Session session = Connection.getServerSession(null, newCreator);
            result = this.setCreatorSynchronized(session);
        }

        return result;
    }

    /**
     * Sets the creator of a the current document file using in a synchronized
     * way with concurrent save operations.
     *
     * @param session
     *  The Session of the user that should be set as the new creator.
     *
     * @return
     *  The method returns true if the creator could be set successfully or
     *  false if the creator couldn't be set.
     */
    private boolean setCreatorSynchronized(Session session) {
        boolean result = false;

        if (null != session) {
            // use the atomic saveDocumentInProgress member to be sure
            // that we don't interfere with a saving operation
            int saveCount = this.m_saveDocumentInProgress.getAndIncrement();
            if (saveCount == 0) {
                m_fileHelper.setCreatedByForFile(session);
            }
            this.m_saveDocumentInProgress.getAndDecrement();
        }

        return result;
    }

    /**
     * Sets the create version state to the provided value
     * in a thread-safe way.
     *
     * @param newValue
     *  The new value of the create version state.
     *
     * @return
     *  The old value of the create version state.
     */
    synchronized private boolean setCreateVersion(boolean newValue) {
        boolean oldValue = m_createVersion;
        m_createVersion = newValue;
        return oldValue;
    }

    /**
     * Retrieves the create version state in a thread-safe way.
     *
     * @return
     *  The current value of the create version state.
     */
    synchronized public boolean getCreateVersion() {
        return m_createVersion;
    }

    // - Members ----------------------------------------------------------------

    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(Connection.class);

    static private final ActionHandler m_handler = new ActionHandler(Connection.class);

    static private final long ACTIONS_UNTIL_FORCE_UPDATE = 20;

    static private final long MAX_TIME_FOR_FLUSHDOCUMENT = 20000; // time in milliseconds

    static private final int DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS = 20; // default in seconds

    static final boolean HANG_UP_ALL = true;
    static final boolean HANG_UP_ID = false;

    private ServiceLookup m_services = null;

    private FileHelper m_fileHelper = null;

    private ResourceManager m_resourceManager = null;

    private final ArrayList<MessageChunk> m_messageChunkList = new ArrayList<MessageChunk>();

    private JSONObject m_uiOperations = null;

    private final ConnectionStatus m_connectionStatus = new ConnectionStatus();

    private Sync m_syncStatus = null;

    private ICalcEngineClient m_calcClient = null;

    private String m_calcDocumentHandle;

    private boolean m_createVersion = true;

    private final AtomicInteger m_saveDocumentInProgress = new AtomicInteger();

    private Date m_lastSaveTime = null;

    private ScheduledTimerTask m_prepareLosingEditRightsTimer = null;

    private long m_threadId = 0;

    private long m_nActionsReceived = 0;

    private int m_nMaxTimeForSendingLastActions = (DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS * 1000);

    private ContextAwareLogHelp m_logHelp = null;
}
