/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.operations;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBException;

import org.apache.commons.logging.Log;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblStylePr;
import org.docx4j.wml.CTTrPrChange;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.Numbering.AbstractNum;
import org.docx4j.wml.Numbering.Num;
import org.docx4j.wml.Numbering.Num.AbstractNumId;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.STBorder;
import org.docx4j.wml.STTblStyleOverrideType;
import org.docx4j.wml.Style;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tabs;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblBorders;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner.TcBorders;
import org.docx4j.wml.Text;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.ComponentContext;
import com.openexchange.office.ooxml.docx.tools.Component.ComponentContextWrapper;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TableComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TcComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;
import com.openexchange.office.ooxml.docx.tools.Component.TrComponent;
import com.openexchange.office.ooxml.docx.tools.Paragraph;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.TextUtils;
import com.openexchange.office.ooxml.docx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

    protected final static Log log = LogFactory.getLog(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final WordprocessingMLPackage wordMLPackage;
    private final MainDocumentPart documentPart;
    private final ObjectFactory objectFactory;
    private final Map<String, String> conditionalTableAttrNames;

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        wordMLPackage = _operationDocument.getPackage();
        documentPart = wordMLPackage.getMainDocumentPart();
        objectFactory = Context.getWmlObjectFactory();

        conditionalTableAttrNames = new HashMap<String, String>();
        conditionalTableAttrNames.put("wholeTable", "wholeTable");
        conditionalTableAttrNames.put("firstRow", "firstRow");
        conditionalTableAttrNames.put("lastRow", "lastRow");
        conditionalTableAttrNames.put("firstCol", "firstCol");
        conditionalTableAttrNames.put("lastCol", "lastCol");
        conditionalTableAttrNames.put("band1Vert", "band1Vert");
        conditionalTableAttrNames.put("band2Vert", "band2Vert");
        conditionalTableAttrNames.put("band1Hor", "band1Horz");
        conditionalTableAttrNames.put("band2Hor", "band2Horz");
        conditionalTableAttrNames.put("northEastCell", "neCell");
        conditionalTableAttrNames.put("northWestCell", "nwCell");
        conditionalTableAttrNames.put("southEastCell", "seCell");
        conditionalTableAttrNames.put("southWestCell", "swCell");
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    /**
     *
     * After applying operations to the document, this method is called
     * once to fix the output so that it fulfills the OOXML specification.
     * (E.g: A Tc element must have at least one P. In our operation model
     * we create Tc and P elements independently, so there might exist
     * a Tc without P... this is fixed here)
     *
     * @author Sven Jacobi <sven.jacobi@open-xchange.com>
     */
    public void postFixNonConformance() {
        ComponentContext componentContext = new Component.MainContext(documentPart);
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            if(component instanceof TableComponent)
                postFixTableComponent((TableComponent)component);
        }
    }

    private void postFixTableComponent(TableComponent tableComponent) {
        ComponentContext componentContext = tableComponent.createComponentContext();
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            if(component instanceof TrComponent)
                postFixTrComponent((TrComponent)component);
        }
    }

    private void postFixTrComponent(TrComponent trComponent) {
        ComponentContext componentContext = trComponent.createComponentContext();
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            if(component instanceof TcComponent)
                postFixTcComponent((TcComponent)component);
        }
    }

    private void postFixTcComponent(TcComponent tcComponent) {
        ComponentContext componentContext = tcComponent.createComponentContext();
        boolean hasP = false;
        while(componentContext.hasNext()) {
            Component component = componentContext.next();
            if(component instanceof TableComponent) {
                postFixTableComponent((TableComponent)component);
                hasP = false;                                           // :-( there must be a P within Tc... and there must be a P behind Tbl
            }
            else if(component instanceof ParagraphComponent)
                hasP = true;
        }
        if(!hasP) {
            P p = objectFactory.createP();
            p.setRsidP("47110815");
            p.setParent(tcComponent.getObject());
            ((Tc)tcComponent.getObject()).getContent().add(p);
        }
    }

    public void delete(JSONArray start, JSONArray end)
        throws JSONException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        ComponentContextWrapper contextWrapper = new ComponentContextWrapper(new Component.MainContext(documentPart));
        Component component = TextUtils.getPosition(contextWrapper, start, start.length());

        // special text handling if parent is a paragraph
        if(component instanceof TextRun_Base)
            TextUtils.deleteText(objectFactory, (P)component.getContextObject(), startComponent, endComponent);
        else {

            ComponentContext componentContext = contextWrapper.getComponentContext();

            // deleting paragraphs, tables, cells and rows
            int startIndex = component.getContextIndex();
            int endIndex = startIndex;
            if(endComponent>startComponent) {
                while(componentContext.hasNext()) {
                    component = componentContext.next();
                    if (component.getComponentNumber()==endComponent) {
                        endIndex = component.getContextIndex();
                        break;
                    }
                }
            }
            if (startIndex>=0&&endIndex>=0) {
                List<Object> content = ((ContentAccessor)component.getContextObject()).getContent();
                for(int i=endIndex;i>=startIndex;i--)
                    content.remove(i);
            }
        }
    }

    /**
     * moving components, only DrawingComponents are supported
     *
     * @param startPosition
     * @param endPosition
     *
     */

    // TODO: it has to be checked if it is allowed to insert the component at the destination position

    public void move(JSONArray startPosition, JSONArray endPosition, JSONArray toPosition)
        throws JSONException {

        if(startPosition.length()!=endPosition.length())
            throw new JSONException("ooxml export: move operation, size of startPosition != size of endPosition");

        int startIndex = startPosition.getInt(startPosition.length()-1);
        int endIndex = endPosition.getInt(startPosition.length()-1);
        int toIndex = toPosition.getInt(toPosition.length()-1);

        if((endIndex - startIndex) < 0)
            throw new JSONException("ooxml export: move operation, component count < 1");

        Component sourceParent = null;
        Component toParent = null;

        // check if split is necessary (only needed if parent is a paragraph, if length == 1 then the parent is always root)
        if(startPosition.length()>1) {
            sourceParent = TextUtils.getPosition(new Component.MainContext(documentPart), startPosition, startPosition.length()-1);

            // only within paragraphs we can have components that might need a split (text runs)
            if(sourceParent instanceof ParagraphComponent) {
                TextUtils.splitTextRun(objectFactory, (P)sourceParent.getObject(), startIndex);
                TextUtils.splitTextRun(objectFactory, (P)sourceParent.getObject(), endIndex+1);
            }
        }
        else
            sourceParent = TextUtils.getPosition(new Component.MainContext(documentPart), startPosition, startPosition.length()-1);

        if(toPosition.length()>1) {
            toParent = TextUtils.getPosition(new Component.MainContext(documentPart), toPosition, toPosition.length()-1);

            if(toParent instanceof ParagraphComponent)
                TextUtils.splitTextRun(objectFactory, (P)toParent.getObject(), toIndex);
        }
        else
            toParent = TextUtils.getPosition(new Component.MainContext(documentPart), toPosition, toPosition.length()-1);

        ComponentContext startContext = sourceParent!=null ? sourceParent.createComponentContext() : new Component.MainContext(documentPart);
        ComponentContext toContext = toParent!=null ? toParent.createComponentContext() : new Component.MainContext(documentPart);

        Component startComponent = TextUtils.getPosition(startContext, startIndex);
        if(startComponent==null)
            throw new JSONException("ooxml export: move operation, could not get component for start position");

        Component toComponent = TextUtils.getInsertPosition(toContext, toIndex);
        if(toComponent==null)
            throw new JSONException("ooxml export: move operation, could not get component for insert position");

        int parentStartIndex = startComponent.getContextIndex();
        int parentEndIndex = parentStartIndex;
        int parentToIndex = toComponent.getContextIndex();

        while(startContext.hasNext()&&startComponent.getNextComponentNumber()<=endIndex) {
            startComponent = startContext.next();
            parentEndIndex = startComponent.getContextIndex();
        }

        List<Object> startContent = ((ContentAccessor)startComponent.getContextObject()).getContent();
        List<Object> toContent = ((ContentAccessor)toComponent.getContextObject()).getContent();

        boolean toIndexIncreasing = true;

        // check if destination index needs to be corrected if operating on the same content list
        if(startContent == toContent) {
            if(parentToIndex > parentStartIndex) {
                toIndexIncreasing = false;
                parentToIndex -= 1;
            }
        }

        for(int i = parentStartIndex; i <= parentEndIndex; i++) {
            Object o = startContent.remove(i);
            if(o instanceof Child)
                ((Child)o).setParent(toComponent.getContextObject());
            toContent.add(parentToIndex, o);
            if(toIndexIncreasing)
                parentToIndex++;
        }
    }

    public void insertParagraph(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        Component o = TextUtils.getInsertPosition(new Component.MainContext(documentPart), start);
        if (o!=null) {
            TextUtils.insertParagraph(o.getContextObject(), o.getContextIndex(), objectFactory.createP());
            if(attrs!=null) {
                setAttributes(attrs, start, null);
            }
        }
    }

    public void splitParagraph(JSONArray start)
        throws JSONException {

        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length()-1);
        if (o instanceof ParagraphComponent) {
                int textPosition = start.getInt(start.length()-1);
                TextUtils.splitParagraph(objectFactory, (P)o.getObject(), textPosition);
        }
    }

    public void mergeParagraph(JSONArray start) {
        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length());
        if(o instanceof ParagraphComponent)
            TextUtils.mergeParagraph((P)o.getObject());
    }

    public void insertText(JSONArray start, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {
        if(text.length()>0) {
            Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length()-1);
            if(o instanceof ParagraphComponent) {
                int textPosition = start.getInt(start.length()-1);
                TextUtils.insertText(objectFactory, (P)o.getObject(), textPosition, text);
                if(attrs!=null) {
                    int i = 0;
                    JSONArray end = new JSONArray();
                    for(i=0;i<start.length()-1;i++)
                        end.put(i, start.getInt(i));
                    end.put(i, start.getInt(i)+(text.length()-1));
                    setAttributes(attrs, start, end);
                }
            }
        }
    }

    public void insertTextRunChild(JSONArray start, JSONObject attrs, Child newTextRunChild)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        // the parent position of start needs to point to a paragraph, otherwise a textRun can't be found or inserted
        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length() - 1);
        if(!(o instanceof ParagraphComponent))
            return;

        int destIndex = 0;
        R destTextRun      = null;
        R referenceTextRun = null;

        P p = (P)o.getObject();
        int position = start.optInt(start.length() - 1, 0);

        ComponentContext context = o.createComponentContext();
        if(!context.hasNext())  // just create and insert R
            destIndex = 0;
        else if(position==0)    // position == 0 -> the textRunChild needs to get the properties from the following textRun
        {
            Component n = context.next();

            // check if we can insert our child within the textRun of the next Component,
            // if not, we will just take the textRun as reference to clone the character properties later
            if(((TextRun_Base)n).isSingleComponentRun())
                referenceTextRun = ((TextRun_Base)n).getTextRun();
            else
                destTextRun = ((TextRun_Base)n).getTextRun();
        }
        else {

            while(context.hasNext()) {
                Component n = context.next();
                if(n.getNextComponentNumber()<position)
                    continue;
                else if(n.getNextComponentNumber()==position) {
                    if(((TextRun_Base)n).isSingleComponentRun()) {
                        referenceTextRun = ((TextRun_Base)n).getTextRun();
                        destIndex = n.getContextIndex() + 1;
                    }
                    else {
                        destTextRun = ((TextRun_Base)n).getTextRun();
                        destIndex = ((TextRun_Base)n).getTextRunIndex() + 1;
                    }
                }
                // we will only come here, if n.nextComponentNumber - n.ComponentNumber is
                // greater than 1, this can only be a textComponent at the moment
                else if(n instanceof TextComponent) {   // only
                        TextUtils.splitText(objectFactory, (Text)n.getObject(), position - n.getComponentNumber(), ((TextComponent) n).getTextRunIndex());
                        destTextRun = ((TextComponent) n).getTextRun();
                        destIndex = ((TextComponent) n).getTextRunIndex() + 1;
                }
                break;
            }
        }

        if(destTextRun==null) {
            destTextRun = objectFactory.createR();
            destTextRun.setParent(p);
            p.getContent().add(destIndex, destTextRun);
            destIndex = 0;
        }
        destTextRun.getContent().add(destIndex, newTextRunChild);
        newTextRunChild.setParent(destTextRun);

        // we have to clone the characterProperties
        if(referenceTextRun!=null) {
            RPr rPr = referenceTextRun.getRPr();
            if(rPr!=null) {
                RPr newRPr = XmlUtils.deepCopy(rPr);
                newRPr.setParent(destTextRun);
                destTextRun.setRPr(newRPr);
            }
        }
        if(attrs!=null)
            setAttributes(attrs, start, null);
    }

    public void insertTab(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        insertTextRunChild(start, attrs, objectFactory.createRTab());
    }

    public void insertHardBreak(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        insertTextRunChild(start, attrs, objectFactory.createBr());
    }

    public void insertDrawing(JSONArray start, @SuppressWarnings("unused") String type, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        insertTextRunChild(start, attrs, objectFactory.createDrawing());
    }

    public void insertTable(JSONArray start, JSONObject attrs)
        throws JSONException {

        Component o = TextUtils.getInsertPosition(new Component.MainContext(documentPart), start);
        if (o!=null)
            Table.insertTable(operationDocument, o.getContextObject(), o.getContextIndex(), objectFactory, attrs);
    }

    public void insertRows(JSONArray start, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
        throws JSONException {

        Component o = TextUtils.getInsertPosition(new Component.MainContext(documentPart),  start);
        if(o!=null && o.getContextObject() instanceof Tbl) {
            Tbl tbl = (Tbl)o.getContextObject();
            List<Object> tblContent = tbl.getContent();
            int destinationIndex = o.getContextIndex();
            Component trReferenceComponent = referenceRow!=-1 ? TextUtils.getPosition(new Component.TableContext(tbl), referenceRow) : null;
            JSONObject rowAttrs = attrs!=null ? attrs.optJSONObject("row") : null;
            for(int i = 0; i<count; i++) {
                Tr tr = objectFactory.createTr();
                tr.setParent(tbl);
                tblContent.add(destinationIndex++, tr);
                if(insertDefaultCells) {
                    JSONArray tableGrid = Table.getTableGrid(tbl);
                    if(tableGrid!=null) {
                        for(int j = 0; j<tableGrid.length(); j++) {
                            Tc tc = objectFactory.createTc();
                            tc.setParent(tr);
                            tr.getContent().add(tc);
                        }
                    }
                }
                else if(trReferenceComponent!=null) {       // we try to create the new row from the referenceRow
                    Tr trReference = (Tr)trReferenceComponent.getObject();
                    TrPr sourceTrPr = trReference.getTrPr();
                    if(sourceTrPr!=null) {
                        TrPr newTrPr = XmlUtils.deepCopy(sourceTrPr);
                        newTrPr.setParent(tr);
                        tr.setTrPr(newTrPr);
                    }
                    Component.TableRowContext tableRowContext = new Component.TableRowContext(trReference);
                    while(tableRowContext.hasNext()) {
                        TcComponent tcComponent = (TcComponent)tableRowContext.next();
                        Tc tcReference = (Tc)tcComponent.getObject();
                        Tc newTc = objectFactory.createTc();
                        newTc.setParent(tr);
                        tr.getContent().add(newTc);
                        TcPr tcPrReference = tcReference.getTcPr();
                        if(tcPrReference!=null) {
                            TcPr newTcPr = XmlUtils.deepCopy(tcPrReference);
                            newTcPr.setParent(newTc);
                            newTc.setTcPr(newTcPr);
                        }
                    }
                }
                if(rowAttrs!=null) {
                    TrPr trPr = tr.getTrPr();
                    if(trPr==null) {
                        trPr = objectFactory.createTrPr();
                        trPr.setParent(tr);
                        tr.setTrPr(trPr);
                    }
                    Table.applyRowProperties(objectFactory, rowAttrs, trPr);
                }
            }
        }
    }

    public void insertCells(JSONArray start, int count, JSONObject attrs)
        throws JSONException {

        Component o = TextUtils.getInsertPosition(new Component.MainContext(documentPart),  start);
        if (o!=null&&o.getContextObject() instanceof Tr) {
            Tr tr = (Tr)o.getContextObject();
            List<Object> trContent = tr.getContent();
            int destinationIndex = o.getContextIndex();

            JSONObject cellAttrs = attrs != null ? attrs.optJSONObject("cell") : null;
            for (int i=0; i<count; i++) {
                Tc tc = objectFactory.createTc();
                tc.setParent(tr);
                if(cellAttrs!=null) {
                    TcPr tcPr = tc.getTcPr();
                    if(tcPr==null) {
                        tcPr = objectFactory.createTcPr();
                        tcPr.setParent(tc);
                        tc.setTcPr(tcPr);
                    }
                    Table.applyCellProperties(operationDocument, objectFactory, cellAttrs, tcPr);
                }
                trContent.add(destinationIndex++, tc);
            }
        }
    }

    public void insertColumn(JSONArray start, JSONArray tableGrid, int gridPosition, String insertMode)
        throws JSONException {

        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length());
        if (o instanceof TableComponent) {

            boolean before = insertMode.equals("before");
            Component.TableContext tableContext = new Component.TableContext((Tbl)o.getObject());
            while(tableContext.hasNext()) {
                TrComponent trComponent = (TrComponent)tableContext.next();

                TcComponent destination = null;
                TcComponent tcReference = null;
                Component.TableRowContext tableRowContext = new Component.TableRowContext((Tr)trComponent.getObject());
                while(tableRowContext.hasNext()) {
                    tcReference = (TcComponent)tableRowContext.next();
                    if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                        destination = tcReference;
                        break;
                    }
                }
                List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                int destinationIndex = destination==null?rowContent.size():before?destination.getContextIndex():destination.getContextIndex()+1;
                Tc tc = objectFactory.createTc();
                tc.setParent(trComponent.getObject());

                if(tcReference!=null) {
                    TcPr referenceTcPr = ((Tc)tcReference.getObject()).getTcPr();
                    if(referenceTcPr!=null) {
                        TcPr newTcPr = XmlUtils.deepCopy(referenceTcPr);
                        newTcPr.setParent(tc);
                        tc.setTcPr(newTcPr);
                    }
                }
                rowContent.add(destinationIndex, tc);
            }
            Table.setTableGrid(objectFactory, (Tbl)o.getObject(), tableGrid);
        }
    }

    public void deleteColumns(JSONArray position, int gridStart, int gridEnd)
        throws JSONException {

        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), position, position.length());
        if (o instanceof TableComponent) {
            Component.TableContext tableContext = new Component.TableContext((Tbl)o.getObject());
            while(tableContext.hasNext()) {
                int startIndex = -1;
                int endIndex = -1;
                TrComponent trComponent = (TrComponent)tableContext.next();
                Component.TableRowContext tableRowContext = new Component.TableRowContext((Tr)trComponent.getObject());
                while(tableRowContext.hasNext()) {
                    TcComponent tcComponent = (TcComponent)tableRowContext.next();
                    if((tcComponent.getGridPosition()>=gridStart&&tcComponent.getGridPosition()<=gridEnd)||
                        (tcComponent.getNextGridPosition()-1>=gridStart&&tcComponent.getNextGridPosition()-1<=gridEnd)){
                        if (startIndex==-1)
                            startIndex = tcComponent.getContextIndex();
                        endIndex = tcComponent.getContextIndex();
                    }
                    if(tcComponent.getNextGridPosition()>gridEnd)
                        break;
                }
                if(startIndex!=-1) {
                    List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                    for(int i=endIndex;i>=startIndex;i--) {
                        rowContent.remove(i);
                    }
                }
            }
            JSONArray tableGrid = Table.getTableGrid((Tbl)o.getObject());
            JSONArray newTableGrid = new JSONArray();
            for(int i=0;i<tableGrid.length();i++) {
                if(i<gridStart||i>gridEnd) {
                    newTableGrid.put(tableGrid.get(i));
                }
            }
            Table.setTableGrid(objectFactory, (Tbl)o.getObject(), newTableGrid);
        }
        else
            System.out.println("OOX Error: ApplyOperationHelper:deleteColumns: Table requested...");
    }

    public void setAttributes(JSONObject attrs, JSONArray start, JSONArray end)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        ComponentContextWrapper contextWrapper = new ComponentContextWrapper(new Component.MainContext(documentPart));
        Component component = TextUtils.getPosition(contextWrapper, start, start.length());
        if(component!=null) {
            int startIndex = start.getInt(start.length()-1);
            int endIndex = startIndex;

            if(end!=null) {
                if(end.length()!=start.length())
                    return;
                endIndex = end.getInt(end.length()-1);
            }
            // special case ... text formatting
            if(component instanceof TextRun_Base)
                TextUtils.setTextRunAttributes(operationDocument, attrs, objectFactory, (P)component.getContextObject(), startIndex, endIndex);
            else
            {
                ComponentContext componentContext = contextWrapper.getComponentContext();
                while(component.getComponentNumber()<=endIndex) {
                    if(component instanceof ParagraphComponent) {
                        P paragraph = (P)component.getObject();
                        PPr pPr = paragraph.getPPr();
                        if (pPr==null) {
                            pPr = objectFactory.createPPr();
                            pPr.setParent(paragraph);
                            paragraph.setPPr(pPr);
                        }
                        if(attrs.has("styleId")) {
                            final Object styleId = attrs.get("styleId");
                            if(styleId instanceof String) {
                                PPrBase.PStyle pStyle = pPr.getPStyle();
                                if(pStyle==null) {
                                    pStyle = objectFactory.createPPrBasePStyle();
                                    pStyle.setParent(pPr);
                                }
                                pStyle.setVal((String)styleId);
                                pPr.setPStyle(pStyle);
                            }
                            else {
                                pPr.setPStyle(null);
                            }
                        }
                        Paragraph.applyParagraphProperties(operationDocument, objectFactory, attrs.optJSONObject("paragraph"), pPr);
                        if(attrs.hasAndNotNull("character")) {
                            ParaRPr paraRPr = pPr.getRPr();
                            if(paraRPr==null) {
                                paraRPr = objectFactory.createParaRPr();
                                paraRPr.setParent(pPr);
                                pPr.setRPr(paraRPr);
                            }
                            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, objectFactory, attrs.optJSONObject("character"), paraRPr);
                        }
                    }
                    else if(component instanceof TableComponent) {
                        if(attrs.has("styleId")) {
                            Table.applyTableStyle(objectFactory, attrs.getString("styleId"), (Tbl)component.getObject());
                        }
                        Table.applyTableProperties(operationDocument, objectFactory, attrs.optJSONObject("table"), (Tbl)component.getObject(), null);
                    }
                    else if(component instanceof TrComponent) {
                        Tr tr = (Tr)component.getObject();
                        TrPr trPr = tr.getTrPr();
                        if(trPr==null) {
                            trPr = objectFactory.createTrPr();
                            trPr.setParent(tr);
                            tr.setTrPr(trPr);
                        }
                        Table.applyRowProperties(objectFactory, attrs.optJSONObject("row"), trPr);
                    }
                    else if(component instanceof TcComponent) {
                        Tc tc = (Tc)component.getObject();
                        TcPr tcPr = tc.getTcPr();
                        if(tcPr==null) {
                            tcPr = objectFactory.createTcPr();
                            tcPr.setParent(tc);
                            tc.setTcPr(tcPr);
                        }
                        Table.applyCellProperties(operationDocument, objectFactory, attrs.optJSONObject("cell"), tcPr);
                    }
                    if(componentContext.hasNext()==false)
                        break;
                    component = componentContext.next();
                }
            }
        }
    }

    public void insertField(JSONArray start, String type, String representation, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        Component o = TextUtils.getPosition(new Component.MainContext(documentPart), start, start.length()-1);
        if (o instanceof ParagraphComponent) {

            int fieldPosition = start.getInt(start.length()-1);
            TextUtils.splitTextRun(objectFactory, (P)o.getObject(), fieldPosition);

            Component destination = null;
            Component.ParagraphContext paragraphContext = new Component.ParagraphContext((P)o.getObject());
            while(paragraphContext.hasNext()) {
                Component component = paragraphContext.next();
                if(fieldPosition==component.getComponentNumber()) {
                    destination = component;
                    break;
                }
                else if (fieldPosition<component.getComponentNumber()) {
                    System.out.println("OOX Error: ApplyOperationHelper:insertField: couldn't insert Field...");
                    break;
                }
            }
            List<Object> paraContent = ((P)o.getObject()).getContent();
            CTSimpleField ctSimpleField = objectFactory.createCTSimpleField();
            ctSimpleField.setParent(o.getObject());
            if(destination==null)
                paraContent.add(ctSimpleField);
            else
                paraContent.add(destination.getContextIndex(), ctSimpleField);
            ctSimpleField.setInstr(type);
            R simpleFieldRun = objectFactory.createR();
            simpleFieldRun.setParent(ctSimpleField);
            ctSimpleField.getContent().add(simpleFieldRun);
            Text simpleFieldText = objectFactory.createText();
            simpleFieldText.setValue(representation);
            simpleFieldText.setParent(simpleFieldRun);
            simpleFieldRun.getContent().add(simpleFieldText);
            if(attrs!=null)
                setAttributes(attrs, start, null);
        }
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, String parentId, Boolean hidden, Integer uipriority, Boolean isDefault)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        final List<Style> stylesList = styles.getStyle();

        // removing style if it already exists
        Iterator<Style> styleIter = stylesList.iterator();
        while(styleIter.hasNext()){
            Style aStyle = styleIter.next();
            if ((aStyle.getType().equals(type)) && (aStyle.getStyleId().equals(styleId))) {
                styleIter.remove();
                break;
            }
        };

        // always creating a new style...
        Style style = objectFactory.createStyle();
        stylesList.add(style);

        // style properties
        style.setParent(styles);
        style.setType(type);
        style.setStyleId(styleId);
        if (styleName != null) {
            Style.Name name = objectFactory.createStyleName();
            name.setParent(style);
            name.setVal(styleName);
            style.setName(name);
        }
        if (hidden != null) {
            BooleanDefaultTrue hiddenValue = objectFactory.createBooleanDefaultTrue();
            hiddenValue.setParent(style);
            hiddenValue.setVal(hidden.booleanValue());
            style.setHidden(hiddenValue);
        }
        if (isDefault != null) {
            style.setDefault(isDefault.booleanValue());
        }
        if (uipriority != null) {
            Style.UiPriority uiPrio = objectFactory.createStyleUiPriority();
            uiPrio.setParent(style);
            uiPrio.setVal(BigInteger.valueOf(uipriority.intValue()));
            style.setUiPriority(uiPrio);
        }
        if (parentId != null) {
            Style.BasedOn basedOn = objectFactory.createStyleBasedOn();
            basedOn.setParent(style);
            basedOn.setVal(parentId);
            style.setBasedOn(basedOn);
        }

        // paragraph properties
        JSONObject paragraphProps = attrs.optJSONObject("paragraph");
        if (paragraphProps != null) {
            String nextStyle = paragraphProps.optString("nextStyleId");
            if (nextStyle != null) {
               Style.Next next = objectFactory.createStyleNext();
               next.setParent(style);
               next.setVal(nextStyle);
               style.setNext(next);
            }
            PPr ppr = style.getPPr();
            if (ppr == null) {
                ppr = objectFactory.createPPr();
                ppr.setParent(style);
                style.setPPr(ppr);
            }
            Paragraph.applyParagraphProperties(operationDocument, objectFactory, paragraphProps, ppr);
        }

        // character properties
        JSONObject characterProps = attrs.optJSONObject("character");
        if (characterProps != null) {
            RPr rpr = style.getRPr();
            if (rpr == null) {
                rpr = objectFactory.createRPr();
                rpr.setParent(style);
                style.setRPr(rpr);
            }
            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, objectFactory, characterProps, rpr);
        }

        if(type.equals("table")) {
            Iterator<String> keys = attrs.keys();
            while(keys.hasNext()) {
                String key = keys.next();
                if(conditionalTableAttrNames.containsKey(key)) {
                    JSONObject conditionalTableProps = attrs.optJSONObject(key);
                    if (conditionalTableProps != null) {
                        String conditionalTableName = conditionalTableAttrNames.get(key);
                        List<CTTblStylePr> conditionalStyles = style.getTblStylePr();
                        CTTblStylePr conditionalStyle = objectFactory.createCTTblStylePr();
                        conditionalStyle.setType(STTblStyleOverrideType.fromValue(conditionalTableName));
                        Iterator<String> tablePropKeys = conditionalTableProps.keys();
                        while(tablePropKeys.hasNext()) {
                            String tablePropKey = tablePropKeys.next();
                            JSONObject tablePropAttrs = conditionalTableProps.optJSONObject(tablePropKey);
                            if(tablePropAttrs!=null) {
                                if(tablePropKey.equals("table")) {
                                    CTTblPrBase tblPrBase = conditionalStyle.getTblPr();
                                    if (tblPrBase==null) {
                                        tblPrBase = objectFactory.createCTTblPrBase();
                                        tblPrBase.setParent(conditionalStyle);
                                        conditionalStyle.setTblPr(tblPrBase);
                                    }
                                    Table.applyTableProperties(operationDocument, objectFactory, tablePropAttrs, null, tblPrBase);
                                } else if(tablePropKey.equals("row")) {
                                     TrPr trPr = conditionalStyle.getTrPr();
                                    if (trPr==null) {
                                        trPr = objectFactory.createTrPr();
                                        trPr.setParent(conditionalStyle);
                                        conditionalStyle.setTrPr(trPr);
                                    }
                                    Table.applyRowProperties(objectFactory, tablePropAttrs, trPr);
                                } else if(tablePropKey.equals("cell")) {
                                    TcPr tcPr = conditionalStyle.getTcPr();
                                   if (tcPr==null) {
                                       tcPr = objectFactory.createTcPr();
                                       tcPr.setParent(conditionalStyle);
                                       conditionalStyle.setTcPr(tcPr);
                                   }
                                   Table.applyCellProperties(operationDocument, objectFactory, tablePropAttrs, tcPr);
                                } else if(tablePropKey.equals("paragraph")) {
                                    PPr pPr = conditionalStyle.getPPr();
                                    if (pPr==null) {
                                        pPr = objectFactory.createPPr();
                                        pPr.setParent(conditionalStyle);
                                        conditionalStyle.setPPr(pPr);
                                    }
                                    Paragraph.applyParagraphProperties(operationDocument, objectFactory, tablePropAttrs, pPr);
                                } else if(tablePropKey.equals("character")) {
                                    RPr rPr = conditionalStyle.getRPr();
                                    if (rPr==null) {
                                        rPr = objectFactory.createRPr();
                                        rPr.setParent(conditionalStyle);
                                        conditionalStyle.setRPr(rPr);
                                    }
                                    com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, objectFactory, tablePropAttrs, rPr);
                                }
                            }
                        }

                        // we will not create a whole_table style, because it does not seem to work,
                        // (even the TblLook does not mention the conditional WholeTable style) instead
                        // we will set the properties direct as tblPr
                        if(conditionalStyle.getType()==STTblStyleOverrideType.WHOLE_TABLE) {
                            style.setTblPr(conditionalStyle.getTblPr());
                            style.setTrPr(conditionalStyle.getTrPr());
                            style.setTcPr(conditionalStyle.getTcPr());
                            style.setPPr(conditionalStyle.getPPr());
                            style.setRPr(conditionalStyle.getRPr());

                            // if no shading is used within the TblPr, then we have to copy the shading from
                            // the cell into TblPr (LO does not take care about cell shading, so we have to do this workaround)
                            CTTblPrBase tblPr = style.getTblPr();
                            if(tblPr==null||tblPr.getShd()==null&&style.getTcPr()!=null) {
                            	final CTShd tableShade = style.getTcPr().getShd();
                            	if(tableShade!=null) {
	                            	if(tblPr==null) {
	                                	tblPr = objectFactory.createCTTblPrBase();
	                                	style.setTblPr(tblPr);
	                                	tblPr.setParent(style);
	                                }
	                            	tblPr.setShd(XmlUtils.deepCopy(tableShade));
	                            	tblPr.getShd().setParent(tblPr);
                            	}
                            }
                        }
                        else {
                            conditionalStyles.add(conditionalStyle);
                        }
                    }
                }
            }
        }
    }

    @SuppressWarnings("unused")
    public void deleteStylesheet(String type, String styleName) {
        //
    }
    public void insertListStyle(String listStyleId, JSONObject listDefinition)
        throws InvalidFormatException, JAXBException, JSONException {

        NumberingDefinitionsPart numDefPart = wordMLPackage.getMainDocumentPart().getNumberingDefinitionsPart();
        if( numDefPart == null ) {
            numDefPart = new NumberingDefinitionsPart();
            numDefPart.unmarshalDefaultNumbering();
            Numbering numbering = numDefPart.getJaxbElement();
            // remove defaults
            numbering.getAbstractNum().clear();
            numbering.getNum().clear();
            wordMLPackage.getMainDocumentPart().addTargetPart(numDefPart);
        }
        Numbering numbering = numDefPart.getJaxbElement();
        List<AbstractNum> abstractNumList = numbering.getAbstractNum();

        //find a 'free' abstractNumId
        Iterator<AbstractNum> abstractIter = abstractNumList.iterator();
        HashSet<Long> existingAbstractIds = new HashSet<Long>();
        while( abstractIter.hasNext() ){
            AbstractNum abstractNum = abstractIter.next();
            long abstractNumId = abstractNum.getAbstractNumId().longValue();
            existingAbstractIds.add(new Long(abstractNumId));
        }
        long newAbstractId = 0;
        while( existingAbstractIds.contains( new Long( newAbstractId ) ) )
            ++newAbstractId;

        AbstractNum newAbstractNum = objectFactory.createNumberingAbstractNum();
        newAbstractNum.setParent(numDefPart);
        BigInteger abstractId = BigInteger.valueOf(newAbstractId);
        newAbstractNum.setAbstractNumId(abstractId);
        List<Lvl> lvlList = newAbstractNum.getLvl();
        for( long lvlIndex = 0; lvlIndex < 9; ++lvlIndex){
            JSONObject levelObject = listDefinition.getJSONObject("listLevel" + lvlIndex);
            Lvl lvl = objectFactory.createLvl();
            lvl.setParent(newAbstractNum);
            lvl.setIlvl(BigInteger.valueOf(lvlIndex));
            if( levelObject.has("numberFormat")) {
                NumFmt numFmt = new NumFmt();
                numFmt.setVal( NumberFormat.fromValue( levelObject.getString("numberFormat") ));
                lvl.setNumFmt(numFmt);
            }
            if( levelObject.has("listStartValue")) {
                Lvl.Start lvlStart = objectFactory.createLvlStart();
                lvlStart.setVal(BigInteger.valueOf(levelObject.getLong("listStartValue")));
                lvl.setStart(lvlStart);
            }

            if( levelObject.has("levelText")) {
                if( lvl.getLvlText() == null )
                    lvl.setLvlText(objectFactory.createLvlLvlText());
                lvl.getLvlText().setVal(levelObject.getString("levelText"));
            }
            if( levelObject.has("textAlign") ){
                lvl.setLvlJc( objectFactory.createJc() );
                lvl.getLvlJc().setVal(JcEnumeration.fromValue(levelObject.getString("textAlign")));
            }
            RPr runProperties = objectFactory.createRPr();
            lvl.setRPr(runProperties);
            if( levelObject.has("fontName") ){
                runProperties.setRFonts(objectFactory.createRFonts());
                String font = levelObject.getString("fontName");
                runProperties.getRFonts().setAscii(font);
                runProperties.getRFonts().setHAnsi(font);
            }
            if( levelObject.has("paraStyle") ){
                Lvl.PStyle pstyle = objectFactory.createLvlPStyle();
                pstyle.setVal(levelObject.getString("paraStyle"));
            }

            PPr paraProperties = objectFactory.createPPr();
            lvl.setPPr(paraProperties);
            Ind ind = objectFactory.createPPrBaseInd();
            if( levelObject.has("indentLeft")) {
                ind.setLeft(Utils.map100THMMToTwip( levelObject.getInt("indentLeft")) );
            }
            if( levelObject.has("indentFirstLine")) {
                int firstline = levelObject.getInt("indentFirstLine");
                if( firstline > 0){
                    ind.setFirstLine( Utils.map100THMMToTwip( firstline ));
                    ind.setHanging( null );
                } else if( firstline < 0) {
                    ind.setFirstLine( null );
                    ind.setHanging( Utils.map100THMMToTwip( -firstline));
                }
            }
            paraProperties.setInd( ind );
            if( levelObject.has("tabStopPosition")){
                Tabs tabs = objectFactory.createTabs();
                CTTabStop tabStop = objectFactory.createCTTabStop();
                tabStop.setPos( Utils.map100THMMToTwip( levelObject.getInt("tabStopPosition") ) );
                tabs.getTab().add(tabStop);
                paraProperties.setTabs(tabs);
            }
            lvlList.add(lvl);
        }
        abstractNumList.add(newAbstractNum);
//        newAbstractNum.setName(value);
//        newAbstractNum.setMultiLevelType(value);

        List<Num> numList = numbering.getNum();
        Num num = objectFactory.createNumberingNum();

        num.setParent(numDefPart);
        AbstractNumId abstractNumId = new AbstractNumId();
        abstractNumId.setVal(abstractId);
        num.setAbstractNumId( abstractNumId );
        BigInteger listId;
        try {
            if(!Character.isDigit(listStyleId.charAt(0))){
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId.substring(1)));
            } else {
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId));
            }
            num.setNumId(listId);

            // removing list style if it already exists
            Iterator<Num> numIter = numList.iterator();
            while(numIter.hasNext()){
                Num aNum = numIter.next();
                if (aNum.getNumId().equals(listId)) {
                    numIter.remove();
                    break;
                }
            };
            numList.add(num);
        }
        catch (NumberFormatException e) {
            log.info("docx export insertListStyle: invalid listStyleId found!");
        }
    }

    public void setDocumentLanguageIfNotSet(String langStr)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        DocDefaults docDefaults = styles.getDocDefaults();
        if (docDefaults == null)
            docDefaults = objectFactory.createDocDefaults();
        if(docDefaults!=null) {
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if (rDefault == null)
                rDefault = objectFactory.createDocDefaultsRPrDefault();
            if(rDefault!=null) {
                RPr rPrDefault = rDefault.getRPr();
                if (rPrDefault == null)
                    rPrDefault = objectFactory.createRPr();
                if(rPrDefault!=null) {
                    CTLanguage ctLanguage = rPrDefault.getLang();
                    if (ctLanguage == null)
                        ctLanguage = objectFactory.createCTLanguage();

                    if (ctLanguage != null) {
                        String lang = ctLanguage.getVal();
                        if (lang == null || lang.length() == 0) {
                            if (langStr != null && langStr.length() > 0)
                                ctLanguage.setVal(langStr);
                            else
                                ctLanguage.setVal("en-US"); // default according to MS
                        }
                        lang = ctLanguage.getEastAsia();
                        if (lang == null || lang.length() == 0)
                            ctLanguage.setEastAsia("en-US"); // default according to MS
                        lang = ctLanguage.getBidi();
                        if (lang == null || lang.length() == 0)
                            ctLanguage.setBidi("ar-SA"); // default according to MS
                        }
                    }
                }
            }
        }
    }

