/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package org.xlsx4j.sml;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;

public final class SmlUtils {

    public static class CellRefRange {
        private final CellRef start;
        private final CellRef end;

        public CellRefRange(CellRef start, CellRef end) {
            this.start = start;
            this.end = end;
        }
        public CellRef getStart() {
            return start;
        }
        public CellRef getEnd() {
            return end;
        }
        public boolean equals(CellRefRange cellRefRange) {
            return start.equals(cellRefRange.getStart())&&end.equals(cellRefRange.getEnd());
        }
        public boolean isInside(int column, int row) {
            return column>=Math.min(start.getColumn(), end.getColumn())&&column<=Math.max(start.getColumn(), end.getColumn())
                && row>=Math.min(start.getRow(), end.getRow())&&row<=Math.max(start.getRow(), end.getRow());
        }
        public boolean intersects(CellRefRange cellRefRange) {

            final int left  = Math.max( start.getColumn(), cellRefRange.getStart().getColumn() );
            final int right = Math.min( end.getColumn(), cellRefRange.getEnd().getColumn() );
            final int top   = Math.max( start.getRow(), cellRefRange.getStart().getRow() );
            final int bottom= Math.min( end.getRow(), cellRefRange.getEnd().getRow() );
            return (right<left||bottom<top) ? false : true;
        }
        public static CellRefRange insertRowRange(CellRefRange cellRefRange, int startRow, int count, boolean append) {
            final CellRef start = cellRefRange.getStart();
            final CellRef end = cellRefRange.getEnd();
            int endRow = end.getRow();
            if(append) {
                endRow++;
            }
            if(startRow<=start.getRow()) {
                return new CellRefRange(new CellRef(start.getColumn(), start.getRow()+count), new CellRef(end.getColumn(), end.getRow()+count));
            }
            else if(startRow<=endRow) {
                return new CellRefRange(new CellRef(start.getColumn(), start.getRow()), new CellRef(end.getColumn(), end.getRow()+count));
            }
            return new CellRefRange(new CellRef(start.getColumn(), start.getRow()), new CellRef(end.getColumn(), end.getRow()));
        }
        public static CellRefRange deleteRowRange(CellRefRange cellRef, int startRow, int endRow) {
            int row1 = cellRef.getStart().getRow();
            int row2 = cellRef.getEnd().getRow();
            final int count = (endRow-startRow)+1;
            if(startRow<=row2) {
                if(startRow<=row1&&endRow>=row2) {
                    return null;                                // merged cell is completely removed
                }
                else if(endRow<row1) {
                    row1 -= count;                              // merged cell is completely moved up by count
                    row2 -= count;
                }
                else if(startRow>row1&&endRow<=row2) {
                    row2 -= count;                              // merged cell is shortened by delete count
                }
                else if(startRow>row1) {
                    row2 = startRow-1;                          // new end is to be set
                }
                else {
                    row1 = startRow;                            // new start is to be set
                    row2 -= count;
                }
            }
            return new CellRefRange(new CellRef(cellRef.getStart().getColumn(), row1), new CellRef(cellRef.getEnd().getColumn(), row2));
        }
        public static CellRefRange insertColumnRange(CellRefRange cellRefRange, int startColumn, int count, boolean append) {
            final CellRef start = cellRefRange.getStart();
            final CellRef end = cellRefRange.getEnd();
            int endColumn = end.getColumn();
            if(append) {
                endColumn++;
            }
            if(startColumn<=start.getColumn()) {
                return new CellRefRange(new CellRef(start.getColumn()+count, start.getRow()), new CellRef(end.getColumn()+count, end.getRow()));
            }
            else if(startColumn<=endColumn) {
                return new CellRefRange(new CellRef(start.getColumn(), start.getRow()), new CellRef(end.getColumn()+count, end.getRow()));
            }
            return new CellRefRange(new CellRef(start.getColumn(), start.getRow()), new CellRef(end.getColumn(), end.getRow()));
        }
        public static CellRefRange deleteColumnRange(CellRefRange cellRef, int startColumn, int endColumn) {
            int column1 = cellRef.getStart().getColumn();
            int column2 = cellRef.getEnd().getColumn();
            final int count = (endColumn-startColumn)+1;
            if(startColumn<=column2) {
                if(startColumn<=column1&&endColumn>=column2) {
                    return null;                                // merged cell is completely removed
                }
                else if(endColumn<column1) {
                    column1 -= count;                           // merged cell is completely moved up by count
                    column2 -= count;
                }
                else if(startColumn>column1&&endColumn<=column2) {
                    column2 -= count;                           // merged cell is shortened by delete count
                }
                else if(startColumn>column1) {
                    column2 = startColumn-1;                    // new end is to be set
                }
                else {
                    column1 = startColumn;                      // new start is to be set
                    column2 -= count;
                }
            }
            return new CellRefRange(new CellRef(column1, cellRef.getStart().getRow()), new CellRef(column2, cellRef.getEnd().getRow()));
        }
    }

    public static class CellRef {

        private int column;
        private int row;

        public CellRef(int column, int row) {
            this.column = column;
            this.row = row;
        }
        public int getColumn() {
            return column;
        }
        public int getRow() {
            return row;
        }
        public void setColumn(int column) {
            this.column = column;
        }
        public void setRow(int row) {
            this.row = row;
        }
        public boolean equals(CellRef cellRef) {
            return this.column==cellRef.column&&this.row==cellRef.row;
        }
        public final JSONArray getJSONArray() {
            final JSONArray jsonArray = new JSONArray();
            jsonArray.put(column);
            jsonArray.put(row);
            return jsonArray;
        }
    }

    public static CellRefRange createCellRefRange(String cellRefRange) {

        if(cellRefRange==null)
            return null;
        final CellRef start = createCellRef(cellRefRange);
        if(start==null)
            return null;
        int indexOfSeparator = cellRefRange.indexOf(':');
        if(indexOfSeparator==-1) {
            return new CellRefRange(start, new CellRef(start.getColumn(), start.getRow()));
        }
        if(indexOfSeparator<2||(indexOfSeparator+1)>=cellRefRange.length())
            return null;
        final CellRef end = createCellRef(cellRefRange.substring(indexOfSeparator+1));
        if(end==null)
            return null;
        return new CellRefRange(start, end);
    }

    public static CellRefRange createCellRefRangeWithSheet(String cellRefRange) {
        final int indexOf = cellRefRange.lastIndexOf("!");
        final String range = cellRefRange.substring(indexOf + 1);
        return createCellRefRange(range);
    }

    /**
     * translates string CellRef like "B23" into a computable form : 1, 22
     *
     * returns null on error or missing cellRef
     *
     */
    public static CellRef createCellRef(String cellRef) {

        if(cellRef==null||cellRef.isEmpty())
            return null;

        CellRef cellPosition = null;

        int characters = 0;
        int characterStartIndex = cellRef.charAt(0) == '$' ? 1 : 0;

        while((characters+characterStartIndex)<cellRef.length()) {
            char character = cellRef.charAt(characterStartIndex+characters);
            if((character<'A')||(character>'Z')) {
                break;
            }
            characters++;
        }
        if(characters>0&&(characters+characterStartIndex)<cellRef.length()&&characters<6) {

            final int[] exMap = { 1, 26, 676, 17576, 456976 };

            int ex = characters - 1;
            int x = 0;

            for(int i = 0; i < characters; i++) {
                x +=((cellRef.charAt(i+characterStartIndex)-'A')+1)*exMap[ex-i];
            }

            int numbers = 0;
            int numberStartIndex = characterStartIndex + characters;

            if(cellRef.charAt(numberStartIndex) == '$') {
                numberStartIndex++;
            }
            while(numbers+numberStartIndex<cellRef.length()) {
                char character = cellRef.charAt(numberStartIndex+numbers);
                if((character<'0')||(character>'9')) {
                    break;
                }
                numbers++;
            }
            if(numbers>0) {
                int y = 0;
                for(int i = 0; i < numbers; i++) {
                    y*=10;
                    y +=cellRef.charAt(i+numberStartIndex)-'0';
                }
                if(y>0) {
                    cellPosition = new CellRef(x-1, y-1);
                }
            }
        }
        return cellPosition;
    }

    private static void addCellRef(StringBuilder buffer, int column, long row) {
        addCellRefColumn(buffer, column);
        addCellRefRow(buffer, row);
    }

    private static void addCellRefColumn(StringBuilder buffer, int column) {

        int pos = buffer.length();
        while(column>=0) {
            buffer.insert(pos, (char)('A' + (column % 26)));
            column /= 26;
            column--;
        }
    }

    private static void addCellRefRow(StringBuilder buffer, long row) {
        buffer.append(Long.toString(row+1));
    }

    public static String getCellRef(CellRef cellRef) {
        return getCellRef(cellRef.getColumn(), cellRef.getRow());
    }
    public static String getCellRef(int column, long row) {
        StringBuilder buffer = new StringBuilder(8);
        addCellRef(buffer, column, row);
        return buffer.toString();
    }

    public static String getCellRefRange(CellRefRange cellRefRange) {
        return getCellRefRange(cellRefRange.getStart().getColumn(), cellRefRange.getStart().getRow(), cellRefRange.getEnd().getColumn(), cellRefRange.getEnd().getRow());
    }

    public static String getAbsoluteCellRefRange(CellRefRange cellRefRange) {
        return getAbsoluteCellRefRange(cellRefRange.getStart().getColumn(), cellRefRange.getStart().getRow(), cellRefRange.getEnd().getColumn(), cellRefRange.getEnd().getRow());
    }

    public static String getCellRefRange(int startColumn, long startRow, int endColumn, int endRow) {
        StringBuilder buffer = new StringBuilder(17);
        addCellRef(buffer, startColumn, startRow);
        buffer.append(':');
        addCellRef(buffer, endColumn, endRow);
        return buffer.toString();
    }
    public static String getAbsoluteCellRefRange(int startColumn, long startRow, int endColumn, int endRow) {
        StringBuilder buffer = new StringBuilder(21);
        buffer.append('$');
        addCellRefColumn(buffer, startColumn);
        buffer.append('$');
        addCellRefRow(buffer, startRow);
        buffer.append(':');
        buffer.append('$');
        addCellRefColumn(buffer, endColumn);
        buffer.append('$');
        addCellRefRow(buffer, endRow);
        return buffer.toString();
    }

    public static class CellRangeIterator implements Iterator<org.xlsx4j.sml.Cell> {

        public CellRangeIterator(int startColumn, int endColumn, CellIterator cellIterator) {
            this.startColumn = startColumn;
            this.endColumn = endColumn;
            this.cellIterator = cellIterator;
            this.nextCell = null;
        }

        @Override
        public org.xlsx4j.sml.Cell next() {
            if (nextCell==null) {
                nextCell = getNext();
            }
            final org.xlsx4j.sml.Cell cell = nextCell;
            nextCell = null;
            return cell;
        }

        @Override
        public boolean hasNext() {
            if (nextCell==null) {
                nextCell = getNext();
            }
            return nextCell!=null;
        }

        @Override
        public void remove() {
            cellIterator.remove();
        }

        private org.xlsx4j.sml.Cell getNext() {

            while(cellIterator.hasNext()) {
                final org.xlsx4j.sml.Cell cell = cellIterator.next();
                final int currentColumnNumber = cellIterator.getColumnNumber();
                if(currentColumnNumber>endColumn) {
                    return null;
                }
                else if (currentColumnNumber>=startColumn) {
                    return cell;
                }
            }
            return null;
        }

        private org.xlsx4j.sml.Cell nextCell;

        private final int startColumn;

        private final int endColumn;

        private final CellIterator cellIterator;
    }

    public static class CellIterator implements Iterator<org.xlsx4j.sml.Cell> {

        public CellIterator(List<org.xlsx4j.sml.Cell> list) {
            this.list = list;
        }
        public CellIterator(List<org.xlsx4j.sml.Cell> list, Map<Integer, Cell> cellMap) {
            this.list = list;
            this.cellMap = cellMap;
            removeSupported = true;
        }
        public int getColumnNumber() {
            return columnNumber;
        }
        public int getListIndex() {
            return listIndex;
        }
        @Override
        public org.xlsx4j.sml.Cell next() {
            cell = list.get(++listIndex);
            final String ref = cell.getR();
            if(ref==null) {
                columnNumber++;
            }
            else {
                final CellRef cr = createCellRef(ref);
                int index = cr.getColumn();
                if(index<columnNumber) {
                    throw new RuntimeException("XLSX: CellIterator, invalid cell index found!!");
                }
                columnNumber = index;
            }
            return cell;
        }
        @Override
        public boolean hasNext() {
            return (listIndex+1) < list.size();
        }
        @Override
        public void remove() {
            if(!removeSupported||cell==null) {
                throw new UnsupportedOperationException();
            }
            if(cellMap!=null) {
                cellMap.remove(columnNumber);
            }
            list.remove(listIndex--);
            columnNumber--;
        }

        private final List<org.xlsx4j.sml.Cell> list;

        private Map<Integer, Cell> cellMap = null;

        private org.xlsx4j.sml.Cell cell = null;

        private boolean removeSupported = false;

        private int columnNumber = -1;

        private int listIndex = -1;
    }

    public static class RowRangeIterator implements Iterator<org.xlsx4j.sml.Row> {

        public RowRangeIterator(int startRow, int endRow, RowIterator rowIterator) {
            this.startRow = startRow;
            this.endRow = endRow;
            this.rowIterator = rowIterator;
            this.nextRow = null;
        }

        @Override
        public org.xlsx4j.sml.Row next() {
            if (nextRow==null) {
                nextRow = getNext();
            }
            final org.xlsx4j.sml.Row row = nextRow;
            nextRow = null;
            return row;
        }

        @Override
        public boolean hasNext() {
            if (nextRow==null) {
                nextRow = getNext();
            }
            return nextRow!=null;
        }

        @Override
        public void remove() {
            rowIterator.remove();
        }

        private org.xlsx4j.sml.Row getNext() {

            while(rowIterator.hasNext()) {
                final org.xlsx4j.sml.Row row = rowIterator.next();
                final int currentRowNumber = rowIterator.getRowNumber();
                if(currentRowNumber>endRow) {
                    return null;
                }
                else if (currentRowNumber>=startRow) {
                    return row;
                }
            }
            return null;
        }

        private org.xlsx4j.sml.Row nextRow;

        private final int startRow;

        private final int endRow;

        private final RowIterator rowIterator;
    }

    public static class RowIterator implements Iterator<org.xlsx4j.sml.Row> {

        public RowIterator(List<org.xlsx4j.sml.Row> list) {
            this.list = list;
        }
        public RowIterator(List<org.xlsx4j.sml.Row> list, Map<Integer, Row> rowMap) {
            this.list = list;
            this.rowMap = rowMap;
            removeSupported = true;
        }
        public int getRowNumber() {
            return rowNumber;
        }
        public int getListIndex() {
            return listIndex;
        }
        @Override
        public org.xlsx4j.sml.Row next() {
            row = list.get(++listIndex);
            final Long r = row.getR();
            if(r==null) {
                rowNumber++;
            }
            else {
                int index = r.intValue() - 1;
                if(index<0||index<rowNumber) {
                    throw new RuntimeException("XLSX: RowIterator, invalid row index found!!");
                }
                rowNumber = index;
            }
            return row;
        }
        @Override
        public boolean hasNext() {
            return (listIndex+1) < list.size();
        }
        @Override
        public void remove() {
            if(!removeSupported||row==null) {
                throw new UnsupportedOperationException();
            }
            if(rowMap!=null) {
                rowMap.remove(rowNumber);
            }
            list.remove(listIndex--);
            rowNumber--;
        }

        private final List<org.xlsx4j.sml.Row> list;

        private Map<Integer, Row> rowMap = null;

        private org.xlsx4j.sml.Row row = null;

        boolean removeSupported = false;

        private int rowNumber = -1;

        private int listIndex = -1;
    }
}
