/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
public class CalcEngineClientSimulator implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientSimulator.class);

    //-------------------------------------------------------------------------
    public static final String OPERATION_4_CRASH    = "{crash:true}";
    public static final String OPERATION_4_LOST_DOC = "{doc:lost-in-space}";

    //-------------------------------------------------------------------------
    private CalcEngineClientSimulator ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientSimulator create()
        throws Exception
    {
    	return new CalcEngineClientSimulator ();
    }

    //-------------------------------------------------------------------------
    @Override
    public String createDocument()
        throws Exception
    {
    	INFO("created new document ...");

    	final Map< String, Document > lDocs   = mem_Documents ();
    	final String                  sHandle = CalcEngineHandleGenerator.newHandle();
    	final Document                aDoc    = new Document ();
    	
    	aDoc.sHandle = sHandle;
        lDocs.put (sHandle, aDoc);

    	INFO("... new document with handle ["+sHandle+"] created.");
        return sHandle;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sHandle)
        throws Exception
    {
        INFO("destroy document with handle ["+sHandle+"] ...");

        if (
        	( ! impl_isHandleValid (sHandle)) ||
        	( ! impl_isHandleKnown (sHandle))
           )
        	return;
        
        final Map< String, Document > lDocs = mem_Documents ();
        lDocs.remove(sHandle);
    };

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sHandle       ,
                                   			 final String       sOperationJSON,
                                   			 final StringBuffer sResultJSON   )
        throws Exception
    {
        INFO("execute operation ( handle='"+sHandle+"', operations='"+sOperationJSON+"' ) ...");

        Validate.isTrue(impl_isHandleValid (sHandle), "Invalid handle.");
        
        if ( ! impl_isHandleKnown (sHandle))
        	return ECalcEngineError.E_DOCUMENT_LOST;

        // special simulation mode : simulate a missing document ... even if its well known .-)
        
        if (impl_isSpecialOp(sOperationJSON, OPERATION_4_LOST_DOC))
        {
        	INFO ("... simulate 'lost doc' by return E_DOCUMENT_LOST !");
        	return ECalcEngineError.E_DOCUMENT_LOST;
        }
        
        final Map< String, Document > lDocs = mem_Documents ();
        final Document                aDoc  = lDocs.get(sHandle);
        
        aDoc.lOperations.append (sOperationJSON);

        // special simulation mode : simulate crash by throwing an exception
        
        if (impl_isSpecialOp(sOperationJSON, OPERATION_4_CRASH))
        	throw new Exception ("Crash was forced by special simulator operation.");
        
        // put the incoming JSON back as result for testing purposes ... might be an unit test rely on that .-)
        sResultJSON.setLength (0);
        sResultJSON.append    (sOperationJSON);

        return ECalcEngineError.E_NONE;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                 final String... lPagedOperationJSONs)
        throws Exception
    {
        final Map< String, Document > lDocs = mem_Documents ();
              Document                aDoc  = lDocs.get(sHandle);
        
        if (aDoc == null)
        {
        	aDoc = new Document ();
        	lDocs.put(sHandle, aDoc);
        }

        for (String sOperationJSON : lPagedOperationJSONs)
        	aDoc.lOperations.append (sOperationJSON);
        
        INFO("document with handle ["+sHandle+"] restored ...");
        return ECalcEngineError.E_NONE;
    }

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError copy(final String                   sSourceDocHandle,
								 final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
		throw new UnsupportedOperationException ("not implemented yet");
	}

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError paste(final String                   sTargetDocHandle,
								  final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
		throw new UnsupportedOperationException ("not implemented yet");
	}

    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
    {
        return "Calc Engine Simulator 1.0 :-)";
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
    	// not implemented real ...
    	// but hey - not useful for a simulator .-)
    }

    //-------------------------------------------------------------------------
    private static boolean impl_isSpecialOp (final String sOp     ,
    										 final String sSpecial)
        throws Exception
    {
    	// we use contains() instead of equals to be prepared for quoting .-)
    	final boolean bIs = StringUtils.contains(sOp, sSpecial);
    	return bIs;
    }
    
    //-------------------------------------------------------------------------
    private boolean impl_isHandleValid (final String sHandle)
        throws Exception
    {
        return CalcEngineHandleGenerator.validateHandle(sHandle);
    }

    //-------------------------------------------------------------------------
    private boolean impl_isHandleKnown (final String sHandle)
        throws Exception
    {
        final Map< String, Document > lDocs = mem_Documents ();
        return lDocs.containsKey(sHandle);
    }

    //-------------------------------------------------------------------------
    private static void INFO (final String sMessage)
        throws Exception
    {
    	//System.out.println (sMessage);
    	LOG.info(sMessage);
    }
    
    //-------------------------------------------------------------------------
    private class Document
    {
    	@SuppressWarnings("unused")
		public String       sHandle     = null;
    	public StringBuffer lOperations = new StringBuffer (256);
    }
    
    //-------------------------------------------------------------------------
    private Map< String, Document > mem_Documents ()
    	throws Exception
    {
    	if (m_lDocuments == null)
    		m_lDocuments = new HashMap< String, Document > ();
    	return m_lDocuments;
    }
    
    //-------------------------------------------------------------------------
    private Map< String, Document > m_lDocuments = null;
}
