package com.openexchange.office.tools;

import org.apache.commons.logging.Log;
import org.json.JSONObject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.jslob.JSlob;
import com.openexchange.jslob.JSlobService;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;


/**
 * A helper class to ease the use of properties settings value
 * retrieval.
 *
 * {@link ConfigurationHelper}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class ConfigurationHelper {

    /**
     * Retrieves a office.properties boolean value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
    static public Boolean getBooleanOfficeConfigurationValue(ServiceLookup serviceLookup, Session session, String settingPath, Boolean defaultValue) {
        String strValue = null;

        strValue = ConfigurationHelper.getOfficeConfigurationValue(serviceLookup, session, settingPath, null);
        if (null == strValue) {
            return defaultValue;
        } else {
            return Boolean.parseBoolean(strValue);
        }
    }

    /**
     * Retrieves a office.properties integer value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
    static public Integer getIntegerOfficeConfigurationValue(ServiceLookup serviceLookup, Session session, String settingPath, Integer defaultValue) {
        String strValue = null;

        strValue = ConfigurationHelper.getOfficeConfigurationValue(serviceLookup, session, settingPath, null);
        if (null == strValue) {
            return defaultValue;
        } else {
            return Integer.parseInt(strValue);
        }
    }

    /**
     * Retrieves a office.properties string value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The string value of the setting or defaultValue if the real value
     *  cannot be retrieved from the configuration.
     */
    static public String getStringOfficeConfigurationValue(ServiceLookup serviceLookup, Session session, String settingPath, String defaultValue) {
        return getOfficeConfigurationValue(serviceLookup, session, settingPath, defaultValue);
    }

    /**
     * Retrieves a office.properties value using the cascade configuration
     * that supports to overwrite settings for a user. A default value can
     * be provided which is returned if the value cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
     static public String getOfficeConfigurationValue(ServiceLookup serviceLookup, Session session, String settingPath, String defaultValue) {
         String result = defaultValue;

         result = ConfigurationHelper.getCascadeConfigurationValue(serviceLookup, session, "io.ox/office", settingPath);
         if (null == result) {
             result = defaultValue;
         }

        return result;
     }

    /**
     * Retrieves a properties value using the cascade configuration
     * that supports to overwrite settings for a user.
     *
     * @param jslobService
     *  The jslob service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval. If the session is null so the user
     *  is unknown, the method uses the configuration service as fallback.
     *
     * @param configuration
     *  The name of the configuration file to used for the retrieval,
     *  e.g. io.ox/office
     *
     * @param path
     *  The path within the configuration file to retrieve the setting
     *  value, e.g. //module/maxTableCells
     *
     * @return
     *  The value of the setting or null if the setting is not available.
     */
     static public String getCascadeConfigurationValue(ServiceLookup serviceLookup, Session session, String configuration, String path) {
        String result = null;

        if (null != serviceLookup && null != configuration && null != path) {

            final JSlobService jslobService = serviceLookup.getService(JSlobService.class);
            if ((null != session) && (null != jslobService)) {
                result = getCascadeConfigurationValue(jslobService, session, configuration, path);
            } else {
                String configurationFullPath = configuration + path;
                final ConfigurationService configService = serviceLookup.getService(com.openexchange.config.ConfigurationService.class);
                if (null != configService) {
                    result = configService.getProperty(configurationFullPath, (String)null);
                }
            }
        }

        return result;
    }

    /**
     * Retrieves a properties value using the cascade configuration
     * that supports to overwrite settings for a user.
     *
     * @param jslobService
     *  The jslob service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param configuration
     *  The name of the configuration file to used for the retrieval,
     *  e.g. io.ox/office
     *
     * @param path
     *  The path within the configuration file to retrieve the setting
     *  value, e.g. //module/maxTableCells
     *
     * @return
     *  The value of the setting or null if the setting is not available.
     */
    static private String getCascadeConfigurationValue(JSlobService jslobService, Session session, String configuration, String path) {
        String result = null;

        if ((null != session) && (null != jslobService)) {
            try {
                final JSlob officeJSlob = jslobService.get(configuration, session);
                JSONObject json = null;

                if (null != officeJSlob) {
                    json = officeJSlob.getJsonObject();
                }

                // cascade through the hierarchical structure to find the correct setting
                if (null != json) {
                    int i = 0;
                    String[] entries = path.split("/");
                    while (i < entries.length) {
                        String entry = entries[i++];
                        // ignore empty strings
                        if (entry.length() > 0) {
                            Object obj = json.opt(entry);
                            if (null == obj) {
                                return null;
                            } else if (obj instanceof JSONObject) {
                                // step down on JSONObject
                                json = (JSONObject)obj;
                            } else {
                                // use other object types as return value
                                result = String.valueOf(obj);
                                break;
                            }
                        }
                    }
                }
            } catch (OXException e) {
                LOG.error("Exception catched while retrieving cascade configuration value", e);
            }
        }

        return result;
    }

    private static final Log LOG = com.openexchange.log.Log.loggerFor(ConfigurationHelper.class);
}
