/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.util.HashMap;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.bind.annotation.XmlElements;

import org.docx4j.dml.CTEffectList;
import org.docx4j.dml.CTLineJoinRound;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTNoFillProperties;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.STLineCap;
import org.docx4j.dml.TextFont;
import org.docx4j.dml.chart.CTArea3DChart;
import org.docx4j.dml.chart.CTAreaChart;
import org.docx4j.dml.chart.CTAxDataSource;
import org.docx4j.dml.chart.CTAxPos;
import org.docx4j.dml.chart.CTBar3DChart;
import org.docx4j.dml.chart.CTBarChart;
import org.docx4j.dml.chart.CTBarDir;
import org.docx4j.dml.chart.CTBoolean;
import org.docx4j.dml.chart.CTBubbleChart;
import org.docx4j.dml.chart.CTBubbleSer;
import org.docx4j.dml.chart.CTCatAx;
import org.docx4j.dml.chart.CTChart;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTDLbls;
import org.docx4j.dml.chart.CTDPt;
import org.docx4j.dml.chart.CTDoughnutChart;
import org.docx4j.dml.chart.CTFirstSliceAng;
import org.docx4j.dml.chart.CTGapAmount;
import org.docx4j.dml.chart.CTHoleSize;
import org.docx4j.dml.chart.CTLegend;
import org.docx4j.dml.chart.CTLegendPos;
import org.docx4j.dml.chart.CTLine3DChart;
import org.docx4j.dml.chart.CTLineChart;
import org.docx4j.dml.chart.CTLineSer;
import org.docx4j.dml.chart.CTMarker;
import org.docx4j.dml.chart.CTNumData;
import org.docx4j.dml.chart.CTNumDataSource;
import org.docx4j.dml.chart.CTNumRef;
import org.docx4j.dml.chart.CTNumVal;
import org.docx4j.dml.chart.CTOfPieChart;
import org.docx4j.dml.chart.CTOrientation;
import org.docx4j.dml.chart.CTOverlap;
import org.docx4j.dml.chart.CTPie3DChart;
import org.docx4j.dml.chart.CTPieChart;
import org.docx4j.dml.chart.CTPlotArea;
import org.docx4j.dml.chart.CTRadarChart;
import org.docx4j.dml.chart.CTScaling;
import org.docx4j.dml.chart.CTScatterChart;
import org.docx4j.dml.chart.CTScatterSer;
import org.docx4j.dml.chart.CTScatterStyle;
import org.docx4j.dml.chart.CTSerTx;
import org.docx4j.dml.chart.CTStockChart;
import org.docx4j.dml.chart.CTStrData;
import org.docx4j.dml.chart.CTStrRef;
import org.docx4j.dml.chart.CTStrVal;
import org.docx4j.dml.chart.CTStyle;
import org.docx4j.dml.chart.CTSurface3DChart;
import org.docx4j.dml.chart.CTSurfaceChart;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.chart.CTUnsignedInt;
import org.docx4j.dml.chart.CTValAx;
import org.docx4j.dml.chart.IAxisDescription;
import org.docx4j.dml.chart.IListSer;
import org.docx4j.dml.chart.ISerContent;
import org.docx4j.dml.chart.ObjectFactory;
import org.docx4j.dml.chart.STAxPos;
import org.docx4j.dml.chart.STLegendPos;
import org.docx4j.dml.chart.STOrientation;
import org.docx4j.dml.chart.STScatterStyle;
import org.docx4j.jaxb.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.tools.Commons;

final public class DMLChartSpace {

	public static void createChartSpaceOperations(JSONArray operationsArray, JSONArray position, CTChartSpace chartSpace) throws JSONException {

		final CTChart chart = chartSpace.getChart();
		if (chart == null) {
			return;
		}
		final CTPlotArea plotArea = chart.getPlotArea();
		if (plotArea != null) {
            final List<IListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
			// TODO: what if there is more than one chart in this list ?
			if (!chartList.isEmpty()) {
                final IListSer chartSer = chartList.get(0);

				final CTBoolean varyColors = chartSer.getVaryColors();

				final JSONObject insertDrawingOperation = operationsArray.getJSONObject(operationsArray.length() - 1);
				boolean valueLabel = false;
				boolean curved = false;

				// we are taking only the first chart ...
				for (int seriesIndex = 0; seriesIndex < chartSer.getSer().size(); seriesIndex++) {
					final ISerContent ser = chartSer.getSer().get(seriesIndex);

					final CTDLbls lbl = ser.getDLbls();
					if (lbl != null && lbl.getShowVal() != null && lbl.getShowVal().isVal()) {
						valueLabel = true;
					}

					// title
					final JSONObject series = new JSONObject();
					final CTSerTx tx = ser.getTx();
					Object title = null;
					if (tx != null) {
						if (tx.getStrRef() != null) {
							title = tx.getStrRef().getF();
                        }
                        else {
							title = new JSONArray();
							((JSONArray) title).put(tx.getV());
						}
					}
					if (title != null) {
						series.put("title", title);
					}

					// names
					Object names = null;
					final CTAxDataSource cat = ser.getCat();
					if (cat != null) {
						if (cat.getStrRef() != null) {
							names = cat.getStrRef().getF();
						}
						else if (cat.getStrLit() != null) {
							final CTStrData strLit = cat.getStrLit();
							final List<CTStrVal> strLitPtList = strLit.getPt();
							if (strLitPtList != null && !strLitPtList.isEmpty()) {
								names = new JSONArray();
								for (final CTStrVal strLitPtEntry : strLitPtList) {
									if (strLitPtEntry != null) {
										((JSONArray) names).put(strLitPtEntry.getV());
                                    }
                                    else {
										((JSONArray) names).put((String) null);
									}
								}
							}
						}
						else if (cat.getNumRef() != null) {
							names = cat.getNumRef().getF();
						}
						else if (cat.getNumLit() != null) {
							final CTNumData numLit = cat.getNumLit();
							final List<CTNumVal> numLitPtList = numLit.getPt();
							if (numLitPtList != null && !numLitPtList.isEmpty()) {
								names = new JSONArray();
								for (final CTNumVal numLitPtEntry : numLitPtList) {
									if (numLitPtEntry != null) {
										((JSONArray) names).put(numLitPtEntry.getV());
									}
									else {
										((JSONArray) names).put((String) null);
									}
								}
							}
						}
						else if (cat.getMultiLvlStrRef() != null) {
							names = cat.getMultiLvlStrRef().getF();
						}
					}
					if (names != null) {
						series.put("names", names);
					}

					// values
					final CTNumDataSource numDataSource = ser.getVal();
					final Object values = getData(numDataSource);

					if (values != null) {
						series.put("values", values);
					}

					if (ser instanceof CTBubbleSer) {
						final CTBubbleSer serBB = (CTBubbleSer) ser;

						final CTNumDataSource bubblesizes = serBB.getBubbleSize();
						if (bubblesizes != null) {
							final Object bubbles = getData(bubblesizes);
							if (bubbles != null) {
								series.put("bubbles", bubbles);
							}
						}
					} else if (ser instanceof CTLineSer) {
						final CTBoolean smooth = ((CTLineSer) ser).getSmooth();
						//no smooth in line chart means it is activated!!!
						if (smooth == null || smooth.isVal()) {
							curved = true;
						}
					} else if (ser instanceof CTScatterSer) {
						final CTBoolean smooth = ((CTScatterSer) ser).getSmooth();
						//no smooth in line chart means it is activated!!!
						if (smooth == null || smooth.isVal()) {
							curved = true;
						}
					}
					final CTMarker marker = ser.getMarker();
					if (marker != null && marker.getSymbol() != null) {
						series.put("marker", marker.getSymbol().getVal().value());
					}
					final JSONObject attrs = Commons.surroundJSONObject("series", series);
                    DMLHelper.createJsonFromChartShapeProperties(attrs, ser.getSpPr());
					if (series.length() > 0) {
						final JSONObject insertChartDataSeriesOperation = new JSONObject();
						insertChartDataSeriesOperation.put("name", "insertChartDataSeries");
						insertChartDataSeriesOperation.put("start", position);
						insertChartDataSeriesOperation.put("series", seriesIndex);
						insertChartDataSeriesOperation.put("attrs", attrs);
						operationsArray.put(insertChartDataSeriesOperation);
					}

				}

				if (curved || valueLabel || varyColors != null || chartSer instanceof CTPieChart) {
					final JSONObject attrs = insertDrawingOperation.getJSONObject("attrs");
					final JSONObject chartProperties = attrs.getJSONObject("chart");

					if (chartSer instanceof CTPieChart) {
						chartProperties.put("rotation", ((CTPieChart) chartSer).getFirstSliceAng().getVal());
					}

					if (varyColors != null) {
						chartProperties.put("varyColors", varyColors.isVal());
					}

					if (valueLabel) {
						chartProperties.put("dataLabel", true);
					}

					if (curved) {
						chartProperties.put("curved", true);
					}
				}
			}
			DMLHelper.addToOperations(plotArea, operationsArray, position);

			final CTTitle title = chart.getTitle();
			final JSONObject attrs = DMLHelper.createJsonFromProperties(title);
			if (attrs != null) {
				final JSONObject op = new JSONObject();
				op.put("name", "setChartTitleAttributes");
				op.put("start", position);
				op.put("axis", "main");
				op.put("attrs", attrs);
				operationsArray.put(op);
			}

			final JSONObject legendAttrs = createLegendAttrs(chart);
			if (legendAttrs != null) {
				final JSONObject op = new JSONObject();
				op.put("name", "setChartLegendAttributes");
				op.put("start", position);
				op.put("attrs", legendAttrs);
				operationsArray.put(op);
			}
		}
	}

	private static JSONObject createLegendAttrs(CTChart chart) throws JSONException {
		final JSONObject ljset = new JSONObject();
		final JSONObject lj = new JSONObject();
		lj.put("pos", "off");

		final CTLegend legend = chart.getLegend();
		if (legend != null) {
			if (legend.getLegendPos() != null && legend.getLegendPos().getVal() != null) {

				final String pos;
				switch (legend.getLegendPos().getVal()) {
				case R:
					pos = "right";
					break;
				case T:
					pos = "top";
					break;
				case B:
					pos = "bottom";
					break;
				case TR:
					pos = "topright";
					break;
				default:
					pos = "left";
					break;
				}

				lj.put("pos", pos);

				final JSONObject character = createJsonFromProperties(legend.getTxPr());
				if (character != null) {
					ljset.put("character", character);
				}
			}
		}
		ljset.put("legend", lj);

		return ljset;
	}

	private static Object getData(CTNumDataSource numDataSource) {
		if (numDataSource == null) {
			Logger.getAnonymousLogger().warning("CTNumDataSource is null");
			return null;
		}
		if (numDataSource.getNumRef() != null) {
			return numDataSource.getNumRef().getF();
		} else if (numDataSource.getNumLit() != null) {
			final CTNumData numLit = numDataSource.getNumLit();
			if (numLit != null) {
				final List<CTNumVal> numLitPtList = numLit.getPt();
				if (numLitPtList != null && !numLitPtList.isEmpty()) {
					final JSONArray values = new JSONArray();
					for (final CTNumVal numLitPt : numLitPtList) {
						if (numLitPt != null) {
							values.put(numLitPt.getV()); // TODO: getIdx
						} else {
							values.put((String) null);
						}
					}
					return values;
				}
			}
		}
		return null;
	}

	public static void applyChartSpaceProperties(JSONObject chartProperties, CTChartSpace chartSpace, JSONObject fill) throws Exception {
		final CTPlotArea plotArea = chartSpace.getChart().getPlotArea();

		if (chartProperties.has("type")) {
			final IListSer oldListSer = plotArea.getAreaChartOrArea3DChartOrLineChart().get(0);
			final IListSer newListSer = createChartType(chartProperties);

			final Class<? extends ISerContent> type = getNewContentType(newListSer);

			if (newListSer != null) {
				final List<CTUnsignedInt> axid = newListSer.getAxId();
				final List<CTUnsignedInt> oldAxid = oldListSer.getAxId();
				if (axid != null && oldAxid != null) {
					axid.clear();
					axid.addAll(oldAxid);
				} else if (axid == null) {
					final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
					if (axes != null) {
						axes.clear();
					}
				} else {
					final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
					axes.add(createCatAx());
					axes.add(createValAx());
				}
				plotArea.getAreaChartOrArea3DChartOrLineChart().set(0, newListSer);
				
				final CTBoolean smooth = new CTBoolean();
				smooth.setVal(false);
				
				for (final ISerContent serContent : oldListSer.getSer()) {
					final ISerContent newContent;
					if (serContent.getClass().equals(type)) {
						newContent = serContent;
					} else {
						newContent = type.newInstance();
						newContent.setCat(serContent.getCat());
						newContent.setExtLst(serContent.getExtLst());
						newContent.setIdx(serContent.getIdx());
						newContent.setOrder(serContent.getOrder());
						newContent.setSpPr(serContent.getSpPr());
						newContent.setTx(serContent.getTx());
						newContent.setVal(serContent.getVal());
						newContent.setDLbls(serContent.getDLbls());
						
						if (newContent instanceof CTDoughnutChart) {
							handleStandardDonutChart((CTDoughnutChart) newContent, chartProperties);
						} else if (newContent instanceof CTPieChart) {
							handleStandardPieChart((CTPieChart) newContent, chartProperties);
						} else if (newContent instanceof CTLineSer) {
							((CTLineSer) newContent).setSmooth(smooth);
						} else if (newContent instanceof CTScatterSer) {
							((CTScatterSer) newContent).setSmooth(smooth);
						}

					}
					newListSer.getSer().add(newContent);

				}
			}
			DMLHelper.refreshTitleRotations(plotArea);
		}

		final IListSer listSer = plotArea.getAreaChartOrArea3DChartOrLineChart().get(0);
		deleteEmptyContentLists(listSer);

		if (chartProperties.has("chartStyleId")) {
			CTStyle chartStyle = chartSpace.getStyle();
			if (chartStyle == null) {
				chartStyle = new CTStyle();
				chartSpace.setStyle(chartStyle);
			}
			chartSpace.setSpPr(null);
			chartStyle.setVal((short) chartProperties.getInt("chartStyleId"));

			// FIXME: this should come with the series-operation
			for (final ISerContent serContent : listSer.getSer()) {
				serContent.setSpPr(null);
				final List<CTDPt> dpt = serContent.getDPt();
				if (dpt != null) {
					for (final CTDPt point : serContent.getDPt()) {
						point.setSpPr(null);
					}
				}
			}
		}
		if (chartProperties.has("stacking")) {
			final String stacking = chartProperties.getString("stacking");
			listSer.setGrouping(stacking);
		}
		
		final String stacking = listSer.getStacking();
		if (listSer instanceof CTBarChart) {
			handleOverlap((CTBarChart) listSer, stacking);
		}
		
		if (chartProperties.has("curved")) {
			if (listSer instanceof CTLineChart || listSer instanceof CTScatterChart) {
				final CTBoolean smooth = new CTBoolean();
				smooth.setVal(chartProperties.getBoolean("curved"));
				for (final ISerContent serContent : listSer.getSer()) {
					if (serContent instanceof CTLineSer) {
						((CTLineSer) serContent).setSmooth(smooth);
					} else if (serContent instanceof CTScatterSer) {
						((CTScatterSer) serContent).setSmooth(smooth);
					}
				}
			}
		}
		if (chartProperties.has("varyColors")) {
			final CTBoolean vary = new CTBoolean();
			vary.setVal(chartProperties.getBoolean("varyColors"));
			listSer.setVaryColors(vary);
		}

		if (chartProperties.has("dataLabel")) {
			final CTDLbls lbl = new CTDLbls();
			lbl.setShowVal(getBoolean(chartProperties.getBoolean("dataLabel")));
			lbl.setShowLegendKey(getBoolean(false));
			lbl.setShowCatName(getBoolean(false));
			lbl.setShowSerName(getBoolean(false));
			lbl.setShowPercent(getBoolean(false));
			lbl.setShowBubbleSize(getBoolean(false));
			lbl.setShowLeaderLines(getBoolean(false));
			
			CTShapeProperties spr = lbl.getSpPr();
			if (spr == null) {
				spr = new CTShapeProperties();
				lbl.setSpPr(spr);
				spr.setNoFill(new CTNoFillProperties());
				final CTLineProperties ln = new CTLineProperties();
				ln.setNoFill(new CTNoFillProperties());
				spr.setLn(ln);
				
				spr.setEffectLst(new CTEffectList());
			}
			
			
						
			CTTextBody txPr = lbl.getTxPr();
			if(txPr==null){
				txPr = new CTTextBody();
				lbl.setTxPr(txPr);
				DMLHelper.handleStandardTypes(txPr);
			}

			final CTTextParagraphProperties prop = DMLHelper.getTextProp();
			final CTTextCharacterProperties def = prop.getDefRPr();

			def.setSz(1000);

			final List<CTTextParagraph> paras = txPr.getP();
			paras.clear();

			final CTTextParagraph p = new CTTextParagraph();
			p.setPPr(prop);
			paras.add(p);
			
			
			
			for (final ISerContent serContent : listSer.getSer()) {
				serContent.setDLbls(lbl);
			}
		}
	
		if (fill != null) {
			CTShapeProperties spr = plotArea.getSpPr();
			if(spr==null){
				spr = new CTShapeProperties();
				plotArea.setSpPr(spr);
			}
			
			final JSONObject fillHolder = new JSONObject();
			fillHolder.put("fill", fill);
			DMLHelper.createChartShapePropertiesFromJson(spr, fillHolder);
			chartSpace.setSpPr(spr);
			chartSpace.setRoundedCorners(getBoolean(false));
		}
		
		if (listSer instanceof CTScatterChart) {
			handleStandardScatterChart((CTScatterChart) listSer, chartProperties);

		}
	}

	private static CTBoolean getBoolean(boolean value){
		final CTBoolean res = new CTBoolean();
		res.setVal(value);
		return res;
	}

	public static void applyShapeProperties(JSONObject attrs, CTChartSpace chartSpace) throws Exception {
		final CTPlotArea plotArea = chartSpace.getChart().getPlotArea();
		CTShapeProperties sp = plotArea.getSpPr();
		if (sp == null) {
			sp = new CTShapeProperties();
			plotArea.setSpPr(sp);
		}
		DMLHelper.createChartShapePropertiesFromJson(sp, attrs);
	}

	public static void createShapeOperations(JSONObject attrs, JSONArray jsonStart, CTChartSpace chartSpace) throws JSONException {
		final CTPlotArea plotArea = chartSpace.getChart().getPlotArea();
		CTShapeProperties sp = plotArea.getSpPr();
		if (sp == null) {
			sp = new CTShapeProperties();
			plotArea.setSpPr(sp);
		}
		DMLHelper.createJsonFromChartShapeProperties(attrs, sp);
	}

	private static void deleteEmptyContentLists(IListSer listSer) {
		for (final ISerContent serContent : listSer.getSer()) {
			if (hasEmptyContents(serContent.getCat())) {
				serContent.setCat(null);
			}
			if (hasEmptyContents(serContent.getTx())) {
				serContent.setTx(null);
			}
			if (serContent instanceof CTBubbleSer) {
				final CTBubbleSer serBub = ((CTBubbleSer) serContent);
				if (hasEmptyContents(serBub.getBubbleSize())) {
					serBub.setBubbleSize(null);
				}
			}
		}
	}

	private static boolean hasEmptyContents(CTNumDataSource data) {
		if (data == null) {
			return true;
		}
		if (data.getNumLit() != null || data.getNumRef() != null) {
			return false;
		}
		return true;
	}

	private static boolean hasEmptyContents(CTAxDataSource data) {
		if (data == null) {
			return true;
		}
		if (data.getMultiLvlStrRef() != null || data.getNumLit() != null || data.getNumRef() != null || data.getStrLit() != null || data.getStrRef() != null) {
			return false;
		}
		return true;
	}

	private static boolean hasEmptyContents(CTSerTx data) {
		if (data == null) {
			return true;
		}
		if (data.getStrRef() != null || data.getV() != null) {
			return false;
		}
		return true;
	}

	private static Class<? extends ISerContent> getNewContentType(IListSer listSer) throws Exception {
		final XmlElements serType = listSer.getClass().getDeclaredField("ser").getAnnotation(XmlElements.class);
		return serType.value()[0].type();
	}

	private static CTUnsignedInt getUnsignedInt(int ptCount) {
		final CTUnsignedInt unsignedInt = Context.getDmlChartObjectFactory().createCTUnsignedInt();
		unsignedInt.setVal(ptCount);
		return unsignedInt;
	}

	private static String getRef(String sheetName, String ref) {
		if (ref.indexOf('!') == -1) {
			return sheetName + '!' + ref;
		}
		return ref;
	}

	// Idx and Order are required elements, so at least we have to create this
	// two elements for our new
	// inserted series... but we will correct indexes of other series, so that
	// they are properly increasing
	private static void updateIndexAndOrder(List<ISerContent> serContentList) {
		for (int i = 0; i < serContentList.size(); i++) {
			final ISerContent serCont = serContentList.get(i);
			serCont.setIdx(getUnsignedInt(i));
			serCont.setOrder(getUnsignedInt(i));
		}
	}
	
	public static void setDataSeriesAttributes(CTChartSpace chartSpace, String sheetName, int series, JSONObject attrs) throws JSONException {

		final IListSer listSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
		final List<ISerContent> serContentList = listSer.getSer();
		final ISerContent serContent = serContentList.get(series);
		setDataseriesAttrs(serContentList, serContent, sheetName, attrs);
		
	}
	
	public static void insertChartSpace(CTChartSpace chartSpace, String sheetName, int series, JSONObject attrs) throws JSONException {

		final IListSer listSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
		final List<ISerContent> serContentList = listSer.getSer();
		final ISerContent serContent = listSer.createSer();
		serContentList.add(series, serContent);
		setDataseriesAttrs(serContentList, serContent, sheetName, attrs);
		
	}

	private static void setDataseriesAttrs(List<ISerContent> serContentList, ISerContent serContent, String sheetName, JSONObject attrs) throws JSONException {

		

		if (attrs != null) {
			final JSONObject attrSeries = attrs.optJSONObject("series");
			if (attrSeries != null) {

				// title
				final Object title = attrSeries.opt("title");
				CTSerTx serTx = serContent.getTx();
				if (title != null)
				{
					serTx = Context.getDmlChartObjectFactory().createCTSerTx();
					if (title instanceof String)
					{
						final CTStrRef strRef = Context.getDmlChartObjectFactory().createCTStrRef();
						strRef.setF(getRef(sheetName, (String) title));
						serTx.setStrRef(strRef);
					}
					else if (title instanceof JSONArray && !((JSONArray) title).isEmpty())
					{
						final Object o = ((JSONArray) title).get(0);
						if (o instanceof String)
						{
							serTx.setV((String) o);
						}
					}
					else
					{
						Logger.getAnonymousLogger().warning("Cant handle Chart Title " + title);
						serTx = null;
					}
				}
				serContent.setTx(serTx);
				
				// names
				final Object names = attrSeries.opt("names");
				CTAxDataSource cat = serContent.getCat();
				if (names != null)
				{

					cat = Context.getDmlChartObjectFactory().createCTAxDataSource();
					if (names instanceof String)
					{
						final CTStrRef strRef = Context.getDmlChartObjectFactory().createCTStrRef();
						strRef.setF(getRef(sheetName, (String) names));
						cat.setStrRef(strRef);
					}
					else if (names instanceof JSONArray)
					{
						final JSONArray titleArray = (JSONArray) names;
						final CTStrData strData = Context.getDmlChartObjectFactory().createCTStrData();
						cat.setStrLit(strData);
						strData.setPtCount(getUnsignedInt(titleArray.length()));
						for (int i = 0; i < titleArray.length(); i++)
						{
							final CTStrVal strVal = Context.getDmlChartObjectFactory().createCTStrVal();
							strVal.setV(titleArray.optString(i, ""));
							strData.getPt().add(strVal);
						}
					}
					else
					{
						Logger.getAnonymousLogger().warning("Cant handle Chart Names " + names);
						cat = null;
					}
				}
				serContent.setCat(cat);

				// values
				CTNumDataSource val = serContent.getVal();
				final Object values = attrSeries.opt("values");
				if (values != null)
				{
					val = Context.getDmlChartObjectFactory().createCTNumDataSource();
					insertSeriesAttribute(sheetName, val, values);
				}
				serContent.setVal(val);
				

				// bubbles
				final Object bubbles = attrSeries.opt("bubbles");
				if (bubbles != null && serContent instanceof CTBubbleSer) {
					final CTNumDataSource bub = Context.getDmlChartObjectFactory().createCTNumDataSource();
					((CTBubbleSer) serContent).setBubbleSize(bub);

					insertSeriesAttribute(sheetName, bub, bubbles);
				}
				
				if (serContent instanceof CTScatterSer) {
					handleStandardScatterSer((CTScatterSer) serContent);
				}
			}
			updateIndexAndOrder(serContentList);
		}
	}

	private static void insertSeriesAttribute(String sheetName, CTNumDataSource source, Object attribute) {
		if (attribute instanceof String) {
			final CTNumRef numRef = Context.getDmlChartObjectFactory().createCTNumRef();
			numRef.setF(getRef(sheetName, (String) attribute));
			source.setNumRef(numRef);
		} else if (attribute instanceof JSONArray) {
			final JSONArray bubblesArray = (JSONArray) attribute;
			final CTNumData numData = Context.getDmlChartObjectFactory().createCTNumData();
			source.setNumLit(numData);
			numData.setPtCount(getUnsignedInt(bubblesArray.length()));
			for (int i = 0; i < bubblesArray.length(); i++) {
				final CTNumVal numVal = Context.getDmlChartObjectFactory().createCTNumVal();
				numVal.setIdx(i);
				numVal.setV(bubblesArray.optString(i, ""));
				numData.getPt().add(numVal);
			}
		}
	}

	public static void deleteChartSpace(CTChartSpace chartSpace, int series) {

		final IListSer listSer = chartSpace.getChart().getPlotArea().getAreaChartOrArea3DChartOrLineChart().get(0);
		final List<ISerContent> serContentList = listSer.getSer();
		serContentList.remove(series);
		updateIndexAndOrder(serContentList);
	}

	public static CTChartSpace createChartSpace(JSONObject chartAttrs) {
		final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
		final CTChartSpace chartSpace = chartObjectFactory.createCTChartSpace();
		final CTChart chart = chartObjectFactory.createCTChart();
		chartSpace.setChart(chart);
		final CTPlotArea plotArea = chartObjectFactory.createCTPlotArea();
		chart.setPlotArea(plotArea);
		final CTLegend legend = chartObjectFactory.createCTLegend();
		chart.setLegend(legend);
		final CTLegendPos legendPos = chartObjectFactory.createCTLegendPos();
		legend.setLegendPos(legendPos);
		legendPos.setVal(STLegendPos.R);
		final CTBoolean booleanValue = chartObjectFactory.createCTBoolean();
		booleanValue.setVal(true);
		chart.setPlotVisOnly(booleanValue);
		plotArea.getAreaChartOrArea3DChartOrLineChart().add(createChartType(chartAttrs));

		/*
		 * CTCatAx CTSerAx CTValAx CTDateAx
		 */
		final List<IAxisDescription> axList = plotArea.getValAxOrCatAxOrDateAx();
		axList.add(createCatAx());
		axList.add(createValAx());
		return chartSpace;
	}

	private static CTCatAx createCatAx() {
		final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
		final CTCatAx catAx = chartObjectFactory.createCTCatAx();
		final CTUnsignedInt axId = chartObjectFactory.createCTUnsignedInt();
		axId.setVal(0);
		catAx.setAxId(axId);
		final CTScaling scaling = chartObjectFactory.createCTScaling();
		final CTOrientation orientation = chartObjectFactory.createCTOrientation();
		orientation.setVal(STOrientation.MIN_MAX);
		scaling.setOrientation(orientation);
		catAx.setScaling(scaling);
		final CTAxPos axPos = chartObjectFactory.createCTAxPos();
		axPos.setVal(STAxPos.B);
		catAx.setAxPos(axPos);
		final CTUnsignedInt crossAx = chartObjectFactory.createCTUnsignedInt();
		crossAx.setVal(1);
		catAx.setCrossAx(crossAx);
		return catAx;
	}

	private static CTValAx createValAx() {
		final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
		final CTValAx valAx = chartObjectFactory.createCTValAx();
		final CTUnsignedInt axId = chartObjectFactory.createCTUnsignedInt();
		axId.setVal(1);
		valAx.setAxId(axId);
		final CTScaling scaling = chartObjectFactory.createCTScaling();
		final CTOrientation orientation = chartObjectFactory.createCTOrientation();
		orientation.setVal(STOrientation.MIN_MAX);
		scaling.setOrientation(orientation);
		valAx.setScaling(scaling);
		final CTAxPos axPos = chartObjectFactory.createCTAxPos();
		axPos.setVal(STAxPos.L);
		valAx.setAxPos(axPos);
		final CTUnsignedInt crossAx = chartObjectFactory.createCTUnsignedInt();
		crossAx.setVal(0);
		valAx.setCrossAx(crossAx);
		return valAx;
	}

	private static void createAxId(List<CTUnsignedInt> axId) {
		final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
		final CTUnsignedInt id0 = chartObjectFactory.createCTUnsignedInt();
		final CTUnsignedInt id1 = chartObjectFactory.createCTUnsignedInt();
		id0.setVal(0);
		id1.setVal(1);
		axId.add(id0);
		axId.add(id1);
	}

	// TODO: creator for pie3d needs to be implemented...

	private static IListSer createChartType(JSONObject chartAttrs) {

		IListSer listSer = null;
		final ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
		final String chartType = chartAttrs != null ? chartAttrs.optString("type", "bar2d") : "bar2d";

		if (chartType.equals("bar3d")) {
			listSer = createBarChartType(chartType, chartAttrs, chartObjectFactory);
		}
		else if (chartType.equals("column2d")) {
			listSer = createBarChartType(chartType, chartAttrs, chartObjectFactory);
		}
		else if (chartType.equals("column3d")) {
			listSer = createBarChartType(chartType, chartAttrs, chartObjectFactory);
		}
		else if (chartType.equals("line2d")) {
			listSer = chartObjectFactory.createCTLineChart();
		}
		else if (chartType.equals("line3d")) {
			listSer = chartObjectFactory.createCTLine3DChart();
		}
		else if (chartType.equals("area2d")) {
			listSer = chartObjectFactory.createCTAreaChart();
		}
		else if (chartType.equals("area3d")) {
			listSer = chartObjectFactory.createCTArea3DChart();
		}
		else if (chartType.equals("radar2d")) {
			listSer = chartObjectFactory.createCTRadarChart();
		}
		else if (chartType.equals("scatter2d")) {
			final CTScatterChart scatter = chartObjectFactory.createCTScatterChart();
			handleStandardScatterChart(scatter, chartAttrs);
			listSer = scatter;
		}
		else if (chartType.equals("bubble2d")) {
			listSer = chartObjectFactory.createCTBubbleChart();
		}
		else if (chartType.equals("pie2d")) {
			final CTPieChart pie = chartObjectFactory.createCTPieChart();
			handleStandardPieChart(pie, chartAttrs);
			listSer = pie;
		}
		else if (chartType.equals("donut2d")) {
			final CTDoughnutChart donut = chartObjectFactory.createCTDoughnutChart();
			handleStandardDonutChart(donut, chartAttrs);
			listSer = donut;
		}
		else { // bar2d -> default
			listSer = createBarChartType(chartType, chartAttrs, chartObjectFactory);
		}
		final List<CTUnsignedInt> axid = listSer.getAxId();
		if (axid != null) {
			createAxId(listSer.getAxId());
		}
		listSer.setGrouping("standard");
		
		return listSer;
	}
	

	private static void handleStandardScatterSer(CTScatterSer serContent) {
		final CTShapeProperties shape = new CTShapeProperties();
		final CTLineProperties line = new CTLineProperties();
		line.setW(19050);
		line.setCap(STLineCap.RND);
		line.setRound(new CTLineJoinRound());
		shape.setLn(line);

		//TODO: just if we want points-only
		//line.setNoFill(new CTNoFillProperties());

		shape.getEffectLst();
		serContent.setSpPr(shape);
	}

	private static void handleStandardScatterChart(CTScatterChart scatter, JSONObject chartAttrs) {
		final CTScatterStyle style = new CTScatterStyle();
		style.setVal(STScatterStyle.LINE_MARKER);
		scatter.setScatterStyle(style);

		for (final ISerContent cont : scatter.getSer()) {
			handleStandardScatterSer((CTScatterSer) cont);
		}
	}
	
	private static void handleStandardPieChart(CTPieChart pie, JSONObject chartAttrs) {
		pie.setFirstSliceAng(new CTFirstSliceAng());
	}
	
	private static void handleStandardDonutChart(CTDoughnutChart donut, JSONObject chartAttrs) {
		final CTHoleSize holeSize = new CTHoleSize();
		holeSize.setVal((short) 60);
		donut.setHoleSize(holeSize);
	}

	private static IListSer createBarChartType(String chartType, JSONObject chartAttrs, ObjectFactory chartObjectFactory) {
		IListSer listSer = null;
		String barDir = null;
		if (chartType.equals("bar3d")) {
			listSer = chartObjectFactory.createCTBar3DChart();
			barDir = "bar";
		}
		else if (chartType.equals("column2d")) {
			listSer = chartObjectFactory.createCTBarChart();
			barDir = "col";
		}
		else if (chartType.equals("column3d")) {
			listSer = chartObjectFactory.createCTBar3DChart();
			barDir = "col";
		}
		else {
			listSer = chartObjectFactory.createCTBarChart();
			barDir = "bar";
		}
		listSer.setBarDirection(barDir);
		
		
		if (listSer instanceof CTBarChart) {
			String stacking = null;
			if (chartAttrs != null) {
				stacking = chartAttrs.optString("stacking", null);
			}
			handleOverlap((CTBarChart) listSer, stacking);
		}
		
		return listSer;
	}

	private static void handleOverlap(CTBarChart barSer, String stacking) {
		final CTGapAmount gap = new CTGapAmount();
		gap.setVal(150);
		final CTOverlap overlap;
		if (stacking == null || stacking.equals("standard") || stacking.equals("clustered")) {
			overlap = null;
		} else {
			overlap = new CTOverlap();
			overlap.setVal((byte) 100);
		}
		barSer.setGapWidth(gap);
		barSer.setOverlap(overlap);

	}

	public static void getChartType(CTChartSpace chartSpace, HashMap<String, Object> map) {

		if (chartSpace == null) {
			return;
		}
		final CTChart chart = chartSpace.getChart();
		if (chart == null) {
			return;
		}
		final CTPlotArea plotArea = chart.getPlotArea();
		if (plotArea == null) {
			return;
		}
		final List<IListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
		if (chartList.isEmpty()) {
			return;
		}

		String chartType = null;
		final IListSer listSer = chartList.get(0);
		if (listSer instanceof CTBubbleChart) {
			chartType = "bubble2d";
		}
		else if (listSer instanceof CTRadarChart) {
			chartType = "radar2d";
		}
		else if (listSer instanceof CTSurface3DChart) {
			chartType = "bar2d";
		}
		else if (listSer instanceof CTArea3DChart) {
			chartType = "area3d";
		}
		else if (listSer instanceof CTOfPieChart) {
			chartType = "bar2d";
		}
		else if (listSer instanceof CTAreaChart) {
			chartType = "area2d";
		}
		else if (listSer instanceof CTBarChart) {
			chartType = handleBarDir(listSer);
		}
		else if (listSer instanceof CTSurfaceChart) {
			chartType = "bar2d";
		}
		else if (listSer instanceof CTLine3DChart) {
			chartType = "line3d";
		}
		else if (listSer instanceof CTDoughnutChart) {
			chartType = "donut2d";
		}
		else if (listSer instanceof CTLineChart) {
			chartType = "line2d";
		}
		else if (listSer instanceof CTScatterChart) {
			chartType = "scatter2d";
		}
		else if (listSer instanceof CTBar3DChart) {
			chartType = handleBarDir(listSer);
		}
		else if (listSer instanceof CTPieChart) {
			chartType = "pie2d";
		}
		else if (listSer instanceof CTPie3DChart) {
			chartType = "pie3d";
		}
		else if (listSer instanceof CTStockChart) {
			chartType = "bar2d";
			map.put("chartGroup", "clustered");
		}
		if (chartType != null) {
			map.put("chartType", chartType);
		}
		final String chartGroup = listSer.getStacking();
		if (chartGroup != null) {
			map.put("chartGroup", chartGroup);
		}
	}

	private static String handleBarDir(IListSer listSer) {
		final CTBarDir dir;
		final String view;
		if (listSer instanceof CTBar3DChart) {
			view = "3d";
			dir = ((CTBar3DChart) listSer).getBarDir();
		} else {
			view = "2d";
			dir = ((CTBarChart) listSer).getBarDir();
		}
		final String dirString;
		switch (dir.getVal()) {
		case BAR:
			dirString = "bar";
			break;
		default:
			dirString = "column";
			break;
		}
		return dirString + view;
	}
	
	public static void setLegendFromAttrs(CTChart chart, JSONObject attrs) throws JSONException {
		final String pos = attrs.getJSONObject("legend").optString("pos", "off");
		if (pos.equals("off")) {
			chart.setLegend(null);
		} else {
			CTLegend legend = chart.getLegend();
			if (legend == null) {
				legend = new CTLegend();
				chart.setLegend(legend);
			}
			
			CTTextBody txPr = legend.getTxPr();
			if (txPr == null) {
				txPr = new CTTextBody();
				legend.setTxPr(txPr);
			}
			DMLHelper.handleStandardTypes(txPr);

			DMLHelper.setCharacterFromAttrs(txPr, attrs);
			
			legend.setOverlay(new CTBoolean());
			legend.getOverlay().setVal(false);
			
			CTLegendPos legendPos = legend.getLegendPos();
			if (legendPos == null) {
				legendPos = new CTLegendPos();
				legend.setLegendPos(legendPos);
			}

			final STLegendPos stpos;
			if (pos.equals("left")) {
				stpos = STLegendPos.L;
			} else if (pos.equals("top")) {
				stpos = STLegendPos.T;
			} else if (pos.equals("right")) {
				stpos = STLegendPos.R;
			} else if (pos.equals("bottom")) {
				stpos = STLegendPos.B;
			} else if (pos.equals("topright")) {
				stpos = STLegendPos.TR;
			} else {
				Logger.getAnonymousLogger().warning("No STLegendPost-enum found for " + pos);
				stpos = STLegendPos.L;
			}
			legendPos.setVal(stpos);
		}
	}
	
	public static JSONObject createJsonFromProperties(CTTextBody props) throws JSONException {
		JSONObject character = null;
		if (props != null) {

			final List<CTTextParagraph> paras = props.getP();
			if (paras != null && paras.size() > 0) {
				final CTTextParagraph para = paras.get(0);

				final CTTextParagraphProperties prop = para.getPPr();
				if (prop != null) {
					final CTTextCharacterProperties defPr = prop.getDefRPr();
					if (defPr != null) {
						
						// color
						if (defPr.getSolidFill() != null) {
							character = DMLHelper.createJsonColorFromSolidColorFillProperties(defPr.getSolidFill());
						} else if (defPr.getNoFill() != null) {
							character = new JSONObject();
							Logger.getAnonymousLogger().warning("CtTextbody has a no fill property, result is set to null");
						}
						if (character == null) {
							character = new JSONObject();
						}

						// font-family
						final TextFont latn = defPr.getLatin();
						if (latn != null) {
							final String ff = latn.getTypeface();
							if (ff != null && !ff.equals("+mn-lt")) {
								character.put("fontName", ff);
							}

						}

						final float size = DMLHelper.fontSizeToPt(defPr.getSz());
						character.put("fontSize", size);
						final Boolean bold = defPr.isB();
						final Boolean italic = defPr.isI();
						if (bold != null && bold.booleanValue()) {
							character.put("bold", true);
						}
						if (italic != null && italic.booleanValue()) {
							character.put("italic", true);
						}
					}

				}
			}
		}

		return character;
	}

}
