/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigInteger;
import java.util.List;

import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTColorScheme;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.wml.Body;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.Color;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Document;
import org.docx4j.wml.STShd;
import org.docx4j.wml.STThemeColor;
import org.docx4j.wml.SectPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;

public class Utils {

    public static String themeColorToString(Theme theme, STThemeColor themeColor) {
        String rgbValue = null;
        if(theme!=null) {
            BaseStyles themeElements = theme.getThemeElements();
            if(themeElements!=null) {
                CTColorScheme clrScheme = themeElements.getClrScheme();
                if(clrScheme!=null) {
                    org.docx4j.dml.CTColor ctColor = null;
                    switch(themeColor) {
                        case ACCENT_1 : ctColor = clrScheme.getAccent1(); break;
                        case ACCENT_2 : ctColor = clrScheme.getAccent2(); break;
                        case ACCENT_3 : ctColor = clrScheme.getAccent3(); break;
                        case ACCENT_4 : ctColor = clrScheme.getAccent4(); break;
                        case ACCENT_5 : ctColor = clrScheme.getAccent5(); break;
                        case ACCENT_6 : ctColor = clrScheme.getAccent6(); break;
                        case BACKGROUND_1 : ctColor = clrScheme.getLt1(); break;
                        case BACKGROUND_2 : ctColor = clrScheme.getLt2(); break;
                        case DARK_1 : ctColor = clrScheme.getDk1(); break;
                        case DARK_2 : ctColor = clrScheme.getDk2(); break;
                        case FOLLOWED_HYPERLINK : ctColor = clrScheme.getFolHlink(); break;
                        case HYPERLINK : ctColor = clrScheme.getHlink(); break;
                        case LIGHT_1 : ctColor = clrScheme.getLt1(); break;
                        case LIGHT_2 : ctColor = clrScheme.getLt2(); break;
                        case TEXT_1 : ctColor = clrScheme.getDk1(); break;
                        case TEXT_2 : ctColor = clrScheme.getDk2(); break;
                        case NONE : break;
                    }
                    if(ctColor!=null) {
                        rgbValue = Commons.ctColorToString(ctColor);
                    }
                }
            }
        }
        return rgbValue;
    }

    private static class TintShadeTransformation {
    	STThemeColor 	themeColor;
    	String 			themeTint;
    	String 			themeShade;
    	String 			Color;
    }

    public static void initBorderColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTBorder border)
        throws JSONException {

    	if(border==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	border.setThemeColor(tintShadeTransformation.themeColor);
        border.setThemeTint(tintShadeTransformation.themeTint);
        border.setThemeShade(tintShadeTransformation.themeShade);
        border.setColor(tintShadeTransformation.Color);
    }

    public static void initShdFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTShd shd)
    	throws JSONException {

    	if(shd==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	shd.setThemeFill(tintShadeTransformation.themeColor);
    	shd.setThemeFillTint(tintShadeTransformation.themeTint);
    	shd.setThemeFillShade(tintShadeTransformation.themeShade);
    	shd.setFill(tintShadeTransformation.Color);
        shd.setVal(STShd.CLEAR);
    }

    public static void initColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, Color color)
        throws JSONException {

    	if(color==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	color.setThemeColor(tintShadeTransformation.themeColor);
    	color.setThemeTint(tintShadeTransformation.themeTint);
    	color.setThemeShade(tintShadeTransformation.themeShade);
    	color.setVal(tintShadeTransformation.Color);
    }
    
    private static TintShadeTransformation createTintShadeTransformationFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor)
    	throws JSONException {

    	final TintShadeTransformation tintShadeTransformation = new TintShadeTransformation();

		final String type = jsonColor.getString("type");
		boolean hasRgbValue = false;
        if(type.equals("rgb")&&jsonColor.has("value")) {
        	tintShadeTransformation.Color = jsonColor.getString("value");
        	hasRgbValue = true;
        } else if (type.equals("scheme")&&jsonColor.has("value")) {
            tintShadeTransformation.themeColor = STThemeColor.fromValue(jsonColor.getString("value"));
            tintShadeTransformation.Color = themeColorToString(operationDocument.getTheme(), tintShadeTransformation.themeColor);
            hasRgbValue = true;
        } else if (type.equals("auto")) {
            tintShadeTransformation.Color = "auto";
        }
        if (jsonColor.has("transformations")) {
        	final JSONArray transformations = jsonColor.getJSONArray("transformations");
            if ((transformations != null) && (transformations.length() > 0)) {
                final Object temp = transformations.opt(0);
                if (temp instanceof JSONObject) {
                    JSONObject transformation = (JSONObject)temp;
                    final String transformationType = transformation.optString("type");
                    final Double value = transformation.optDouble("value");
                    if ((transformationType != null) && (value != null)) {
                        if (transformationType.equals("tint")) {
                        	if(hasRgbValue) {
                        		final double[] hsl = Commons.rgbToHsl(tintShadeTransformation.Color);
                        		hsl[2] = (hsl[2]-1)*(value/100000) + 1;
                        		tintShadeTransformation.Color = Commons.hslToRgb(hsl);
                        	}
                        	if(tintShadeTransformation.themeColor!=null) {
                        		tintShadeTransformation.themeTint = Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv));
                        	}
                        }
                        else if (transformationType.equals("shade")) {
                        	if(hasRgbValue) {
                        		final double[] hsl = Commons.rgbToHsl(tintShadeTransformation.Color);
                        		hsl[2] = hsl[2] * (value/100000);
                        		tintShadeTransformation.Color = Commons.hslToRgb(hsl);
                        	}
                        	if(tintShadeTransformation.themeColor!=null) {
                        		tintShadeTransformation.themeShade = Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv));
                        	}
                        }
                    }
                }
            }
        }
	    return tintShadeTransformation;
    }

    public static boolean fillJsonColor(JSONObject jsonColor, STThemeColor themeColor, String STHexColor)
        throws JSONException {

        if(themeColor!=null&&themeColor.value()!=null&&themeColor.value().length()>0) {
            jsonColor.put("type", "scheme");
            jsonColor.put("value", themeColor.value());
        } else if (STHexColor!=null&&STHexColor.length()>0) {
            if(STHexColor.equals("auto"))
                jsonColor.put("type", "auto");
            else if(Commons.highlightColorToRgb.containsKey(STHexColor)){
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.highlightColorToRgb.get(STHexColor));
            }
            else {
                jsonColor.put("type", "rgb");
                jsonColor.put("value", STHexColor);
            }
        }
        return jsonColor.length()>0;
    }

    public static JSONObject createColor(STThemeColor themeColor, String STHexColor)
        throws JSONException {

        JSONObject jsonColor = new JSONObject();
        return fillJsonColor(jsonColor, themeColor, STHexColor) ? jsonColor : null;
    }

    public static JSONObject createFillColor(CTShd shd)
        throws JSONException {

        if (shd!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createFillTransformations(shd);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, shd.getThemeFill(), shd.getFill()) ? jsonColor : null;
        }
        return null;
    }

    public static JSONObject createColor(Color color)
        throws JSONException {

        if (color!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createTransformations(color);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, color.getThemeColor(), color.getVal()) ? jsonColor : null;
        }
        return null;
    }

    private static JSONArray createFillTransformations(CTShd shd)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (shd!=null) {
            String shade = shd.getThemeFillShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = shd.getThemeFillTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONArray createTransformations(Color color)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (color!=null) {
            String shade = color.getThemeShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = color.getThemeTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONObject createShadeTransformation(String shade)
        throws JSONException {

        JSONObject shadeTransform = null;
        if (shade!=null) {
            shadeTransform = new JSONObject();
            shadeTransform.put("type", "shade");
            shadeTransform.put("value", Math.round(Integer.parseInt(shade, 16) * Commons.themeTransformFactor));
        }
        return shadeTransform;
    }

    private static JSONObject createTintTransformation(String tint)
        throws JSONException {

        JSONObject tintTransform = null;
        if (tint!=null) {
            tintTransform = new JSONObject();
            tintTransform.put("type", "tint");
            tintTransform.put("value", Math.round(Integer.parseInt(tint, 16) * Commons.themeTransformFactor));
        }
        return tintTransform;
    }

    public static SectPr getDocumentProperties(MainDocumentPart mainDocPart) {

        SectPr sectPr = null;
        if(mainDocPart!=null) {
            Document doc = mainDocPart.getJaxbElement();
            if(doc!=null) {
                Body body = doc.getBody();
                if(body!=null)
                    sectPr = body.getSectPr();
            }
        }
        return sectPr;
    }

    public static long mapTwipTo100THMM(BigInteger bigInt) {
        return bigInt!=null?((bigInt.longValue() * 2540) / 72) / 20:0;
    }

    public static long mapTwipTo100THMM(long longValue) {
        return ((longValue * 2540) / 72) / 20;
    }

    public static BigInteger map100THMMToTwip(long longValue) {
        return BigInteger.valueOf( longValue * 1440 /2540 );
    }

    public static BigInteger map100THMMToTwip(double doubleValue) {
        return BigInteger.valueOf( (long)(doubleValue * 1440. /2540.) );
    }

    public static boolean equals(final String s1, final String s2) {
        return (s1==null&&s2==null)||((s1!=null)&&s1.equals(s2));
    }

    public static List<Object> getContent(Object o) {
        List<Object> r = null;
        if(o instanceof ContentAccessor)
            r = ((ContentAccessor)o).getContent();
        return r;
    }
}
