/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.worker;

import java.net.InetSocketAddress;
import java.util.Set;

import javax.ws.rs.core.Application;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;

import com.openexchange.office.calcengine.CalcEngineConst;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntityReader;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntityWriter;
import com.openexchange.office.calcengine.client.ECalcEngineMode;
import com.openexchange.office.calcengine.client.impl.CalcEngineLogContextConst;
import com.openexchange.office.calcengine.servlets.CalcEngineServlet;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogFramework;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;
import com.openexchange.office.tools.logging.impl.RequestLogFilter;
import com.sun.jersey.api.core.DefaultResourceConfig;
import com.sun.jersey.spi.container.servlet.ServletContainer;

//==============================================================================
/** Main for our Calc Engine Worker process.
 * 
 *  Those process use JNI to load libcalcengine.so into process
 *  and map its API from/to Java.
 */
public class WorkerMain
{
    //--------------------------------------------------------------------------
	private static Log LOG = null; // it's important to create LOG in main() AFTER logging was instrumented !!!
	
    //--------------------------------------------------------------------------
	/** listen by default on all interface for local and remote connections */
	public static final String DEFAULT_INTERFACE = "0.0.0.0";
	
    //--------------------------------------------------------------------------
	public static void main (String[] lArgs)
	{
		// make sure WE log on stdout/stderr ...
		// We are used as simple command line tool within a bigger process ...
		// and those process will parse our stdout/stderr output o forward them
		// to it's own logging framework .-)

		LogFactory.establishLogBackStdOutMode ();
		LogFactory.setLogFramework            (ELogFramework.E_STD);

		JettyToApacheLoggingBridge.register   (     );
		JettyToApacheLoggingBridge.enableDebug(false); // this line is important ! debug output disturb stdout parsing for detecting if worker was started successfully or not ! see {@link CalcEngineJob}
		
		LOG = LogFactory.getJclLog(WorkerMain.class);
		
		try
		{
			final WorkerMain aWorker =  new WorkerMain ();
			aWorker.impl_start (lArgs);
		}
		catch (java.net.BindException exBind)
		{
			LOG.error ("port already used ... exit("+CalcEngineConst.EXITCODE_PORT_IN_USE+")", exBind);
			System.exit (CalcEngineConst.EXITCODE_PORT_IN_USE);
		}
		catch (Throwable ex)
		{
			LOG.error ("worker finished unexpected ... exit("+CalcEngineConst.EXITCODE_GENERAL_ERROR+")", ex);
			System.exit (CalcEngineConst.EXITCODE_GENERAL_ERROR);
		}

		LOG.info ("worker finished successfully ...");
		System.exit (CalcEngineConst.EXITCODE_OK);
	}

    //--------------------------------------------------------------------------
	private void impl_start (final String[] lArgs)
		throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		
		WorkerCommandLine aCmdLine = new WorkerCommandLine ();
		aCmdLine.parse(lArgs);

		if (aCmdLine.needsHelp())
		{
			aCmdLine.printHelp();
			System.exit (CalcEngineConst.EXITCODE_INVALID_CMDLINE);
		}
		
		String  sInterface    = aCmdLine.getInterface   ();
		int     nPort         = aCmdLine.getPort        ();
		boolean bDoSimulation = aCmdLine.isSimulationOn ();
		boolean bDebugOn      = aCmdLine.isDebugOn      ();
		
		if (StringUtils.isEmpty(sInterface))
			sInterface = DEFAULT_INTERFACE;

		aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_NETINTERFACE, sInterface             );
		aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_PORT        , Integer.toString(nPort));
		
		if (bDoSimulation)
		{
			LOG.warn(aLog.forLevel(ELogLevel.E_WARNING                                                 )
					     .toLog   ("worker run in simulation mode ... Please verify it was expected !"));
			CalcEngineServlet.MODE = ECalcEngineMode.E_SIMULATOR;
		}
		else
		{
			LOG.info(aLog.forLevel(ELogLevel.E_WARNING             )
				         .toLog   ("worker run in native mode ..."));
			CalcEngineServlet.MODE = ECalcEngineMode.E_NATIVE;
		}
		
		impl_startFromCode   (sInterface, nPort, bDebugOn);
		//impl_startFromWebXml (sInterface, nPort);
	}
	
    //--------------------------------------------------------------------------
	private void impl_startFromCode (final String  sInterface,
									 final int     nPort     ,
									 final boolean bDebugOn  )
		throws Exception
	{
		final ContextAwareLogHelp      aLog     = mem_Log ();
		final Server                   aServer  = new Server( new InetSocketAddress(sInterface, nPort) );
        final ServletContextHandler    aHandler = new ServletContextHandler  ();
        final CalcEngineServlet        aServlet = new CalcEngineServlet      ();
        final Application              aApp     = new DefaultResourceConfig  ();

        aServlet.setServerPort(nPort);
        
        LOG.info (aLog.forLevel(ELogLevel.E_INFO                        )
			          .toLog   ("configure marshalling environment ..."));

		// ensure our own set of message body reader/writer is on start .-)
        Set< Class< ? > > lClasses = aApp.getClasses();
        lClasses.add(CalcEngineHttpEntityWriter.class);
        lClasses.add(CalcEngineHttpEntityReader.class);

        LOG.info (aLog.forLevel(ELogLevel.E_INFO               )
		              .toLog   ("worker register servlets ..."));
		
		aApp.getSingletons().add(aServlet);
        ServletContainer aServletContainer = new ServletContainer(aApp);
        
        aHandler.setContextPath(CalcEngineConst.CALCENGINE_CONTEXTPATH);
        aHandler.addServlet    (new ServletHolder(aServletContainer), "/*");
        
        if (bDebugOn)
        {
            LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG                                                )
		                   .toLog   ("worker register log filter to record request/reponse pairs ..."));
        	aHandler.addFilter (RequestLogFilter.class, "/*", null);
        }
        
        aServer.setHandler(aHandler);

        LOG.info (aLog.forLevel(ELogLevel.E_INFO    )
	                  .toLog   ("worker starts ..."));
        aServer.start();

        LOG.info (aLog.forLevel(ELogLevel.E_INFO                  )
	                  .toLog   ("worker started successfully ..."));
        aServer.join ();
    }

    //--------------------------------------------------------------------------
//	/** @deprecated do not use :-)
//	 */
//	private void impl_startFromWebXml (final String sInterface,
//									  final int    nPort     )
//		throws Exception
//	{
//		URL    aRoot = WorkerMain.class.getResource("/WEB-INF");
//		String sRoot = aRoot.toURI().toASCIIString();
//		       sRoot = StringUtils.removeStartIgnoreCase(sRoot, "jar:");
//		       sRoot = StringUtils.substringBeforeLast  (sRoot, "!"   );
//		
//		final Server aServer = new Server( new InetSocketAddress(sInterface, nPort) );
//		final WebAppContext aContext = new WebAppContext();
//
//        LOG.info ("worker register servlets ...");
//
//		aContext.setWar        (sRoot);
//        //aContext.setDescriptor("WEB-INF/web.xml");
//		aContext.setContextPath("/"     );
//		aServer.setHandler     (aContext);
//
//		LOG.info ("worker starts ...");
//
//		aServer.start();
//        aServer.join ();
//	}

    //--------------------------------------------------------------------------
	private ContextAwareLogHelp mem_Log ()
	    throws Exception
	{
		if (m_aLog == null)
			m_aLog = new ContextAwareLogHelp (LOG);
		return m_aLog;
	}

    //--------------------------------------------------------------------------
	private ContextAwareLogHelp m_aLog = null;
}
