/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.servlets;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.CalcEngineConst;
import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntity;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ECalcEngineMode;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.calcengine.client.impl.CalcEngineLogContextConst;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** bind the calc engine JNI code to REST.
*/
@Path    (CalcEngineConst.CALCENGINE_SERVLETPATH)
@Consumes(CalcEngineConst.MEDIATYPE_4_STREAMING )
@Produces(CalcEngineConst.MEDIATYPE_4_STREAMING )
public class CalcEngineServlet
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineServlet.class);

    //-------------------------------------------------------------------------
    public static final ECalcEngineMode DEFAULT_MODE = ECalcEngineMode.E_NATIVE;
    
    //-------------------------------------------------------------------------
    public static ECalcEngineMode MODE = DEFAULT_MODE;
    
    //-------------------------------------------------------------------------
    public CalcEngineServlet ()
    {}
    
    //-------------------------------------------------------------------------
    @GET
    public Response createDocument()
        throws Exception
    {
        try
        {
    		final ICalcEngineClient    iImpl     = mem_Impl ();
        	final String               sHandle   = iImpl.createDocument();
            final String               sJson     = CalcEngineHandleGenerator.toJSON(sHandle);
            final CalcEngineHttpEntity aResponse = CalcEngineHttpEntity.OK (sJson, null);
            return Response.ok (aResponse).build();
        }
        catch (Throwable ex)
        {
            LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
        }
    }

    //-------------------------------------------------------------------------
    @DELETE
    @Path("{handle}")
    public Response destroyDocument(@PathParam("handle") final String sHandle)
        throws Exception
    {
    	try
    	{
        	final ContextAwareLogHelp aLog = mem_Log ();
        	LOG.info(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			     .forLevel    (ELogLevel.E_INFO                                     )
        			     .toLog       ("destroy document"                                   ));
        	
    		final ICalcEngineClient iImpl = mem_Impl ();
    		iImpl.destroyDocument(sHandle);

            final CalcEngineHttpEntity aResponse = CalcEngineHttpEntity.OK (null, null);
            return Response.ok (aResponse).build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path("{handle}/"+CalcEngineConst.RELPATH_EXECUTE)
    public Response executeOperation(@PathParam("handle") final String sHandle,
                                                          final String sOpJSON)
        throws Exception
    {
    	try
    	{
        	final ContextAwareLogHelp aLog = mem_Log ();
        	LOG.info(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			     .forLevel    (ELogLevel.E_INFO                                     )
        			     .toLog       ("execute operation"                                  ));
        	
    		final ICalcEngineClient    iImpl     = mem_Impl ();
    		final StringBuffer         sResult   = new StringBuffer (256);
    		final ECalcEngineError     eError    = iImpl.executeOperation(sHandle, sOpJSON, sResult);
    		      CalcEngineHttpEntity aResponse = null;

    		if (eError == ECalcEngineError.E_NONE)
    		{
	    		final String sOpResult = sResult.toString ();
	                         aResponse = CalcEngineHttpEntity.OK (sOpResult, null);
    		}
    		else
    		{
        		aResponse = CalcEngineHttpEntity.ERROR (eError);
    		}

			return Response.ok (aResponse).build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path("{handle}/"+CalcEngineConst.RELPATH_RESTORE)
    public Response restoreDocument(@PathParam("handle") final String sHandle,
                                                         final String sOpJSON)
        throws Exception
    {
    	try
    	{
        	final ContextAwareLogHelp aLog = mem_Log ();
        	LOG.info(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			     .forLevel    (ELogLevel.E_INFO                                     )
        			     .toLog       ("restore document"                                   ));

    		final ICalcEngineClient    iImpl     = mem_Impl ();
    		final ECalcEngineError     eError    = iImpl.restoreDocument(sHandle, sOpJSON);
		          CalcEngineHttpEntity aResponse = null;

			if (eError == ECalcEngineError.E_NONE)
			{
	            aResponse = CalcEngineHttpEntity.OK (null, null);
			}
			else
			{
				aResponse = CalcEngineHttpEntity.ERROR (eError);
			}

            return Response.ok (aResponse).build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path("{handle}/"+CalcEngineConst.RELPATH_COPY)
    public Response copy(@PathParam("handle") final String               sHandle,
                                              final CalcEngineHttpEntity aEntity)
        throws Exception
    {
    	try
    	{
        	final ContextAwareLogHelp aLog = mem_Log ();
        	LOG.info(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			     .forLevel    (ELogLevel.E_INFO                                     )
        			     .toLog       ("copy"                                               ));

    		final ICalcEngineClient        iImpl     = mem_Impl ();
    		final CalcEngineClipBoardEvent aEvent    = aEntity.getClipboardEvent();
    		final ECalcEngineError         eError    = iImpl.copy(sHandle, aEvent);
    		      CalcEngineHttpEntity     aResponse = null;

    		if (eError == ECalcEngineError.E_NONE)
    		{
                 aResponse = CalcEngineHttpEntity.OK (null, aEvent);
    		}
    		else
    		{
        		aResponse = CalcEngineHttpEntity.ERROR (eError);
    		}

			return Response.ok (aResponse).build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path("{handle}/"+CalcEngineConst.RELPATH_PASTE)
    public Response paste(@PathParam("handle") final String               sHandle,
                                               final CalcEngineHttpEntity aEntity)
        throws Exception
    {
    	try
    	{
        	final ContextAwareLogHelp aLog = mem_Log ();
        	LOG.info(aLog.enterContext(CalcEngineLogContextConst.CONTEXT_DOC_HANDLE, sHandle)
        			     .forLevel    (ELogLevel.E_INFO                                     )
        			     .toLog       ("paste"                                              ));

    		final ICalcEngineClient        iImpl     = mem_Impl ();
    		final CalcEngineClipBoardEvent aEvent    = aEntity.getClipboardEvent();
    		final ECalcEngineError         eError    = iImpl.paste(sHandle, aEvent);
    		      CalcEngineHttpEntity     aResponse = null;

    		if (eError == ECalcEngineError.E_NONE)
    		{
                 aResponse = CalcEngineHttpEntity.OK (null, aEvent);
    		}
    		else
    		{
        		aResponse = CalcEngineHttpEntity.ERROR (eError);
    		}

			return Response.ok (aResponse).build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path(CalcEngineConst.RELPATH_SETLOGLEVEL)
    public Response setLogLevel(final CalcEngineHttpEntity aEntity)
        throws Exception
    {
    	try
    	{
    		final ICalcEngineClient iImpl     = mem_Impl ();
    		final String            sLogLevel = aEntity.getJSONPayload();
    		final ELogLevel         eLogLevel = ELogLevel.fromJSON(sLogLevel, null);

    		if (eLogLevel == null)
    			return Response.status(Response.Status.BAD_REQUEST).build ();

    		iImpl.setLogLevel(eLogLevel);

			return Response.ok ().build();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path(CalcEngineConst.RELPATH_ISALIVE)
    public Response isAlive()
        throws Exception
    {
    	final ContextAwareLogHelp aLog = mem_Log ();

    	try
    	{
    		LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG          )
    				       .toLog   ("got is-alive request ..."));
    		synchronized (this)
    		{
    			if (m_bInShutdown)
    			{
    				LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG                                    )
     				               .toLog   ("... is 'not alive' - is in shutdown mode already !"));
    				return Response.status(Response.Status.SERVICE_UNAVAILABLE).build ();
    			}
    		}
    		
    		LOG.debug (aLog.forLevel(ELogLevel.E_DEBUG )
				           .toLog   ("... is alive"   ));
			return Response.ok ().build ();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path(CalcEngineConst.RELPATH_SHUTDOWN)
    public Response shutdown()
        throws Exception
    {
    	final ContextAwareLogHelp aLog = mem_Log ();
    	
    	try
    	{
    		LOG.info (aLog.forLevel (ELogLevel.E_INFO           )
    				      .toLog    ("got shutdown request ..."));

    		synchronized (this)
			{
				// already in shutdown ?
				if (m_bInShutdown)
				{
					LOG.warn (aLog.forLevel (ELogLevel.E_WARNING                                                       )
	    				          .toLog    ("... already in shutdown. New shutdown request will be ignored silently."));
					return Response.ok ().build ();
				}
				
				m_bInShutdown = true;
			}

    		LOG.info (aLog.forLevel (ELogLevel.E_INFO                  )
				          .toLog    ("... start shutdown asynchronous"));
    		new Thread ()
    		{
    			@Override
    			public void run ()
    			{
    				try
    				{
    					// let jetty(jersey some time to handle the response ... .-)
    					final int TIME = 250;
    					LOG.info (aLog.forLevel (ELogLevel.E_INFO              )
    	    				          .toLog    ("... shutdown in "+TIME+" ms"));
    					synchronized (this)
    					{
    						wait (TIME);
    					}

    					LOG.info (aLog.forLevel (ELogLevel.E_INFO                     )
    	    				          .toLog    ("... shutdown finished (exit now) !"));
    					System.exit (CalcEngineConst.EXITCODE_OK); // TODO find better solution then these .-)
    				}
    				catch (Throwable ex)
    				{}
    			}
    		}.start ();
			return Response.ok ().build ();
    	}
    	catch (Throwable ex)
    	{
    		LOG.error(ex.getMessage(), ex);
            return Response.serverError().entity(ex.getMessage()).build();
    	}
    }

    //-------------------------------------------------------------------------
    @POST
    @Path(CalcEngineConst.RELPATH_KILL)
    public void kill()
        throws Exception
    {
    	// no logging
    	// no checks
    	// no questions
    	// KILL THIS PROCESS ! THATS WHAT SHOULD HAPPEN HERE ONLY !
    	System.exit(CalcEngineConst.EXITCODE_KILLBILL);
    }

    //-------------------------------------------------------------------------
    public void setServerPort (final int nPort)
        throws Exception
    {
    	m_nServerPort = nPort;
    }
    
    //-------------------------------------------------------------------------
    private ICalcEngineClient mem_Impl ()
        throws Exception
    {
        if (m_iImpl == null)
        {
            final CalcEngineDescriptor aDesc = new CalcEngineDescriptor ();
            aDesc.m_eMode = MODE;
            m_iImpl = CalcEngineClientFactory.getDirect(aDesc);
        }
        return m_iImpl;
    }

    //-------------------------------------------------------------------------
    private ContextAwareLogHelp mem_Log ()
        throws Exception
    {
        if (m_aLog == null)
        {
        	m_aLog = new ContextAwareLogHelp (LOG);
        	m_aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_PORT, Integer.toString(m_nServerPort));
        }
        return m_aLog;
    }
    
    //-------------------------------------------------------------------------
    private ICalcEngineClient m_iImpl = null;

    //-------------------------------------------------------------------------
    private @Context ServletContext m_aContext = null;

    //-------------------------------------------------------------------------
    private int m_nServerPort = 0;

    //-------------------------------------------------------------------------
    private boolean m_bInShutdown = false;

    //-------------------------------------------------------------------------
    private ContextAwareLogHelp m_aLog = null;
}
