/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/basemenu',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/basepopup'
    ], function (Utils, KeyCodes, BasePopup) {

    'use strict';

    // class BaseMenu =========================================================

    /**
     * Wrapper class for a DOM node used as interactive pop-up menu element,
     * shown on top of the application window, and relative to an arbitrary DOM
     * node.
     *
     * @constructor
     *
     * @extends BasePopup
     *
     * @param {Object} [initOptions]
     *  Initial options to control the properties of the pop-up menu element.
     *  Supports all options also supported by the base class BasePopup.
     *  Additionally, the following options are supported:
     *  @param {String|Function|Node|jQuery} [initOptions.preferFocusFilter]
     *      A jQuery selector that filters the available focusable control
     *      elements to a list of preferred controls, that will be used when
     *      moving the browser focus into this pop-up menu using the method
     *      BaseMenu.grabFocus(). The selector will be passed to the jQuery
     *      method jQuery.filter(). If this selector is a function, it will be
     *      called with the DOM node bound to the symbol 'this'. See the jQuery
     *      API documentation at http://api.jquery.com/filter for details.
     */
    function BaseMenu(initOptions) {

        var // self reference
            self = this,

            // filter selector for preferred focusable elements
            preferFocusFilter = Utils.getOption(initOptions, 'preferFocusFilter');

        // base constructor ---------------------------------------------------

        BasePopup.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Hides the pop-up menu, if the ESCAPE key has been pressed inside.
         */
        function keyDownHandler(event) {
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.hide();
                return false;
            }
        }

        /**
         * Scrolls the pop-up menu to make a focused control element visible.
         */
        function focusInHandler(event) {
            self.scrollToChildNode(event.target);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this pop-up menu has the browser focus.
         * The check starts from the pop-up menu root node itself and continues
         * to its children nodes.
         *
         * @returns {Boolean}
         *  Whether this pop-up menu has the browser focus.
         */
        this.hasFocus = function () {
            // the root node itself may be focused, e.g. in IE while dragging menu scroll bar
            return Utils.hasOrContainsFocus(this.getNode());
        };

        /**
         * Sets the browser focus into the first focusable control element of
         * this pop-up menu element.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.bottom=false]
         *      If set to true, the last available entry of the pop-up menu
         *      will be focused instead of the first.
         *
         * @returns {BaseMenu}
         *  A reference to this instance.
         */
        this.grabFocus = function (options) {

            var // all focusable control elements
                focusableNodes = this.hasFocus() ? $() : this.getNode().find(Utils.REALLY_VISIBLE_SELECTOR + Utils.FOCUSABLE_SELECTOR),
                // the preferred controls returned by the callback function
                preferredNodes = preferFocusFilter ? focusableNodes.filter(preferFocusFilter) : focusableNodes,
                // whether to select the last control
                bottom = Utils.getBooleanOption(options, 'bottom', false);

            // fall back to all focusable controls if no preferred controls are available
            if (preferredNodes.length === 0) { preferredNodes = focusableNodes; }
            preferredNodes[bottom ? 'last' : 'first']().focus();
            return this;
        };

        this.registerDestructor(function () {
            self = null;
            initOptions = null;
            preferFocusFilter = null;
        });

        // initialization -----------------------------------------------------

        // add basic pop-up menu styling
        this.getNode().addClass('popup-menu');

        // register event handlers
        this.getNode().on({ keydown: keyDownHandler, focusin: focusInHandler });

    } // class BaseMenu

    // exports ================================================================

    // derive this class from class BasePopup
    return BasePopup.extend({ constructor: BaseMenu });

});
