/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/menumixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes'
    ], function (Utils, KeyCodes) {

    'use strict';

    var // marker CSS class for elements containing an open drop-down menu
        OPEN_CLASS = 'dropdown-open';

    // class MenuMixin ========================================================

    /**
     * Extends a Group object with a drop-down menu element. Creates a new
     * drop-down button element in the group and extends it with a caret sign.
     * Implements mouse and keyboard event handling for that drop-down button
     * (open and close the drop-down menu, and closing the drop-down menu
     * automatically on focus navigation). Adds new methods to the group
     * instance to control the drop-down button and menu.
     *
     * Note: This is a mix-in class supposed to extend an existing instance of
     * the class Group or one of its derived classes. Expects the symbol 'this'
     * to be bound to an instance of Group.
     *
     * Instances of this class trigger the following events:
     * - 'menu:beforeopen': Before the drop-down menu will be opened.
     * - 'menu:open': After the drop-down menu has been opened.
     * - 'menu:beforeclose': Before the drop-down menu will be closed.
     * - 'menu:close': After the drop-down menu has been closed.
     * - 'menu:beforelayout': If the absolute position of the group or the
     *      browser window size has changed, and the position and size of the
     *      drop-down menu node needs to be adjusted.
     * - 'menu:layout': If the absolute position of the group or the browser
     *      window size has changed, and the position and size of the drop-down
     *      menu node has been adjusted (in auto-layout mode).
     *
     * @constructor
     *
     * @param {BaseMenu} menu
     *  The pop-up menu instance that will be bound to the group. Can be the
     *  instance of any subclass of BaseMenu. ATTENTION: This mix-in takes
     *  ownership of the pop-up menu instance and will destroy it by itself.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the behavior of the group. The following
     *  options are supported:
     *  @param {HTMLElement|jQuery} [initOptions.button]
     *      The drop-down button element that will toggle the drop-down menu
     *      when clicked. If omitted, the drop-down menu cannot be toggled
     *      using the mouse; the client implementation has to provide its own
     *      logic to show and hide the menu.
     *  @param {Boolean} [initOptions.caret=true]
     *      If set to true or omitted, a caret icon will be added to the right
     *      border of the drop-down button specified with the option 'button'
     *      (see above).
     */
    function MenuMixin(menu, initOptions) {

        var // self reference (the Group instance)
            self = this,

            // the DOM node this drop-down menu is anchored to
            groupNode = this.getNode(),

            // the drop-down button
            menuButton = $(Utils.getOption(initOptions, 'button'));

        // private methods ----------------------------------------------------

        /**
         * Handles mouse clicks everywhere on the page. Closes the drop-down
         * menu automatically.
         */
        function globalClickHandler(event) {

            // Returns whether one of the specified nodes contains the DOM
            // element in event.target, or if it is the event.target by itself.
            function isTargetIn(nodes) {
                return (nodes.filter(event.target).length > 0) || (nodes.has(event.target).length > 0);
            }

            // close the menu unless a 'mousedown' event occurred on the drop-down button
            // (this would lead to reopening the menu immediately with the following click)
            if (!isTargetIn(menu.getNode()) && !((event.type === 'mousedown') && isTargetIn(menuButton))) {
                menu.hide();
            }
        }

        /**
         * Initializes the drop-down menu after it has been opened, and
         * triggers a 'menu:open' event at the group.
         */
        function popupShowHandler() {

            // add CSS marker class to the group, and notify own listeners
            groupNode.addClass(OPEN_CLASS);
            self.trigger('menu:open');

            // Add a global click handler to close the menu automatically when
            // clicking somewhere in the page. Listening to the 'mousedown'
            // event will catch all real mouse clicks and close the menu
            // immediately ('click' events will not be generated by the browser
            // when selecting a text range over several text spans or
            // paragraphs). It is still needed to listen to real 'click' events
            // which may be triggered indirectly, e.g. after pressing a button
            // with the keyboard. The event handler must be attached deferred,
            // otherwise it would close the drop-down menu immediately after it
            // has been opened with a mouse click on the drop-down button while
            // the event bubbles up to the document root.
            _.defer(function () {
                if (self.isVisible()) {
                    $(document).on('mousedown click', globalClickHandler);
                }
            });
        }

        /**
         * Deinitializes the drop-down menu before it will be closed, and
         * triggers a 'menu:beforeclose' event at the group.
         */
        function popupBeforeHideHandler() {

            // move focus to drop-down button, if drop-down menu is focused
            if (menu.hasFocus()) {
                menuButton.focus();
            }

            // unregister the global click handler
            $(document).off('mousedown click', globalClickHandler);

            // remove CSS marker class from the group, and notify own listeners
            self.trigger('menu:beforeclose');
            groupNode.removeClass(OPEN_CLASS);
        }

        /**
         * Handles click events from the drop-down button, and toggles the
         * drop-down menu.
         */
        function menuButtonClickHandler(event) {

            // do nothing (but trigger the 'group:cancel' event) if the group is disabled
            if (self.isEnabled()) {

                // WebKit does not focus the clicked button, which is needed to
                // get keyboard control in the drop-down menu
                menuButton.focus();

                // toggle the drop-down menu, this triggers the appropriate event
                menu.toggle();

                // move focus into drop-down menu, if triggered with SPACE or ENTER key
                if (menu.isVisible() && _.isNumber(event.keyCode)) {
                    menu.grabFocus();
                }
            }

            // trigger 'group:cancel' event, if the menu has been closed with mouse click
            if (!menu.isVisible()) {
                self.triggerCancel();
            }
        }

        /**
         * Handles keyboard events from any control in the group object.
         */
        function groupKeyHandler(event) {

            var // distinguish between event types (ignore keypress events)
                keydown = event.type === 'keydown';

            switch (event.keyCode) {
            case KeyCodes.DOWN_ARROW:
            case KeyCodes.PAGE_DOWN:
                if (keydown && self.isEnabled()) {
                    menu.show().grabFocus();
                }
                return false;
            case KeyCodes.UP_ARROW:
            case KeyCodes.PAGE_UP:
                if (keydown && self.isEnabled()) {
                    menu.show().grabFocus({ bottom: true });
                }
                return false;
            }
        }

        /**
         * Handles keyboard events in the focused drop-down button.
         */
        function menuButtonKeyDownHandler(event) {
            if ((event.keyCode === KeyCodes.ESCAPE) && menu.isVisible()) {
                menu.hide();
                // do not let the Group base class trigger the 'group:cancel' event
                event.preventDefault();
            }
        }

        /**
         * Handles keyboard events inside the open drop-down menu.
         */
        function menuKeyHandler(event) {

            var // distinguish between event types
                keydown = event.type === 'keydown';

            switch (event.keyCode) {
            case KeyCodes.TAB:
                if (keydown && KeyCodes.matchModifierKeys(event, { shift: null })) {
                    menu.hide();
                    // To prevent problems with event bubbling (Firefox continues
                    // to bubble to the parent of the menu node, while Chrome
                    // always bubbles from the focused DOM node, in this case
                    // from the menu *button*), stop propagation of the original
                    // event, and simulate a TAB key event from the menu button
                    // to move the focus to the next control.
                    menuButton.trigger(event);
                    return false;
                }
                break;
            case KeyCodes.ESCAPE:
                if (keydown) {
                    menu.hide();
                }
                return false;
            case KeyCodes.F6:
                // Hide drop-down menu on global F6 focus traveling. This will set
                // the focus back to the drop-down button, so that F6 will jump to
                // the next view component. Jumping directly from the drop-down menu
                // (which is located near the body element in the DOM) would select
                // the wrong element. Ignore all modifier keys here, even on
                // non-MacOS systems where F6 traveling is triggered by Ctrl+F6
                // (browsers will move the focus away anyway).
                if (keydown) {
                    menu.hide();
                }
                break;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the button element that toggles the drop-down menu.
         *
         * @returns {jQuery}
         *  The drop-down button element, as jQuery object.
         */
        this.getMenuButton = function () {
            return menuButton;
        };

        /**
         * Returns the instance of the pop-up menu class representing the
         * drop-down menu of this group.
         *
         * @returns {BaseMenu}
         *  The pop-up menu instance representing the drop-down menu of this
         *  group.
         */
        this.getMenu = function () {
            return menu;
        };

        /**
         * Returns the root DOM node of the drop-down menu, as jQuery object.
         *
         * @returns {jQuery}
         *  The root DOM node of the drop-down menu.
         */
        this.getMenuNode = function () {
            return menu.getNode();
        };

        /**
         * Returns whether the drop-down menu is currently visible.
         */
        this.isMenuVisible = function () {
            return menu.isVisible();
        };

        /**
         * Shows the drop-down menu.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.showMenu = function () {
            menu.show();
            return this;
        };

        /**
         * Hides the drop-down menu.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.hideMenu = function () {
            menu.hide();
            return this;
        };

        /**
         * Removes all contents from the drop-down menu.
         *
         * @returns {MenuMixin}
         *  A reference to this instance.
         */
        this.clearMenu = function () {
            menu.clearContents();
            return this;
        };

        // initialization -----------------------------------------------------

        // convert ENTER and SPACE keys to click events
        Utils.setButtonKeyHandler(menuButton);

        // initialize the caret icon and drop-down menu
        if (Utils.getBooleanOption(initOptions, 'caret', true) && (menuButton.length > 0)) {
            menuButton.addClass('caret-button').append($('<span>').addClass('caret-span').append(Utils.createIcon('fa-caret-down')));
        }

        // register menu node for focus handling (group retains focused state while focus is in menu node)
        this.registerFocusableNodes(menu.getNode());

        // register event handlers for the group
        this.on({
            'group:blur': function () { menu.hide(); },
            'group:show group:enable': function (event, state) { if (!state) { menu.hide(); } }
        });
        self.listenTo(groupNode, 'keydown keypress keyup', groupKeyHandler);
        menuButton.on({
            click: menuButtonClickHandler,
            keydown: menuButtonKeyDownHandler
        });

        // register event handlers for the drop-down menu
        menu.on({
            'popup:beforeshow': function () { self.trigger('menu:beforeopen'); },
            'popup:show': popupShowHandler,
            'popup:beforehide': popupBeforeHideHandler,
            'popup:hide': function () { self.trigger('menu:close'); },
            'popup:beforelayout': function () { self.trigger('menu:beforelayout'); },
            'popup:layout': function () { self.trigger('menu:layout'); }
        });
        menu.getNode().on('keydown keypress keyup', menuKeyHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            $(document).off('mousedown click', globalClickHandler);

            menuButton.remove();
            menu.destroy();

            initOptions = globalClickHandler = groupNode = menuButton = menu = null;
        });

    } // class MenuMixin

    // exports ================================================================

    return MenuMixin;

});
