/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/button',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/group'
    ], function (Utils, Group) {

    'use strict';

    // class Button ===========================================================

    /**
     * Creates a container element used to hold a push button or a toggle
     * button.
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the button. Supports all
     *  options of the Group base class and all generic formatting options for
     *  buttons (see method Utils.createButton() for details). Additionally,
     *  the following options are supported:
     *  @param {Boolean} [initOption.toggle=false]
     *      If set to true, the button represents a boolean value and
     *      toggles its state when clicked.
     *  @param {Function} [initOptions.highlight]
     *      A predicate function that will be called every time after the value
     *      of the button has been set. The button will be highlighted if this
     *      handler function returns true. Receives the value of the button as
     *      first parameter. Will be called in the context of this button
     *      instance.
     */
    function Button(initOptions) {

        var // self reference
            self = this,

            // create the DOM button element
            buttonNode = Utils.createButton(initOptions),

            // toggle button or push button
            toggle = Utils.getBooleanOption(initOptions, 'toggle', false),

            // custom predicate callback for button highlighting
            highlightHandler = Utils.getFunctionOption(initOptions, 'highlight');

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * The update handler for this button.
         */
        function updateHandler(value) {

            if (_.isFunction(highlightHandler)) {
                Utils.selectButtons(buttonNode, highlightHandler.call(self, value));
            } else if (toggle) {
                // Translate null (special 'ambiguous' state) to false to
                // prevent toggling the button as implemented by the static
                // method Utils.selectButtons().
                Utils.selectButtons(buttonNode, value);
            }

            // do not change the initial value of push buttons (several buttons
            // can be used to change the value of an item independently)
        }

        /**
         * Returns the value of the button after it has been clicked.
         */
        function clickHandler() {
            // change boolean value of a toggle button, otherwise return fixed value
            return toggle ? !self.getValue() : Utils.getControlValue(buttonNode);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the DOM button element, as jQuery object.
         *
         * @returns {jQuery}
         *  The DOM button element, as jQuery object.
         */
        this.getButtonNode = function () {
            return buttonNode;
        };

        /**
         * Changes the icon of this button control.
         *
         * @param {String} [icon]
         *  The CSS class name of the new icon. If omitted, the current icon
         *  will be removed from the button.
         *
         * @returns {Button}
         *  A reference to this instance.
         */
        this.setIcon = function (icon) {
            Utils.setControlCaption(buttonNode, Utils.extendOptions(initOptions, { icon: icon }));
            return this;
        };

        /**
         * Changes the label text of this button control.
         *
         * @param {String} [labelText]
         *  The new label text. If omitted, the current label text will be
         *  removed from the button control.
         *
         * @returns {Button}
         *  A reference to this instance.
         */
        this.setLabelText = function (labelText) {
            Utils.setControlCaption(buttonNode, Utils.extendOptions(initOptions, { label: labelText }));
            return this;
        };

        // initialization -----------------------------------------------------

        // convert ENTER and SPACE keys to click events
        Utils.setButtonKeyHandler(buttonNode);

        // insert the button into this group, and register event handlers
        this.addFocusableControl(buttonNode)
            .registerUpdateHandler(updateHandler)
            .registerChangeHandler('click', { source: buttonNode, valueResolver: clickHandler });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            buttonNode.remove();

            // restore original focus() method of the root node DOM element (prevent leaks)
            buttonNode = self = initOptions = highlightHandler = null;
        });

    } // class Button

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: Button });

});
