/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/editframework/model/format/lineheight',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/text/app/config',
     'io.ox/office/text/view/controls',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox',
     'gettext!io.ox/office/text',
     'less!io.ox/office/text/view/style'
    ], function (Utils, KeyCodes, LineHeight, EditView, TextConfig, Controls, BaseControls, Pane, ToolBox, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        TextField = Controls.TextField,
        RadioGroup = Controls.RadioGroup,
        RadioList = Controls.RadioList;

    // class TextView =========================================================

    /**
     * @constructor
     *
     * @extends EditView
     */
    function TextView(app) {

        var // self reference
            self = this,

            // the document model
            model = null,

            // the scrollable document content node
            contentRootNode = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // debug operations for replay
            replayOperations = null,

            // edit field for the debug HTML code for the page
            pageHtmlCode = null,

            // the status label for the page number and zoom factor
            statusLabel = new BaseControls.StatusLabel(app, { type: 'info', fadeOut: true }),
            
            // tool pane floating over the bottom of the application pane
            bottomOverlayPane = null,

            // the tool box containing page and zoom control groups
            bottomToolBox = null,

            // predefined zoom factors
            ZOOM_FACTORS = [35, 50, 75, 100, 150, 200, 300, 400, 600, 800],

            // current zoom type (percentage or keyword)
            zoomType = 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            deferredInitHandler: deferredInitHandler,
            grabFocusHandler: grabFocusHandler,
            contentScrollable: true,
            contentMargin: 30,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler() {
            model.getNode().focus();
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection() {

            var // the boundaries of the text cursor
                boundRect = model.getSelection().getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect);
            }
        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            model.getSelection().registerInsertText(options);
        }

        /**
         * Configure the global 'Search and Replace' tool bar.
         */
        function configureSearchAndReplace() {

            var win = app.getWindow(),
                controller = app.getController(),
                scrollTimer = null,
                lastKeyTab = false,
                // the ox search panel
                searchDiv = win.nodes.search,
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                searchOffset,
                // additional controls for search & replace
                nextButton = $('<button>')
                    .addClass('btn btn-default margin-right search-next')
                    .append(Utils.createIcon('fa-chevron-right'))
                    .css({ borderRadius: '0 4px 4px 0', verticalAlign: 'top'})
                    .attr('tabindex', 0),
                previousButton = $('<button>')
                    .addClass('btn btn-default search-prev')
                    .append(Utils.createIcon('fa-chevron-left'))
                    .css({ borderRadius: '4px 0 0 4px', verticalAlign: 'top'})
                    .attr('tabindex', 0),
                replaceInput = $('<input>', { type: 'text', name: 'replace' })
                    .attr({placeholder: gt('Replace with ...'), tabindex: 0 })
                    .addClass('form-control replace-query'),
                replaceButton = $('<button>')
                    .addClass('btn btn-default')
                    .css({'border-radius': '0'})
                    .attr('tabindex', 0)
                    .text(gt('Replace')),
                replaceAllButton = $('<button>')
                    .addClass('btn btn-default')
                    .attr('tabindex', 0)
                    .text(/*#. in search&replace: replace all occurrences */ gt('Replace all')),
                clearReplaceInputIcon = Utils.createIcon('fa fa-times clear-query'),
                replaceInputGroup = $('<span>').addClass('input-group-btn'),
                replaceQueryContainer =  $('<div>')
                    .addClass('search-query-container replace-query-container input-group')
                    .css({ width: '400px', display: 'inline-table', verticalAlign: 'top'});

            replaceInputGroup.append(replaceButton, replaceAllButton);
            replaceQueryContainer.append(replaceInput, clearReplaceInputIcon, replaceInputGroup);

            // add next/prev search hit,replace input, replace and replace all buttons
            searchForm.find('.search-query-container').css({ display: 'inline-table'}).after(previousButton, nextButton, replaceQueryContainer);

            // Handles the 'search' event from the search panel.
            function searchHandler(event, searchQuery) {
                controller.executeItem('document/search/highlight', searchQuery);
            }

            // Handles the 'search:close' event from the search panel.
            function searchCloseHandler() {
                // since we add the replace input ourself, we need to clear it on close
                searchDiv.css('z-index', '1');
                win.nodes.search.find('input[name="replace"]').val('');
                controller.executeItem('document/search/close');
                if (scrollTimer) { scrollTimer.abort(); }
            }

            // Handles the 'search:open' event from the search panel.
            function searchOpenHandler() {
                searchDiv.css('z-index', '100');

                // set smaller search bar for mobile devices
                if (window.matchMedia('(orientation: portrait)').matches) { setSmallSearchPane(); }

                app.executeDelayed(function () {
                    // execute delayed, otherwise the offset is wrong
                    searchOffset = win.nodes.search.offset();
                    scrollTimer = app.repeatDelayed(scrollHandler, { delay: 1000, repeatDelay: 500 });
                });
            }

            // Handles the 'orientationchange' event of mobile devices
            function orientationChangeHandler(event) {
                // use event.orientation and media queries for the orientation detection.
                // 'window.orientation' depends on where the device defines it's top
                // and therefore says nothing reliable about the orientation.
                if (event && event.orientation === 'landscape') {
                    setLargeSearchPane();
                } else if (event && event.orientation === 'portrait') {
                    setSmallSearchPane();
                } else if (window.matchMedia('(orientation: landscape)').matches) {
                    setLargeSearchPane();
                } else if (window.matchMedia('(orientation: portrait)').matches) {
                    setSmallSearchPane();
                }
            }

            // Handles the 'scroll' event from the window.
            function scrollHandler() {

                var offset = _.clone(searchOffset);
                // if the window has been scrolled move by scrollY
                if (window.scrollY > searchOffset.top) {
                    offset.top = window.scrollY;
                }
                // set offset for the side pane
                if (!self.isSidePaneVisible()) {
                    offset.left = 0;
                }
                searchDiv.offset(offset);
                return win.search.active;
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    lastKeyTab = true;
                    event.preventDefault();

                    items = $(this).find('[tabindex]');
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;

                } else if (event.keyCode === KeyCodes.ESCAPE) {
                    win.search.close();
                }

                lastKeyTab = false;
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                if (lastKeyTab) {
                    event.stopImmediatePropagation();
                    // if we stop the event propagation we need to update the query here
                    win.search.query = win.search.getQuery();
                    return false;
                }
            }

            function setLargeSearchPane() {
                searchForm.find('.search-query-container').css({width: '300px'});
                searchForm.find('.replace-query-container').css({width: '400px'});
            }

            function setSmallSearchPane() {
                searchForm.find('.search-query-container').css({width: '180px'});
                searchForm.find('.replace-query-container').css({width: '330px'});
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            searchForm.addClass('f6-target');

            // set the tool tips
            Utils.setControlTooltip(searchInput, gt('Find text'), 'bottom');
            Utils.setControlTooltip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Utils.setControlTooltip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Utils.setControlTooltip(previousButton, gt('Select previous search result'), 'bottom');
            Utils.setControlTooltip(nextButton, gt('Select next search result'), 'bottom');
            Utils.setControlTooltip(replaceInput, gt('Replacement text'), 'bottom');
            Utils.setControlTooltip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Utils.setControlTooltip(replaceAllButton, gt('Replace all search results'), 'bottom');
            Utils.setControlTooltip(searchForm.find('a.close'), gt('Close search'), 'bottom');


            // event handling
            self.listenTo(previousButton, 'click', function () { controller.executeItem('document/search/previousResult'); });
            self.listenTo(nextButton, 'click', function () { controller.executeItem('document/search/nextResult'); });
            self.listenTo(replaceButton, 'click', function () { controller.executeItem('document/search/replaceSelected'); });
            self.listenTo(replaceAllButton, 'click', function () { controller.executeItem('document/search/replaceAll'); });
            self.listenTo(searchForm, 'keydown', keyHandler);
            self.listenTo(replaceInput, 'change', function () { win.search.replacement = Utils.cleanString(this.value || ''); });
            self.listenTo(searchInput, 'keydown', function (event) { if (KeyCodes.matchKeyCode(event, 'ENTER')) { event.preventDefault(); } });
            self.listenTo(clearReplaceInputIcon, 'click', function () { replaceInput.val(''); win.search.replacement = ''; });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler, self);

            self.listenTo(win, 'search', searchHandler);
            self.listenTo(win, 'search:close', searchCloseHandler);

            self.listenTo(app.getModel(), 'change:editmode', editmodeChangeHandler);

            // when scrolling on mobile devices we need to
            // place the search & replace into the view port
            if (Modernizr.touch) {
                self.listenTo(win, 'search:open', searchOpenHandler);
                self.listenTo($(window), 'orientationchange', orientationChangeHandler);
                orientationChangeHandler();
            }
        }

        /**
         * Shows the current zoom factor in the status label for a short time
         *
         */
        function updateZoomStatus() {
            _.defer(function () {
                statusLabel.setValue(
                    //#. %1$d is the current zoom factor, in percent
                    //#, c-format
                    gt('Zoom: %1$d%', _.noI18n(Math.round(zoomFactor)))
                );
            });
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // the root node of the entire application pane
                appPaneNode = self.getAppPaneNode(),
                // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1,
                // deferred and debounced scroll handler, registering insertText operations immediately
                scrollToSelectionDebounced = app.createDebouncedMethod(registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500 });

            // collect operations
            function collectOperations(event, operations, external) {
                if (model.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) {
                    return;
                } else {
                    delayedRecordingActive = true;
                    app.executeDelayed(function () {
                        var // full JSON string
                            fullJSONString = replayOperations.getFieldValue(),
                            // json part string to be appended
                            jsonPartString = '',
                            // current part to be processed
                            operations = pendingOperations.splice(0, 20),
                            // first entry
                            firstEntry = (fullJSONString.length === 0);

                        _(operations).each(function (entry) {

                            // skipping operations, that were 'optimized' in 'sendActions' in the
                            // optimizeOperationHandler. Otherwise 'merged operations' are still
                            // followed by the operations, that will not be sent to
                            // the server, because they were removed by the 'optimizeOperationHandler'.
                            var skipOperation = false,
                                opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                            // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                            // only happen for internal operations (task 29601)
                            if (! entry.external) {
                                if (_.isNumber(opl) && (opl > 1)) {
                                    oplCounter = opl;
                                } else if (oplCounter > 1) {
                                    oplCounter--;
                                    skipOperation = true;
                                }
                            }

                            if (! skipOperation) {
                                if (!firstEntry) {
                                    jsonPartString += ',';
                                }
                                jsonPartString += JSON.stringify(entry.operation);
                                firstEntry = false;
                            }
                        });

                        if (fullJSONString.length >= 2) {
                            if (fullJSONString.charAt(0) === '[') {
                                fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                            }
                        }
                        fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                        replayOperations.setValue(fullJSONString);

                        delayedRecordingActive = false;
                        if (pendingOperations.length > 0) {
                            recordOperations();
                        }
                    }, {delay: 50});
                }
            }

            // initialize other instance fields
            model = app.getModel();
            contentRootNode = self.getContentRootNode();

            // configure the OX search bar for OX Text search&replace
            configureSearchAndReplace();

            // insert the editor root node into the application pane
            self.insertContentNode(model.getNode());

            // make the application pane focusable for global navigation with F6 key
            appPaneNode.addClass('f6-target');

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    _.defer(function () { appPaneNode.css('opacity', 1); });
                }
            });

            // disable drag&drop for the content root node to prevent dropping images which are
            // loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);

            // handle selection change events
            self.listenTo(model, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor
                scrollToSelectionDebounced(options);
                // do user data update
                var userData = {
                    selection : {
                        type: selection.getSelectionType(),
                        start: selection.getStartPosition(),
                        end: selection.getEndPosition()
                    }
                };
                if (app.getActiveClients() > 1) {
                    // fire and forget user data to server, only if we are in collaborative situation
                    app.updateUserData(userData);
                }
            });

            // log current selection in debug mode
            if (Utils.DEBUG) {

                // create the output nodes in the debug pane
                self.addDebugInfoHeader('selection', { tooltip: 'Current selection' })
                    .addDebugInfoNode('type', { tooltip: 'Type of the selection' })
                    .addDebugInfoNode('start', { tooltip: 'Start position' })
                    .addDebugInfoNode('end', { tooltip: 'End position' })
                    .addDebugInfoNode('dir', { tooltip: 'Direction' });

                // log all selection events
                self.listenTo(model, 'selection', function (event, selection) {
                    self.setDebugInfoText('selection', 'type', selection.getSelectionType())
                        .setDebugInfoText('selection', 'start', selection.getStartPosition().join(', '))
                        .setDebugInfoText('selection', 'end', selection.getEndPosition().join(', '))
                        .setDebugInfoText('selection', 'dir', selection.isTextCursor() ? 'cursor' : selection.isBackwards() ? 'backwards' : 'forwards');
                });

                // operations to replay
                replayOperations = new TextField({ tooltip: _.noI18n('Paste operations here'), select: true, fullWidth: false });

                // page html code
                pageHtmlCode = new TextField({ tooltip: _.noI18n('Page html code'), select: true, fullWidth: false });

                // record operations after importing the document
                app.on('docs:import:after', function () {
                    // connect the 'operations:after' event to the collectOperations function
                    self.listenTo(app.getModel(), 'operations:after', collectOperations);
                });
            }
        }


        /**
         * Calculating maximum values for the columns and the rows for the TableSizePicker.
         * Some extra calculation is necessary, if the maximum value for the cells is smaller
         * than the product of the maximum values for the rows and columns.
         */
        function calculateTableSizeOptions(maxCols, maxRows, maxCells) {

            var cols = maxCols,
                rows = maxRows;

            // calculating values for max rows and max columns taking care of max cells. This is necessary, if the maximum
            // number of cells is smaller than the product of maximum number of rows and columns (28805).
            if (_.isNumber(cols) && _.isNumber(rows) && _.isNumber(maxCells) && (maxCells < (cols * rows))) {

                // First check: Using the square root of the number of cells, so that there are equal values for columns and rows.
                rows = Utils.roundDown(Math.sqrt(maxCells), 1);
                cols = rows;

                // Second check: Comparing this new calculated values with the given maximum values.
                if (rows > maxRows) {
                    rows = maxRows;
                    cols = Utils.roundDown(maxCells / rows, 1);
                } else if (cols > maxCols) {
                    cols = maxCols;
                    rows = Utils.roundDown(maxCells / cols, 1);
                }

                Utils.log('Recalculating maxRows and maxCols corresponding to maxCells for inserting new table. New values: Cells: ' + maxCells + ' Rows: ' + rows + ' Cols: ' + cols);
            }

            return { maxCols: cols, maxRows: rows, maxCells: maxCells };
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            var // ODF does not support inner borders at paragraphs -> special option only for this case
                displayInnerBorders = !app.isODF(),
                // the table tool box container
                tableToolBox = null,
                // the insert tool box container
                insertToolBox = null,
                // optional table size settings
                tableSizeOptions = calculateTableSizeOptions(TextConfig.getMaxTableColumns(), TextConfig.getMaxTableRows(), TextConfig.getMaxTableCells()),
                // whether it shall be possible to insert tables (Fix for 28945)
                hideInsertTable = (_.isNumber(tableSizeOptions.maxCols) && tableSizeOptions.maxCols === 0) ||
                                  (_.isNumber(tableSizeOptions.maxRows) && tableSizeOptions.maxRows === 0) ||
                                  (_.isNumber(tableSizeOptions.maxCells) && tableSizeOptions.maxCells === 0);

            self.createToolBox('font', { label: gt('Font'), visible: 'document/editable/text' })
                .addGroup('character/fontname', new Controls.FontFamilyPicker(app, { width: 117 }))
                .addRightTab()
                .addGroup('character/fontsize', new Controls.FontHeightPicker({ width: 47 }))
                .newLine()
                .addGroup('character/bold',      new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic',    new Button(Controls.ITALIC_OPTIONS))
                .addGroup('character/underline', new Button(Controls.UNDERLINE_OPTIONS))
                .addGroup('character/strike',    new Button(Controls.STRIKEOUT_OPTIONS))
                .addGap(11)
                .addGroup('character/vertalign', new RadioGroup({ toggleValue: 'baseline' })
                    .createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript') })
                    .createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript') })
                )
                .newLine()
                .addGroup('character/color',     new Controls.ColorPicker(app, 'text', { icon: 'docs-font-color',      tooltip: gt('Text color') }))
                .addGap()
                .addGroup('character/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-font-fill-color', tooltip: /*#. fill color behind single characters */ gt('Text highlight color') }))
                .addRightTab()
                .addGroup('character/resetAttributes', new Button(Controls.CLEAR_FORMAT_OPTIONS));

            self.createToolBox('paragraph', { label: gt('Paragraph'), visible: 'document/editable/text' })
                .addGroup('paragraph/stylesheet', new Controls.ParagraphStylePicker(app, { fullWidth: true }))
                .newLine()
                .addGroup('paragraph/alignment', new RadioList({ icon: 'docs-para-align-left', tooltip: gt('Paragraph alignment'), updateCaptionMode: 'icon' })
                    .createOptionButton('', 'left',    { icon: 'docs-para-align-left',    label: /*#. alignment of text in paragraphs or cells */ gt('Left') })
                    .createOptionButton('', 'center',  { icon: 'docs-para-align-center',  label: /*#. alignment of text in paragraphs or cells */ gt('Center') })
                    .createOptionButton('', 'right',   { icon: 'docs-para-align-right',   label: /*#. alignment of text in paragraphs or cells */ gt('Right') })
                    .createOptionButton('', 'justify', { icon: 'docs-para-align-justify', label: /*#. alignment of text in paragraphs or cells */ gt('Justify') })
                )
                .addGap()
                .addGroup('paragraph/lineheight', new RadioList({ icon: 'docs-para-line-spacing-100', tooltip: gt('Line spacing'), updateCaptionMode: 'icon' })
                    .createOptionButton('', LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') })
                    .createOptionButton('', LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') })
                    .createOptionButton('', LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') })
                    .createOptionButton('', LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') })
                )
                .addGap()
                .addGroup('paragraph/borders', new Controls.BorderPicker({ tooltip: gt('Paragraph borders'), showInsideHor: displayInnerBorders }))
                .addGap()
                .addGroup('paragraph/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-para-fill-color', tooltip: gt('Paragraph fill color') }))
                .newLine()
                .addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(app))
                .addGap()
                .addGroup('paragraph/list/numbered', new Controls.NumberedListStylePicker(app))
                .addRightTab()
                .addGroup('paragraph/list/decindent', new Button({ icon: 'docs-list-dec-level', tooltip: /*#. indentation of lists (one list level up) */ gt('Demote one level') }))
                .addGroup('paragraph/list/incindent', new Button({ icon: 'docs-list-inc-level', tooltip: /*#. indentation of lists (one list level down) */ gt('Promote one level') }));

            tableToolBox = self.createToolBox('table', { label: gt('Table'), visible: 'document/editable/table' });

            if (!app.isODF()) {
                tableToolBox
                    .addGroup('table/stylesheet', new Controls.TableStylePicker(app))
                    .addRightTab();
            }

            tableToolBox
                .addGroup('table/insert/row',    new Button({ icon: 'docs-table-insert-row',    tooltip: gt('Insert row') }))
                .addGroup('table/insert/column', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('table/delete/row',    new Button({ icon: 'docs-table-delete-row',    tooltip: gt('Delete selected rows') }))
                .addGroup('table/delete/column', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }));


            if (!app.isODF()) {
                tableToolBox
                    .newLine()
                    .addGroup('table/cellborder', new Controls.BorderPicker({ tooltip: gt('Cell borders'), showInsideHor: true, showInsideVert: true }))
                    .addGap()
                    .addGroup('table/borderwidth', new Controls.TableBorderWidthPicker({ tooltip: gt('Cell border width') }))
                    .addGap()
                    .addGroup('table/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }));
            }

            self.createToolBox('drawing', { label: /*#. drawing objects: images, diagrams, ... */ gt('Drawing'), visible: 'document/editable/drawing' })
                .addGroup('drawing/position', new RadioList({ icon: 'docs-drawing-inline', tooltip: /*#. alignment and text floating of drawing objects in text */ gt('Drawing position'), updateCaptionMode: 'icon' })
                    .createMenuSection('inline')
                    .createOptionButton('inline', 'inline',       { icon: 'docs-drawing-inline',       label: gt('Inline with text') })
                    .createMenuSection('left')
                    .createOptionButton('left', 'left:none',      { icon: 'docs-drawing-left-none',    label: gt('Left aligned, no text wrapping') })
                    .createOptionButton('left', 'left:right',     { icon: 'docs-drawing-left-right',   label: gt('Left aligned, text wraps at right side') })
                    .createMenuSection('right')
                    .createOptionButton('right', 'right:none',    { icon: 'docs-drawing-right-none',   label: gt('Right aligned, no text wrapping') })
                    .createOptionButton('right', 'right:left',    { icon: 'docs-drawing-right-left',   label: gt('Right aligned, text wraps at left side') })
                    .createMenuSection('center')
                    .createOptionButton('center', 'center:none',  { icon: 'docs-drawing-center-none',  label: gt('Centered, no text wrapping') })
                    .createOptionButton('center', 'center:left',  { icon: 'docs-drawing-center-left',  label: gt('Centered, text wraps at left side') })
                    .createOptionButton('center', 'center:right', { icon: 'docs-drawing-center-right', label: gt('Centered, text wraps at right side') })
                )
                .addRightTab()
                .addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_OPTIONS));

            insertToolBox = self.createToolBox('insert', { label: gt('Insert'), visible: 'document/editable/text' });
            if (! hideInsertTable) {
                insertToolBox
                    .addGroup('table/insert', new Controls.TableSizePicker(app, tableSizeOptions))
                    .addGap();
            }
            insertToolBox
                .addGroup('character/hyperlink/dialog', new Button({ icon: 'docs-hyperlink',        tooltip: gt('Insert/Edit hyperlink') }))
                .addGap()
                .addGroup('image/insert/dialog',        new Button({ icon: 'docs-image-insert',     tooltip: gt('Insert image') }))
                .addRightTab()
                .addGroup('character/insertTab',        new Button({ icon: 'docs-insert-tab',       tooltip: gt('Insert tab') }))
                .addGroup('character/insertLineBreak',  new Button({ icon: 'docs-insert-linebreak', tooltip: gt('Insert line break') }));

            self.createToolBox('spelling', { label: gt('Spelling'), visible: 'document/spelling/enabled' })
                .addGroup('character/language', new Controls.LanguagePicker({ width: 178 }))
                .addGap()
                .addGroup('document/onlinespelling', new Button({ icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true }));

            self.getOverlayToolBox()
                .addGroup('character/bold',   new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic', new Button(Controls.ITALIC_OPTIONS));

            if (Utils.DEBUG) {
                self.createDebugToolBox()
                    .addGap()
                    .addGroup('document/cut',   new Button({ icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                    .addGroup('document/copy',  new Button({ icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                    .addGroup('document/paste', new Button({ icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }))
                    .newLine()
                    .addGroup('format/spellchecking', new Button({ label: _.noI18n('ab'), tooltip: gt('Check spelling of selected text') }))
                    .addGap()
                    .addGroup('debug/accessrights', new Controls.AccessRightsPicker(app))
                    .newLine()
                    .addGroup('debug/recordoperations', new Button({ icon: 'fa-play-circle-o', css: { color: 'red' }, toggle: true, tooltip: _.noI18n('Record operations')}))
                    .addGroup('debug/replayoperations', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Replay operations')}))
                    .addGroup('debug/operations', replayOperations)
                    .newLine()
                    .addGroup('debug/getpagehtmlcode', new Button({ icon: 'docs-undo', tooltip: _.noI18n('Get page HTML mark-up')}))
                    .addGroup('debug/setpagehtmlcode', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Set page HTML mark-up')}))
                    .addGroup('debug/pagehtmlcode', pageHtmlCode)
                    .newLine()
                    .addGroup('debug/togglepagebreaks', new Button({ label: _.noI18n('Paper roll'), tooltip: _.noI18n('Toggle page/paper roll layout'), toggle: true }));
            }

            self.createToolBox('zoom', { fixed: 'bottom' })
            .addGroup('zoom/dec',  new Button(BaseControls.ZOOMOUT_OPTIONS))
            .addGroup('zoom/type', new Controls.ZoomTypePicker({ position: 'top', align: 'center' }))
            .addGroup('zoom/inc',  new Button(BaseControls.ZOOMIN_OPTIONS));

            // create the bottom overlay pane
            self.addPane(bottomOverlayPane = new Pane(app, 'overlaybottom', { position: 'bottom', classes: 'inline right', overlay: true, transparent: true, hoverEffect: true })
                .addViewComponent(bottomToolBox = new ToolBox(app, 'overlaypages'))
            );

            bottomToolBox
                .addGroup('zoom/dec',  new Button(BaseControls.ZOOMOUT_OPTIONS))
                .addGroup('zoom/type', new Controls.ZoomTypePicker())
                .addGroup('zoom/inc',  new Button(BaseControls.ZOOMIN_OPTIONS));

            // create the status overlay pane
            self.addPane(new Pane(app, 'statuslabel', { position: 'bottom', classes: 'inline right', overlay: true, transparent: true })
                .addViewComponent(new ToolBox(app, 'statuslabel', { landmark: false }).addPrivateGroup(statusLabel))
            );

            // update visibility of bottom overlay pane
            self.listenTo(self.getSidePane(), 'pane:show', function (event, visible) { bottomOverlayPane.toggle(!visible); });
            bottomOverlayPane.toggle(!self.isSidePaneVisible());

            self.on('refresh:layout', function () {
                var // the app content node
                    appContentRootNode = self.getContentRootNode(),
                    // the page attributes contain the width of the page in hmm
                    pageAttribute = app.getModel().getDocumentStyles().getStyleSheets('page').getElementAttributes(app.getModel().getNode()),
                    // whether the page padding need to be reduced
                    reducePagePadding = (pageAttribute.page.width > Utils.convertLengthToHmm(appContentRootNode.width(), 'px'));

                // check, if there is sufficient space for left and right padding on the page
                appContentRootNode.toggleClass('small-device', reducePagePadding);

                if (zoomType === 'width') {
                    _.defer(function () { self.setZoomType('width'); });
                }
            });

            self.listenTo(app.getModel(), 'operations:after', this.recalculateDocumentMargin);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node) {
            Utils.scrollToChildNode(contentRootNode, node, { padding: 15 });
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            Utils.scrollToPageRectangle(contentRootNode, pageRect, { padding: 15 });
            return this;
        };

        /**
         * Provides the debug operations as string for the debugging
         * feature replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field or null
         *  if there is no debug toolbox.
         */
        this.getDebugOperationsContent = function () {
            if (replayOperations) {
                return replayOperations.getFieldValue();
            }
            return null;
        };

        /**
         * Receiving the html code of the edit field to insert it
         * into the document page.
         *
         * @returns {String}
         *  The content of the debug html code field or null
         *  if there is no debug toolbox or no content in it.
         */
        this.getPageHtmlCode = function () {
            var value = null;
            if (pageHtmlCode) {
                value = pageHtmlCode.getFieldValue();
                if (value === '') { value = null; }
            }
            return value;
        };

        /**
         * Setting html code into the debug edit field.
         *
         * @param {String} htmlCode
         *  The html code as string that will be displayed in
         *  the debug edit field.
         */
        this.setPageHtmlCode = function (htmlCode) {
            if (pageHtmlCode) {
                pageHtmlCode.setFieldValue(htmlCode);
            }
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            var
                cachedDocumentParentNode = this.getAppPaneNode(),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text')),
                documentWidth = cachedDocumentNode.width() + parseInt(cachedDocumentNode.css('padding-left'), 10) + parseInt(cachedDocumentNode.css('padding-right'), 10),
                parentWidth = cachedDocumentParentNode.width(),
                outerDocumentMargin = parseInt(cachedDocumentParentNode.find('.app-content').css('margin-left'), 10) + parseInt(cachedDocumentParentNode.find('.app-content').css('margin-right'), 10);

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'width') {
                this.increaseZoomLevel(((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMargin))) * 100); //20 is width of scrollbar
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                //marginAddition = 0,
                cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text'));

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
            Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform-origin', 'top');

            this.recalculateDocumentMargin();
            cachedAppContentRoot.css({
                'overflow': zoomInsertion > 99 ? '' : 'hidden'
            });

            // Use zoom as soon as all browsers support it.
            // cachedDocumentNode.css({
            //    'zoom': zoomInsertion / 100
            // });

            updateZoomStatus();
            zoomType = zoomFactor;
            scrollToSelection();
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                //marginAddition = 0,
                cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text'));

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
            Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform-origin', 'top');

            this.recalculateDocumentMargin();
            cachedAppContentRoot.css({
                'overflow': zoomInsertion > 99 ? '' : 'hidden'
            });

            // Use zoom as soon as all browsers support it.
            //  cachedDocumentNode.css({
            //      'zoom': zoomInsertion / 100
            //  });

            updateZoomStatus();
            zoomType = zoomFactor;
            scrollToSelection();
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function () {
            var
                marginAddition = 0,
                //cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = self.getAppPaneNode().find('.page.user-select-text'),
                initialWidth = cachedDocumentNode.width() + parseInt(cachedDocumentNode.css('padding-left'), 10) + parseInt(cachedDocumentNode.css('padding-right'), 10),
                initialHeight = cachedDocumentNode.height() + parseInt(cachedDocumentNode.css('padding-top'), 10) + parseInt(cachedDocumentNode.css('padding-bottom'), 10);

            if (zoomFactor < 99) {
                marginAddition = 3;
            }
            if (_.browser.Chrome && zoomFactor > 100) {
                cachedDocumentNode.css({
                    'margin':  marginAddition + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px ' + (30 / (zoomFactor / 100)) + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px'
                });
            } else {
                cachedDocumentNode.css({
                    'margin':  marginAddition + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px ' + (initialHeight * (zoomFactor / 100 - 1) + marginAddition) + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px'
                });
            }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
            case 'tablesizerow':
                this.yell('info', gt('The table reached its maximum size. You cannot insert further rows.'));
                break;

            case 'tablesizecolumn':
                this.yell('info', gt('The table reached its maximum size. You cannot insert further columns.'));
                break;
            }

            return this;
        };

        /**
         * Toggles Document layout - with or without page breaks
         *
         */
        this.setPageBreaks = function () {
            model.togglePageBreaks();
        };

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = pageHtmlCode = replayOperations = model = null;
        });

    } // class TextView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: TextView });

});
