/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/headertrackingmixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils'
    ], function (Utils, SheetUtils, PaneUtils) {

    'use strict';

    // class HeaderTrackingMixin ==============================================

    /**
     * Mix-in class for the class HeaderPane that implements all kinds of mouse
     * and touch tracking, for example selection, or manipulation of column/row
     * sizes.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this header pane.
     *
     * @param {String} paneSide
     *  The pane side identifier of this header pane.
     *
     * @param {jQuery} contentNode
     *  The content node with the header cell and resizer nodes
     */
    function HeaderTrackingMixin(app, paneSide, contentNode) {

        var // self reference
            self = this,

            // whether this header pane contains columns
            columns = PaneUtils.isColumnSide(paneSide),

            // the spreadsheet model and view
            model = null,
            view = null,

            // the model and collections of the active sheet
            sheetModel = null,
            collection = null,

            // the type of the current tracking
            trackingType = null,

            // whether current tracking can be continued in read-only mode
            readOnlyTracking = false;

        // private methods ----------------------------------------------------

        /**
         * Handles a changed document edit mode. If switched to read-only mode,
         * cancels in-place cell edit mode, or a running tracking action to
         * move or resize drawing objects.
         */
        function editModeHandler(event, editMode) {
            // stop tracking if edit rights have been lost
            if (!editMode && !readOnlyTracking) {
                $.cancelTracking();
            }
        }

        /**
         * Initializes this grid pane after the active sheet has been changed.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();
        }

        /**
         * Cancels resize tracking when the selection changes.
         */
        function changeSelectionHandler() {
            if (trackingType === 'resize') {
                $.cancelTracking();
            }
        }

        /**
         * Returns the layout data of the column/row matching the 'data-index'
         * attribute of the target DOM node in the passed 'tracking:start'
         * event.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event.
         *
         * @returns {Object}
         *  The layout data of the matching column/row, in the properties
         *  'index', 'offset', and 'size'.
         */
        function getTrackingStartEntryData(event) {
            var indexNode = $(event.target).closest('[data-index]');
            return collection.getEntry(Utils.getElementAttributeAsInteger(indexNode, 'data-index', -1));
        }

        /**
         * Returns the current absolute sheet position in the header pane for
         * the passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Number}
         *  The absolute sheet position, in pixels.
         */
        function getTrackingOffset(event) {
            return (columns ? (event.pageX - contentNode.offset().left) : (event.pageY - contentNode.offset().top)) + self.getIntervalPosition().offset;
        }

        /**
         * Returns the column/row collection entry matching the position in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object}
         *  The column/row collection entry for the tracking position.
         */
        function getTrackingEntry(event, options) {
            return collection.getEntryByOffset(getTrackingOffset(event), Utils.extendOptions(options, { pixel: true }));
        }

        /**
         * Handles all tracking events while cell selection is active, and
         * triggers the appropriate selection events to the own listeners.
         */
        var selectionTrackingHandler = (function () {

            var // current index (prevent updates while tracking over the same column/row)
                currentIndex = 0;

            // initializes selection tracking
            function initializeTracking(event) {
                currentIndex = getTrackingStartEntryData(event).index;
                self.scrollToEntry(currentIndex);
                sheetModel.setViewAttributes({
                    activePane: PaneUtils.getNextPanePos(sheetModel.getViewAttribute('activePane'), paneSide),
                    activePaneSide: paneSide
                });
                self.trigger('select:start', currentIndex, PaneUtils.getSelectionMode(event));
            }

            // updates the current index according to the passed tracking event
            function updateTracking(event) {
                var index = getTrackingEntry(event, { outerHidden: true }).index;
                if (index !== currentIndex) {
                    currentIndex = index;
                    self.trigger('select:move', currentIndex);
                }
            }

            // finalizes selection tracking
            function finalizeTracking(event, apply) {
                self.scrollToEntry(currentIndex);
                sheetModel.setViewAttribute('activePaneSide', null);
                self.trigger('select:end', currentIndex, apply);
            }

            // return the actual selectionTrackingHandler() method
            return function (event) {
                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    event.preventDefault(); // prevent native scrolling on touch devices
                    break;
                case 'tracking:move':
                    updateTracking(event);
                    event.preventDefault(); // prevent native scrolling on touch devices
                    break;
                case 'tracking:scroll':
                    self.scrollRelative(columns ? event.scrollX : event.scrollY);
                    updateTracking(event);
                    break;
                case 'tracking:end':
                    finalizeTracking(event, true);
                    break;
                case 'tracking:cancel':
                    finalizeTracking(event, false);
                    break;
                }
            };
        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events while resizing a column/row is active.
         */
        var resizeTrackingHandler = (function () {

            var // the layout data of the resized column/row
                entryData = null,
                // difference between exact start position and trailing position of column/row
                correction = 0,
                // the current size of the column/row while tracking, in pixels
                currentSize = 0,
                // minimum/maximum scroll position allowed while resizing
                minScrollPos = 0, maxScrollPos = 0,
                // maximum size of columns/rows
                MAX_SIZE = Utils.convertHmmToLength(columns ? SheetUtils.MAX_COLUMN_WIDTH : SheetUtils.MAX_ROW_HEIGHT, 'px', 1);

            // initializes column/row resizing according to the passed tracking event
            function initializeTracking(event) {

                var position = self.getVisiblePosition(),
                    hiddenSize = position.offset - self.getScrollPos();

                entryData = getTrackingStartEntryData(event);
                // calculate difference between exact tracking start position and end of column/row
                // (to prevent unwanted resize actions without moving the tracking point)
                correction = getTrackingOffset(event) - (entryData.offset + entryData.size);
                currentSize = entryData.size;
                minScrollPos = Math.max(0, entryData.offset - 20 - hiddenSize);
                maxScrollPos = Math.max(minScrollPos, entryData.offset + MAX_SIZE + 20 - position.size - hiddenSize);
                self.trigger('resize:start', entryData.offset, currentSize);
            }

            // updates column/row resizing according to the passed tracking event
            function updateTracking(event) {
                currentSize = Utils.minMax(getTrackingOffset(event) - correction - entryData.offset, 0, MAX_SIZE);
                self.trigger('resize:move', entryData.offset, currentSize);
            }

            // updates scroll position according to the passed tracking event
            function updateScroll(event) {

                var scrollPos = self.getScrollPos(),
                    newScrollPos = scrollPos + (columns ? event.scrollX : event.scrollY);

                if ((newScrollPos < scrollPos) && (scrollPos > minScrollPos)) {
                    self.scrollTo(Math.max(newScrollPos, minScrollPos));
                } else if ((newScrollPos > scrollPos) && (scrollPos < maxScrollPos)) {
                    self.scrollTo(Math.min(newScrollPos, maxScrollPos));
                }
            }

            // finalizes the resize tracking
            function finalizeTracking() {
                self.trigger('resize:end');
            }

            // execute the resize operation
            function applyResize() {
                if (currentSize !== entryData.size) {
                    // pass tracked column/row as target index, will cause to modify all columns/rows selected completely
                    view[columns ? 'setColumnWidth' : 'setRowHeight'](sheetModel.convertPixelToHmm(currentSize), { custom: true, target: entryData.index });
                }
            }

            // return the actual resizeTrackingHandler() method
            return function (event) {
                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    event.preventDefault(); // prevent native scrolling on touch devices
                    break;
                case 'tracking:move':
                    updateTracking(event);
                    event.preventDefault(); // prevent native scrolling on touch devices
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateTracking(event);
                    break;
                case 'tracking:end':
                    finalizeTracking(event);
                    applyResize();
                    break;
                case 'tracking:cancel':
                    finalizeTracking(event);
                    break;
                }
            };
        }()); // end of resizeTrackingHandler() local scope

        /**
         * Handles 'tracking:start' events end decides whether to select cells,
         * or to resize columns or rows.
         */
        function trackingStartHandler(event) {

            var // target node
                targetNode = $(event.target);

            // starts tracking with the specified type and event handler
            function startTracking(type, trackingHandler, options) {

                // store the tracking type
                trackingType = type;
                // store whether tracking in read-only mode is supported
                readOnlyTracking = Utils.getBooleanOption(options, 'readOnlyTracking', false);

                // handle all tracking events for this tracking cycle
                PaneUtils.processTrackingCycle(event, trackingHandler, function () {
                    trackingType = null;
                    view.grabFocus();
                });
            }

            if (targetNode.closest('.cell').length > 0) {
                startTracking('select', selectionTrackingHandler, { readOnlyTracking: true });
            } else if (targetNode.closest('.resizer,.touchresizer').length > 0) {
                startTracking('resize', resizeTrackingHandler);
            }
        }

        /**
         * Handles 'dblclick' events on resizer nodes and resizes the
         * column/row to the optimal width/height depending on cell content.
         */
        function doubleClickHandler(event) {

            var // the column/row index
                index = collection.getEntry(Utils.getElementAttributeAsInteger(event.target, 'data-index', -1)).index;

            if (columns) {
                view.setOptimalColumnWidth(index);
            } else {
                view.setOptimalRowHeight(index);
            }

            // return focus to active grid pane, after the event has been processed
            _.defer(function () { view.grabFocus(); });
        }

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', function () {

            // the spreadsheet model and view instances
            model = app.getModel();
            view = app.getView();

            // enable/disable operations that modify the document according to edit mode
            model.on('change:editmode', editModeHandler);

            // listen to view events
            view.on({
                'change:activesheet': changeActiveSheetHandler,
                'change:selection': changeSelectionHandler
            });

            // tracking for column/row selection with mouse/touch
            self.getNode().on('tracking:start', trackingStartHandler);

            // double click on resizer for column/row
            self.getNode().on('dblclick', '.resizer', doubleClickHandler);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = view = sheetModel = collection = null;
        });

    } // class HeaderTrackingMixin

    // exports ================================================================

    return HeaderTrackingMixin;

});
