/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/view/cellstylepicker',
     'io.ox/office/spreadsheet/view/mixin/orderdndmixin',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, EditControls, SheetUtils, CellStylePicker, OrderDnDMixin, gt) {

    'use strict';

    var // class name shortcuts
        Button = EditControls.Button,
        RadioGroup = EditControls.RadioGroup,
        RadioList = EditControls.RadioList,

        // predefined number format categories
        NUMBER_FORMAT_CATEGORIES = [
            { value: 'standard',   label: /*#. Number format category in spreadsheets: no special format, show numbers as they are */ gt('General') },
            { value: 'number',     label: /*#. Number format category in spreadsheets: numbers with specific count of decimal places */ gt('Number') },
            { value: 'currency',   label: /*#. Number format category in spreadsheets: number with currency symbol */ gt('Currency') },
            { value: 'date',       label: /*#. Number format category in spreadsheets: date formats */ gt('Date') },
            { value: 'time',       label: /*#. Number format category in spreadsheets: time formats */ gt('Time') },
            { value: 'datetime',   label: /*#. Number format category in spreadsheets: combined date/time formats */ gt('Date and time') },
            { value: 'percent',    label: /*#. Number format category in spreadsheets: numbers with percent sign */ gt('Percentage') },
            { value: 'text',       label: /*#. Number format category in spreadsheets: text only */ gt('Text')},
            { value: 'scientific', label: /*#. Number format category in spreadsheets: scientific notation (e.g. 1.23E+10) */ gt('Scientific') },
            { value: 'fraction',   label: /*#. Number format category in spreadsheets: fractional numbers (e.g. 3 1/4) */ gt('Fraction') },
            { value: 'custom',     label: /*#. Number format category in spreadsheets: all user-defined number formats */ gt('Custom') }
        ];

    // static class SpreadsheetControls =======================================

    /**
     * Additional classes defining specialized GUI controls for the OX
     * Spreadsheet application.
     *
     * @extends EditControls
     */
    var SpreadsheetControls = _.clone(EditControls);

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    SpreadsheetControls.ActiveSheetGroup = RadioGroup.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // get application model and view
            model = app.getModel(),
            view = app.getView(),

            // the scroll button nodes
            prevButton = Utils.createButton({ icon: 'fa-angle-left' }).addClass('scroll-button'),
            nextButton = Utils.createButton({ icon: 'fa-angle-right' }).addClass('scroll-button'),

            // index of the first and last sheet tab button currently visible
            firstIndex = 0,
            lastIndex = 0,

            // whether scroll mode is enabled
            scrollable = false;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'active-sheet-group' });
        OrderDnDMixin.call(this, app, { cursor: 'ew-resize', icon: 'fa-caret-down' });

        // private methods ----------------------------------------------------

        /**
         * Shows all sheet buttons and removes explicit width, and hides the
         * scroll buttons.
         */
        function showAllSheetButtons() {
            self.getOptionButtons().removeClass('hidden').css('width', '');
            prevButton.addClass('hidden');
            nextButton.addClass('hidden').css({ position: '', right: '' });
            scrollable = false;
        }

        /**
         * Shows the sheet buttons starting at the start sheet index currently
         * cached.
         */
        function renderSheetButtons() {

            var // the root node of this group
                groupNode = self.getNode(),
                // the inner width of the group (space available for buttons)
                groupWidth = groupNode.width(),
                // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // the width of all sheet buttons
                buttonsWidth = 0,
                // relative start/end position of all sheet buttons
                buttonPositions = null,
                // total width of all buttons, including scroll buttons
                totalWidth = 0,
                // remaining width available in the group node for sheet buttons
                remainingWidth = 0,
                // maximum index of button that can be shown at the left border (without gap to right border)
                maxIndex = 0;

            // do nothing if there are no sheets available (yet)
            if (sheetButtons.length === 0) { return; }

            // initialize all sheet buttons, hide scroll buttons
            showAllSheetButtons();
            buttonsWidth = sheetButtons.last().offset().left + sheetButtons.last().outerWidth() - sheetButtons.first().offset().left;
            if (buttonsWidth <= groupWidth) { return; }

            // collect position information of all buttons (performance: once for all buttons, before manipulating DOM again)
            var leadingOffset = sheetButtons.first().offset().left;
            buttonPositions = sheetButtons.map(function () {
                var buttonOffset = $(this).offset().left - leadingOffset;
                return {
                    offset: buttonOffset,
                    leadingWidth: buttonOffset + $(this).outerWidth(),
                    trailingWidth: buttonsWidth - buttonOffset
                };
            });

            // do not show scroll buttons, if there is only one visible sheet
            if (sheetButtons.length > 1) {

                // show scroll buttons
                prevButton.removeClass('hidden');
                nextButton.removeClass('hidden');
                scrollable = true;

                // get total width of all buttons, including scroll buttons, and inner remaining width
                totalWidth = nextButton.offset().left + nextButton.outerWidth() - prevButton.offset().left;
                remainingWidth = groupWidth - (totalWidth - buttonsWidth);

                // hide leading sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.trailingWidth <= remainingWidth; }, { sorted: true });
                firstIndex = Utils.minMax(firstIndex, 0, Math.max(0, maxIndex));
                sheetButtons.slice(0, firstIndex).addClass('hidden');

                // adjust button positions relative to first visible button
                var relOffset = buttonPositions[firstIndex].offset;
                _(buttonPositions).each(function (position) {
                    position.offset -= relOffset;
                    position.leadingWidth -= relOffset;
                });

                // hide trailing sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.leadingWidth > remainingWidth; }, { sorted: true });
                lastIndex = ((maxIndex < 0) ? sheetButtons.length : maxIndex) - 1;
                sheetButtons.slice(lastIndex + 1).addClass('hidden');

                // enable/disable scroll buttons
                prevButton.toggleClass(Utils.DISABLED_CLASS, firstIndex === 0);
                nextButton.toggleClass(Utils.DISABLED_CLASS, lastIndex + 1 === sheetButtons.length);

            } else {
                // single sheet: initialize indexes
                firstIndex = lastIndex = 0;
                totalWidth = buttonsWidth;
                remainingWidth = groupWidth;
            }

            // set fixed position for the right scroll button
            nextButton.css({ position: 'absolute', right: 0 });

            // shrink single visible sheet button, if still oversized
            var sheetButton = sheetButtons.eq(firstIndex);
            if ((firstIndex === lastIndex) && (remainingWidth < sheetButton.outerWidth())) {
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
                return;
            }

            // add the first hidden sheet button after the last visible, if there is some space left
            if (lastIndex + 1 === buttonPositions.length) { return; }
            remainingWidth -= buttonPositions[lastIndex + 1].offset;
            if (remainingWidth >= 30) {
                sheetButton = sheetButtons.eq(lastIndex + 1);
                sheetButton.removeClass('hidden');
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
            }
        }

        /**
         * Scrolls to the left to make at least one preceding sheet tab button
         * visible.
         */
        function scrollToPrev() {

            var // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to previous sheets until the index of the last sheet has changed
            while ((firstIndex > 0) && (oldLastIndex === lastIndex)) {
                firstIndex -= 1;
                renderSheetButtons();
            }
        }

        /**
         * Scrolls to the right to make at least one following sheet tab button
         * visible.
         */
        function scrollToNext() {

            var // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to next sheets until the index of the last sheet has changed
            while ((lastIndex + 1 < sheetButtons.length) && (oldLastIndex === lastIndex)) {
                firstIndex += 1;
                renderSheetButtons();
            }
        }

        /**
         * Handles tracking on the scroll-back button. Stops event propagation,
         * to not interfere with sheet move tracking.
         */
        function prevTrackingHandler(event) {
            scrollToPrev();
            event.stopPropagation();
        }

        /**
         * Handles tracking on the scroll-forward button. Stops event
         * propagation, to not interfere with sheet move tracking.
         */
        function nextTrackingHandler(event) {
            scrollToNext();
            event.stopPropagation();
        }

        /**
         * Handles end of tracking on the scroll buttons. Moves the browser
         * focus back to the application, and stops event propagation, to not
         * interfere with sheet move tracking.
         */
        function trackingEndHandler(event) {
            view.grabFocus();
            event.stopPropagation();
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(newIndex, options, oldIndex) {

            // do not change anything, if the sheet index did not change
            if (!scrollable || (newIndex === oldIndex)) {
                return;
            }

            // scroll backwards, if the active sheet is located before the first visible
            if (newIndex < firstIndex) {
                firstIndex = newIndex;
                renderSheetButtons();
                return;
            }

            // scroll forwards, if the active sheet is located after the last visible
            if (newIndex > lastIndex) {
                firstIndex = newIndex;
                renderSheetButtons();
            }
        }

        /**
         * Handles double click events on sheet tab buttons and initiates
         * renaming the sheet.
         */
        function doubleClickHandler(event) {

            var // the zero-based index of the sheet to be renamed
                sheet = Utils.getControlValue($(event.currentTarget));

            // ignore double clicks on scroll buttons
            if (_.isNumber(sheet)) {
                app.getController().executeItem('sheet/rename/dialog');
            }
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            // create buttons for all visible sheets
            self.clearOptionButtons();
            view.iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock small-icon' : null,
                    label: _.noI18n(sheetName),
                    tooltip: _.noI18n(sheetName)
                });
            });

            // move trailing scroll button to the end
            self.addChildNodes(nextButton);
            self.trigger('refresh:layout');
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to automatic width, shows all sheet tab buttons,
         * and hides the scroll buttons.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setFullWidth = function () {
            this.getNode().css('width', '');
            showAllSheetButtons();
            return this;
        };

        /**
         * Sets this control to a fixed width. Performs further initialization
         * according to the available space: shows/hides the scroll buttons,
         * and draws the available sheet tabs according to the last cached
         * scroll position.
         *
         * @param {Number} width
         *  The new outer width of this control, in pixels.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().width(width);
            renderSheetButtons();
            return this;
        };

        // initialization -----------------------------------------------------

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // rebuild the list of available sheets after it has been changed
        app.on('docs:import:success', function () {
            fillList();
            model.on('change:sheet:attributes', fillList); // e.g. sheet protection
            view.on('change:sheets', fillList);
        });

        // add scroll buttons to the group node
        this.addFocusableControl(prevButton).addFocusableControl(nextButton);

        // handle tracking events of scroll buttons
        prevButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', prevTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', trackingEndHandler);
        nextButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', nextTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', trackingEndHandler);

        // double click shows rename dialog (TODO: in-place editing)
        this.getNode().on('dblclick', Utils.BUTTON_SELECTOR, doubleClickHandler);

        // change sheet order when dragging sheet buttons around
        this.on('change:order', function (evt, from, to) {
            view.leaveCellEditMode('cell');
            model.moveSheet(from, to);
        });

    }}); // class ActiveSheetGroup

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.ActiveSheetList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: 'fa-bars',
            tooltip: gt('Select sheet'),
            caret: false,
            position: 'top',
            updateCaptionMode: Utils.getBooleanOption(initOptions, 'showNames', false) ? 'label' : 'none'
        });

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            app.getView().iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton('', index, {
                    icon: (sheetModel.isLocked() ? 'fa-lock' : 'docs-empty') + ' small-icon',
                    label: _.noI18n(sheetName),
                    tooltip: _.noI18n(sheetName)
                });
            });
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to fixed width mode.
         *
         * @param {Number} width
         *  The new outer width of this control.
         *
         * @returns {ActiveSheetList}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().css({ width: width });
            // prevent wrong position/size of the button element due to padding/margin
            this.getMenuButton().css({ position: 'absolute', left: 0, right: 0 });
            return this;
        };

        // initialization -----------------------------------------------------

        app.on('docs:import:success', function () {
            fillList();
            app.getModel().on('change:sheet:attributes', fillList); // e.g. sheet protection
            app.getView().on('change:sheets', fillList);
        });

    }}); // class ActiveSheetList

    // class CellBorderPicker =================================================

    /**
     * The selector for cell borders. Hides specific items in the drop-down
     * menu, according to the current cell selection.
     *
     * @constructor
     *
     * @extends EditControls.BorderPicker
     */
    SpreadsheetControls.CellBorderPicker = EditControls.BorderPicker.extend({ constructor: function (app, initOptions) {

        var // the spreadsheet view instance
            view = app.getView();

        // base constructor ---------------------------------------------------

        EditControls.BorderPicker.call(this, Utils.extendOptions({
            tooltip: gt('Cell borders')
        }, initOptions, {
            showInsideHor: function () { return view.hasMultipleRowsSelected(); },
            showInsideVert: function () { return view.hasMultipleColumnsSelected(); }
        }));

    }}); // class CellBorderPicker

    // class CellBorderColorPicker ============================================

    /**
     * The selector for the cell border color.
     *
     * @constructor
     *
     * @extends EditControls.ColorPicker
     */
    SpreadsheetControls.CellBorderColorPicker = EditControls.ColorPicker.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditControls.ColorPicker.call(this, app, getGridColor(), Utils.extendOptions({
            icon: 'docs-cell-fill-color',
            tooltip: gt('Border color')
        }, initOptions));

        // private methods ----------------------------------------------------

        function getGridColor() {
            return app.getView().getEffectiveGridColor();
        }

        // initialization -----------------------------------------------------

        // refresh the 'automatic' border color (depends on sheet grid color)
        this.listenTo(app.getView(), 'change:activesheet change:sheet:viewattributes', function () {
            self.setAutoColor(getGridColor());
        });

    }}); // class CellBorderColorPicker

    // class MergePicker ======================================================

    /**
     * The selector for merged ranges, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.MergePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the spreadsheet view instance
            view = app.getView(),

            // whether to show the 'merge' menu item
            showMergeItem = false,
            // whether to show the 'merge horizontal' menu item
            showHMergeItem = false,
            // whether to show the 'merge vertical' menu item
            showVMergeItem = false,
            // whether to show the 'unmerge' menu item
            showUnmergeItem = false;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-merged-cells-on',
            tooltip: gt('Merge or unmerge cells')
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the merged state of the selection.
         */
        function updateHandler() {
            Utils.setControlCaption(self.getCaptionButton(), { icon: showUnmergeItem ? 'docs-merged-cells-off' : 'docs-merged-cells-on' });
        }

        /**
         * Updates the visibility flags of the list items after the selection
         * in the active sheet has been changed, or after the merge collection
         * of the active sheet has changed.
         */
        function updateVisibilityFlags() {

            var // the collection of merged ranges in the active sheet
                mergeCollection = view.getActiveSheetModel().getMergeCollection(),
                // the selected cell ranges
                ranges = view.getSelectedRanges();

            // show the 'merge' list item, if at least one range consists of more than a cell or a merged range
            showMergeItem = _(ranges).any(function (range) {
                var mergedRange = (SheetUtils.getCellCount(range) > 1) ? mergeCollection.getMergedRange(range.start) : null;
                return !mergedRange || !SheetUtils.rangeContainsRange(mergedRange, range);
            });

            // show the 'horizontal' list item, if at least one range does not consist entirely of horizontally merged cells
            showHMergeItem = _(ranges).any(function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isHorizontallyMerged(range);
            });

            // show the 'vertical' list item, if at least one range does not consist entirely of vertically merged cells
            showVMergeItem = _(ranges).any(function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isVerticallyMerged(range);
            });

            // show the 'unmerge' list item, if at least one range overlaps with a merged range
            showUnmergeItem = mergeCollection.rangesOverlapAnyMergedRange(ranges);
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('', 'merge',      { icon: 'docs-merged-cells-on',         label: gt('Merge cells'),              visible: function () { return showMergeItem; } })
            .createOptionButton('', 'horizontal', { icon: 'docs-merged-cells-horizontal', label: gt('Merge cells horizontally'), visible: function () { return showHMergeItem; } })
            .createOptionButton('', 'vertical',   { icon: 'docs-merged-cells-vertical',   label: gt('Merge cells vertically'),   visible: function () { return showVMergeItem; } })
            .createOptionButton('', 'unmerge',    { icon: 'docs-merged-cells-off',        label: gt('Unmerge cells'),            visible: function () { return showUnmergeItem; } });

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // update state flags used for visibility of list items (to keep the 'visible' callback functions simple and fast)
        view.on('change:selection insert:merged delete:merged', updateVisibilityFlags);

    }}); // class MergePicker

    // class NumberFormatPicker ===============================================

    /**
     * A drop-down list control for choosing number format categories, that
     * will load predefined format codes in the FormatCodePicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the drop-down list.
     *  Supports all options of the base class RadioList.
     */
    SpreadsheetControls.NumberFormatPicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-border',
            tooltip: /*#. in paragraphs and tables cells */ gt('Number Format')
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Insert all supported list entries according to the passed options.
         */
        function fillList() {
            // insert all supported list items
            _(NUMBER_FORMAT_CATEGORIES).each(function (entry) {
                self.createOptionButton('', entry.value, { label: entry.label, dataValue: entry.value });
            });
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {
            // defaults for e.g. not supported number formats (temporary)
            // TODO specify behavior with unsupported number formats
            var newLabel = gt('Unsupported');
            if (_.isString(value) && value.length > 0) {
                var activeEntry = _.find(NUMBER_FORMAT_CATEGORIES, function (elem) {
                    return elem.value === value;
                });
                if (!_.isUndefined(activeEntry)) {
                    newLabel = activeEntry.label;
                }
                // Temporary till we have nice icons
                Utils.setControlCaption(self.getMenuButton(), { label: newLabel });
            }
        }

        // initialization -----------------------------------------------------

        // initialize the drop-down list items
        fillList();

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

    }}); // class NumberFormatPicker

    // class FormatCodePicker =================================================

    /**
     * A drop-down list control for selection of predefined number formats
     * depending to currently selected category in the NumberFormatPicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class RadioList.
     */
    SpreadsheetControls.FormatCodePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the model and view
            model = app.getModel(),
            view = app.getView(),

            // the category of the format codes currently shown in the list
            listCategory = null;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Format codes')
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Updates the label of the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {

            var // the current number format category
                currentCategory = view.getNumberFormatCategory(),
                // all codes for the current number format
                categoryCodes = model.getNumberFormatter().getCategoryCodes(currentCategory),
                // the active format code
                formatCodeEntry = _(categoryCodes).findWhere({ value: value }),
                // the caption options for the drop-down button
                options = null;

            // refill the list, if the category has changed
            if (listCategory !== currentCategory) {
                self.clearMenu();
                _(categoryCodes).each(function (entry) {
                    self.createOptionButton('', entry.value, {
                        label: _.noI18n(entry.label),
                        labelCss: entry.red ? { color: 'red' } : null
                    });
                });
                listCategory = currentCategory;
            }

            // build caption options and update the caption
            if (_.isObject(formatCodeEntry)) {
                options = {
                    label: formatCodeEntry.label,
                    labelCss: formatCodeEntry.red ? { color: '#ff5555' } : null
                };
            } else {
                options = { label: gt('Custom') };
            }
            Utils.setControlCaption(self.getMenuButton(), options);
        }

        // initialization -----------------------------------------------------

        // register custom update handler to update button caption
        this.registerUpdateHandler(updateHandler);

    }}); // class FormatCodePicker

    // class SubtotalList =====================================================

    /**
     * The subtotal 'customizer'. A yet another drop down list for displaying
     * desired subtotal value.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.SubtotalList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the view instance
            view = app.getView(),

            // the number formatter of the document
            numberFormatter = app.getModel().getNumberFormatter(),

            // whether this control is globally activated
            activated = true,

            // whether this control is currently visible
            visible = true;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Select type of subtotal value')
        }, initOptions));

        // private methods ----------------------------------------------------

        function updateVisibility() {

            var // the parent container node (workaround for tool box usage: get rid of right margin)
                containerNode = self.getNode().closest('.group-container'),
                // whether the control is currently visible
                wasVisible = !containerNode.hasClass('hidden'),
                // whether the control will be visible
                isVisible = activated && visible;

            // workaround for tool box usage: hide parent container node to get rid of its right margin
            if (wasVisible !== isVisible) {
                containerNode.toggleClass('hidden', !isVisible);
                self.layout();
            }
        }

        function toggleVisibility(state) {
            visible = state;
            updateVisibility();
        }

        /**
         * Returns caption options of a list entry for the specified subtotal
         * type and value.
         */
        function getCaptionOptions(type, value) {

            var // the passed value, formatted in 'General' number format
                label = (_.isNumber(value) && _.isFinite(value)) ? numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_CELL).text : String(value);

            switch (type) {
            case 'sum':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Sum of all selected numeric cells */ gt('Sum: %1$s', _.noI18n(label)),
                    tooltip: gt('Sum of selected cells')
                };
            case 'min':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Minimum of all selected numeric cells */ gt('Min: %1$s', _.noI18n(label)),
                    tooltip: gt('Minimum value in selection')
                };
            case 'max':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Maximum of all selected numeric cells */ gt('Max: %1$s', _.noI18n(label)),
                    tooltip: gt('Maximum value in selection')
                };
            case 'cells':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells AND text cells */ gt('Count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain data')
                };
            case 'numbers':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Count of all selected numeric cells only (NOT text cells) */ gt('Numerical count: %1$s', _.noI18n(label)),
                    tooltip: gt('Number of selected cells that contain numerical data')
                };
            case 'average':
                return {
                    label: /*#. Subtotal result in OX Spreadsheet's status bar: Average of all selected numeric cells */ gt('Average: %1$s', _.noI18n(label)),
                    tooltip: gt('Average of selected cells')
                };
            }
        }

        /**
         * Updates the visible label of this control.
         */
        function updateHandler(type) {

            var // get fresh subtotals
                subtotals = view.getSubtotals();

            // do not show the control for a single value cell
            if (!(type in subtotals) || (subtotals.cells <= 1)) {
                toggleVisibility(false);
                return;
            }

            // if selection only consists of string values, show cell count
            if (subtotals.numbers === 0) { type = 'cells'; }

            // workaround for tool box usage: show parent container node to get its right margin
            toggleVisibility(true);

            // update the caption of the menu button
            Utils.setControlCaption(self.getMenuButton(), getCaptionOptions(type, subtotals[type]));

            // trigger layout listener to update the view component
            self.layout();
        }

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            var // get fresh subtotals
                subtotals = view.getSubtotals();

            function createOptionButton(sectionId, type) {
                self.createOptionButton(sectionId, type, getCaptionOptions(type, subtotals[type]));
            }

            // always create the cell count entry
            self.clearMenu().createMenuSection('cells');
            createOptionButton('cells', 'cells');

            // create numerical entries on demand
            if (subtotals.numbers > 0) {
                self.createMenuSection('numbers');
                createOptionButton('numbers', 'numbers');
                createOptionButton('numbers', 'sum');
                createOptionButton('numbers', 'min');
                createOptionButton('numbers', 'max');
                createOptionButton('numbers', 'average');
            }
        });

        // methods ------------------------------------------------------------

        this.setActivated = function (state) {
            activated = state;
            updateVisibility();
            if (state) { this.refresh(); }
            return this;
        };

        // initialization -----------------------------------------------------

        app.on('docs:import:success', function () {
            fillList();
            view.on('change:layoutdata', fillList);
        });

        this.registerUpdateHandler(updateHandler);
        toggleVisibility(false);

    }}); // class SubtotalList

    // class CellStylePicker ==================================================

    SpreadsheetControls.CellStylePicker = CellStylePicker;

    // class DynamicSplitButton ===============================================

    /**
     * A button to toggle the dynamic split view.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.DynamicSplitButton = Button.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({
            label: /*#. Short (!) label for a button used to split a spreadsheet into 2 or 4 different parts that can be scrolled independently */ gt('Split'),
            tooltip: gt('Split the sheet above and left of the cursor')
        }, initOptions, {
            toggle: true
        }));

    }}); // class DynamicSplitButton

    // class FrozenSplitPicker ================================================

    /**
     * The selector for frozen view panes, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.FrozenSplitPicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: /*#. Short (!) label for a button used to freeze leading columns/rows in a spreadsheet (make them non-scrollable) */ gt('Freeze'),
            tooltip: gt('Freeze the rows above and the columns left of the cursor')
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('', { cols: 0, rows: 1 }, { label: gt('Freeze first row'), tooltip: gt('Freeze the first visible row') })
            .createOptionButton('', { cols: 1, rows: 0 }, { label: gt('Freeze first column'), tooltip: gt('Freeze the first visible column') })
            .createOptionButton('', { cols: 1, rows: 1 }, { label: gt('Freeze first row and column'), tooltip: gt('Freeze the first visible row and the first visible column') });

    }}); // class FrozenSplitPicker

    // exports ================================================================

    return SpreadsheetControls;

});
