/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/tokenarray',
     'io.ox/office/drawinglayer/view/chartstyleutil',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, SheetUtils, Operations, TokenArray, ChartStyleUtil, gt) {

    'use strict';

    // class ChartCreator =================================================

    var DEFAULT_TEXTCOLOR = {type: 'auto'};
    var DEFAULT_LINECOLOR = {transformations: [{type: 'lumMod', value: 15000}, {type: 'lumOff',value: 85000}],type: 'scheme', value: 'text1'};
    var STANDARD_SHAPE = {type: 'solid', color: DEFAULT_LINECOLOR, width: 1};
    var NONE_SHAPE = {type: 'none'};
    var STANDARD_CHAR = {color: DEFAULT_TEXTCOLOR};
    var HEAD_CHAR = {color: DEFAULT_TEXTCOLOR, fontSize: 16};


    var INSERT_DEFAULTS = (function () {

        var AXIS_ENABLED =  { axis: {label: true},  line: NONE_SHAPE, character: STANDARD_CHAR};
        var AXIS_DISABLED = { axis: {label: false}, line: NONE_SHAPE};
        var GRID_ENABLED =  { line: STANDARD_SHAPE};
        var GRID_DISABLED = { line: NONE_SHAPE};

        return {
            normal: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED,
            },
            xy: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_ENABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED,
            },
            pie: {
                xAxis: AXIS_DISABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_DISABLED,
                yGrid: GRID_DISABLED,
            }
        };
    }());

    function getFormula(app, sheet, from, to) {
        if (!from) {
            throw 'no from assigned';
        }
        var range = {};
        range.start = from;
        if (to) {
            range.end = to;
        } else {
            range.end = range.start;
        }
        var tokenArray = new TokenArray(app, null, { silent: true });
        tokenArray.appendRange(range, { abs: true, sheet: sheet });
        var formula = tokenArray.getFormula();
        tokenArray.destroy();

        return formula;
    }

    function addDataSeries(app, generator, sheet, position, series, keyFrom, keyTo, title, valueFrom, valueTo, bubbleFrom, bubbleTo, bubbleArray) {
        var insert = {
            values: getFormula(app, sheet, valueFrom, valueTo)
        };
        if (title) {
            insert.title = getFormula(app, sheet, title);
        }
        if (keyFrom) {
            insert.names = getFormula(app, sheet, keyFrom, keyTo);
        }
        if (bubbleFrom) {
            insert.bubbles = getFormula(app, sheet, bubbleFrom, bubbleTo);
        } else if (bubbleArray) {
            insert.bubbles = bubbleArray;
        }
        var options = {
            series: series,
            attrs: {
                series: insert,
                line: {type: 'solid'},
                fill: {type: 'solid'},
            }
        };
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, sheet, position, options);
    }

    function generateSeriesOperations(app, range, sheet, generator, position, cells, axis, chart) {
        if (_.isUndefined(chart.varyColors)) {
            chart.varyColors = true;
        }
        generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, sheet, position, {attrs: { chart: chart}});

        var start = range.start;
        var end = range.end;

        var useAxis;
        if (axis !== 0 && axis !== 1) {
            if (cells[0] && !cells[1]) {
                useAxis = 0;
            } else if (!cells[0] && cells[1]) {
                useAxis = 1;
            } else {
                var width = end[0] - start[0];
                var height = end[1] - start[1];
                if (width >= height) {
                    useAxis = 1;
                } else {
                    useAxis = 0;
                }
            }
        } else {
            useAxis = axis;
        }

        var seriesCount;
        if (chart.type && chart.type.indexOf('bubble') === 0) {
            seriesCount = generateBubbleSeriesOperations(app, range, sheet, generator, position, cells, useAxis);
        } else {
            seriesCount =  generateStandardSeriesOperations(app, range, sheet, generator, position, cells, useAxis);
        }
        if ((seriesCount === 1 && !chart.varyColors) || (chart.type === 'scatter2d' || chart.type === 'line2d')) {
            var send = {};
            if (seriesCount === 1 && !chart.varyColors) {
                send.varyColors = true;
            }
            //CURVED is a workaround, filter has not all the information it needs!
            send.curved = chart.curved;
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, sheet, position, {attrs: { chart: send}});
        }
        return seriesCount;
    }

    /**
     * makes an address array, needed for the code, which does not know if its row or column oriented
     */
    function makeAddress(activeAxis, otherAxis, activeValue, otherValue) {
        var res = [];
        res[activeAxis] = activeValue;
        res[otherAxis] = otherValue;
        return res;
    }

    /**
     * generate all series from start to end.
     * Depending if first row/column contains a String,
     * it is defined as Names (datapoint) and Titles (dataseries)
     *
     *  @param isNumber {Array} first is a Boolean for the cell under the first cell
     *                         second is a Boolean for the cell right of the first cell
     *  @param activeAxis [1 or 0] column or row for the direction of any series data
     */
    function generateStandardSeriesOperations(app, range, sheet, generator, position, cells, activeAxis) {
        var start = range.start;
        var end = range.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        for (var i = 0; i < max; i++) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            addDataSeries(app, generator, sheet, position, i, keyFrom, keyTo, title, valueFrom, valueTo);
        }
        return max;
    }

    /**
     * generate all bubble-series from start to end.
     * Every second row/columns are values, the others are bubbles (sizes).
     * Depending if first row/column contains a String,
     * it is defined as Titles (dataseries).
     * If the count of the series is odd,
     * the first row/column is defined as Names (datapoints)
     *
     *  @param isNumber {Array} first is a Boolean for the cell under the first cell
     *                         second is a Boolean for the cell right of the first cell
     *  @param activeAxis [1 or 0] column or row for the direction of any series data
     */
    function generateBubbleSeriesOperations(app, range, sheet, generator, position, cells, activeAxis) {
        var start = range.start;
        var end = range.end;

        var max = end[activeAxis] - start[activeAxis];
        if (max === 0) {
            //exception only for 1D - Bubbles, behaves like excel
            activeAxis = (activeAxis - 1) * -1;
            max = end[activeAxis] - start[activeAxis];
        }
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var size = 0;

        for (var i = 0; i < max; i += 2) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            var bubbleFrom = null;
            var bubbleTo = null;
            var bubbleArray = null;

            if (i===max-1) {
                bubbleArray = [];
                var arraySize = 1 + (valueTo[otherAxis] - valueFrom[otherAxis]);
                for (var j = 0; j < arraySize; j++) {
                    bubbleArray.push(1);
                }
            } else {
                bubbleFrom = makeAddress(activeAxis, otherAxis, valueFrom[activeAxis] + 1, valueFrom[otherAxis]);
                bubbleTo = makeAddress(activeAxis, otherAxis, valueTo[activeAxis] + 1, valueTo[otherAxis]);
            }

            addDataSeries(app, generator, sheet, position, size, keyFrom, keyTo, title, valueFrom, valueTo, bubbleFrom, bubbleTo, bubbleArray);
            size++;
        }
        return size;
    }


    function getContentForRange(app, range, sheet, def, follow) {
        var view = app.getView();

        var start = range.start;
        var end = range.end;
        var width = end[0] - start[0];
        var height = end[1] - start[1];

        if (width > 50 || height > 50) {
            view.yell('warning', gt('It is not possible to create a chart out of more than 50 input cells.'));
            if (def) {
                def.reject();
            }
        } else {
			var bottAddress = [start[0], start[1] + 1];
            var rightAddress = [start[0] + 1, start[1]];
			var sData = [
                [{ sheet: sheet, start: bottAddress }],
                [{ sheet: sheet, start: rightAddress }],
                [{ sheet: sheet, start: start }]
            ];

			var request = app.queryCellResults(sData, {attributes: true});
            request.done(function (res) {
                var bottom = false;
                var right = false;
                var axis = null;

                if (width === 0 || height === 0) {
                    var firstString = (res[2][0] && !_.isNumber(res[2][0].result));
                    if (width === 0) {
                        axis = 0;
                        if (firstString) {
                            right = true;
                        }
                    } else {
                        axis = 1;
                        if (firstString) {
                            bottom = true;
                        }
                    }
                } else {
                    if (!res[2][0] || !res[2][0].display || !res[2][0].display.length) {
                        bottom = true;
                        right = true;
                    } else {
                        if (res[0][0]) {
                            bottom = ChartStyleUtil.isDate(res[0][0]) || (!_.isNumber(res[0][0].result));
                        }
                        if (res[1][0]) {
                            right = ChartStyleUtil.isDate(res[1][0]) || (!_.isNumber(res[1][0].result));
                        }
                    }
                }
                follow([bottom, right], axis);
            });
        }
    }

    /**
     * ChartCreator makes an insert-Chart-Op and Insert-Chart-DataSeries chosen by current Selection
     * and changes the Selection to a Multiselection by title, names and values
     */
    function ChartCreator(app, chart) {
        var view = app.getView();
        var gridPane = view.getActiveGridPane();

        var def = $.Deferred();

        var view = app.getView();
        var sheetModel = view.getActiveSheetModel();
        var drawingCollection = sheetModel.getDrawingCollection();
        var sheet = view.getActiveSheet();
        var selection = view.getSelection();

        var range = selection.ranges[selection.activeRange];


        getContentForRange(app, range, sheet, def, function (cells, axis) {
            var rect = gridPane.getVisibleRectangle();
            var clip = {
                left: Math.round(rect.left + rect.width / 3),
                top: Math.round(rect.top + rect.height / 4),
                width: Math.round(rect.width / 3),
                height: Math.round(rect.height / 2)
            };
            var attrs = drawingCollection.getAttributesForRectangle(clip);

            view.insertDrawing('chart', attrs, function (generator, sheet, position) {
                if (!chart.chartStyleId) {
                    chart.chartStyleId = 2;
                }
                var seriesCount = generateSeriesOperations(app, range, sheet, generator, position, cells, axis, chart);

                var start = [sheet, position[0]];

                var legendPos;
                if (seriesCount === 1) {
                    //TODO: title
                    legendPos = 'off';
                    generator.generateOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, {start: start, axis: 'main', attrs: { character: HEAD_CHAR}});
                } else {
                    legendPos = 'bottom';
                }
                generator.generateOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, {start: start, attrs: { legend: { pos: legendPos}, character: STANDARD_CHAR}});

                var defaults;
                switch (chart.type) {
                case 'bubble2d':
                case 'scatter2d':
                    defaults = INSERT_DEFAULTS.xy;
                    break;
                case 'pie2d':
                case 'donut2d':
                    defaults = INSERT_DEFAULTS.pie;
                    break;
                default:
                    defaults = INSERT_DEFAULTS.normal;
                    break;
                }

                generator.generateOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, {axis: 'x', start: start, attrs: defaults.xAxis});
                generator.generateOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, {axis: 'y', start: start, attrs: defaults.yAxis});
                generator.generateOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, {axis: 'x', start: start, attrs: defaults.xGrid});
                generator.generateOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, {axis: 'y', start: start, attrs: defaults.yGrid});
            });
            def.resolve();
        });


        /**
         * the deferred Object handles noting at the Moment,
         * its just for Future cases, when the situation could be changed
         * @return {Jquery|Deferred}
         */
        this.deferred = function () {
            return def;
        };
    }


    /**
     *
     * @param app
     * @param range whole start to end area of the new series
     * @param sheet
     * @param chartModel is used for the position and id in the document and as fallback for chart-type-checks
     * @param axis series direction
     * @param chartAttrs is used for changing other attributes of the chart and for the type info (bubble or not)
     */
    ChartCreator.generateOperations = function (app, range, sheet, chartModel, axis, chartAttrs) {
        var pos = chartModel.getPosition();
        var position = pos[1];
        var chart;
        if (chartAttrs) {
            chart = chartAttrs;
        } else {
            var attrs = chartModel.getMergedAttributes();
            var type = attrs.chart.type;
            //CURVED is a workaround, filter has not all the information it needs!
            chart = {
                type: type,
                curved: attrs.chart.curved
            };
        }


        getContentForRange(app, range, sheet, null, function (cells, overwrittenAxis) {

            //only overwritten if source range height is 1 oder width is 1
            if (overwrittenAxis === 0 || overwrittenAxis === 1) {
                axis = overwrittenAxis;
            }
            var model = app.getModel();
            var generator = model.createOperationsGenerator();

            var count = chartModel.getSeriesCount();
            _.times(count, function () {
                generator.generateOperation(Operations.DELETE_CHART_DATASERIES, {start: pos, series: 0});
            });


            generateSeriesOperations(app, range, sheet, generator, position, cells, axis, chart);
            model.applyOperations(generator.getOperations());
        });

    };

    ChartCreator.getStandardShape = function() {
        return STANDARD_SHAPE;
    };

    ChartCreator.getStandardChar = function() {
        return STANDARD_CHAR;
    };

    ChartCreator.getHeadChar = function() {
        return HEAD_CHAR;
    };

    ChartCreator.getNoneShape = function() {
        return NONE_SHAPE;
    };
    // exports ================================================================

    return ChartCreator;

});
