/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/paneutils', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // configuration for pane sides (header panes)
        PANE_SIDE_DATA = {
            left:   { nextSide: 'right',  panePos1: 'topLeft',    panePos2: 'bottomLeft',  columns: true,  anchorName: 'anchorLeft'   },
            right:  { nextSide: 'left',   panePos1: 'topRight',   panePos2: 'bottomRight', columns: true,  anchorName: 'anchorRight'  },
            top:    { nextSide: 'bottom', panePos1: 'topLeft',    panePos2: 'topRight',    columns: false, anchorName: 'anchorTop'    },
            bottom: { nextSide: 'top',    panePos1: 'bottomLeft', panePos2: 'bottomRight', columns: false, anchorName: 'anchorBottom' }
        },

        // configuration for pane positions (grid panes)
        PANE_POS_DATA = {
            topLeft:     { colSide: 'left',  rowSide: 'top',    nextColPos: 'topRight',    nextRowPos: 'bottomLeft'  },
            topRight:    { colSide: 'right', rowSide: 'top',    nextColPos: 'topLeft',     nextRowPos: 'bottomRight' },
            bottomLeft:  { colSide: 'left',  rowSide: 'bottom', nextColPos: 'bottomRight', nextRowPos: 'topLeft'     },
            bottomRight: { colSide: 'right', rowSide: 'bottom', nextColPos: 'bottomLeft',  nextRowPos: 'topRight'    }
        },

        // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:repeat tracking:scroll tracking:end tracking:cancel';

    // static class PaneUtils =================================================

    var PaneUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Default settings for mouse/touch tracking in header and grid panes.
     *
     * @constant
     */
    PaneUtils.DEFAULT_TRACKING_OPTIONS = {
        autoScroll: true,
        borderMargin: -30,
        borderSize: 60,
        minSpeed: 5,
        maxSpeed: 500,
        acceleration: 1.5
    };

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed pane side represents a column header pane.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The position 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    PaneUtils.isColumnSide = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].columns;
    };

    /**
     * Returns the name of the scroll anchor view property for the passed pane
     * side.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The name of the scroll anchor sheet view property.
     */
    PaneUtils.getScrollAnchorName = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].anchorName;
    };

    /**
     * Returns the column pane side identifier for the passed grid pane
     * position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    PaneUtils.getColPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].colSide;
    };

    /**
     * Returns the row pane side identifier for the passed grid pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'top' for the pane positions 'topLeft' and 'topRight', or
     *  'bottom' for the pane positions 'bottomLeft' and 'bottomRight'.
     */
    PaneUtils.getRowPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].rowSide;
    };

    /**
     * Returns the pane position of the horizontal neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the horizontal neighbor (for example, returns
     *  'bottomLeft' for the pane position 'bottomRight').
     */
    PaneUtils.getNextColPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextColPos;
    };

    /**
     * Returns the pane position of the vertical neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the vertical neighbor (for example, returns
     *  'topRight' for the pane position 'bottomRight').
     */
    PaneUtils.getNextRowPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextRowPos;
    };

    /**
     * Returns the pane position identifier according to the passed pane side
     * identifiers.
     *
     * @param {String} colPaneSide
     *  The column pane side ('left' or 'right').
     *
     * @param rowPaneSide
     *  The row pane side ('top' or 'bottom').
     *
     * @returns {String}
     *  The pane position for the passed pane side identifiers.
     */
    PaneUtils.getPanePos = function (colPaneSide, rowPaneSide) {
        return rowPaneSide + Utils.capitalize(colPaneSide);
    };

    /**
     * Returns the pane position nearest to the specified source pane position,
     * but matching the specified target pane side.
     *
     * @param {String} panePos
     *  The source pane position (one of 'topLeft', 'topRight', 'bottomLeft',
     *  or 'bottomRight').
     *
     * @param {String} paneSide
     *  The target pane side (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The grid pane position of the nearest neighbor, matching the specified
     *  pane side (for example, returns 'topRight' for the source pane position
     *  'bottomRight' and target pane side 'top').
     */
    PaneUtils.getNextPanePos = function (panePos, paneSide) {
        var panePosData = PANE_POS_DATA[panePos],
            paneSideData = PANE_SIDE_DATA[paneSide];
        return ((paneSide === panePosData.colSide) || (paneSide === panePosData.rowSide)) ? panePos :
            (paneSideData.columns ? panePosData.nextColPos : panePosData.nextRowPos);
    };

    // CSS helper functions ---------------------------------------------------

    /**
     * Returns the CSS position and size attributes for the passed rectangle in
     * pixels, as HTML mark-up value for the 'style' element attribute.
     *
     * @param {Object} rectangle
     *  The rectangle to be converted to CSS position and size attributes, in
     *  the properties 'left', 'top', 'width', and 'height'.
     *
     * @returns {String}
     *  The value for the 'style' element attribute in HTML mark-up text.
     */
    PaneUtils.getRectangleStyleMarkup = function (rectangle) {
        return 'left:' + rectangle.left + 'px;top:' + rectangle.top + 'px;width:' + rectangle.width + 'px;height:' + rectangle.height + 'px;';
    };

    /**
     * Returns the effective CSS text alignment of the passed cell data,
     * according to the cell result value and the alignment formatting
     * attributes.
     *
     * @param {Object} cellData
     *  The cell descriptor object, as received from the cell collection.
     *
     * @param {Boolean} [editMode=false]
     *  If set to true, formula cells will be left-aligned (used for example in
     *  in-place cell edit mode).
     *
     * @returns {String}
     *  The effective horizontal CSS text alignment ('left', 'center', 'right',
     *  or 'justify').
     */
    PaneUtils.getCssTextAlignment = function (cellData, editMode) {

        switch (cellData.attributes.cell.alignHor) {
        case 'left':
        case 'center':
        case 'right':
        case 'justify':
            return cellData.attributes.cell.alignHor;
        case 'auto':
            // formulas in cell edit mode always left aligned, regardless of result type
            if (editMode && _.isString(cellData.formula)) { return 'left'; }
            // errors (centered) or strings (left aligned)
            if (_.isString(cellData.result)) { return (cellData.result[0] === '#' ? 'center' : 'left'); }
            // numbers (right aligned)
            if (_.isNumber(cellData.result)) { return 'right'; }
            // booleans (centered)
            if (_.isBoolean(cellData.result)) { return 'center'; }
            // empty cells (left aligned)
            return 'left';
        }

        Utils.warn('PaneUtils.getTextAlignment(): unknown alignment attribute value "' + cellData.attributes.cell.alignHor + '"');
        return 'left';
    };

    /**
     * Returns the CSS text decoration value according to the formatting
     * attributes in the passed cell descriptor.
     *
     * @param {Object} cellData
     *  The cell descriptor object, as received from the cell collection.
     *
     * @returns {String}
     *  The effective CSS text decoration (underline and strike-out mode).
     */
    PaneUtils.getCssTextDecoration = function (cellData) {
        var textDecoration = 'none';
        if (cellData.attributes.character.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (cellData.attributes.character.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        return textDecoration;
    };

    // tracking events --------------------------------------------------------

    /**
     * Handles a single tracking cycle initiated by a 'tracking:start' event
     * received from a tracking node. Invokes the passed tracking event handler
     * for the 'tracking:start' event and all following related tracking events
     * that will be triggered while this tracking cycle is active. After the
     * event 'tracking:end' or 'tracking:cancel' has been triggered, the
     * tracking event handler will not be invoked anymore, allowing to use
     * another tracking event handler for the next tracking cycle.
     *
     * Additionally, this method will guard the tracked DOM node when tracking
     * on a touch device is active. On touch devices, the tracked DOM node (the
     * target node of the initial 'touchstart' browser event) must remain in
     * the DOM as long as touch tracking is active. If the node will be removed
     * from the DOM, the browser will not trigger any further 'touchmove' or
     * 'touchend' events. This method listens to the 'DOMNodeRemoved' browser
     * event, and reinserts the tracked DOM node, as long as touch tracking is
     * active. This ensures that all cases are covered that remove the tracked
     * DOM node, for example when re-rendering parts of the DOM while tracking.
     * This works even for code that removes the tracked node deferred in
     * browser timeouts etc.
     *
     * @param {jQuery.Event} startEvent
     *  The initial 'tracking:start' event for the tracking cycle to be handled
     *  by this method invocation.
     *
     * @param {Function} trackingHandler
     *  The event handler function that will be invoked for all tracking events
     *  triggered for this tracking cycle, including the 'tracking:start' event
     *  passed to this method. Receives the jQuery event object as first
     *  parameter.
     *
     * @param {Function} [endCallback]
     *  A callback function that will be invoked after the tracking cycle has
     *  finished and the passed event handler has been invoked the last time.
     *  Receives the jQuery event object as first parameter.
     */
    PaneUtils.processTrackingCycle = function (startEvent, trackingHandler, endCallback) {

        var // the event target node
            targetNode = $(startEvent.target),
            // event delegation target node
            delegateNode = $(startEvent.delegateTarget),
            // whether to keep the target node in the DOM (touch tracking)
            guardTargetNode = (startEvent.trackingType === 'touch') && Utils.containsNode(delegateNode, targetNode);

        // registers or unregisters the 'DOMNodeRemoved' event handler
        function registerDOMNodeRemovedHandler(add) {
            delegateNode[0][add ? 'addEventListener' : 'removeEventListener']('DOMNodeRemoved', nodeRemovedHandler, false);
        }

        // touch tracking: called when target node has been removed from the DOM
        function nodeRemovedHandler(removeEvent) {
            if (targetNode.is(removeEvent.target) || Utils.containsNode(removeEvent.target, targetNode)) {
                registerDOMNodeRemovedHandler(false);
                guardTargetNode = false;
                delegateNode.append(targetNode.addClass('touch-tracker-hidden'));
            }
        }

        // ends current tracking cycle
        function endTracking(event) {
            delegateNode.off(TRACKING_EVENT_NAMES, trackingHandler).off('tracking:end tracking:cancel', endTracking);
            if (guardTargetNode) { registerDOMNodeRemovedHandler(false); }
            if (targetNode.hasClass('touch-tracker-hidden')) { targetNode.remove(); }
            if (_.isFunction(endCallback)) { endCallback(event); }
        }

        // touch tracking: keep target node in the DOM
        if (guardTargetNode) { registerDOMNodeRemovedHandler(true); }

        // start listening to tracking events
        delegateNode.on(TRACKING_EVENT_NAMES, trackingHandler).on('tracking:end tracking:cancel', endTracking);

        // invoke the event handler explicitly for the 'tracking:start' event
        trackingHandler.call(startEvent.currentTarget, startEvent);
    };

    /**
     * Returns the GUI selection mode according to the modifier keys in the
     * passed GUI event.
     *
     * @param {jQuery.Event} event
     *  The event object received by any jQuery event handlers (keyboard,
     *  mouse, tracking).
     *
     * @returns {String}
     *  The selection mode according to the keyboard modifier keys:
     *  - 'select': Standard selection without modifier keys.
     *  - 'append': Append new range to current selection (CTRL/META key).
     *  - 'extend': Extend current active range (SHIFT key).
     */
    PaneUtils.getSelectionMode = function (event) {
        if (!event.shiftKey && (event.ctrlKey || event.metaKey)) { return 'append'; }
        if (event.shiftKey && !event.ctrlKey && !event.metaKey) { return 'extend'; }
        return 'select';
    };

    // exports ================================================================

    return PaneUtils;

});
