/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/tokenarray'
    ], function (Utils, ModelObject, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    // class EntryModel =======================================================

    /**
     * Stores data validation settings for a single range in a specific sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this data validation range.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this data validation range.
     *
     * @param {Array} ranges
     *  The logical addresses of all cell ranges that contain the data
     *  validation settings.
     *
     * @param {Object} [attrs]
     *  The data validation attribute set.
     */
    var EntryModel = AttributedModel.extend({ constructor: function (app, sheetModel, ranges, attrs) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = app.getModel(),

            // token array representing the validation attribute 'value1'
            tokenArray1 = new TokenArray(app, sheetModel),

            // token array representing the validation attribute 'value2'
            tokenArray2 = new TokenArray(app, sheetModel);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, attrs, { additionalFamilies: 'validation' });

        // private methods ----------------------------------------------------

        /**
         * Updates internal settings after the validation attributes of this
         * collection entry have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the validation attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributes()).validation;
            oldAttributes = oldAttributes ? oldAttributes.validation : {};

            // parse a changed formula expression in attribute 'value1'
            if (newAttributes.value1 !== oldAttributes.value1) {
                tokenArray1.invokeSilently('parseFormula', newAttributes.value1);
            }

            // parse a changed formula expression in attribute 'value2'
            if (newAttributes.value2 !== oldAttributes.value2) {
                tokenArray2.invokeSilently('parseFormula', newAttributes.value2);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the addresses of all cell ranges of this collection entry.
         *
         * @returns {Array}
         *  The addresses of all cell ranges.
         */
        this.getRanges = function () {
            return _.copy(ranges, true);
        };

        /**
         * Returns whether the passed cell address is contained by the cell
         * ranges of this collection entry.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this collection entry.
         */
        this.containsCell = function (address) {
            return SheetUtils.rangesContainCell(ranges, address);
        };

        /**
         * Returns the reference addresses of this collection entry. The
         * reference address is the top-left cell of the bounding range of all
         * ranges of this collection entry.
         *
         * @returns {Number[]}
         *  The logical address of the reference cell of this collection entry.
         */
        this.getRefAddress = function () {
            return SheetUtils.getBoundingRange(ranges).start;
        };

        /**
         * Returns the list contents for validations of type 'list' or 'source'
         * according to the formula expression contained in the attribute
         * 'value1'.
         *
         * @param {Number[]} targetAddress
         *  The target address used to resolve relative references in the
         *  formula expression.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  array of strings, if this collection entry is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list of
         *  strings. Otherwise, the promise will be rejected. Additionally, the
         *  promise will contain the method 'abort()' that is able to abort a
         *  running server request. In this case, the promise will be rejected
         *  immediately.
         */
        this.queryListValues = function (targetAddress) {

            var // the original server query for 'source' validation
                serverQuery = null,
                // the resulting promise
                resultPromise = null;

            // resolve formula to constant strings (validation type 'list')
            function resolveList() {
                var listValues = tokenArray1.resolveStringList();
                return _.isArray(listValues) ? $.when(listValues) : $.Deferred().reject();
            }

            // resolve formula to source ranges (validation type 'source')
            function resolveSource() {
                var refAddress = self.getRefAddress(),
                    sourceRanges = tokenArray1.resolveRangeList({ refAddress: refAddress, targetAddress: targetAddress });
                if (!_.isArray(sourceRanges)) { return $.Deferred().reject(); }
                serverQuery = app.queryCellResults([sourceRanges]);
                return serverQuery.then(function (resultContents) {
                    return _(resultContents[0]).pluck('display');
                });
            }

            // resolve source data by validation type
            switch (this.getMergedAttributes().validation.type) {
            case 'list':
                resultPromise = resolveList();
                break;
            case 'source':
                resultPromise = resolveSource();
                break;
            default:
                // wrong validation type
                resultPromise = $.Deferred().reject();
            }

            // add an 'abort()' method to the promise that aborts the server query
            return _(resultPromise).extend({
                abort: function () {
                    if (serverQuery) { serverQuery.abort(); }
                    return this;
                }
            });
        };

        /**
         * Changes the addresses of all cell ranges of this collection entry.
         *
         * @param {Array} newRanges
         *  The addresses of the new cell ranges. The old ranges of this
         *  collection entry will be replaced completely.
         *
         * @returns {EntryModel}
         *  A reference to this instance.
         */
        this.setRanges = function (newRanges) {
            ranges = _.copy(newRanges, true);
            return this;
        };

        /**
         * Reduces the cell ranges containing the data validation settings to
         * the difference of these ranges and the passed ranges.
         *
         * @returns {Boolean}
         *  Whether the reduced ranges are still valid (not all deleted). If
         *  false is returned, this collection entry must be deleted from the
         *  validation collection.
         */
        this.reduceRanges = function (otherRanges) {
            ranges = SheetUtils.getRemainingRanges(ranges, otherRanges);
            return ranges.length > 0;
        };

        /**
         * Transforms the addresses of the cell ranges containing the data
         * validation settings, after columns or rows have been inserted into
         * or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed ranges are still valid (not all deleted).
         *  If false is returned, this collection entry must be deleted from
         *  the validation collection.
         */
        this.transformRanges = function (interval, insert, columns) {
            ranges = _.chain(ranges)
                .map(function (range) { return model.transformRange(range, interval, insert, columns, true); })
                .filter(_.isObject)
                .value();
            return ranges.length > 0;
        };

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            tokenArray1.transformFormula(sheet, interval, insert, columns);
            tokenArray2.transformFormula(sheet, interval, insert, columns);
            return true;
        };

        // initialization -----------------------------------------------------

        // unify the range addresses
        ranges = SheetUtils.getUnifiedRanges(ranges);

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // update source values after one of the token array has changed (but prevent updating the token array again)
        tokenArray1.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value1: tokenArray1.getFormula() } });
            self.on('change:attributes', changeAttributesHandler);
        });
        tokenArray2.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value2: tokenArray2.getFormula() } });
            self.on('change:attributes', changeAttributesHandler);
        });

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            // pass own entries as hidden argument to the constructor
            return new EntryModel(app, newSheetModel, ranges, this.getExplicitAttributes(), { oldSheet: sheetModel.getIndex(), newSheet: newSheet });
        });

        // adjust formulas according to sheet indexes passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[EntryModel.length];
            if (_.isObject(cloneData)) {
                tokenArray1.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
                tokenArray2.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members
        this.registerDestructor(function () {
            tokenArray1.destroy();
            tokenArray2.destroy();
            model = tokenArray1 = tokenArray2 = null;
            app = sheetModel = ranges = null;
        });

    }}); // class EntryModel

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in cell ranges, and
     * to show tooltips and error messages.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function ValidationCollection(app, sheetModel) {

        var // the spreadsheet document model
            model = app.getModel(),

            // all collection entries with validation settings
            entries = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Returns the collection entry that covers the specified cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {EntryModel|Null}
         *  The collection entry for the specified cell. If the cell is not
         *  validated at all, returns null.
         */
        function getEntry(address) {
            var entry = _(entries).find(function (entry2) {
                return entry2.containsCell(address);
            });
            return entry ? entry : null;
        }

        /**
         * Invokes the specified method on all collection entries.
         *
         * @param {String} methodName
         *  The name of the method to be invoked on each collection entry. If
         *  the method returns the value false, the respective entry will be
         *  removed from the collection.
         *
         * @param {EntryModel|Null} excludeEntry
         *  A collection entry that will be skipped (its method will not be
         *  invoked).
         *
         * @param {Any} ...
         *  Other parameters that will be passed to the specified method.
         */
        function invokeEntryMethod(methodName, excludeEntry) {
            var args = _.toArray(arguments).slice(invokeEntryMethod.length);
            Utils.iterateArray(entries, function (entry, index) {
                if ((entry !== excludeEntry) && (entry[methodName].apply(entry, args) === false)) {
                    entries[index].destroy();
                    entries.splice(index, 1);
                }
            }, { reverse: true }); // reverse to be able to delete in-place
        }

        /**
         * Recalculates the position of all validation ranges, after columns or
         * rows have been inserted into or deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            invokeEntryMethod('transformRanges', null, interval, insert, columns);
        }

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        function transformValues(sheet, interval, insert, columns) {
            invokeEntryMethod('transformValues', null, sheet, interval, insert, columns);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {Array} ranges
         *      The logical addresses of all related cell ranges containing the
         *      same validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var entry = getEntry(address);
            return entry ? {
                ranges: entry.getRanges(),
                attributes: entry.getMergedAttributes().validation
            } : null;
        };

        /**
         * Returns the list contents for a data validation of type 'list' or
         * 'source' at the specified cell address.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  array of strings, if this collection entry is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list of
         *  strings. Otherwise, the promise will be rejected. Additionally, the
         *  promise will contain the method 'abort()' that is able to abort a
         *  running server request. In this case, the promise will be rejected
         *  immediately.
         */
        this.queryListValues = function (address) {
            var entry = getEntry(address);
            return entry ? entry.queryListValues(address) : $.Deferred().reject();
        };

        /**
         * Restricts the valid values of a range of cells according to the
         * specified settings.
         *
         * @param {Array} ranges
         *  The addresses of all cell ranges that contain the data validation
         *  settings.
         *
         * @param {Object} attrs
         *  The validation attributes.
         *
         * @param {Number} [index]
         *  The zero-based insertion index for the new collection entry. If
         *  omitted, the entry will be appended to the collection.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been created successfully.
         */
        this.insertValidation = function (ranges, attrs, index) {

            var // the new collection entry
                entry = new EntryModel(app, sheetModel, ranges, { validation: attrs });

            // reduce ranges of existing entries (this may result in removing entries completely)
            invokeEntryMethod('reduceRanges', null, entry.getRanges());

            // insert the new collection entry
            index = _.isNumber(index) ? Utils.minMax(index, 0, entries.length) : entries.length;
            entries.splice(index, 0, entry);

            // TODO: trigger change events

            // no known error case
            return true;
        };

        /**
         * Changes the ranges and/or validation attributes of an existing
         * collection entry.
         *
         * @param {Number} index
         *  The zero-based index of the collection entry to be changed.
         *
         * @param {Array|Null} ranges
         *  The new addresses of the cell ranges for the collection entry. If
         *  set to null, the original ranges will not be modified. Otherwise,
         *  the
         *
         * @param {Object} attrs
         *  The validation attributes.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been changed successfully.
         */
        this.changeValidation = function (index, ranges, attrs) {

            var // the collection entry to be modified
                entry = entries[index],
                // the new ranges for the entry
                ranges = Utils.getArrayOption(attrs, 'ranges', null);

            // validate passed collection index
            if (!_.isObject(entry)) { return false; }

            // change validation ranges
            if (_.isArray(ranges) && (ranges.length > 0)) {

                // reduce ranges of other existing entries (this may result in removing entries completely)
                invokeEntryMethod('reduceRanges', entry, ranges);

                // set the new ranges at the collection entry
                entry.setRanges(ranges);
            }

            // change all other validation properties
            entry.setAttributes({ validation: attrs });

            return true;
        };

        /**
         * Deletes validation settings from this collection.
         *
         * @param {Number} index
         *  The zero-based index of the collection entry to be deleted.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been deleted successfully.
         */
        this.deleteValidation = function (index) {

            var // the collection entry to be deleted
                entry = entries[index];

            // validate passed collection index
            if (!_.isObject(entry)) { return false; }

            // destroy and remove the collection entry
            entry.destroy();
            entries.splice(index, 1);

            return true;
        };

        // initialization -----------------------------------------------------

        // update ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // update source formulas after inserting/deleting columns or rows in any sheet
        model.registerTransformationHandler(transformValues);

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            // pass own entries as hidden argument to the constructor
            return new ValidationCollection(app, newSheetModel, { entries: entries, newSheet: newSheet });
        });

        // clone collection entries passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[ValidationCollection.length];
            if (_.isObject(cloneData)) {
                entries = ModelObject.cloneArray(cloneData.entries, sheetModel, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members
        this.registerDestructor(function () {
            model.unregisterTransformationHandler(transformValues);
            _(entries).invoke('destroy');
            model = entries = null;
        });

    } // class ValidationCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ValidationCollection });

});
