/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/drawingcontroller',
    ['io.ox/office/tk/config',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/chartcreator',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/model/tokenarray',
     'gettext!io.ox/office/spreadsheet'
    ], function (Config, Operations, Controls, ChartCreator, Label, Utils, TokenArray, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        CheckBox = Controls.CheckBox,
        TextField = Controls.TextField,

        SOURCE_CANCEL_VIEW = 'before:activesheet change:sheets change:layoutdata celledit:enter celledit:change celledit:leave insert:merged delete:merged',
        SOURCE_CANCEL_MODEL = 'change:sheet:attributes',
        SOURCE_CANCEL_GRID = 'select:end';


    // class DrawingController =================================================

    /**
     * DrawingController (will) encapsulate(s) all Drawing-Features of the Controller and of the View (Sidepane-toolbox)
     * For now Chart are part of Drawings!
     */
    function DrawingController(app) {

        var self = this,
            model = null,
            view = null,
            chartTypes = Controls.CHART_TYPES,
            sourceSelector = false,
            chartSourceToken = null,
            // the token arrays containing the highlighted ranges of the selected drawing object
            tokenArrays = [];

        // private methods ----------------------------------------------------

        /**
         * initToolBox initiates the view components
         * -delete Drawing
         * -change Chart Type
         * -change Chart Sub Type (clustered, stacked, percent)
         * -change chart curves (for line charts)
         * -change Chart Colors (patterns with grayscale and accents, multicolors for single series)
         * -change Chart Style (background, highlighteffect)
         */
        function initToolBox() {
            var xTitle = /*#. Title of the horizontal axis of charts/diagrams */ gt('X axis title');
            var yTitle = /*#. Title of the vertical axis of charts/diagrams */ gt('Y axis title');
            var xLine = /*#. Line of the horizontal axis of charts/diagrams */ gt('X axis line');
            var yLine = /*#. Line of the vertical axis of charts/diagrams */ gt('Y axis line');
            var xLabels = /*#. Horizontal axis-labels of charts/diagrams */ gt('X labels');
            var yLabels = /*#. Vertical axis-labels of charts/diagrams */ gt('Y labels');
            var xGrid = /*#. Horizontal gridline of charts/diagrams */ gt('X grid');
            var yGrid = /*#. Vertical gridline of charts/diagrams */ gt('Y grid');

            var drawingbox = view.createToolBox('drawing', { label: /*#. drawing objects: images, diagrams, ... */ gt('Drawing'), visible: 'document/editable/drawing' });
            drawingbox.addRightTab();
            drawingbox.addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_OPTIONS));

            var chartbox = view.createToolBox('chart', { label: /*#. drawing objects: images, diagrams, ... */ gt('Chart'), visible: 'drawing/chart' });

            chartbox.addGroup('drawing/charttype', new Controls.ChartTypePicker(app, {fullWidth: true, icon: 'fa-bar-chart-o'}));

            chartbox.newLine();
            chartbox.addGroup('drawing/chartcolorset', new Controls.ChartColorsetPicker(app, { width: 110 }));
            chartbox.addRightTab();
            chartbox.addGroup('drawing/chartstyleset', new Controls.ChartStylesetPicker({ width: 110 }));

            chartbox.newLine();
            chartbox.addGroup('drawing/chartsource', new Button(Controls.CHART_SOURCE_OPTIONS));

            chartbox.newLine();
            chartbox.addGroup('drawing/chartexchange', new Button(Controls.CHART_SOURCE_EXCHANGE_OPTIONS));

            var labelsbox = view.createToolBox('chartlabels', { label: /*#. Labels section of charts/diagrams */ gt('Labels'), visible: 'drawing/chart' });
            labelsbox.addGroup('drawing/charttitle', new TextField(Controls.CHART_TITLE_OPTIONS));

            labelsbox.newLine();
            addAxisTitle('x', xTitle, labelsbox);

            labelsbox.newLine();
            addAxisTitle('y', yTitle, labelsbox);

            labelsbox.newLine();
            addAxisLabel('x', xLabels, labelsbox);
            labelsbox.addRightTab();
            addAxisLabel('y', yLabels, labelsbox);

            labelsbox.newLine();
            addAxisLine('x', xLine, labelsbox);
            labelsbox.addRightTab();
            addAxisLine('y', yLine, labelsbox);

            labelsbox.newLine();
            addAxisGrid('x', xGrid, labelsbox);
            labelsbox.addRightTab();
            addAxisGrid('y', yGrid, labelsbox);

            labelsbox.newLine();
            labelsbox.addGroup('drawing/chartvarycolor', new CheckBox({ label: /*#. Check box label: if active, a bar/pie chart will have different colors for each data point (default is the same color for all points) */ gt('Vary colors by point'), width: 230 }));

            labelsbox.newLine();
            labelsbox.addGroup('drawing/chartdatalabel', new CheckBox({ label: /*#. Small labels over the data points in charts objects */ gt('Data labels'), width: 110 }));
            labelsbox.addRightTab();
            labelsbox.addGroup('drawing/chartlegend/pos', new Controls.ChartLegendPicker({width: 100}));

        }

        function addAxisTitle(id, name, toolBox) {
            toolBox.addGroup('drawing/chartaxes/' + id + '/title', new TextField({ fullWidth: true, tooltip: name, placeholder: name, select: true }));
        }

        function addAxisLabel(id, name, toolBox) {
            toolBox.addGroup('drawing/chartaxes/' + id + '/label', new CheckBox({ label: name, width: 110 }));
        }

        function addAxisLine(id, name, toolBox) {
            toolBox.addGroup('drawing/chartaxes/' + id + '/visible', new CheckBox({ label: name, width: 110 }));
        }

        function addAxisGrid(id, name, toolBox) {
            toolBox.addGroup('drawing/chartgrids/' + id, new CheckBox({ label: name, width: 110 }));
        }

        /**
         * changeSelection
         * if only one Drawing is selected and this is a Chart, the function highlights its source-cells via TokenArray
         */
        var changeSelection = (function () {

            return function (evt, selection) {
                if (tokenArrays.length > 0) {
                    view.endRangeHighlighting();
                    tokenArrays = [];
                }

                if (selection.drawings.length === 1) {

                    var drawingCollection = view.getActiveSheetModel().getDrawingCollection(),
                        chartModel = drawingCollection.findModel(selection.drawings[0], { type: 'chart' });

                    if (chartModel) {
                        chartModel.getTokenArrays(tokenArrays);
                    }
                }

                if (tokenArrays.length > 0) {
                    view.startRangeHighlighting(tokenArrays, { draggable: false, resolveNames: true });
                }
            };
        }());

        /**
         * @return the ID of the current selected Drawing
         */
        function getDrawingIndices() {
            var selection = view.getSelection();
            if (selection.drawings.length === 1) {
                return selection.drawings[0];
            }
        }

        /**
         * @return the current selected Drawing
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingIndices();
            if (drawingIndices) {
                var drawingCollection = view.getActiveSheetModel().getDrawingCollection();
                return drawingCollection.findModel(drawingIndices, { type: type });
            }
        }

        /**
         * returns the axis-ID except it is a bar-chart, then x returns y and y returns x
         */
        function getCorrectAxisId(id) {
            var chart = getDrawingModel('chart');
            if (chart) {
                var attr = chart.getMergedAttributes().chart;
                if (attr.type.indexOf('bar') === 0) {
                    if (id === 'x') {
                        id = 'y';
                    } else if(id === 'y'){
                        id = 'x';
                    }
                }
            }
            return id;
        }

        /**
         * setAxisAttributes generates the Operation for changing Attributes of Charts Axes
         */
        function setAxisAttributes(op, axis, attrs) {
            axis = getCorrectAxisId(axis);
            generateDrawingOperation(op, { attrs: attrs, axis: axis });
        }

        function generateDrawingOperation(opName, params) {
            var generator = model.createOperationsGenerator();
            var position = getDrawingIndices();
            generator.generateDrawingOperation(opName, view.getActiveSheet(), position, params);
            model.applyOperations(generator.getOperations());
        }

        /**
         * generates an insert-Chart-Op and Insert-Chart-DataSeries chosen by
         * current Selection
         *
         * @returns {jQuery.Deferred}
         *  a deferred object, which is already resolved
         */
        function insertChart(id) {
            return new ChartCreator(app, chartTypes[id]).deferred();
        }

        function initSourceCancelListener(end) {
            view.on(SOURCE_CANCEL_VIEW, end);
            view.getActiveGridPane().on(SOURCE_CANCEL_GRID, end);
            model.on(SOURCE_CANCEL_MODEL, end);

        }

        function endSourceCancelListener(end) {
            view.off(SOURCE_CANCEL_VIEW, end);
            model.off(SOURCE_CANCEL_MODEL, end);
        }

        function endSourceFinishListener(end) {
            view.getActiveGridPane().off(SOURCE_CANCEL_GRID, end);
            endSourceCancelListener(end);
        }

        function selectEnd() {
            endSourceFinishListener(selectEnd);

            if (chartSourceToken) {
                var rangeHolder = chartSourceToken.resolveRangeList()[0];
                var chart = getDrawingModel('chart');

                var possSources = chart.getExchangeInfo();
                if (possSources.warn) {
                    ChartCreator.generateOperations(app, rangeHolder, rangeHolder.sheet, chart);
                } else {
                    ChartCreator.generateOperations(app, rangeHolder, rangeHolder.sheet, chart, (possSources.axis - 1) * - 1);
                }
                chartSourceToken = null;
            }

            view.endRangeHighlighting();
            view.unregisterCellSelectionHandlers();
            view.getActiveSheetModel().setViewAttribute('activeSelection', null);
            view.startRangeHighlighting(tokenArrays, { draggable: false, resolveNames: true });

            sourceSelector = false;
        }

        function setChartSource() {
            if (sourceSelector) {
                chartSourceToken = null;
                selectEnd();
                return;
            }
            chartSourceToken = null;
            sourceSelector = true;

            view.registerCellSelectionHandlers(function () {
                endSourceCancelListener(selectEnd);
            }, function (selection) {
                var range = selection.ranges[0];
                var sheet = view.getActiveSheet();

                if (!chartSourceToken) {
                    chartSourceToken = new TokenArray(app, view.getActiveSheetModel());
                    view.startRangeHighlighting(chartSourceToken, { resolveNames: true });
                }

                chartSourceToken.clear().appendRange(range, { sheet: sheet, abs: true });

                view.getActiveSheetModel().setViewAttribute('activeSelection', selection);

                return true;
            });

            initSourceCancelListener(selectEnd);



            return sourceSelector;

        }

        /**
         * changeChartType maps the assigned id to chart-data
         * all ids are mappend in DrawingControls.CHART_TYPES
         *
         * There is a special behavior for bubble-chart, change to bubble or from bubble.
         * all series-data will be removed an initialized complete new by the ChartCreator
         */
        function changeChartType(id) {
            var chart = getDrawingModel('chart');
            var data = chartTypes[id];
            var oldData = chart.getMergedAttributes().chart;
            if (data.type !== oldData.type && (data.type.indexOf('bubble') === 0 || oldData.type.indexOf('bubble') === 0)) {
                //special behavior for bubble charts!!!
                var possSources = chart.getExchangeInfo();
                if (possSources.warn) {
                    if (possSources.warn === 'sheets') {
                        view.yell('warning', Controls.WARN_DIFF_SHEETS);
                    } else if (possSources.warn === 'directions') {
                        view.yell('warning', Controls.WARN_DIFF_DIRS);
                    }
                } else {
                    ChartCreator.generateOperations(app, possSources.range, possSources.sheet, chart, (possSources.axis - 1) * - 1, data);
                }
            } else {
                generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: data} });
            }
        }

        function getAxisModel(id) {
            var chart = getDrawingModel('chart');
            if (chart && chart.isAxesEnabled()) {
                id = getCorrectAxisId(id);
                return chart.getAxisModel(id);
            }
        }

        function addAxisDef(id, defs) {
            defs['drawing/chartaxes/' + id + '/visible'] = {
                parent: 'drawing/chartaxes',
                set: function (visible) {
                    var line;
                    if (visible) {
                        line = ChartCreator.getStandardShape();
                    } else {
                        line = ChartCreator.getNoneShape();
                    }
                    setAxisAttributes(Operations.SET_CHART_AXIS_ATTRIBUTES, id, {
                        line: line
                    });
                },
                get: function () {
                    var axis = getAxisModel(id);
                    if (axis) { return axis.getMergedAttributes().line.type !== 'none'; } else { return false; }
                }
            };

            defs['drawing/chartaxes/' + id + '/label'] = {
                parent: 'drawing/chartaxes',
                set: function (visible) {
                    setAxisAttributes(Operations.SET_CHART_AXIS_ATTRIBUTES, id, {
                        axis: {label: visible}
                    });
                },
                get: function () {
                    var axis = getAxisModel(id);
                    if (axis) { return axis.getMergedAttributes().axis.label; } else { return ''; }
                }
            };

            defs['drawing/chartaxes/' + id + '/title'] = {
                parent: 'drawing/chartaxes',
                set: function (title) {
                    setAxisAttributes(Operations.SET_CHART_TITLE_ATTRIBUTES, id, {
                        text: {link: [title]}, character: ChartCreator.getHeadChar()
                    });
                },
                get: function () {
                    var axis = getAxisModel(id);
                    if (axis) { return axis.getTitle().getMergedAttributes().text.link[0] || ''; } else { return ''; }
                }
            };

            defs['drawing/chartgrids/' + id] = {
                parent: 'drawing/chartaxes',
                set: function (state) {
                    if (state) {
                        setAxisAttributes(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, id, {line: ChartCreator.getStandardShape()});
                    } else {
                        setAxisAttributes(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, id, {line: ChartCreator.getNoneShape()});
                    }

                },
                get: function () {
                    var axis = getAxisModel(id);
                    if (axis) { return axis.getGrid().getMergedAttributes().line.type !== 'none'; } else { return false; }
                }
            };

        }


        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['drawing/operations'] = {
                parent: ['sheet/unlocked', 'view/cell/editmode/off'] // allow insertion of drawings only if sheet is not locked
            };

            defs['image/insert/dialog'] = {
                parent: 'drawing/operations',
                set: function () {
                    return view.showInsertImageDialog();
                },
                focus: 'wait' // wait for the dialog before returning focus to application
            };

            defs['chart/insert'] = {
                parent: 'drawing/operations',
                set: insertChart,
                enable: function () {
                    var ranges = view.getSelectedRanges();
                    return (ranges.length === 1) && !view.isSingleCellInRange(ranges[0]);
                }
            };

            defs['document/editable/drawing'] = {
                parent: 'sheet/unlocked',
                enable: function () { return view.hasDrawingSelection(); }
            };

            defs.drawing = {
                parent: 'document/editable/drawing',
                get: getDrawingModel
            };

            defs['drawing/delete'] = {
                parent: 'document/editable/drawing',
                set: function () { view.deleteDrawings(); }
            };

            defs['drawing/chart'] = {
                parent: 'document/editable/drawing',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            };

            defs['drawing/chartexchange'] = {
                parent: 'document/editable/drawing',
                set: function () {
                    var chart = getDrawingModel('chart');
                    var possSources = chart.getExchangeInfo();
                    if (possSources.warn) {
                        if (possSources.warn === 'sheets') {
                            view.yell('warning', Controls.WARN_DIFF_SHEETS);
                        } else if (possSources.warn === 'directions') {
                            view.yell('warning', Controls.WARN_DIFF_DIRS);
                        }
                    } else {
                        ChartCreator.generateOperations(app, possSources.range, possSources.sheet, chart, possSources.axis);
                    }
                },
                enable: function () { var ch = getDrawingModel('chart'); if (ch) { return ch.getSeriesCount() > 0; }},
            };

            defs['drawing/chartaxes'] = {
                parent: 'drawing/chart',
                enable: function () { var ch = this.getValue(); return _.isObject(ch) && ch.isAxesEnabled(); }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getChartTypeForGui() : null; },
                set: changeChartType
            };

            defs['drawing/chartvarycolor'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return (chartModel && chartModel.isVaryColorEnabled()) ? chartModel.getMergedAttributes().chart.varyColors : false; },
                set: function (state) { getDrawingModel('chart').changeVaryColors(state); },
                enable: function () {
                    var chartModel = getDrawingModel('chart');
                    return chartModel && chartModel.isVaryColorEnabled();
                }
            };

            defs['drawing/chartcolorset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeColorSet(colorset); }
            };

            defs['drawing/chartstyleset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeStyleSet(colorset); }
            };

            defs['drawing/chartsource'] = {
                parent: 'drawing/chart',
                set: setChartSource,
                get: function () { return sourceSelector; }
            };

            defs['drawing/chartlegend/pos'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getLegendModel().getMergedAttributes().legend.pos : null; },
                set: function (pos) { generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, {attrs: {legend: {pos: pos}, character: ChartCreator.getStandardChar()}}); }
            };

            defs['drawing/chartdatalabel'] = {
                parent: 'drawing/chart',
                get: function (ch) { return (ch && ch.isDataLabelEnabled()) ? ch.getMergedAttributes().chart.dataLabel : false; },
                set: function (state) {  generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: {dataLabel: state}}}); },
                enable: function () { var ch = getDrawingModel('chart'); return _.isObject(ch) && ch.isDataLabelEnabled(); }
            };

            addAxisDef('x', defs);
            addAxisDef('y', defs);


            defs['drawing/charttitle'] = {
                parent: 'drawing/chart',
                set: function (title) {
                    setAxisAttributes(Operations.SET_CHART_TITLE_ATTRIBUTES, 'main', {text: {link: [title]}, character: ChartCreator.getHeadChar()});
                },
                get: function (chartModel) {
                    if (chartModel) {
                        return chartModel.getTitleModel('main').getMergedAttributes().text.link[0] || '';
                    }
                }
            };


            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        //View and Listener initialization
        app.on('docs:init', function () {
            model = app.getModel();
            view = app.getView();

            initToolBox();
            view.on('change:selection', changeSelection);
        });

    } // class DrawingController

    // exports ================================================================

    return DrawingController;

});
