/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2012 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/app/actionshelper',
    ['io.ox/core/capabilities',
     'io.ox/core/extensions',
     'io.ox/core/extPatterns/links',
     'io.ox/preview/main',
     'io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/extensionregistry',
     'gettext!io.ox/office/preview'
    ], function (capabilities, ext, links, preview, Utils, ExtensionRegistry, gt) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns whether the file selection described in the passed callback data
     * object is viewable in the OX Viewer application.
     *
     * @param {Object} data
     *  The data passed to the 'requires' callback function of an action.
     *
     * @returns {Boolean}
     *  Whether the passed data describes exactly one file that is viewable in
     *  the OX Viewer application.
     */
    function isViewable(data) {
        return data.collection.has('one') && ExtensionRegistry.isViewable(data.context.filename);
    }

    /**
     * Converts the passed data object of a mail attachment to a normalized
     * file descriptor object that can be used to launch an OX Viewer
     * application instance.
     *
     * @param {Object} data
     *  The data object to be converted.
     *
     * @param {String} name
     *  The name of the property in the passed data object which contains the
     *  mail identifier and mail folder name.
     *
     * @returns {Object}
     *  The converted file descriptor for the OX Viewer application.
     */
    function convertMailData(data, name) {
        return {
            source: 'mail',
            folder_id: data[name].folder_id,
            id: data[name].id,
            attached: data.id,
            filename: data.filename,
            file_size: data.size,
            file_mimetype: data.content_type
        };
    }

    /**
     * Converts the passed baton data object to a file descriptor object that
     * can be used to launch an OX Viewer application instance.
     *
     * @param {Object} data
     *  The original baton data object to be converted.
     *
     * @param {String} type
     *  The type of the passed data object. Must be one of the following types:
     *  - 'file'
     *      A file descriptor used by OX Drive.
     *  - 'mail'
     *      An attachment descriptor used by OX Mail.
     *  - 'task'
     *      An attachment descriptor used by OX Tasks.
     *
     * @returns {Object|Null}
     *  The converted file descriptor for the OX Viewer application; or null,
     *  if the file descriptor could not be created.
     */
    function convertBatonData(data, type) {

        switch (type) {

        // file from OX Drive: use the original baton data object
        case 'file':
            return data;

        // mail attachment: convert file properties to the property names used in OX Drive
        case 'mail':
            return convertMailData(data, 'mail');

        // task attachment: convert file properties to the property names used in OX Drive
        case 'task':
            return {
                source: 'task',
                folder_id: data.folder,
                id: data.id,
                attached: data.attached,
                module: data.module,
                filename: data.filename,
                file_size: data.file_size,
                file_mimetype: data.file_mimetype
            };
        }

        Utils.error('ActionsHelper.convertBatonData(): unknown descriptor type: "' + type + '"');
        return null;
    }

    /**
     * Converts the passed data object received from a side pop-up to a file
     * descriptor object that can be used to launch an OX Viewer application
     * instance.
     *
     * @param {Object} data
     *  The original file descriptor object to be converted.
     *
     * @returns {Object|Null}
     *  The converted file descriptor for the OX Viewer application; or null,
     *  if the file descriptor could not be created.
     */
    function convertPopupData(data) {

        // attachment from OX Tasks, OX Calendar etc. contain a 'module' property
        if ('module' in data) {
            Utils.error('ActionsHelper.convertPopupData(): descriptor for other modules not implemented yet');
            return null;
        }

        // attachment of existing mails: 'data' sub object has same format as baton data
        if (_.isObject(data.data) && _.isObject(data.data.mail)) {
            return convertMailData(data.data, 'mail');
        }

        // new mail with attachment, saved as draft: file descriptor marked with 'source' property
        if (data.source === 'mail' && _.isObject(data.data) && _.isObject(data.data.parent)) {
            return convertMailData(data.data, 'parent');
        }

        // other sources need to be implemented
        if ('source' in data) {
            Utils.error('ActionsHelper.convertPopupData(): descriptor for source "' + data.source + '" not implemented yet');
            return null;
        }

        // fall-back to file descriptors of OX Drive
        return {
            folder_id: data.folder_id,
            id: data.id,
            filename: data.filename,
            file_size: data.size,
            file_mimetype: data.mimetype
        };
    }

    /**
     * Launches a new OX Viewer application with the passed file descriptor.
     */
    function launchApplication(file) {
        ox.launch('io.ox/office/preview/main', { action: 'load', file: file });
    }

    // static class ActionsHelper =============================================

    /**
     * Defines static methods to create new actions and links to launch the OX
     * Viewer application from various other applications in OX AppSuite.
     */
    var ActionsHelper = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates a new action that will show a single document in the OX Viewer
     * application.
     *
     * @param {String} actionId
     *  The identifier of the new action.
     *
     * @param {Function} fileDescriptorHandler
     *  A converter callback function that receives the Baton object and has to
     *  return the file descriptor passed to the launcher of OX Viewer.
     */
    ActionsHelper.createViewerAction = function (actionId, type) {
        new links.Action(actionId, {
            requires: isViewable,
            action: function (baton) {
                var file = convertBatonData(baton.data, type);
                if (file) { launchApplication(file); }
            }
        });
    };

    /**
     * Creates a new clickable link that launches OX Viewer for a specific
     * file.
     *
     * @param {String} pointId
     *  The identifier of the extension point containing the links.
     *
     * @param {String} actionId
     *  The identifier of the action launching the OX Viewer application.
     *
     * @param {Object} [options]
     *  Additional options passed to the constructor of the link.
     */
    ActionsHelper.createViewerLink = function (pointId, actionId, options) {
        ext.point(pointId).extend(new links.Link(Utils.extendOptions({
            id: 'office_view',
            index: 100,
            label: gt('View'),
            ref: actionId
        }, options)));
    };

    /**
     * Extends the specified action. It will be disabled, if the file passed to
     * the action is viewable in the OX Viewer application.
     *
     * @param {String} actionId
     *  The identifier of the action to be disabled for files viewable in the
     *  OX Viewer application.
     */
    ActionsHelper.disableActionForViewable = function (actionId) {
        new links.Action(actionId, {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                if (isViewable(data)) {
                    data.stopPropagation();
                    return false;
                }
            }
        });
    };

    // static initialization ==================================================

    // register preview renderer for documents supported by OX Viewer
    if (capabilities.has('document_preview')) {
        (function () {

            function getWidth(options) {
                return Utils.getIntegerOption(options, 'width', 400);
            }

            function getUrl(data, options) {
                var url = data.dataURL || data.url;
                return url + '&format=preview_image&width=' + getWidth(options) + '&delivery=view&scaleType=contain' + '&dummy=' + Date.now();
            }

            function drawPreview(data, options) {

                var // the outer container node for the preview image
                    containerNode = this,
                    // the normalized file descriptor
                    file = _.isObject(data) ? convertPopupData(data) : null,
                    // the link node that will launch OX Viewer when clicked
                    linkNode = preview.protectedMethods.clickableLink(data, function (e) {
                        e.preventDefault();
                        if (file) { launchApplication(file); }
                    }).appendTo(containerNode),
                    // the image node showing the first page of the document
                    imgNode = $('<img>', { alt: '' }).addClass('io-ox-clickable').appendTo(linkNode),
                    // a Deferred object waiting for the image
                    def = $.Deferred();

                containerNode.css({ minHeight: 20 }).busy();

                // prepare and load the preview image
                imgNode
                    .css({ width: getWidth(options), maxWidth: getWidth(options), visibility: 'hidden' })
                    .on('load', function () { def.resolve(); })
                    .on('error', function () { def.reject(); })
                    .attr('src', getUrl(data, options));

                // react on the result of the image
                def
                .always(function () { containerNode.css({ minHeight: '' }).idle(); })
                .done(function () { imgNode.css('visibility', ''); })
                .fail(function () { containerNode.empty(); });

                preview.protectedMethods.dragOutHandler(linkNode);
            }

            // create a new rendering engine for all supported documents
            preview.Renderer.point.extend(new preview.Engine({
                id: 'office',
                index: 10,
                supports: ExtensionRegistry.getViewableExtensions(), // TODO: mimetypes too?
                getUrl: getUrl,
                draw: drawPreview,
                omitDragoutAndClick: true
            }));

        }());
    }

    // exports ================================================================

    return ActionsHelper;

});
