/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/portal/view/documentview',
    ['gettext!io.ox/office/portal',
     'io.ox/office/portal/utils',
     'io.ox/core/extensions',
     'io.ox/office/portal/action/recentsactions',
     'io.ox/core/capabilities'
    ],
function (gt, Utils, Ext, RecentsActions) {

    'use strict';

    var documentView = Backbone.View.extend({

        tagName: 'div',

        className: 'office-portal-document',

        documentNode: null,

        events: {

        },

        initialize: function () {

            var // the document view object
                self = this,
                // the handler for Window resize events
                resizeHandler = _.debounce(function () { self.showOptimalCount(self, { isResize: true }); }, 250);

            this.listenTo(this.collection, 'reset', this.render);
            this.listenTo(this.collection, 'destroy', this.remove);
            this.render(Utils.getActiveApp(this.model));

            // Registering resize handler for the recent document list
            $(window).on('resize', resizeHandler);

            this.model.getWindow().on('hide', function () {
                // Deregistering resize handler for the recent document list, if the app window is hidden
                $(window).off('resize', resizeHandler);
            });

        },

        /**
         * Renders recent documents section of supported applications
         *
         * @param {String} appBaseName
         *  The base name of the current application.
         *
         * @returns {documentsView}
         */
        render: function (appBaseName) {

            var documentCollection = this.collection,
                self = this,
                emptyNotificationNode = $('<p>').addClass('office-portal-notification bg-info').text(gt('You have no recent documents.'));

            if (documentCollection.length === 0) {
                self.$el.append(emptyNotificationNode);
                self.$el.removeClass('f6-target');
                return;
            }

            self.$el.empty();

            _.each(documentCollection.models, function (model) {
                var // the application handling files with the specified name
                    fileAppName = Utils.getAppBaseName(model.get('filename')),
                    // the extension of files
                    errorSuffix = '_ox';
                // quit if the current visible application is not the file application
                // or if the file name ends with the error extension '_ox'
                if ((appBaseName !== fileAppName) || Utils.stringEndsWith(model.get('filename'), errorSuffix)) { return; }
                // draw recent document items and bind actions  through extension invoke
                RecentsActions.documentPoint.invoke('draw', self.$el, Ext.Baton({ data : model.attributes }));
            });

            this.showOptimalCount(self);

            // if no recent file item is drawn to the DOM after app type filter, show empty notification,
            // remove f6-target so that the recent file list is skipped while iterating with keyboard.
            if (self.$el.find('.document-link').length === 0) {
                self.$el.append(emptyNotificationNode);
                self.$el.removeClass('f6-target');
            }

            return this;
        },

        // show as many recent documents, that they all fit in the current window height.
        // count is set fixed at 10 for small and medium displays.
        showOptimalCount: function (self, options) {

            var // whether this list creation is triggered by a resize event
                isResize = Utils.getBooleanOption(options, 'isResize', false),
                // the node element with class 'office-portal-recents-list'
                list = self.$el,
                // the height in pixel of the complete recent list
                listPaneHeight = list.parent().outerHeight(),
                // the existing links to the recent documents
                documents = list.children('a.document-link'),
                // the height in pixel of one link
                documentHeight = documents.outerHeight(),
                // the length of the collection of recent documents
                collectionLength = documents.length,
                // whether the view is displayed in mode for small devices (width < 1024px)
                isSmallDevice = (_.device('small') || _.device('medium')),
                // the height in pixel of the header line
                headlineHeight = list.parent().find('h2').outerHeight(),
                // calculate optimal recent documents to show
                documentsToShow = isSmallDevice ? collectionLength : Math.floor((listPaneHeight - headlineHeight) / documentHeight) - 2,
                // the current number of displayed documents
                oldLength = 0,
                // whether it is necessary to rerender the recent document list
                rerender = true;

            // set a fixed document count for small and medium displays
            if ((documentsToShow > 10) && isSmallDevice) { documentsToShow = 10; }

            // checking, if rerendering the list of recent files is neccessary
            if (isResize) {
                oldLength = _.filter(documents, function (oneLink) {
                    return $(oneLink).css('display') !== 'none';
                }).length;

                // no rerendering required, if list length is not modified, or
                // if already all documents are shown and the size is increased
                if ((documentsToShow === oldLength) ||
                    ((documentsToShow > oldLength) && (oldLength >= collectionLength))) { rerender = false; }
            }

            if (rerender) {
                // show optimal count of documents
                documentsToShow = (documentsToShow > collectionLength) ? collectionLength : documentsToShow;

                _.each(documents, function (oneLink, index) {
                    if (index + 1 <= documentsToShow) {
                        $(oneLink).css('display', 'block');
                    } else {
                        $(oneLink).css('display', 'none');
                    }
                });
            }

        }

    });

    return documentView;

});