/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/portal/mainhelper',
    ['settings!io.ox/office',
     'gettext!io.ox/office/portal',
     'io.ox/core/date',
     'io.ox/core/capabilities',
     'io.ox/core/commons',
     'io.ox/core/notifications',
     'io.ox/office/portal/collection/documentcollection',
     'io.ox/office/portal/collection/templatecollection',
     'io.ox/office/portal/view/documentview',
     'io.ox/office/portal/view/templateview',
     'io.ox/office/portal/view/toolbarview'],
function (Settings, gt, CoreDate, Capabilities, Commons, Notifications, DocumentCollection, TemplateCollection, DocumentView, TemplateView, ToolbarView) {

    'use strict';

    var // the exported MainHelper class
        MainHelper = {};

    /**
     * Creating the launcher for the portal applications.
     *
     * @param {Object} app
     *  The OX Appsuite application object.
     *
     * @param {String} appName
     *  The application name.
     *
     * @param {String} appTitle
     *  The application title.
     *
     * @returns
     *  The application launcher.
     */
    MainHelper.getAppLauncher = function (app, appName, appTitle) {

        // configure application launcher
        return function (options) {

            // create main window object
            var portalWindow = ox.ui.createWindow({
                name: appName,
                title: appTitle,
                chromeless: true
            });

            // the bootstrap of the OX Documents app
            function initOXDocuments() {

                // prepare app skeleton. Maybe this could be the AppView in the future.
                var rootNode = portalWindow.nodes.main,
                    windowBodyNode = portalWindow.nodes.body,
                    recentsNode = $('<div>').addClass('office-portal-recents').append($('<h2>').text(gt('Recent Documents'))),
                    recentsList = $('<div>').addClass('office-portal-recents-list f6-target'),
                    templatesNode = $('<div>').addClass('office-portal-templates').append($('<h2>').text(gt('New from Template'))),
                    // the document collection for the recent documents list
                    documentCollection = new DocumentCollection(),
                    // the template collection for the listed templates
                    templateCollection = new TemplateCollection(),
                    // whether this is an Android device with reduced functionality
                    reducedFunctionality = (_.browser.Android) && (appTitle === 'Text');

                // rootNode.empty().append(recentsNode.append(recentsList), templatesNode);
                rootNode.empty().append(recentsNode.append(recentsList));

                // create app views
                new ToolbarView({ el: windowBodyNode, model: app});
                new DocumentView({collection: documentCollection, el: recentsList, model: app});

                if (reducedFunctionality) {
                    recentsNode.addClass('reduced-recents-functionality');
                } else {
                    rootNode.append(templatesNode); // no templateView for Android
                    new TemplateView({collection: templateCollection, el: templatesNode, model: app});
                }

            }

            // save settings to the app object
            app.settings = Settings;

            portalWindow.addClass('io-ox-office-portal-main');

            // set it as the application window
            app.setWindow(portalWindow);

            // show portal window
            portalWindow.show();

            // Reinit app on portal window show. This is needed for this use case:
            // User opens a document from the recents list -> loses app focus,
            // and close the document -> gained focus to this app without an app 'resume' event).
            portalWindow.on('show', initOXDocuments);

            // refresh recent document list on app 'resume' event.
            // resume event-> app name in the topbar is clicked.
            //app.on('resume', initOXDocuments);

            // refreshing after a file was modified
            // -> not always create completely new template view
            // FilesApi is 'io.ox/files/api'
            // FilesApi.on('update', initOXDocuments);

            // init OX Documents in folder support addition success.
            Commons.addFolderSupport(app, null, 'infostore', options.folder)
                .always(function always() {
                    initOXDocuments();
                })
                .fail(function fail(result) {
                    var errorMsg = (result && result.error) ? result.error + ' ' : '';
                    errorMsg += gt('Application may not work as expected until this problem is solved.');
                    Notifications.yell('error', errorMsg);
                });
        };

    };

    return MainHelper;

});