/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/editcontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/basecontroller'
    ], function (Utils, BaseController) {

    'use strict';

    // class EditController ===================================================

    /**
     * The base class for controller classes of all OX Documents allowing to
     * edit a document.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {EditApplication} app
     *  The OX Documents application that has created this controller instance.
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the behavior of this controller.
     *  Supports all options supported by the base class BaseController.
     */
    function EditController(app, initOptions) {

        var // self reference
            self = this,

            // the model instance of the passed application
            model = null,

            // the view instance of the passed application
            view = null,

            // all the little controller items
            items = {

                // application and view ---------------------------------------

                // enabled unless the application is in 'internal error' state
                'app/valid': {
                    enable: function () { return app.getState() !== 'error'; }
                },

                // toggle the main side pane
                'view/sidepane': {
                    parent: 'app/valid',
                    get: function () { return app.getView().isSidePaneVisible(); },
                    set: function (state) { app.getView().toggleSidePane(state); },
                    shortcut: { keyCode: 'F3', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                // document ---------------------------------------------------

                'document/acquireedit': {
                    parent: 'app/valid',
                    enable: function () { return !app.isLocked() && (app.getState() !== 'offline') && !model.getEditMode(); },
                    set: function () { app.acquireEditRights(); }
                },

                'document/reload': {
                    enable: function () { return app.getState() === 'error'; },
                    set: function () { app.reloadDocument(); }
                },

                // to be used as parent item for all items that require edit mode
                'document/editable': {
                    parent: 'app/valid',
                    enable: function () { return model.getEditMode(); }
                },

                'document/rename': {
                    parent: 'document/editable',
                    get: function () { return app.getShortFileName(); },
                    set: function (fileName) { return app.rename(fileName); }
                },

                'document/undo': {
                    parent: 'document/editable',
                    enable: function () { return view.isUndoAvailable(); },
                    set: function () { return view.undo(); },
                    shortcut: [
                       // bug 33077: restrict to application pane, to not interfere with text field's
                       // native undo/redo and automatic item execution after losing focus
                        { keyCode: 'Z', ctrlOrMeta: true, selector: '.app-pane' },
                        { keyCode: 'BACKSPACE', alt: true, selector: '.app-pane' }
                    ],
                    busy: true
                },

                'document/redo': {
                    parent: 'document/editable',
                    enable: function () { return view.isRedoAvailable(); },
                    set: function () { return view.redo(); },
                    shortcut: [
                        // bug 33077: restrict to application pane, to not interfere with text field's
                        // native undo/redo and automatic item execution after losing focus
                        { keyCode: 'Y', ctrlOrMeta: true, selector: '.app-pane' },
                        { keyCode: 'BACKSPACE', shift: true, alt: true, selector: '.app-pane' }
                    ],
                    busy: true
                },

                'document/saveas': {
                    parent: 'document/editable',
                    enable: function () { return model.getEditMode(); },
                    set: function () { return app.copyDocument(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'document/saveastemplate': {
                    parent: 'document/editable',
                    enable: function () { return model.getEditMode(); },
                    set: function () { return app.copyDocument({asTemplate: true}); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'document/autosave': {
                    parent: 'document/editable',
                    enable: function () { return model.getEditMode(); },
                    set: function () { app.autoSave(); }
                },

                // debug mode -------------------------------------------------

                'debug/enabled': {
                    enable: function () { return Utils.DEBUG; }
                },

                'debug/toggle': {
                    parent: 'debug/enabled',
                    get: function () { return view.isDebugPaneVisible(); },
                    set: function (state) { view.toggleDebugPane(state); }
                },

                'debug/highlight': {
                    parent: 'debug/enabled',
                    get: function () { return view.isDebugHighlight(); },
                    set: function (state) { view.toggleDebugHighlight(state); }
                },

                'debug/clipboard': {
                    parent: 'debug/enabled',
                    get: function () { return view.isClipboardPaneVisible(); },
                    set: function (state) { view.toggleClipboardPane(state); }
                },

                'debug/realtime/trigger': {
                    parent: 'debug/enabled',
                    set: function (type) { app.debugTriggerRealtimeEvent(type); }
                },

                'debug/realtime/sendinvalidoperation': {
                    parent: 'debug/enabled',
                    enable: function () { return model.getEditMode(); },
                    set: function () { app.debugSendInvalidOperation(); }
                },

                'debug/operations/apply': {
                    parent: 'debug/enabled',
                    enable: function () { return model.getEditMode(); },
                    set: function (operation) { model.applyOperations(operation); }
                },

                'debug/quit': {
                    parent: 'debug/enabled',
                    set: function (type) { app.debugQuit(type); }
                }
            };

        // base constructor ---------------------------------------------------

        BaseController.call(this, app, initOptions);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction (model not available at construction time)
        app.on('docs:init', function () {
            model = app.getModel();
            view = app.getView();
        });

        // update GUI after operations, or changed state of edit mode
        app.on('docs:import:after', function () {
            model.on('operations:after change:editmode', function () { self.update(); });
            model.getUndoManager().on('change:count', function () { self.update(); });
        });

    } // class EditController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: EditController });

});
