/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/charttypepicker',
    ['io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/drawinglayer/lib/canvasjs.min',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/drawing'], function (BaseControls, CanvasJS, Utils, gt) {

    'use strict';

    var // class name shortcuts
        RadioList = BaseControls.RadioList,
        idCounter = 0;

    /**
     * ChartTypePicker generates preview Images for all assigned Chart-Types
     * With CanvasJS it uses the same Drawer like the Chart in the Sheets
     * Colors are Accents 1 till 4 and the preview are refresh if the theme is changed
     */
    function ChartTypePicker(app, chartTypes, options) {
        var self = this,
            //the array for the deffered attribute sets, to draw the chart-canvas
            chain = [],
            //data-model for CanvasJS to drop all changed value before drawing
            data = {},
            scatterPoints = [2.5, 4, 1.5, 3.5, 2, 2.5];


        RadioList.call(this,  Utils.extendOptions({itemDesign: 'grid', tooltip: gt('Chart type'), gridColumns: 6}, options));


        /**
         * inits the data-model with disabled axes and random-values
         */
        (function initData() {
            data.axisX = {};
            data.axisY = {};
            data.creditHref = '';
            data.creditText = '';
            data.title = { text: '' };
            data.data = [];


            var pointList = [[6, 4, 5, 3, 4, 1], [1, 2, 3, 4, 6, 7]];

            _.each(pointList, function (points, series) {
                var dataPoints = [];
                data.data[series] = { markerSize: 2, indexLabelFontColor: 'transparent', indexLabelLineColor: 'transparent', showInLegend: false, dataPoints: dataPoints, sourcePoints: {}};
                _.each(points, function (point, index) {
                    dataPoints.push({ label: 'bla ' + (index + 1), y: point, z: (2 + Math.random() * 100)|0});
                });
            });

            data.axisX.labelFontColor = data.axisX.lineColor = data.axisX.tickColor = data.axisX.gridColor = 'transparent';
            data.axisX.labelFontSize = 1;
            data.axisY.labelFontColor = data.axisY.lineColor = data.axisY.tickColor = data.axisY.gridColor = 'transparent';
            data.axisY.labelFontSize = 1;
        })();




        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode, sectionId, id, options) {
            var value = options.data;
            buttonNode.empty();
            buttonNode.css({
                width: '100px',
                height: '100px',
                textAlign: 'center'
            });
            buttonNode.addClass('mini-caption');
            if (value) {
                var use = value.cjs;

                var div = $('<div>');
                div.addClass('caption');
                var span = $('<span>');
                span.text(options.tooltip);
                span.attr('data-role', 'label');
                div.append(span);

                chain.push({
                    id: id,
                    div: buttonNode,
                    cjs: use,
                    single: (use === 'pie' || use === 'doughnut')
                });
                buttonNode.append(div);
            }
        }
        self.getMenu().on('create:item', createItemHandler);


        /**
         * called initial and on theme-change
         * creates CanvasJS structure
         * and draws deffered all types in the chain to images
         */
        function updateTable() {
            var id = 'chart-dummy-' + idCounter;
            idCounter++;
            var dummy = $('<div>');
            dummy.attr('id', id);
            dummy.appendTo('body');
            dummy.css({
                position: 'fixed',
                visibility: 'hidden',
                left: '-100px',
                top: '-100px',
                width: '100px',
                height: '100px'
            });
            var chart = new CanvasJS.Chart(id, data);
            var canvas = dummy.find('canvas')[0];

            if (!canvas) {
                Utils.warn('no canvas could be found, no chart-type*preview will be generated');
                return;
            }

            function render(args) {
                var cjs = args[0].cjs;
                var div = args[0].div;
                var single = args[0].single;
                var cells = app.getModel().getStyleSheets('cell');

                _.each(data.data, function (dataSeries, index) {
                    dataSeries.type = cjs;
                    var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                    dataSeries.color = cells.getCssColor(cl, 'fill');

                });

                _.each(data.data[0].dataPoints, function (dataPoint, index) {
                    if (single) {
                        var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                        dataPoint.color = cells.getCssColor(cl, 'fill');
                    } else {
                        dataPoint.color = null;
                    }
                });

                _.each(data.data[1].dataPoints, function (dataPoint, index) {
                    if (args[0].id.indexOf('scatter') === 0) {
                        dataPoint.x = scatterPoints[index];
                    } else {
                        dataPoint.x = index;
                    }
                });

                chart.render();

                var dataURL = canvas.toDataURL();

                var img = new Image();
                img.src = dataURL;

                $(img).css({
                    marginLeft: '-11px',
                    display: 'block',
                    width: '100px',
                    height: '100px',
                });

                div.children('img').each(function () {
                    $(this).remove();
                });
                div.prepend(img);
            }

            var def = app.processArrayDelayed(render, chain, {chunkLength: 1, repeatDelay: 100});
            def.done(function () {
                dummy.css({
                    width: '1px',
                    height: '1px'
                });
                dummy.remove();
            });
        }

        /**
         * initializes the RadioList Buttons by iterating over the assigned chartTypes
         */
        function initializeTable() {
            self.clearMenu();

            _(chartTypes).each(function (chartType, id) {
                self.createOptionButton(chartType.cat, id, { label: chartType.title, tooltip: chartType.title, data: chartType});
            });

        }


        initializeTable();
        // insert color buttons in the drop-down menu after import and changed themes
        app.on('docs:import:success', updateTable);
        app.getModel().getThemeCollection().on('triggered', updateTable);

    }



    return RadioList.extend({ constructor: ChartTypePicker });

});
