/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/toolbaractions',
    ['io.ox/core/extPatterns/links',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/main'
    ], function (links, Utils, gt) {

    'use strict';

    // static class ToolBarActions ============================================

    /**
     * Provides static methods to configure the global window tool bar for
     * various application types.
     */
    var ToolBarActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates a new icon in the global window tool bar of the specified
     * application type. Registration is done globally once per application
     * type, using the extension framework.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Object} options
     *  A map with options for the new tool bar icon. The following options
     *  are supported:
     *  @param {String} options.id
     *      The unique identifier of the tool bar icon.
     *  @param {String} options.icon
     *      The CSS class of the icon symbol.
     *  @param {String} options.label
     *      The tool tip label of the icon.
     *  @param {Number} options.index
     *      The insertion index, compared to other icons on the same level.
     *  @param {String} options.itemKey
     *      The key of the controller item to be executed for the button.
     *  @param {Any} [options.itemValue]
     *      The value to be passed to the controller item.
     *  @param {Object} [options.itemOptions]
     *      Additional options to be passed to the controller item.
     */
    ToolBarActions.createIcon = function (moduleName, options) {

        var // extract all passed options
            id = Utils.getStringOption(options, 'id'),
            icon = Utils.getStringOption(options, 'icon'),
            label = Utils.getStringOption(options, 'label', ''),
            index = Utils.getIntegerOption(options, 'index', 0),
            itemKey = Utils.getStringOption(options, 'itemKey', ''),
            itemValue = Utils.getOption(options, 'itemValue'),
            itemOptions = Utils.getObjectOption(options, 'itemOptions'),

            // unique name of the action
            ACTION_POINT = moduleName + '/actions/' + id,
            // root path for tool bar links
            TOOLBAR_PATH = moduleName + '/links/toolbar',
            // unique name of the tool bar link
            TOOLBAR_POINT = TOOLBAR_PATH + '/' + id;

        // create the action (call with application instead of baton)
        new links.Action(ACTION_POINT, {
            action: function (baton) {
                return baton.app.getController().executeItem(itemKey, itemValue, itemOptions);
            }
        });

        // create an action group for the action link in the tool bar
        new links.ActionGroup(TOOLBAR_PATH, {
            id: id,
            index: index,
            icon: function () { return Utils.createIcon(icon); }
        });

        // create the action link in the group (single links are shown as
        // icons in the tool bar, not as drop-down menu)
        new links.ActionLink(TOOLBAR_POINT, {
            label: label,
            index: 100,
            ref: ACTION_POINT
        });
    };

    /**
     * Creates a new 'search' icon in the global window tool bar of the
     * specified application type. The created action toggles the search tool
     * bar of the application window. Registration is done globally once per
     * application type, using the extension framework.
     *
     * @param {String} moduleName
     *  The application type identifier.
     */
    ToolBarActions.createSearchIcon = function (moduleName) {
        ToolBarActions.createIcon(moduleName, {
            id: 'search',
            icon: 'fa-search',
            label: gt('Toggle search'),
            itemKey: 'view/searchpane'
        });
    };

    /**
     * Creates a new 'download' icon in the global window tool bar of the
     * specified application type. The created action calls the method
     * 'BaseApplication.download()'. Registration is done globally once per
     * application type, using the extension framework.
     *
     * @param {String} moduleName
     *  The application type identifier.
     */
    ToolBarActions.createDownloadIcon = function (moduleName) {
        ToolBarActions.createIcon(moduleName, {
            id: 'download',
            icon: 'fa-cloud-download',
            label: gt('Download'),
            itemKey: 'document/download'
        });
    };

    /**
     * Creates a new 'print' icon in the global window tool bar of the
     * specified application type. The created action calls the method
     * 'BaseApplication.print()'. Registration is done globally once per
     * application type, using the extension framework.
     *
     * @param {String} moduleName
     *  The application type identifier.
     */
    ToolBarActions.createPrintIcon = function (moduleName) {
        ToolBarActions.createIcon(moduleName, {
            id: 'print',
            icon: 'fa-print',
            label: gt('Print'),
            itemKey: 'document/print'
        });
    };

    /**
     * Creates a new 'mail' icon in the global window tool bar of the
     * specified application type. The created action calls the method
     * 'BaseApplication.sendMail()'. Registration is done globally once per
     * application type, using the extension framework.
     *
     * @param {String} moduleName
     *  The application type identifier.
     */
    ToolBarActions.createMailIcon = function (moduleName) {
        ToolBarActions.createIcon(moduleName, {
            id: 'send',
            icon: 'fa-envelope-o',
            label: gt('Send as mail'),
            itemKey: 'document/sendmail'
        });
    };

    // exports ================================================================

    return ToolBarActions;

});
