/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2014 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([], function () {

    'use strict';

    describe('Chai extensions', function () {

        // constants ==========================================================

        describe('constant "EPSILON"', function () {
            it('should exist', function () {
                expect(chai.EPSILON).to.be.a('number');
                expect(chai.EPSILON).to.be.above(0);
                expect(chai.EPSILON).to.be.below(1);
            });
        });

        describe('constant "ITERATOR"', function () {
            it('should exist', function () {
                expect(chai.ITERATOR).to.exist;
            });
        });

        // properties =========================================================

        describe('property "almostZero"', function () {
            it('should pass for numbers almost zero', function () {
                expect(0).to.be.almostZero;
                expect(chai.EPSILON * 0.9).to.be.almostZero;
                expect(-chai.EPSILON * 0.9).to.be.almostZero;
            });
            it('should fail for numbers not almost zero', function () {
                expect(chai.EPSILON).not.to.be.almostZero;
                expect(chai.EPSILON * 1.1).not.to.be.almostZero;
                expect(-chai.EPSILON).not.to.be.almostZero;
                expect(-chai.EPSILON * 1.1).not.to.be.almostZero;
            });
        });

        // methods ============================================================

        describe('method "almostEqual"', function () {
            it('should pass for numbers almost equal to the expected number', function () {
                expect(1).to.almostEqual(1);
                expect(1 + chai.EPSILON * 0.9).to.almostEqual(1);
                expect(1 - chai.EPSILON * 0.9).to.almostEqual(1);
                expect(-1).to.almostEqual(-1);
                expect(-1 + chai.EPSILON * 0.9).to.almostEqual(-1);
                expect(-1 - chai.EPSILON * 0.9).to.almostEqual(-1);
                expect(1000).to.almostEqual(1000);
                expect(1000 + chai.EPSILON * 900).to.almostEqual(1000);
                expect(1000 - chai.EPSILON * 900).to.almostEqual(1000);
                expect(-1000).to.almostEqual(-1000);
                expect(-1000 + chai.EPSILON * 900).to.almostEqual(-1000);
                expect(-1000 - chai.EPSILON * 900).to.almostEqual(-1000);
            });
            it('should fail for numbers not close to the expected numbers', function () {
                expect(-1).not.to.almostEqual(1);
                expect(1 + chai.EPSILON * 1.1).not.to.almostEqual(1);
                expect(1 - chai.EPSILON * 1.1).not.to.almostEqual(1);
                expect(1).not.to.almostEqual(-1);
                expect(-1 + chai.EPSILON * 1.1).not.to.almostEqual(-1);
                expect(-1 - chai.EPSILON * 1.1).not.to.almostEqual(-1);
                expect(-1000).not.to.almostEqual(1000);
                expect(1000 + chai.EPSILON * 1100).not.to.almostEqual(1000);
                expect(1000 - chai.EPSILON * 1100).not.to.almostEqual(1000);
                expect(1000).not.to.almostEqual(-1000);
                expect(-1000 + chai.EPSILON * 1100).not.to.almostEqual(-1000);
                expect(-1000 - chai.EPSILON * 1100).not.to.almostEqual(-1000);
            });
        });

        // static functions ===================================================

        describe('method "runIterator"', function () {

            var CONTEXT = {};

            function iterate(array, iterator, options) {
                _(array).each(function (value) { return iterator.call(CONTEXT, value, options); });
                return 42;
            }

            it('should exist', function () {
                expect(chai).itself.to.respondTo('runIterator');
            });
            it('should invoke an iterating function', function () {
                var results = chai.runIterator(iterate, null, [[1, 2, 3], chai.ITERATOR, {}]);
                expect(results).to.be.an('object');
                expect(results.runs).to.be.an('array');
                expect(results.runs).to.have.length(3);
                expect(results.runs[0]).to.be.an('object');
                expect(results.runs[0].context).to.equal(CONTEXT);
                expect(results.runs[0].args).to.deep.equal([1, {}]);
                expect(results.runs[1]).to.be.an('object');
                expect(results.runs[1].context).to.equal(CONTEXT);
                expect(results.runs[1].args).to.deep.equal([2, {}]);
                expect(results.runs[2]).to.be.an('object');
                expect(results.runs[2].context).to.equal(CONTEXT);
                expect(results.runs[2].args).to.deep.equal([3, {}]);
                expect(results.args).to.be.an('array');
                expect(results.args).to.have.length(3);
                expect(results.args[0]).to.deep.equal([1, {}]);
                expect(results.args[1]).to.deep.equal([2, {}]);
                expect(results.args[2]).to.deep.equal([3, {}]);
                expect(results.ret).to.equal(42);
            });
            it('should invoke the breaker callback function', function () {
                var breakerArgs = [];
                function breaker(value) { breakerArgs.push(value); return value + 1; }
                var results = chai.runIterator(iterate, null, [[1, 2, 3], chai.ITERATOR], breaker);
                expect(results).to.be.an('object');
                expect(results.runs).to.be.an('array');
                expect(results.runs).to.have.length(3);
                expect(results.ret).to.equal(42);
                expect(breakerArgs).to.deep.equal([1, 2, 3]);
            });
        });

        // --------------------------------------------------------------------

    });
});
