/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/text/dom',
     'io.ox/office/text/format/listcollection',
     'gettext!io.ox/office/text'
    ], function (Utils, EditControls, DOM, ListCollection, gt) {

    'use strict';

    var // class name shortcuts
        RadioList = EditControls.RadioList;

    // static class TextControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX Text
     * application.
     *
     * @extends EditControls
     */
    var TextControls = _.clone(EditControls);

    // class ParagraphStylePicker =============================================

    /**
     * A drop-down menu control for paragraph style sheets.
     *
     * @constructor
     *
     * @extends EditControls.StyleSheetPicker
     */
    TextControls.ParagraphStylePicker = EditControls.StyleSheetPicker.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // paragraph style sheets
            paragraphStyles = app.getModel().getStyleSheets('paragraph'),

            // paragraph node inserted into the caption of the menu button
            captionParagraph = createParagraphNode('');

        // base constructor ---------------------------------------------------

        EditControls.StyleSheetPicker.call(this, app, 'paragraph', Utils.extendOptions({
            icon: 'docs-pilcrow',
            tooltip: gt('Paragraph style'),
            gridColumns: 4,
            updateCaptionMode: 'none',
            i18nModulePath: 'io.ox/office/text/resource/paragraphstylenames'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Creates a new paragraph element with the passed label.
         */
        function createParagraphNode(label) {
            return $('<span>').append(label ? DOM.createTextSpan().text(_.noI18n(label)) : DOM.createTextSpan());
        }

        /**
         * Formats the passed paragraph element according to the specified
         * paragraph style sheet.
         */
        function setParagraphFormatting(paragraph, styleId) {
            paragraphStyles.setElementAttributes(paragraph, { styleId: styleId }, { preview: true });
        }

        /**
         * Initializes a new list item according to the paragraph style sheet.
         */
        function createItemHandler(event, buttonNode, sectionId, styleId) {

            var // the paragraph node for the preview
                paragraph = createParagraphNode(Utils.getControlLabel(buttonNode));

            buttonNode.empty().append(paragraph);
            setParagraphFormatting(paragraph, styleId);
        }

        /**
         * Updates the drop-down menu button.
         */
        function updateHandler(styleId) {

            var // the span element taking the style sheet name
                span = captionParagraph.children('span').first(),
                // the name of the style sheet
                styleName = paragraphStyles.getName(styleId),
                // the translated style name
                translatedStyleName = null;

            if (_.isString(styleId) && _.isString(styleName)) {
                setParagraphFormatting(captionParagraph, styleId);
                translatedStyleName = self.translateStyleName(styleName);
                if (translatedStyleName) {
                    span.text(translatedStyleName);
                } else {
                    DOM.ensureExistingTextNode(span);
                }
                span.css('color', '');
            } else {
                DOM.ensureExistingTextNode(span);
            }
        }

        // initialization -----------------------------------------------------

        this.getMenuButton().children('div.caption').append(captionParagraph);
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a paragraph element into each list item
        this.getMenu().on('create:item', createItemHandler);

    }}); // class ParagraphStylePicker

    // class ListStylePicker ==================================================

    TextControls.ListStylePicker = RadioList.extend({ constructor: function (app, predefinedListStyles, initOptions) {

        var // the collection of fonts of the edited document
            fontCollection = app.getModel().getFontCollection(),
            // font attributes for the drop-down list entries
            labelCss = { fontFamily: fontCollection.getCssFontFamily('Cambria'), fontSize: '18px', minWidth: '18px' };

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            highlight: function (listStyleId) { return listStyleId !== ''; },
            splitValue: ListCollection.DEFAULT_VALUE,
            updateCaptionMode: 'none',
            itemDesign: 'grid',
            sortItems: Utils.getControlValue // sort by list identifiers
        }, initOptions));

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('list-style-picker');

        this.createMenuSection('default').createMenuSection('special', { gridColumns: 1 });
        _(predefinedListStyles).each(function (listStyle, listStyleId) {
            if (Utils.trimAndCleanString(listStyle.listLabel).length > 0) {
                this.createOptionButton('default', listStyleId, {
                    label: _.noI18n(listStyle.listLabel),
                    tooltip: _.noI18n(listStyle.tooltip),
                    css: { textAlign: 'center' },
                    labelCss: labelCss,
                    dataValue: listStyle.listKey
                });
            } else {
                this.createOptionButton('special', listStyleId, {
                    label: _.noI18n(listStyle.tooltip),
                    dataValue: listStyle.listKey
                });
            }
        }, this);

    }}); // class ListStylePicker

    // class BulletListStylePicker ============================================

    TextControls.BulletListStylePicker = TextControls.ListStylePicker.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        TextControls.ListStylePicker.call(this, app, ListCollection.getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            gridColumns: 4
        }, initOptions));

    }}); // class BulletListStylePicker

    // class NumberedListStylePicker ==========================================

    TextControls.NumberedListStylePicker = TextControls.ListStylePicker.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        TextControls.ListStylePicker.call(this, app, ListCollection.getPredefinedNumberedListStyles(), Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            gridColumns: 4
        }, initOptions));

    }}); // class NumberedListStylePicker


    // class ZoomTypePicker ===================================================

    /**
     * The selector for additional zoom types.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.ZoomTypePicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'fa-ellipsis-v',
            tooltip: gt('More zoom settings'),
            caret: false,
            updateCaptionMode: 'none'
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createMenuSection('fix')
            .createOptionButton('fix',  50, { label: gt('50%') })
            .createOptionButton('fix',  75, { label: gt('75%') })
            .createOptionButton('fix', 100, { label: gt('100%') })
            .createOptionButton('fix', 150, { label: gt('150%') })
            .createOptionButton('fix', 200, { label: gt('200%') })
            .createMenuSection('fit')
            .createOptionButton('fit', 'width', { label: gt('Fit to screen width') });

    }}); // class ZoomTypePicker

    // class TableStylePicker =================================================

    /**
     * @constructor
     *
     * @extends EditControls.StyleSheetPicker
     */
    TextControls.TableStylePicker = EditControls.StyleSheetPicker.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // table style sheets
            tableStyles = app.getModel().getStyleSheets('table'),

            /**
             * Registers a new table that will be formatted delayed with a
             * specific table style sheet.
             *
             * @param {jQuery} table
             *  The new table to be formatted.
             *
             * @param {String} styleId
             *  The identifier of the table style sheet.
             */
            updateTableFormatting = (function () {

                var // which tables in list items need formatting
                    pendingTables = [],
                    // background loop processing all pending tables
                    formatTimer = null;

                // direct callback: called every time when updateTableFormatting() has been called
                function registerTable(tableNode, styleId) {
                    pendingTables.push({ tableNode: tableNode, styleId: styleId });
                }

                // deferred callback: called once after the specified timeout
                function formatTables() {

                    // check if the background loop is already running
                    if (formatTimer) { return; }

                    // create a new background loop that processes all pending tables
                    formatTimer = app.repeatDelayed(function () {

                        var // table info with table and style identifier of the next table to be formatted
                            tableInfo = null;

                        // find a pending table that is still in the drop-down menu (menu may have been cleared
                        // in the meantime), abort the background loop, if no more tables are available
                        while ((tableInfo = pendingTables.shift()) && !Utils.containsNode(self.getMenuNode(), tableInfo.tableNode)) {}
                        if (!tableInfo) { return Utils.BREAK; }

                        // update the formatting of the table
                        tableStyles.setElementAttributes(tableInfo.tableNode, {
                            styleId: tableInfo.styleId,
                            table: { exclude: ['lastRow', 'lastCol', 'bandsVert'] }
                        }, { preview: true });
                        tableInfo.tableNode.find(DOM.PARAGRAPH_NODE_SELECTOR).css({ margin: 0 });  // no margins in micro tables allowed
                    });

                    // forget reference to the timer, when all tables have been formatted
                    formatTimer.always(function () { formatTimer = null; });
                }

                // create and return the debounced updateTableFormatting() method (with a delay of 10ms per table)
                return app.createDebouncedMethod(registerTable, formatTables, { delay: 10 });

            }()); // updateTableFormatting()

        // base constructors --------------------------------------------------

        EditControls.StyleSheetPicker.call(this, app, 'table', Utils.extendOptions({
            icon: 'docs-table-style',
            tooltip: gt('Table style'),
            gridColumns: 7,
            updateCaptionMode: 'none',
            i18nModulePath: 'io.ox/office/text/resource/tablestylenames'
        }, initOptions));

        // private methods ----------------------------------------------------

        function createMicroTable(minParaWidth, minParaHeight, cols, rows) {

            var paragraph = DOM.createParagraphNode().css({ minWidth: minParaWidth, minHeight: minParaHeight }),
                cellNode = DOM.createTableCellNode(paragraph, { plain: true }),
                tableRow = $('<tr>').append(cellNode),
                tableNode = $('<table>').append(tableRow);

            _(cols - 1).times(function () { tableRow.append(cellNode.clone()); });
            _(rows - 1).times(function () { tableNode.append(tableRow.clone()); });
            return tableNode;
        }

        /**
         * Initializes a new list item according to the table style sheet.
         */
        function createItemHandler(event, buttonNode, sectionId, styleId) {

            var // the preview table element
                tableNode = createMicroTable('8px', '4px', 5, 5);

            buttonNode.addClass('mini-caption').prepend(tableNode);
            updateTableFormatting(tableNode, styleId);
        }

        // initialization -----------------------------------------------------

        // register a handler that inserts a table element into each list item
        this.getMenu().on('create:item', createItemHandler);

    }}); // class TableStylePicker

    // exports ================================================================

    return TextControls;

});
