/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/selection',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/text/dom',
     'io.ox/office/text/position',
     'io.ox/office/text/drawingResize',
     'io.ox/office/drawinglayer/view/imageutil'
    ], function (Utils, KeyCodes, TriggerObject, BaseView, DrawingFrame, DOM, Position, DrawingResize, Image) {

    'use strict';

    var // whether the browser selection object supports the collapse() and expand() methods needed to create a backwards selection
        SELECTION_COLLAPSE_EXPAND_SUPPORT = (function () {
            var selection = window.getSelection();
            return _.isFunction(selection.collapse) && _.isFunction(selection.extend);
        }()),

        // if set to true, DOM selection will be logged to console
        LOG_SELECTION = false;

    // private static functions ===============================================

    /**
     * A jQuery selector for non-empty text spans and inline component
     * nodes.
     */
    function inlineNodeSelector() {
        return DOM.isInlineComponentNode(this) || (DOM.isTextSpan(this) && !DOM.isEmptySpan(this));
    }

    /**
     * Returns the first non-empty inline node of the specified paragraph (text
     * component nodes or non-empty text spans) if existing. Otherwise, if the
     * paragraph is empty, returns the trailing text span (following the
     * leading floating drawing objects).
     */
    function getFirstTextCursorNode(paragraph) {
        return Utils.findDescendantNode(paragraph, inlineNodeSelector, { children: true }) || DOM.findLastPortionSpan(paragraph);
    }

    /**
     * Returns the last non-empty inline node of the specified paragraph (text
     * component nodes or non-empty text spans) if existing. Otherwise, if the
     * paragraph is empty, returns the trailing text span (following the
     * leading floating drawing objects).
     */
    function getLastTextCursorNode(paragraph) {
        return Utils.findDescendantNode(paragraph, inlineNodeSelector, { children: true, reverse: true }) || DOM.findLastPortionSpan(paragraph);
    }

    // class Selection ========================================================

    /**
     * An instance of this class represents a selection in the edited document,
     * consisting of a logical start and end position representing a half-open
     * text range, or a rectangular table cell range.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     */
    function Selection(app, rootNode) {

        var // self reference
            self = this,

            // logical start position
            startPosition = [],

            // logical end position (half-open range)
            endPosition = [],

            // whether the current text range has been selected backwards
            backwards = false,

            // drawing node currently selected, as jQuery collection
            selectedDrawing = $(),

            // table currently selected, as jQuery object
            selectedTable = $(),

            // whether this selection represents a rectangular table cell range
            cellRangeSelected = false,

            // whether the cell range selection covers the entire table
            tableSelected = false,

            // the anchor of a cell range of a cell selection in a table
            anchorCellRange = null,

            // copy of drawing contents with browser selection, for copy&paste
            clipboardNode = null,

            // the original focus() method of the root node DOM element
            originalFocusMethod = rootNode[0].focus,

            // Performance: Saving paragraph and offset from insertText operations
            insertTextParagraph = null, insertTextOffset = 0,

            // Performance: Saving info from insertText operation to simplify scroll calculation
            insertTextPoint = null, isInsertTextOperation = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the current logical anchor position.
         */
        function getAnchorPosition() {
            return backwards ? endPosition : startPosition;
        }

        /**
         * Returns the current logical focus position.
         */
        function getFocusPosition() {
            return backwards ? startPosition : endPosition;
        }

        /**
         * Returns the first logical position in the document.
         */
        function getFirstPosition() {
            var firstParagraph = Utils.findDescendantNode(rootNode, DOM.PARAGRAPH_NODE_SELECTOR);
            return Position.getOxoPosition(rootNode, firstParagraph, 0).concat([0]);
        }

        /**
         * Returns the last logical position in the document.
         */
        function getLastPosition() {
            var lastSpan = Utils.findDescendantNode(rootNode, function () { return DOM.isPortionSpan(this); }, { reverse: true });
            return Position.getOxoPosition(rootNode, lastSpan, lastSpan.firstChild.nodeValue.length);
        }

        /**
         * Returns the first logical text cursor position in the document
         * (skips leading floating drawing objects in the first paragraph).
         */
        function getFirstTextPosition() {
            var firstInlineNode = getFirstTextCursorNode(Utils.findDescendantNode(rootNode, DOM.PARAGRAPH_NODE_SELECTOR));
            return Position.getOxoPosition(rootNode, firstInlineNode, 0);
        }

        /**
         * Clears the internal clipboard node.
         */
        function clearClipboardNode() {
            app.destroyImageNodes(clipboardNode);
            clipboardNode.empty().data('source-node', null);
        }

        /**
         * Checking whether a position is a possible text position.
         * This is a weak, but very fast check.
         * Another check is available in 'this.isValidTextSelection()',
         * that checks, whether the startPosition and the endPosition
         * contain valid positions.
         *
         * @param {Number[]} position
         *  The logical position to be checked.
         *
         * @returns {Boolean}
         *  Whether the given selection is a possible text selection.
         */
        function isPossibleTextSelection(position) {
            // checking length or checking if the shortened position is a paragraph position.
            return ((position.length - 2) % 3) === 0;  // -> fast solution
            // var localPos = _.clone(position);
            // localPos.pop();
            // return (Position.getParagraphElement(rootNode, localPos) !== null);
        }

        /**
         * Returns an array of DOM ranges representing the current browser
         * selection inside the passed container node.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node containing the returned selection ranges. Ranges not
         *  contained in this node will not be included into the result.
         *
         * @returns {Object}
         *  An object that contains a property 'active' with a DOM.Range object
         *  containing the current anchor point in its 'start' property, and
         *  the focus point in its 'end' property. The focus point may precede
         *  the anchor point if selecting backwards with mouse or keyboard. The
         *  returned object contains another property 'ranges' which is an
         *  array of DOM.Range objects representing all ranges currently
         *  selected. Start and end points of these ranges are already adjusted
         *  so that each start point precedes the end point.
         */
        function getBrowserSelection(containerNode) {

            var // the browser selection
                selection = window.getSelection(),
                // the result object
                result = { active: null, ranges: [] },
                // a single range object
                range = null,
                // the limiting points for valid ranges
                containerRange = DOM.Range.createRangeForNode(containerNode);

            // creates a DOM.Range object
            function createRange(startNode, startOffset, endNode, endOffset) {

                var // the range to be pushed into the array
                    range = DOM.Range.createRange(startNode, startOffset, endNode, endOffset),
                    // check that the nodes are inside the root node (with adjusted clone of the range)
                    adjustedRange = range.clone().adjust();

                return ((DOM.Point.comparePoints(containerRange.start, adjustedRange.start) <= 0) && (DOM.Point.comparePoints(adjustedRange.end, containerRange.end) <= 0)) ? range : null;
            }

            if (LOG_SELECTION) { Utils.info('Selection.getBrowserSelection(): reading browser selection...'); }

            // get anchor range which preserves direction of selection (focus node
            // may be located before anchor node)
            if (selection.rangeCount >= 1) {
                result.active = createRange(selection.anchorNode, selection.anchorOffset, selection.focusNode, selection.focusOffset);
                if (LOG_SELECTION) { Utils.log('  anchor=' + result.active); }
            }

            // read all selection ranges
            for (var index = 0; index < selection.rangeCount; index += 1) {
                // get the native selection Range object
                range = selection.getRangeAt(index);
                // translate to the internal text range representation
                range = createRange(range.startContainer, range.startOffset, range.endContainer, range.endOffset);
                // push, if range is valid
                if (range) {
                    if (LOG_SELECTION) { Utils.log('  ' + result.ranges.length + '=' + range); }
                    result.ranges.push(range.adjust());
                }
            }

            return result;
        }

        /**
         * Sets the browser selection to the passed DOM ranges.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node that must contain the passed selection ranges. This
         *  node must be focusable, and will be focused after the browser
         *  selection has been set (except if specified otherwise, see options
         *  below).
         *
         * @param {DOM.Range[]|DOM.Range} ranges
         *  The DOM ranges representing the new browser selection. May be an
         *  array of DOM range objects, or a single DOM range object.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         */
        function setBrowserSelection(containerNode, ranges, options) {

            var // the browser selection
                selection = window.getSelection(),
                // whether to restore old focus element (always restore, if application is not active)
                restoreFocus = !app.getView().isVisible() || Utils.getBooleanOption(options, 'preserveFocus', false),
                // the current focus element
                focusNode = $(window.document.activeElement);

            if (LOG_SELECTION) { Utils.info('Selection.setBrowserSelection(): writing browser selection...'); }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            // Performance: This is very expensive and should only be executed, if
            // it is really necessary. Setting text selection after insertOperations
            // should not execute this code.
            if ((_.browser.Firefox) && (! Utils.getBooleanOption(options, 'simpleTextSelection', false))) { $(containerNode).focus(); }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            BaseView.clearBrowserSelection();

            // convert to array
            ranges = _.getArray(ranges);

            // single range: use attributes of the Selection object (anchor/focus)
            // directly to preserve direction of selection when selecting backwards
            if (SELECTION_COLLAPSE_EXPAND_SUPPORT && (ranges.length === 1) && !$(ranges[0].start.node).is('tr')) {
                if (LOG_SELECTION) { Utils.log('  0=' + ranges[0]); }
                try {
                    selection.collapse(ranges[0].start.node, ranges[0].start.offset);
                    selection.extend(ranges[0].end.node, ranges[0].end.offset);
                    ranges = [];
                } catch (ex) {
                    if (!(_.browser.Firefox && $(ranges[0].start.node).is('div.clipboard'))) {  // Fix for 26645, no warning required in Firefox
                        Utils.warn('Selection.setBrowserSelection(): failed to collapse/expand to range: ' + ranges[0]);
                    }
                    // retry with regular code below
                    selection.removeAllRanges();
                }
            }

            // create a multi-selection
            _(ranges).each(function (range, index) {

                var docRange = null;

                if (LOG_SELECTION) { Utils.log('  ' + index + '=' + range); }
                try {
                    range.adjust();  // 26574, backward selection
                    docRange = window.document.createRange();
                    docRange.setStart(range.start.node, range.start.offset);
                    docRange.setEnd(range.end.node, range.end.offset);
                    selection.addRange(docRange);
                } catch (ex) {
                    Utils.error('Selection.setBrowserSelection(): failed to add range to selection: ' + range);
                }
            });

            // restore the old focus element if specified
            if (restoreFocus && !focusNode.is(containerNode)) {
                focusNode.focus();
            }
        }

        /**
         * Updates the contents of the clipboard node, according to the current
         * selection.
         */
        function updateClipboardNode() {

            // inserts a clone of the passed image node into the clipboard
            function copyImageNode(imgNode) {

                var // the source URL of the image
                    sourceUrl = null;

                // insert the image clone into the clipboard node
                imgNode = $(imgNode).clone();
                clipboardNode.append(imgNode);

                // IE needs attributes for width/height instead of styles
                imgNode.attr({ width: imgNode.width(), height: imgNode.height() });

                if (DOM.isDocumentImageNode(imgNode)) {

                    // additional attributes to check for copy&paste inside the same editor instance
                    sourceUrl = DOM.getUrlParamFromImageNode(selectedDrawing, 'get_filename');
                    imgNode.attr('alt', JSON.stringify({
                        altsrc: sourceUrl,
                        sessionId: DOM.getUrlParamFromImageNode(selectedDrawing, 'session'),
                        fileId: DOM.getUrlParamFromImageNode(selectedDrawing, 'id')
                    }));

                    // replace source URL with Base64 data URL, doesn't work correctly on IE
                    if (!_.browser.IE) {
                        imgNode.attr('src', DOM.getBase64FromImageNode(imgNode, Image.getMimeTypeFromImageUri(sourceUrl)));
                    }
                }
            }

            // prevent multiple clones of the same drawing frame
            if ((selectedDrawing.length > 0) && selectedDrawing.is(clipboardNode.data('source-node'))) { return; }

            // clear clipboard node, register drawing node at clipboard node
            clearClipboardNode();
            clipboardNode.toggle(selectedDrawing.length > 0).data('source-node', selectedDrawing[0]);

            // clone contained image nodes
            selectedDrawing.find('img').each(function () { copyImageNode(this); });
        }

        /**
         * Initializes this selection with the passed start and end points, and
         * validates the browser selection by moving the start and end points
         * to editable nodes.
         *
         * @param {Object} browserSelection
         *  The new browser selection descriptor. See the method
         *  Selection.getBrowserSelection() for details.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *  @param {Object} [options.event]
         *      The original browser event that has caused the changed
         *      selection.
         *
         *  @param {Object} [newPositions]
         *   An object with a property 'start' and optionally a property 'end',
         *   that describes the logical position of the selection range. If this
         *   optional object is defined, it must be used for the simplified process
         *   of applying the browser selection. This simplified process avoids the
         *   recalculation of logical positions via Position.getTextLevelOxoPosition.
         *
         */
        function applyBrowserSelection(browserSelection, options, newPositions) {

            var // active range from selection, preserving direction
                anchorPoint = null,
                focusPoint = null,
                // adjusted points (start before end)
                startPoint = null, endPoint = null,
                // whether position is a single cursor
                isCursor = false,
                // table containing the selection
                tableNode = null,
                // selected drawing node
                nodeInfo = null,
                // the browser event passed to this method
                browserEvent = Utils.getOption(options, 'event'),
                // browser event from a touch device
                browserTouchEvent = Modernizr.touch && _.isObject(browserEvent),
                // old selection, used to detect changed selection
                oldStartPosition = _.clone(startPosition),
                oldEndPosition = _.clone(endPosition),
                // whether the browser selection shall be restored
                restoreSelection = Utils.getBooleanOption(options, 'restoreSelection', true),
                // whether this function was triggered by an insert operation, for example insertText
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                // whether this function was triggered by an split paragraph operation
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                // whether this function was triggered by an 'simplified text selection' operation, for example insertText, backspace, delete or return
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // whether the simplified process can be used to set the new text selection
                simplifiedProcess = !!newPositions,
                // whether the 'change' event must be triggered -> informing remote users about the current selection, even if it was not modified (31832)
                forceTrigger = Utils.getBooleanOption(options, 'forceTrigger', false),
                // whether an existing cell selection shall be restored
                keepCellSelection = Utils.getBooleanOption(options, 'keepCellSelection', false);

            if (simplifiedProcess) {

                // check for cell range selection (must be in the same table)
                cellRangeSelected =  false;
                tableSelected = false;

                if (!_.isArray(newPositions.start)) {
                    Utils.error('Selection.applyBrowserSelection(): no start position set in newPositions: ' + JSON.stringify(newPositions));
                }

                startPosition = _.clone(newPositions.start);
                endPosition = newPositions.end ? _.clone(newPositions.end) : _.clone(newPositions.start);

            } else if (keepCellSelection) {

                // simplified behavior for modifying position on remote clients -> this must not destroy local cell selection
                cellRangeSelected = true;

            } else {

                anchorPoint = browserSelection.active.start;
                focusPoint = browserSelection.active.end;

                // check for cell range selection (must be in the same table)
                cellRangeSelected = ($(anchorPoint.node).is('tr') && $(focusPoint.node).is('tr') && (anchorPoint.node.parentNode === focusPoint.node.parentNode));
                if (cellRangeSelected) {

                    // cell range selection is always ordered, no need to check for direction
                    backwards = false;

                    // convert multi-selection for cells to rectangular cell selection
                    startPoint = _(browserSelection.ranges).first().start;
                    endPoint = _(browserSelection.ranges).last().end;

                    // entire table selected, if number of cell range objects in selection is equal to number of table cells
                    tableNode = Utils.findClosest(rootNode, focusPoint.node, DOM.TABLE_NODE_SELECTOR);
                    tableSelected = browserSelection.ranges.length === DOM.getTableCells(tableNode).length;

                } else {

                    // get range direction (check for real range, DOM.Point.comparePoints() is expensive)
                    isCursor = browserSelection.active.isCollapsed();

                    // Fix for 28344 in combination with special Firefox behavior for ranges ending directly behind list labels (see Position.getTextNodeFromCurrentNode)
                    if ((anchorPoint.node === focusPoint.node) && (anchorPoint.offset === focusPoint.offset - 1) && (DOM.isParagraphNode(anchorPoint.node)) && (DOM.isListLabelNode(anchorPoint.node.childNodes[anchorPoint.offset]))) {
                        // removing listlabels from selection
                        anchorPoint.offset = 1;
                        isCursor = true;
                    }

                    backwards = !isCursor && (DOM.Point.comparePoints(anchorPoint, focusPoint) > 0);
                    tableSelected = false;

                    // adjust start and end position
                    startPoint = backwards ? focusPoint : anchorPoint;
                    endPoint = backwards ? anchorPoint : focusPoint;
                }

                // calculate start and end position (always text positions, also in cell range mode)
                startPosition = Position.getTextLevelOxoPosition(startPoint, rootNode, false);
                endPosition = isCursor ? _.clone(startPosition) : Position.getTextLevelOxoPosition(endPoint, rootNode, true, !isCursor);

                if (!_.isArray(startPosition)) {
                    Utils.error('Selection.applyBrowserSelection(): no position for node ' + startPoint);
                }
                if (!_.isArray(endPosition)) {
                    Utils.error('Selection.applyBrowserSelection(): no position for node ' + endPoint);
                }
                if (!_.isArray(startPosition) || !_.isArray(endPosition) || !isPossibleTextSelection(startPosition) || !isPossibleTextSelection(endPosition)) {
                    startPosition = getFirstTextPosition();
                    endPosition = _.clone(startPosition);
                    isCursor = true;
                }
            }

            // check for drawing selection
            if (selectedDrawing.length > 0) {
                DrawingFrame.clearSelection(selectedDrawing);
                selectedDrawing = $();
            }

            if (!simplifiedProcess && !cellRangeSelected && self.isSingleComponentSelection()) {
                nodeInfo = Position.getDOMPosition(rootNode, startPosition, true);
                if (nodeInfo && DrawingFrame.isDrawingFrame(nodeInfo.node)) {
                    selectedDrawing = $(nodeInfo.node);
                    DrawingResize.drawDrawingSelection(app, selectedDrawing);
                }
            }

            // update table selection
            // Performance: NOT required for insertText, insertTab, splitParagraph, ..., but might be necessary for backspace, delete and return
            if (! insertOperation && ! splitOperation) {
                selectedTable.removeClass('selected');
                selectedTable = $(self.getEnclosingTable()).addClass('selected');

                // update the clipboard (clone drawing contents, otherwise clear)
                // Performance: NOT required for insertText
                updateClipboardNode();
            }

            // draw correct browser selection (but not if browser does not
            // support backward selection mode, or on touch devices to keep the
            // text selection menu open after a double tap event)
            if (restoreSelection && !browserTouchEvent && (isCursor || !backwards || SELECTION_COLLAPSE_EXPAND_SUPPORT)) {
                options = Utils.extendOptions(options, { isCursor: isCursor });
                self.restoreBrowserSelection(options);
            }

            // notify listeners
            if ((!_.isEqual(startPosition, oldStartPosition) || !_.isEqual(endPosition, oldEndPosition)) || (forceTrigger)) {
                self.trigger('change', { insertOperation: insertOperation, splitOperation: splitOperation, simpleTextSelection: simpleTextSelection });
            }
        }

        /**
         * Changes the current text position or selection by one character or
         * inline component.
         *
         * @param {Object} [options]
         *  A map with options controlling the operation. The following options
         *  are supported:
         *  @param {Boolean} [options.extend=false]
         *      If set to true, the current selection will be extended at the
         *      current focus point. Otherwise, the text cursor will be moved
         *      starting from the current focus point.
         *  @param {Boolean} [options.backwards=false]
         *      If set to true, the selection will be moved back by one
         *      character; otherwise the selection will be moved ahead.
         *  @param {Boolean} [options.verticalCellSelection=false]
         *      If set to true, the selection will be converted from a
         *      text selection to a vertical cell selection. This can be used
         *      in Firefox only, where multi-cell selection is possible.
         */
        function moveTextCursor(options) {

            var // whether to extend the selection
                extend = Utils.getBooleanOption(options, 'extend', false),
                // whether to move cursor back
                backwards = Utils.getBooleanOption(options, 'backwards', false),
                // whether the selection has to be converted into a vertical cell selection
                verticalCellSelection = Utils.getBooleanOption(options, 'verticalCellSelection', false),
                // text node at the current focus position
                focusNodeInfo = Position.getDOMPosition(rootNode, getFocusPosition()),
                // text node at the current anchor position (changes with focus node without SHIFT key)
                anchorNodeInfo = extend ? Position.getDOMPosition(rootNode, getAnchorPosition()) : focusNodeInfo,
                // the text node at the anchor position
                anchorTextNode = anchorNodeInfo && anchorNodeInfo.node && (anchorNodeInfo.node.nodeType === 3) ? anchorNodeInfo.node : null,
                // the text node at the focus position
                focusTextNode = focusNodeInfo && focusNodeInfo.node && (focusNodeInfo.node.nodeType === 3) ? focusNodeInfo.node : null,
                // space for other nodes
                node = null;

            // find the closest following inline node of the passed node
            function findNextNonEmptyInlineNode(node) {
                return Utils.findNextSiblingNode(node, inlineNodeSelector);
            }

            // find the closest preceding inline node of the passed node
            function findPreviousNonEmptyInlineNode(node) {
                return Utils.findPreviousSiblingNode(node, inlineNodeSelector);
            }

            // find the first inline node or empty node of the following paragraph
            function findFirstInlineNodeInNextParagraph(node) {
                var paragraph = Utils.findNextNode(rootNode, node, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR);
                return paragraph && getFirstTextCursorNode(paragraph);
            }

            // find the last inline node or empty node of the preceding paragraph
            function findLastInlineNodeInPreviousParagraph(node) {
                var paragraph = Utils.findClosest(rootNode, node, DOM.PARAGRAPH_NODE_SELECTOR);
                paragraph = paragraph && Utils.findPreviousNode(rootNode, paragraph, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR);
                return paragraph && getLastTextCursorNode(paragraph);
            }

            // move to start of text span; or to end of text span preceding the inline component
            function jumpBeforeInlineNode(node) {
                if (DOM.isTextSpan(node)) {
                    // text span: jump to its beginning
                    focusNodeInfo.node = node.firstChild;
                    focusNodeInfo.offset = 0;
                } else if (DOM.isTextSpan(node.previousSibling)) {
                    // jump to end of the span preceding the inline component
                    focusNodeInfo.node = node.previousSibling.firstChild;
                    focusNodeInfo.offset = focusNodeInfo.node.nodeValue.length;
                } else {
                    Utils.warn('Selection.moveTextCursor.jumpBeforeInlineNode(): missing text span preceding a component node');
                }
            }

            // skip inline component; move to end or to specific offset of text span
            function jumpOverInlineNode(node, offset) {
                if (DOM.isTextSpan(node)) {
                    // text span: jump to passed offset, or to its end
                    focusNodeInfo.node = node.firstChild;
                    focusNodeInfo.offset = _.isNumber(offset) ? offset : focusNodeInfo.node.nodeValue.length;
                } else if (DOM.isTextSpan(node.nextSibling)) {
                    // jump to beginning of the span following the inline component
                    // (may be an empty span before floating node)
                    focusNodeInfo.node = node.nextSibling.firstChild;
                    focusNodeInfo.offset = 0;
                } else {
                    Utils.warn('Selection.moveTextCursor.jumpOverInlineNode(): missing text span following a component node');
                }
            }

            // setting a cell selection instead of a text selection
            function switchToCellSelectionHorz() {

                var // an array of ranges for cell selections
                    ranges = [],
                    // the previous or following row node required for cell selections
                    siblingRow = null,
                    // a temporarily required node
                    helperNode = null;

                // try to find the nearest table cell containing the text node
                focusNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);
                anchorNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);

                if (backwards) {
                    // selecting the cell and its previous sibling, if available (the previous sibling has to be the active cell)
                    focusNodeInfo.offset += 1;
                    if (anchorNodeInfo.offset > 0) {

                        helperNode = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);
                        anchorNodeInfo.offset -= 1;

                        ranges.push({ start: anchorNodeInfo, end: helperNode });
                        ranges.push({ start: helperNode, end: focusNodeInfo });

                        self.setAnchorCellRange(ranges[1]);

                    } else {
                        // the complete row and the previous row need to be selected
                        if ($(focusNodeInfo.node).prev().length > 0) {
                            siblingRow = $(focusNodeInfo.node).prev();
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, 0), new DOM.Point(siblingRow, siblingRow.children().length)));
                            ranges.push(new DOM.Range(new DOM.Point(focusNodeInfo.node, 0), new DOM.Point(focusNodeInfo.node, $(focusNodeInfo.node).children().length)));
                            self.setAnchorCellRange(new DOM.Range(ranges[1].start, new DOM.Point(ranges[1].start.node, ranges[1].start.offset + 1)));
                        } else {
                            // Fix for 29402, not changing selection
                            if (self.isTextLevelSelection()) {
                                self.setTextSelection(self.getEndPosition(), Position.getFirstPositionInCurrentCell(rootNode, self.getEndPosition()));
                            }
                        }
                    }
                } else {
                    // selecting the cell and its following sibling, if available
                    if (focusNodeInfo.offset + 1 < $(focusNodeInfo.node).children().length) {
                        focusNodeInfo.offset += 2;  // selecting the current and the following cell
                        ranges.push({ start: anchorNodeInfo, end: focusNodeInfo });
                        self.setAnchorCellRange(new DOM.Range(anchorNodeInfo, new DOM.Point(anchorNodeInfo.node, anchorNodeInfo.offset + 1)));
                    } else {
                        // the complete row and the next row need to be selected
                        if ($(focusNodeInfo.node).next().length > 0) {
                            siblingRow = $(focusNodeInfo.node).next();
                            ranges.push(new DOM.Range(new DOM.Point(focusNodeInfo.node, 0), new DOM.Point(focusNodeInfo.node, $(focusNodeInfo.node).children().length)));
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, 0), new DOM.Point(siblingRow, siblingRow.children().length)));
                            self.setAnchorCellRange(new DOM.Range(new DOM.Point(ranges[0].end.node, ranges[0].end.offset - 1), ranges[0].end.node));
                        }
                    }
                }

                self.setCellSelection(ranges);
            }

            // setting a cell selection instead of a text selection
            function switchToCellSelectionVert() {

                var // an array of ranges for cell selections
                    ranges = [],
                    // the previous or following row node required for cell selections
                    siblingRow = null;

                // try to find the nearest table cell containing the text node
                if ($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR).length === 0) {
                    // Fix for 29402, not changing selection
                    if (self.isTextLevelSelection()) {
                        if (backwards) {
                            self.setTextSelection(Position.getFirstPositionInCurrentCell(rootNode, self.getEndPosition()), self.getEndPosition());
                        } else {
                            self.setTextSelection(self.getStartPosition(), Position.getLastPositionInCurrentCell(rootNode, self.getStartPosition()));
                        }
                    }
                } else {

                    focusNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);

                    if (backwards) {
                        // selecting the cell and its previous sibling in the row above, if available (the previous sibling has to be the active cell)
                        if ($(focusNodeInfo.node).prev().length > 0) {
                            siblingRow = $(focusNodeInfo.node).prev();
                            while (siblingRow.hasClass('pb-row') && siblingRow.prev().length > 0) { siblingRow = siblingRow.prev(); } // skip over page break row
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, focusNodeInfo.offset), new DOM.Point(siblingRow, focusNodeInfo.offset + 1)));
                            ranges.push(new DOM.Range(focusNodeInfo, new DOM.Point(focusNodeInfo.node, focusNodeInfo.offset + 1)));
                            self.setAnchorCellRange(ranges[ranges.length - 1]);
                        }
                    } else {
                        if ($(focusNodeInfo.node).next().length > 0) {
                            siblingRow = $(focusNodeInfo.node).next();
                            while (siblingRow.hasClass('pb-row') && siblingRow.next().length > 0) { siblingRow = siblingRow.next(); } // skip over page break row
                            ranges.push(new DOM.Range(focusNodeInfo, new DOM.Point(focusNodeInfo.node, focusNodeInfo.offset + 1)));
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, focusNodeInfo.offset), new DOM.Point(siblingRow, focusNodeInfo.offset + 1)));
                            self.setAnchorCellRange(ranges[0]);
                        }
                    }

                    self.setCellSelection(ranges);
                }
            }

            // check anchor and focus position
            if (!anchorTextNode || !focusTextNode) {
                Utils.warn('Selection.moveTextCursor(): missing valid text position');
                return false;
            }

            // update focusNodeInfo according to the passed direction
            if (backwards) {

                if ((verticalCellSelection) && (! cellRangeSelected) && Position.isSameTableLevel(rootNode, self.getFocusPosition(), self.getAnchorPosition())) {
                    switchToCellSelectionVert();
                    return true;
                }

                // move back inside non-empty text span, but not always to the beginning
                else if (focusNodeInfo.offset > 1) {
                    focusNodeInfo.offset -= 1;
                }

                // try to find the previous non-empty inline node in the own paragraph
                else if ((node = findPreviousNonEmptyInlineNode(focusTextNode.parentNode))) {

                    // offset is 1, or preceding node is a text span: move cursor behind the previous inline node
                    if ((focusNodeInfo.offset === 1) || DOM.isTextSpan(node)) {
                        if ($(focusTextNode.parentNode).prev().hasClass('page-break') && focusNodeInfo.offset === 0) {
                            // cursor is on first position before page break inside paragraph,
                            // and it has to jump to last position on previous page
                            jumpOverInlineNode(node, node.firstChild.nodeValue.length - 1); //TODO: special chars cannot be enumerated with text length of node
                        } else {
                            jumpOverInlineNode(node);
                        }
                    // offset is 0, skip the previous inline component (jump to end of its preceding inline node)
                    } else {
                        // jump to end of preceding text span
                        jumpOverInlineNode(node.previousSibling);

                        // try to find the correct trailing text span of the pre-preceding inline component
                        // (this may jump from the preceding empty span over floating nodes to the trailing
                        // text span of the previous inline component)
                        if (DOM.isEmptySpan(node.previousSibling) && (node = findPreviousNonEmptyInlineNode(node))) {
                            // skipping an inline node: jump to end of preceding text span (try to
                            // find text span following the next preceding component)
                            jumpOverInlineNode(node);
                        }
                    }
                }

                // after first character in paragraph: go to beginning of the first text span
                else if (focusNodeInfo.offset === 1) {
                    focusNodeInfo.offset = 0;
                }

                // in Mozilla based browsers a cell selection can be created inside tables
                else if (extend && _.browser.Firefox && !cellRangeSelected &&
                        Position.isPositionInTable(rootNode, getFocusPosition()) &&
                        Position.isFirstPositionInTableCell(rootNode, getFocusPosition()) &&
                        Position.isSameTableLevel(rootNode, getFocusPosition(), self.getAnchorPosition())) {
                    switchToCellSelectionHorz();
                    return true;
                }

                // try to find the last text position in the preceding paragraph
                else if ((node = findLastInlineNodeInPreviousParagraph(focusTextNode.parentNode))) {
                    jumpOverInlineNode(node);
                }

            } else {

                if ((verticalCellSelection) && (! cellRangeSelected) && Position.isSameTableLevel(rootNode, self.getAnchorPosition(), self.getFocusPosition())) {
                    switchToCellSelectionVert();
                    return true;
                }

                // move ahead inside non-empty text span (always up to the end of the span)
                else if (focusNodeInfo.offset < focusTextNode.nodeValue.length) {
                    focusNodeInfo.offset += 1;
                }

                // try to find the next non-empty inline node in the own paragraph
                else if ((node = findNextNonEmptyInlineNode(focusTextNode.parentNode))) {
                    // skip only the first character of following non-empty text span
                    jumpOverInlineNode(node, 1);
                }

                // in Mozilla based browsers a cell selection can be created inside tables
                else if (extend && _.browser.Firefox && !cellRangeSelected &&
                        Position.isPositionInTable(rootNode, getFocusPosition()) &&
                        Position.isLastPositionInTableCell(rootNode, getFocusPosition()) &&
                        Position.isSameTableLevel(rootNode, self.getAnchorPosition(), getFocusPosition())) {
                    switchToCellSelectionHorz();
                    return true;
                }

                // try to find the first text position in the next paragraph
                else if ((node = findFirstInlineNodeInNextParagraph(focusTextNode.parentNode))) {
                    jumpBeforeInlineNode(node);
                }
            }

            // update browser selection if focusNodeInfo still valid
            if (focusNodeInfo.node) {
                applyBrowserSelection({ active: new DOM.Range(anchorNodeInfo, focusNodeInfo), ranges: [] });
            }
            return true;
        }

        /**
         * Converts the passed logical text position to a DOM point pointing to
         * a DOM text node as used by the internal browser selection.
         *
         * @param {Number[]} position
         *  The logical position of the target node. Must be the position of a
         *  paragraph child node, either a text span, a text component (fields,
         *  tabs), or a drawing node.
         *
         * @param {Object} [point]
         *  Performance: The point containing node and offset as calculated
         *  from Position.getDOMPosition. Saving the object, that was calculated
         *  before, saves the call of Position.getDOMPosition.
         *
         * @returns {DOM.Point|Null}
         *  The DOM.Point object representing the passed logical position, or
         *  null, if the passed position is invalid.
         */
        function getPointForTextPosition(position, point) {

            var nodeInfo = point ? point : Position.getDOMPosition(rootNode, position, true),
                node = nodeInfo ? nodeInfo.node : null;

            if (point && node && node.nodeType === 3) {
                node = node.parentNode;  // saved node could be text node instead of text span
            }

            // check that the position selects a paragraph child node
            if (!node || !DOM.isParagraphNode(node.parentNode)) {
                return null;
            }

            if (DOM.isTextSpan(node)) {
                return new DOM.Point(node.firstChild, nodeInfo.offset);
            }

            node = Utils.findPreviousSiblingNode(node, function () { return DOM.isTextSpan(this); });
            if (DOM.isTextSpan(node)) {
                return new DOM.Point(node.firstChild, node.firstChild.nodeValue.length);
            }

            return null;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this selection contains a valid start and end
         * position.
         */
        this.isValid = function () {
            return (startPosition) && (endPosition) && (startPosition.length > 0) && (endPosition.length > 0);
        };

        /**
         * Returns the current logical start position. The start position is
         * always located before the end position (or, in case of a text
         * cursor, is equal to the end position), regardless of the direction
         * of the selection. To receive positions dependent on the direction,
         * see the methods Selection.getAnchorPosition() and
         * Selection.getFocusPosition().
         *
         * @returns {Number[]}
         *  The logical start position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getStartPosition = function () {
            return _.clone(startPosition);
        };

        /**
         * Returns the current logical end position. The end position is always
         * located behind the start position (or, in case of a text cursor,
         * is equal to the start position), regardless of the direction of the
         * selection. To receive positions dependent on the direction, see the
         * methods Selection.getAnchorPosition() and
         * Selection.getFocusPosition().
         *
         * @returns {Number[]}
         *  The logical end position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getEndPosition = function () {
            return _.clone(endPosition);
        };

        /**
         * Returns the current logical anchor position. The anchor position is
         * the position where the selection of a text range (by mouse or cursor
         * keys) has been started. The anchor position will be located after
         * the focus position if selecting backwards. To receive positions
         * independent from the direction, see the methods
         * Selection.getStartPosition() and Selection.getEndPosition().
         *
         * @returns {Number[]}
         *  The logical anchor position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getAnchorPosition = function () {
            return _.clone(getAnchorPosition());
        };

        /**
         * Returns the current logical focus position. The focus position is
         * the position that changes while modifying the selection of a range
         * (by mouse or cursor keys). The focus position will be located before
         * the anchor position if selecting backwards. To receive positions
         * independent from the direction, see the methods
         * Selection.getStartPosition() and Selection.getEndPosition().
         *
         * @returns {Number[]}
         *  The logical focus position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getFocusPosition = function () {
            return _.clone(getFocusPosition());
        };

        /**
         * Returns whether this selection represents a simple text cursor.
         *
         * @returns {Boolean}
         *  True, if this selection represents a simple text cursor.
         */
        this.isTextCursor = function () {
            return !cellRangeSelected && _.isEqual(startPosition, endPosition);
        };

        /**
         * Returns whether this selection represents a range that covers some
         * document contents. The result is the exact opposite of the method
         * Selection.isTextCursor().
         *
         * @returns {Boolean}
         *  True, if this selection represents a range in the document.
         */
        this.hasRange = function () {
            return !this.isTextCursor();
        };

        /**
         * Returns whether this selection has been created while selecting the
         * document contents backwards (by cursor keys or by mouse).
         *
         * @returns {Boolean}
         *  True, if the selection has been created backwards.
         */
        this.isBackwards = function () {
            return backwards;
        };

        /**
         * Returns the type of this selection as string.
         *
         * @returns {String}
         *  Returns 'text' for a text range or text cursor, or 'cell' for a
         *  rectangular cell range in a table, or 'drawing' for a drawing
         *  selection.
         */
        this.getSelectionType = function () {
            return cellRangeSelected ? 'cell' : (selectedDrawing.length > 0) ? 'drawing' : 'text';
        };

        /**
         * Returns whether the current selection is selection inside a paragraph.
         * So currently the selection type must be 'text' or 'drawing'.
         *
         * @returns {Boolean}
         *  Whether the current selection type is 'text' or 'drawing'.
         */
        this.isTextLevelSelection = function () {
            return !cellRangeSelected;
        };

        /**
         * Checking, if the current values for startPosition and endPosition
         * represent a valid text position. It can happen, that these values
         * point to a non existing position ( -> task 28520).
         * Returns whether the current selection is a valid text selection.
         *
         * @returns {Boolean}
         *   Whether the current selection is a valid text selection.
         */
        this.isValidTextSelection = function () {

            var startPos = _.clone(startPosition),
                textPos = startPos.pop(),
                paragraph =  Position.getParagraphElement(rootNode, startPos),
                endPos = null;

            if ((paragraph === null) || (Position.getParagraphNodeLength(paragraph) < textPos)) {
                return false;
            }

            if (!_.isEqual(startPosition, endPosition)) {
                endPos = _.clone(endPosition);
                textPos = endPos.pop();
                paragraph =  Position.getParagraphElement(rootNode, endPos);

                if ((paragraph === null) || (Position.getParagraphNodeLength(paragraph) < textPos)) {
                    return false;
                }
            }

            return true;
        };

        /**
         * Returns whether the start and end position of this selection are
         * located in the same parent component (all array elements but the
         * last are equal).
         *
         * @param {Number} [parentLevel=1]
         *  The number of parent levels. If omitted, the direct parents of the
         *  start and end position will be checked (only the last element of
         *  the position array will be ignored). Otherwise, the specified
         *  number of trailing array elements will be ignored (for example, a
         *  value of 2 checks the grand parents).
         *
         * @returns {Boolean}
         *  Whether the start and end position are located in the same parent
         *  component.
         */
        this.hasSameParentComponent = function (parentLevel) {
            return Position.hasSameParentComponent(startPosition, endPosition, parentLevel);
        };

        /**
         * Returns whether this selection covers exactly one component.
         *
         * @returns {Boolean}
         *  Returns whether the selection is covering a single component. The
         *  start and end position must refer to the same parent component, and
         *  the last array element of the end position must be the last array
         *  element of the start position increased by the value 1.
         */
        this.isSingleComponentSelection = function () {
            return this.hasSameParentComponent() && (_.last(startPosition) === _.last(endPosition) - 1);
        };

        /**
         * Returns the logical position of the closest common component
         * containing all nodes covered by this selection (the leading array
         * elements that are equal in the start and end position arrays).
         *
         * @returns {Number[]}
         *  The logical position of the closest common component containing
         *  this selection. May be the empty array if the positions already
         *  differ in their first element.
         */
        this.getClosestCommonPosition = function () {

            var index = 0, length = Math.min(startPosition.length, endPosition.length);

            while ((index < length) && (startPosition[index] === endPosition[index])) {
                index += 1;
            }

            return startPosition.slice(0, index);
        };

        /**
         * Returns the closest paragraph that contains all nodes of this
         * selection completely.
         *
         * @returns {HTMLElement|Null}
         *  The closest paragraph containing this selection; or null, if the
         *  selection is not contained in a single paragraph.
         */
        this.getEnclosingParagraph = function () {

            var // position of closest common parent component containing the selection
                commonPosition = this.getClosestCommonPosition();

            // the closest paragraph containing the common parent component
            return (commonPosition.length > 0) ? Position.getCurrentParagraph(rootNode, commonPosition) : null;
        };

        /**
         * Returns the closest table that contains all nodes of this selection
         * completely.
         *
         * @returns {HTMLTableElement|Null}
         *  The closest table containing this selection; or null, if the
         *  selection is not contained in a single table.
         */
        this.getEnclosingTable = function () {

            var // position of closest common parent component containing the selection
                commonPosition = this.getClosestCommonPosition();

            // the closest table containing the common parent component
            return (commonPosition.length > 0) ? Position.getCurrentTable(rootNode, commonPosition) : null;
        };

        /**
         * Provides the text contents from the selection.
         *
         * @returns {String|Null}
         *  The text of the current selection or null if no text is available.
         */
        this.getSelectedText = function () {
            var text = null;

            this.iterateNodes(function (node, pos, start, length) {
                if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                    var nodeText = $(node).text();
                    if (nodeText) {
                        text = text || '';
                        text = text.concat(nodeText.slice(start, start + length));
                    }
                }
            });

            return text;
        };

        /**
         * Returns an object describing the table cell range that is currently
         * selected.
         *
         * @returns {Object|Null}
         *  If this selection is contained completely inside a table, returns
         *  an object containing the following attributes:
         *  - {HTMLTableElement} tableNode: the table element containing the
         *      selection,
         *  - {Number[]} tablePosition: the logical position of the table,
         *  - {Number[2]} firstCellPosition: the logical position of the first
         *      cell, relative to the table (contains exactly two elements:
         *      row, column),
         *  - {Number[2]} lastCellPosition: the logical position of the last
         *      cell, relative to the table (contains exactly two elements:
         *      row, column),
         *  - {Number} width: the number of columns covered by the cell range,
         *  - {Number} height: the number of rows covered by the cell range.
         *  Otherwise, this method returns null.
         */
        this.getSelectedCellRange = function () {

            var // the result object containing all info about the cell range
                result = { tableNode: this.getEnclosingTable() };

            if (!result.tableNode) {
                return null;
            }

            // logical position of the table
            result.tablePosition = Position.getOxoPosition(rootNode, result.tableNode, 0);

            // convert selection positions to cell positions relative to table
            if ((startPosition.length < result.tablePosition.length + 2) || (endPosition.length < result.tablePosition.length + 2)) {
                Utils.error('Selection.getSelectedCellRange(): invalid start or end position');
                return null;
            }
            result.firstCellPosition = startPosition.slice(result.tablePosition.length, result.tablePosition.length + 2);
            result.lastCellPosition = endPosition.slice(result.tablePosition.length, result.tablePosition.length + 2);

            // width and height of the range for convenience
            result.width = result.lastCellPosition[1] - result.firstCellPosition[1] + 1;
            result.height = result.lastCellPosition[0] - result.firstCellPosition[0] + 1;

            return result;
        };

        /**
         * Returns the anchor cell range of a cell selection.
         *
         * @returns {DOM.Range}
         *  A DOM range describing the anchor cell of a cell selection,
         *  or null, if not set.
         */
        this.getAnchorCellRange = function () {
            return anchorCellRange;
        };

        /**
         * Returns the anchor cell range of a cell selection.
         *
         * @param {DOM.Range} range
         *  The DOM range, that describes the position of the anchor
         *  cell in a cell selection.
         */
        this.setAnchorCellRange = function (range) {
            anchorCellRange = range;
        };

        /**
         * Returns the drawing node currently selected.
         *
         * @returns {jQuery}
         *  A jQuery collection containing the currently selected drawing, if
         *  existing; otherwise an empty jQuery collection.
         */
        this.getSelectedDrawing = function () {
            return selectedDrawing;
        };

        /**
         * Setting the drawing node currently selected.
         *
         * @param {jQuery} drawing
         *  A jQuery collection containing the currently selected drawing.
         */
        this.setSelectedDrawing = function (drawing) {
            selectedDrawing = drawing;
        };

        /**
         * Returns the DOM clipboard node used to store a copy of the contents
         * of a selected drawing frame for copy&paste.
         *
         * @returns {jQuery}
         *  The clipboard node, as jQuery object.
         */
        this.getClipboardNode = function () {
            return clipboardNode;
        };

        /**
         * Returns the bounding rectangle of the focus position relative to the
         * entire document page, if this selection is of type 'text'.
         */
        this.getFocusPositionBoundRect = function () {

            var // the boundaries of the cursor
                boundRect = null,
                // the DOM point of the focus position
                focusPoint = null,
                // the DOM Range object
                docRange = null;

            if (this.getSelectionType() === 'text') {

                // Build DOM Range object from calculated focus point instead
                // of the settings in the core browser selection object (the
                // focusNode and focusOffset attributes). Needed because Chrome
                // reports wrong DOM nodes if cursor is located in an empty
                // paragraph (the browser selection object sometimes returns
                // the last text node of the preceding paragraph instead of the
                // empty text node of the current paragraph).
                // Performance: Special handling for insertText operation, where
                // the node was already saved before
                if (isInsertTextOperation && insertTextPoint) {
                    focusPoint = getPointForTextPosition(null, insertTextPoint);
                } else {
                    focusPoint = getPointForTextPosition(getFocusPosition());
                }

                if (focusPoint) {

                    // If the cursor points into an empty text span, calculate its
                    // position directly instead of using the getClientRects()
                    // method of the DOM Range object. This is needed because
                    // Chrome has problems calculating the bounding rectangle of an
                    // empty text node.
                    // Zoomed documents in Firefox needs this correction also, but only
                    // if the zoom factor is not 100.
                    // Allowing this for zoom factor 100 in Firefox leads to task 32397
                    if ((focusPoint.node.nodeValue.length === 0) || (_.browser.Firefox && app.getView().getZoomFactor() !== 100)) {
                        boundRect = Utils.getNodePositionInPage(focusPoint.node.parentNode);
                    } else {
                        try {
                            // Build a DOM range object needed to get the actual
                            // position of the focus position between two characters.
                            docRange = window.document.createRange();
                            docRange.setStart(focusPoint.node, focusPoint.offset);
                            docRange.setEnd(focusPoint.node, focusPoint.offset);
                            boundRect = docRange.getClientRects()[0];
                        } catch (e) {
                            Utils.error('Selection.getFocusPositionBoundRect(): failed to initialize focus range');
                        }
                    }
                }
            }

            return boundRect;
        };

        // low-level browser selection ----------------------------------------

        /**
         * Returns an array of DOM ranges representing the current browser
         * selection inside the editor root node.
         *
         * @returns {Object}
         *  An object that contains a property 'active' with a DOM.Range object
         *  containing the current anchor point in its 'start' property, and
         *  the focus point in its 'end' property. The focus point may precede
         *  the anchor point if selecting backwards with mouse or keyboard. The
         *  returned object contains another property 'ranges' which is an
         *  array of DOM.Range objects representing all ranges currently
         *  selected. Start and end points of these ranges are already adjusted
         *  so that each start point precedes the end point.
         */
        this.getBrowserSelection = function () {
            return getBrowserSelection(rootNode);
        };

        /**
         * Returns the text covered by the current browser selection as plain
         * text string.
         *
         * @returns {String}
         *  The text covered by the current browser selection.
         */
        this.getTextFromBrowserSelection = function () {
            return window.getSelection().toString();
        };

        /**
         * Returns the currently selected contents as HTML mark-up.
         *
         * @returns {String}
         *  The selected content, as HTML mark-up string.
         */
        this.getHtmlFromBrowserSelection = function () {

            var // the browser selection
                selection = window.getSelection(),
                // the result container
                resultNode = $('<div>');

            // read all selection ranges
            for (var index = 0; index < selection.rangeCount; index += 1) {
                resultNode.append(selection.getRangeAt(index).cloneContents());
            }
            resultNode.find('tr.pb-row').remove(); //filter rows that contains page breaks - they should not be copied

            return resultNode.html();
        };

        /**
         * Sets the browser selection to the passed DOM ranges in the editor
         * root node.
         *
         * @param {DOM.Range[]|DOM.Range} ranges
         *  The DOM ranges representing the new browser selection. May be an
         *  array of DOM range objects, or a single DOM range object. Must be
         *  contained in the editor root node.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setBrowserSelection = function (ranges, options) {

            var // save custom implementation of the focus() method
                customFocusMethod = rootNode[0].focus;

            rootNode[0].focus = originalFocusMethod;
            setBrowserSelection(rootNode, ranges, options);
            rootNode[0].focus = customFocusMethod;
            return this;
        };

        /**
         * Sets the browser selection to the contents of the passed DOM element
         * node.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node whose contents will be completely selected. This node
         *  must be focusable, and will be focused after the browser selection
         *  has been set (except if specified otherwise, see options below).
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setBrowserSelectionToContents = function (containerNode, options) {
            containerNode = Utils.getDomNode(containerNode);
            setBrowserSelection(containerNode, DOM.Range.createRange(containerNode, 0, containerNode, containerNode.childNodes.length), options);
            return this;
        };

        /**
         * Performance: Saving paragraph and final number of the logical position during
         * insertText operation. This values can be reused during cursor setting after
         * insertText operation. This reduces Position.getDOMPosition to the final
         * value of the logical position.
         *
         * @param {HTMLElement|jQuery|} paragraphNode
         *  The paragraph element as DOM node or jQuery object.
         *
         * @param {Number} offset
         *  The offset inside the paragraph node.
         */
        this.setInsertTextInfo = function (paragraphNode, offset) {
            insertTextParagraph = paragraphNode;
            insertTextOffset = offset;
        };

        /**
         * Performance: Returning paragraph and final number of the logical position during
         * insertText or splitParagraph operation. This values can be reused during cursor
         * setting after insertText operation. This reduces Position.getDOMPosition to the
         * final value of the logical position.
         *
         * @returns {Object}
         *  A DOM.Point object containing the node and offset properties. The node is the
         *  most inner paragraph, so that this paragraph together with the offset describes
         *  the complete cursor position and simplifies the usage of Postion.getDOMPosition
         *  to the last value.
         */
        this.getInsertTextInfo = function () {
            return new DOM.Point(insertTextParagraph, insertTextOffset);
        };

        /**
         * Performance: Saving point calculated from Position.getDOMPosition for
         * insertText and splitParagraph operations.
         *
         * @param {Object} point
         *  The point calculated from Position.getDOMPosition, containing the
         *  html node element and the offset.
         */
        this.setInsertTextPoint = function (point) {
            insertTextPoint = point;
        };

        /**
         * Performance: Getting point calculated from Position.getDOMPosition for
         * insertText and splitParagraph operations.
         *
         * @returns {Object}
         *  The point calculated from Position.getDOMPosition, containing the
         *  html node element and the offset.
         */
        this.getInsertTextPoint = function () {
            return insertTextPoint;
        };

        /**
         * Performance: Register, whether the current operation is an insertText
         * operation. In this case the function 'getFocusPositionBoundRect' can use
         * the saved DOM in 'insertTextPoint'.
         */
        this.registerInsertText = function (options) {
            isInsertTextOperation = Utils.getBooleanOption(options, 'insertOperation', false);
        };

        // selection manipulation ---------------------------------------------

        /**
         * Restores the browser selection according to the current logical
         * selection represented by this instance.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.restoreBrowserSelection = function (options) {

            var // the DOM ranges representing the logical selection
                ranges = [],
                // start and end DOM point for text selection
                startPoint = null, endPoint = null,
                // the DOM element currently focused
                focusNode = $(window.document.activeElement),
                // Performance: whether this is an insertText operation
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                // Performance: whether this is an insertText operation
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                // Performance: whether this is a 'simpleTextSelection' operation (backspace, delete, return, insertText)
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // Performance: whether this is a cursor selection
                isCursor = Utils.getBooleanOption(options, 'isCursor', false),
                // logical text position for temporary usage
                localStartPos;

            // Do not preserve focus, if it is inside root node or clipboard
            // (focus may switch nodes depending on new selection type).
            // Bug 28195: IE contains table cells that are content-editable on
            // their own and therefore are focused instead of the root node.
            // Another workaround: IE sometimes focuses the hidden root node if
            // drawing objects are deselected although this node is not focusable.
            if ((focusNode.length === 0) || focusNode.is(rootNode) || Utils.containsNode(rootNode, focusNode) || focusNode.is(clipboardNode) || focusNode.is(app.getView().getHiddenRootNode())) {
                options = Utils.extendOptions(options, { preserveFocus: false });
            }

            switch (this.getSelectionType()) {

            // text selection: select text range
            case 'text':
                // Performance: Simple process for cursor selections, calling Position.getDOMPosition only once
                if ((insertOperation || splitOperation) && insertTextParagraph && _.isNumber(insertTextOffset)) {
                    startPoint = Position.getDOMPosition(insertTextParagraph, [insertTextOffset]);  // Performance: Using saved paragraph node with last oxo position value -> very fast getDOMPosition
                    self.setInsertTextPoint(startPoint); // Performance: Save for later usage
                } else {
                    startPoint = Position.getDOMPosition(rootNode, startPosition);
                    // Fix for 32002, skipping floated drawings at beginning of document, webkit only, only complete selections (Ctrl-A)
                    if (_.browser.WebKit && !isCursor && _.isEqual(startPosition, getFirstPosition()) && _.isEqual(endPosition, getLastPosition()) &&
                        DOM.isEmptySpan(startPoint.node.parentNode) &&
                        startPoint.node.parentNode.nextSibling &&
                        (DOM.isFloatingDrawingNode(startPoint.node.parentNode.nextSibling) || DOM.isOffsetNode(startPoint.node.parentNode.nextSibling))) {
                        localStartPos = Position.skipFloatedDrawings(rootNode, startPosition); // Skipping floated drawings without modifying the original start position (32002)
                        startPoint = Position.getDOMPosition(rootNode, localStartPos);
                    }
                }
                if (startPoint && (simpleTextSelection || isCursor)) {
                    ranges.push(new DOM.Range(startPoint, startPoint));
                } else {
                    endPoint = Position.getDOMPosition(rootNode, endPosition);
                    if (startPoint && endPoint) {
                        ranges.push(new DOM.Range(backwards ? endPoint : startPoint, backwards ? startPoint : endPoint));
                    } else {
                        Utils.error('Selection.restoreBrowserSelection(): missing text selection range');
                    }
                }
                break;

            // cell selection: iterate all cells
            case 'cell':
                this.iterateTableCells(function (cell) {
                    ranges.push(DOM.Range.createRangeForNode(cell));
                });
                break;

            // drawing selection: select the clipboard node
            case 'drawing':
                break;

            default:
                Utils.error('Selection.restoreBrowserSelection(): unknown selection type');
            }

            // Set the new selection ranges. Bug 26283: when a drawing frame is
            // selected, do not focus root node to prevent scroll flickering.
            if (ranges.length > 0) {
                self.setBrowserSelection(ranges, options);
            } else {
                self.setBrowserSelectionToContents(clipboardNode, options);
            }
            return this;
        };

        /**
         * Processes a browser event that will change the current selection.
         * Supported are 'mousedown' events and 'keydown' events originating
         * from cursor navigation keys and 'selectionchange' events.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after the
         *  browser has processed the passed event, and this selection instance
         *  has updated itself according to the new browser selection.
         */
        this.processBrowserEvent = function (event, options) {

            var // deferred return value
                def = $.Deferred(),
                // whether event is a keydown event
                keyDownEvent = event.type === 'keydown',
                // whether event is a keydown event without modifier keys except SHIFT
                simpleKeyDownEvent = keyDownEvent && !event.ctrlKey && !event.altKey && !event.metaKey,
                // whether event is a Left/Right cursor key
                leftRightCursor = (event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW),
                // whether event is a Left/Right cursor key
                upDownCursor = (event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW),
                // original selection, for post-processing
                originalSelection = { start: this.getStartPosition(), end: this.getEndPosition() },
                // whether the document is in read-only mode
                readOnly = Utils.getBooleanOption(options, 'readonly', false);

            // skipping shrinked implicit paragraphs in read-only mode (28563)
            function skipImplicitShrinkedParagraph() {

                // Checking, if the new position is not inside an implicit paragraph with height 0
                var para = Position.getParagraphElement(rootNode, _.initial(startPosition)),
                    eventTriggered = false;

                if (para && DOM.isImplicitParagraphNode(para) && $(para).height() === 0) {
                    // simply skip this paragraph -> triggering this event again
                    $(rootNode).trigger(event);
                    eventTriggered = true;
                }

                return eventTriggered;
            }

            // handle simple left/right cursor keys (with and without SHIFT) manually
            if (simpleKeyDownEvent && leftRightCursor) {
                // do not move selection manually with SHIFT, if there is no support for backward selection
                if (!event.shiftKey || SELECTION_COLLAPSE_EXPAND_SUPPORT) {
                    if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                        def.resolve();
                    } else {
                        def.reject();
                    }

                    if (readOnly && skipImplicitShrinkedParagraph()) {
                        def.resolve();
                        return;
                    }

                    event.preventDefault();
                    return def.promise();
                }

            }

            // Setting cursor to end of document for webkit browsers (32007)
            if (_.browser.WebKit && event.ctrlKey && !event.shiftKey && (event.keyCode === 35)) {
                self.setTextSelection(getLastPosition());
                def.resolve();
                return def.promise();
            }

            // browser needs to process pending events before its selection can be queried
            app.executeDelayed(function () {

                var // the current browser selection
                    browserSelection = self.getBrowserSelection(),
                    // a temporarily required start position
                    start,
                    // a temporarily required information about a position containing node and offset
                    nodeInfo,
                    // whether the browser selection shall be restored
                    restoreSelection = true,
                    // a current cell node in which the event occurred
                    currentCell,
                    // a current cell node in which the event occurred
                    currentCellContentNode,
                    // IE: A helper logical position
                    tempPosition = null;

                // helper functions for cursor positioning in Firefox in tables with page break
                function findNextPosInTd(node, pos) {
                    if (!node.hasClass('pb-row')) {
                        node = node.parents('.pb-row');
                    }
                    if (node.next().find('td').eq(pos).length > 0) {
                        return node.next().find('td').eq(pos).find('div.p').first();
                    } else {
                        return node.next().find('td').last().find('div.p').first();
                    }
                }
                function findPrevPosInTd(node, pos) {
                    if (!node.hasClass('pb-row')) {
                        node = node.parents('.pb-row');
                    }
                    if (node.prev().find('td').eq(pos).length > 0) {
                        return node.prev().find('td').eq(pos).find('div.p').last();
                    } else {
                        return node.prev().find('td').last().find('div.p').last();
                    }
                }

                if (browserSelection.active) {
                    // do not modify selection in IE (< 11) when starting to select with mouse
                    if (!((Modernizr.touch || _.browser.IE) && ((event.type === 'mousedown') || (event.type === 'touchstart')))) {

                        // Fix for 28222: Avoid calling restoreBrowserSelection for Ctrl-Shift-CursorLeft in Chrome and Firefox
                        if (event.shiftKey && event.ctrlKey && event.keyCode === KeyCodes.LEFT_ARROW && SELECTION_COLLAPSE_EXPAND_SUPPORT) {
                            restoreSelection = false;
                        }

                        // Firefox - table on two pages cursor traversal
                        if (($(browserSelection.active.start.node).is('td .page-break, .pb-row td, .pb-row') || $(browserSelection.active.end.node).is('td .page-break, .pb-row td, .pb-row'))) {
                            var
                                modifiedBrowserSel = _.clone(browserSelection),
                                pos = originalSelection.start[2],
                                modifiedBrowserSelActStartNode = modifiedBrowserSel.active.start.node,
                                modifiedBrowserSelActEndNode = modifiedBrowserSel.active.end.node,
                                index; // holds position of row with page breaks inside ranges array

                            if (event.keyCode === KeyCodes.DOWN_ARROW) {
                                if ($(browserSelection.active.start.node).is('td .page-break, .pb-row td')) {
                                    modifiedBrowserSelActStartNode = findNextPosInTd($(browserSelection.active.start.node), pos);
                                    while (modifiedBrowserSelActStartNode.is('.page-break, .clear-floating')) {
                                        modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.next();
                                    }
                                    modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                                }
                                if ($(browserSelection.active.end.node).is('td .page-break, .pb-row td')) {
                                    modifiedBrowserSelActEndNode = findNextPosInTd($(browserSelection.active.end.node), pos);
                                    while (modifiedBrowserSelActEndNode.is('.page-break, .clear-floating')) {
                                        modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.next();
                                    }
                                    modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                                }
                            } else {
                                if ($(browserSelection.active.start.node).is('td .page-break, .pb-row td')) {
                                    modifiedBrowserSelActStartNode = findPrevPosInTd($(browserSelection.active.start.node), pos);
                                    while (modifiedBrowserSelActStartNode.is('.page-break, .clear-floating')) {
                                        modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.prev();
                                    }
                                    modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                                }
                                if ($(browserSelection.active.end.node).is('td .page-break, .pb-row td')) {
                                    modifiedBrowserSelActEndNode = findPrevPosInTd($(browserSelection.active.end.node), pos);
                                    while (modifiedBrowserSelActEndNode.is('.page-break, .clear-floating')) {
                                        modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.prev();
                                    }
                                    modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                                }

                                if ($(browserSelection.active.start.node).hasClass('pb-row')) { // we dont know direction (to move to next or previous node), so we will get it from ranges
                                    _.each(browserSelection.ranges, function (range, i) {
                                        if ($(range.start.node).is('tr.pb-row')) {
                                            index = i;
                                        }
                                    });
                                    if (browserSelection.ranges[index + 1]) {
                                        modifiedBrowserSel.active.start.node = browserSelection.ranges[index + 1].start.node;
                                        modifiedBrowserSel.active.start.offset = browserSelection.ranges[index + 1].start.offset;
                                    } else {
                                        modifiedBrowserSel.active.start.node = browserSelection.ranges[index - 1].start.node;
                                        modifiedBrowserSel.active.start.offset = browserSelection.ranges[index - 1].start.offset;
                                    }
                                }
                                if ($(browserSelection.active.end.node).hasClass('pb-row')) { // we dont know direction (to move to next or previous node), so we will get it from ranges
                                    if (_.isUndefined(index)) {
                                        _.each(browserSelection.ranges, function (range, i) {
                                            if ($(range.end.node).is('tr.pb-row')) {
                                                index = i;
                                            }
                                        });
                                    }
                                    if (browserSelection.ranges[index + 1]) {
                                        modifiedBrowserSel.active.end.node = browserSelection.ranges[index + 1].end.node;
                                        modifiedBrowserSel.active.end.offset = browserSelection.ranges[index + 1].end.offset;
                                    } else {
                                        modifiedBrowserSel.active.end.node = browserSelection.ranges[index - 1].end.node;
                                        modifiedBrowserSel.active.end.offset = browserSelection.ranges[index - 1].end.offset;
                                    }
                                }
                                if (!_.isUndefined(index)) { // remove range that contains page break row
                                    modifiedBrowserSel.ranges.splice(index, 1);
                                }
                            }
                            // calculates the new logical selection
                            applyBrowserSelection(modifiedBrowserSel, { event: event, restoreSelection: restoreSelection });

                        } else if (($(browserSelection.active.start.node).hasClass('page-break') || $(browserSelection.active.end.node).hasClass('page-break'))) {
                            var modifiedBrowserSel = _.clone(browserSelection),
                                modifiedBrowserSelActStartNode = modifiedBrowserSel.active.start.node,
                                modifiedBrowserSelActEndNode = modifiedBrowserSel.active.end.node;

                            if (event.keyCode === KeyCodes.DOWN_ARROW) {
                                modifiedBrowserSelActStartNode = $(browserSelection.active.start.node).next();
                                while (modifiedBrowserSelActStartNode.is('.page-break, .clear-floating')) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.next();
                                }
                                if (modifiedBrowserSelActStartNode.is('table')) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.find('div.p').first();
                                }
                                modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                                modifiedBrowserSelActEndNode = $(browserSelection.active.end.node).next();
                                while (modifiedBrowserSelActEndNode.is('.page-break, .clear-floating')) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.next();
                                }
                                if (modifiedBrowserSelActEndNode.is('table')) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.find('div.p').first();
                                }
                                modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                            } else {
                                modifiedBrowserSelActStartNode = $(browserSelection.active.start.node).prev();
                                while (modifiedBrowserSelActStartNode.is('.page-break, .clear-floating')) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.prev();
                                }
                                if (modifiedBrowserSelActStartNode.hasClass('page-break')) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.parent();
                                }
                                if (modifiedBrowserSelActStartNode.is('table')) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.find('div.p').last();
                                }
                                modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                                modifiedBrowserSelActEndNode = $(browserSelection.active.end.node).prev();
                                while (modifiedBrowserSelActEndNode.is('.page-break, .clear-floating')) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.prev();
                                }
                                if (modifiedBrowserSelActEndNode.hasClass('page-break')) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.parent();
                                }
                                if (modifiedBrowserSelActEndNode.is('table')) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.find('div.p').last();
                                }
                                modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                            }
                            // calculates the new logical selection
                            applyBrowserSelection(modifiedBrowserSel, { event: event, restoreSelection: restoreSelection });
                        } else {
                            // calculates the new logical selection
                            applyBrowserSelection(browserSelection, { event: event, restoreSelection: restoreSelection });
                        }

                        // Only for IE: Expanding selection after a double click or triple click (Fix for 29751)
                        if ((self.getSelectionType() === 'text') && (self.isTextCursor())) {
                            if (event.type === 'dblclick') {
                                self.setTextSelection(Position.getWordBoundary(rootNode, self.getStartPosition(), true), Position.getWordBoundary(rootNode, self.getStartPosition(), false));
                            } else if (event.type === 'tripleclick') {
                                self.setTextSelection(Position.appendNewIndex(_.initial(self.getStartPosition()), 0), Position.appendNewIndex(_.initial(self.getStartPosition()), Position.getParagraphLength(rootNode, self.getStartPosition())));
                            }
                        }

                        // Fixing the position after pressing the 'End' key for MS IE and Webkit (26863) and Firefox (28309).
                        // The selection has to be reduced by 1 (if it is not the end of the paragraph) -> task 26863 and 28309
                        if ((event.keyCode === KeyCodes.END) && (! event.shiftKey)) {
                            start = self.getStartPosition();
                            nodeInfo = Position.getDOMPosition(rootNode, start);

                            if ($(nodeInfo.node).text().charCodeAt(nodeInfo.offset - 1) === 32) {
                                if (start[start.length - 1] < Position.getParagraphLength(rootNode, start)) {
                                    start[start.length - 1] -= 1;
                                    self.setTextSelection(start);
                                }
                            } else if ((_.browser.Firefox) && (nodeInfo.offset === 0) && (nodeInfo.node.parentNode && nodeInfo.node.parentNode.previousSibling && $(nodeInfo.node.parentNode.previousSibling).is(DOM.HARDBREAK_NODE_SELECTOR))) {
                                // Fix for 28544
                                start[start.length - 1] -= 1;
                                self.setTextSelection(start);
                            }
                        }

                        // Fixing in MS IE the handling of shift + 'end', that selects until the end of the document, if pressed
                        // in the last line of a paragraph (Fix for 28218).
                        if ((event.keyCode === KeyCodes.END) && event.shiftKey && _.browser.IE) {
                            if (! Position.hasSameParentComponent(startPosition, endPosition)) {
                                self.setTextSelection(startPosition, Position.getLastPositionInParagraph(rootNode, _.clone(startPosition)));
                            }
                        }

                        // adjust selection, if key event did not have any effect
                        if (keyDownEvent && _.isEqual(originalSelection, { start: startPosition, end: endPosition })) {

                            // CTRL+Left/Right cursor: execute simple left/right step
                            if (leftRightCursor && event.ctrlKey && !event.altKey && !event.metaKey) {
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            }

                            // Setting cell selections in Mozilla, if the selection did not change the content is separated into different cells
                            if (_.browser.Firefox && upDownCursor && event.shiftKey && Position.isPositionInTable(rootNode, startPosition) && (! cellRangeSelected)) {
                                // simple Up/Down cursor: execute simple left/right step
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW, verticalCellSelection: true })) {
                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            } else if (upDownCursor && !event.ctrlKey && !event.altKey && !event.metaKey) {
                                // simple Up/Down cursor: execute simple left/right step
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW })) {
                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            }
                        } else {
                            // setting cell selections in Mozilla, if the content is separated into different cells
                            // Typically this is most often handled with 'verticalCellSelection' in the 'if', because
                            // the selection did not change. But sometimes it changes, and this can be recognized by
                            // the following call of moveTextCursor.
                            if (_.browser.Firefox && upDownCursor && event.shiftKey && Position.isPositionInTable(rootNode, startPosition) && (! cellRangeSelected)) {
                                // are start and end position in the same table cell?
                                if (! _.isEqual(Position.getLastPositionFromPositionByNodeName(rootNode, startPosition, DOM.TABLE_CELLNODE_SELECTOR), Position.getLastPositionFromPositionByNodeName(rootNode, endPosition, DOM.TABLE_CELLNODE_SELECTOR))) {
                                    if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW, verticalCellSelection: true })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                }
                            }

                            // setting selection in IE using shift and cursor-Right and jumping from one paragraph to the following paragraph
                            if (_.browser.IE && event.shiftKey) {
                                if (event.keyCode === KeyCodes.RIGHT_ARROW && Position.isLastPositionInParagraph(rootNode, endPosition)) {
                                    if (moveTextCursor({ extend: true, backwards: false, verticalCellSelection: false })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                } else if (event.keyCode === KeyCodes.LEFT_ARROW && startPosition[startPosition.length - 1] === 0) {
                                    if (moveTextCursor({ extend: true, backwards: true, verticalCellSelection: false })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                }
                            }
                        }

                        // Skipping not expanded implicit paragraphs in read-only mode during up or down cursor (left or right cursor was already handled before)
                        if (simpleKeyDownEvent && upDownCursor && readOnly && skipImplicitShrinkedParagraph()) {
                            def.resolve();
                            return;
                        }

                    }

                    // special IE behavior
                    if (_.browser.IE) {
                        if ((event.type === 'mousedown') && (selectedDrawing.length > 0)) {
                            // deselect drawing node in IE (mousedown event has been ignored, see above)
                            DrawingFrame.clearSelection(selectedDrawing);
                            selectedDrawing = $();
                            updateClipboardNode();
                        } else if ((event.type === 'mousedown') || (event.type === 'mouseup')) {
                            // handling clicks behind last paragraph in table cell node in IE 11 (27287, 29409)
                            currentCell = $(event.target).closest(DOM.TABLE_CELLNODE_SELECTOR);
                            currentCellContentNode = $(event.target).closest(DOM.CELLCONTENT_NODE_SELECTOR);
                            if ((currentCell.length > 0) && (currentCellContentNode.length === 0)  && (! DOM.isExceededSizeTableNode(currentCell.closest(DOM.TABLE_NODE_SELECTOR)))) {
                                // the mouse event happened behind the last paragraph in the cell. Therefore the new position has to be the last position inside the cell
                                self.setTextSelection(Position.getLastPositionInCurrentCell(rootNode, Position.getOxoPosition(rootNode, currentCell[0], 0)));
                            }

                            // Handling of selections over several cells in tables in IE (evaluation in mouseup event)
                            if ((event.type === 'mouseup') && (Position.isPositionInTable(rootNode, self.getStartPosition())) && DOM.isCellContentNode(event.target)) {

                                // using event.offsetX and event.offsetY to calculate, if the mouseup event happened in the same table cell
                                // -> if not, a simplified process for selections outside the cell is required
                                if ((event.offsetX < 0) || (event.offsetY < 0) || (event.offsetX > $(event.target.parentNode).width()) || (event.offsetY > $(event.target.parentNode).height())) {

                                    tempPosition = Position.getOxoPositionFromPixelPosition(rootNode, event.pageX, event.pageY, { backwards: backwards, textlevel: true });

                                    if (tempPosition && (tempPosition.length > 0)) {
                                        self.setTextSelection(backwards ? tempPosition : self.getStartPosition(), backwards ? self.getEndPosition() : tempPosition);
                                    }
                                }
                            }
                        }
                    }

                    def.resolve();

                } else if (selectedDrawing.length > 0) {

                    // IE destroys text selection in clipboard node immediately after
                    // the mouseup event (this cannot be prevented by returning false)
                    if (_.browser.IE && (event.type === 'mouseup') && (selectedDrawing.length > 0)) {
                        self.restoreBrowserSelection(event);
                    }

                    def.resolve();
                } else {
                    Utils.warn('Selection.processBrowserEvent(): missing valid browser selection');
                    def.reject();
                }

            });

            return def.promise();
        };

        /**
         * Applies the passed cell selection.
         *
         * @param {DOM.Range[]} ranges
         *  The DOM ranges of the cell selection.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setCellSelection = function (ranges) {
            if (ranges.length > 0) {
                // option 'active' required: point to any cell to be able (used to detect cell selection)
                applyBrowserSelection({ active: ranges[0], ranges: ranges });
            }
            return this;
        };

        /**
         * Selects the passed logical text range in the document.
         *
         * @param {Number[} newStartPosition
         *  The logical position of the first text component in the selection.
         *  Must be the position of a paragraph child node, either a text span,
         *  a text component (fields, tabs), or a drawing node.
         *
         * @param {Number[]} [newEndPosition]
         *  The logical position behind the last text component in the
         *  selection (half-open range). Must be the position of a paragraph
         *  child node, either a text span, a text component (fields, tabs), or
         *  a drawing node. If omitted, sets a text cursor according to the
         *  passed start position.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.backwards=false]
         *      If set to true, the selection will be created backwards. Note
         *      that not all browsers allow to create a reverse DOM browser
         *      selection. Nonetheless, the internal selection direction will
         *      always be set to backwards.
         *  @param {Boolean} [options.simpleTextSelection=false]
         *      If set to true, the selection will be set using newStartPosition
         *      and newEndPosition directly. The logical positions are not converted
         *      to a browserSelection and then calculated back in applyBrowserSelection
         *      using Position.getTextLevelOxoPosition. This is especially important
         *      for the very fast operations like 'insertText' (also used for 'Enter',
         *      'Delete' and 'Backspace').
         *  @param {Boolean} [options.insertOperation=false]
         *      If set to true, this setTextSelection is called from an insert operation,
         *      for example an insertText operation. This is important for performance
         *      reasons, because the cursor setting can be simplified.
         *  @param {Boolean} [options.splitOperation=false]
         *      If set to true, this setTextSelection is called from an 'Enter' operation.
         *      This is important for performance reasons, because the cursor setting can
         *      be simplified.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setTextSelection = function (newStartPosition, newEndPosition, options) {

            var // DOM points for start and end position
                startPoint = null, endPoint = null,
                // whether to create the selection backwards
                selectBackwards = false,
                // the range containing start and end point
                range = null,
                // Performance: Whether the simplified text selection can be used
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // whether the selection shall be updated after an external operation (then a 'change' must be triggered, task 31832)
                remoteSelectionUpdate = Utils.getBooleanOption(options, 'remoteSelectionUpdate', false),
                // whether this call was triggered by the userDataUpdateHandler
                userDataUpdateHandler = Utils.getBooleanOption(options, 'userDataUpdateHandler', false),
                // whether an existing cell selection must not be destroyed
                keepCellSelection = userDataUpdateHandler && self.getSelectionType() === 'cell' && _.isEqual(newStartPosition, this.getStartPosition()) && _.isEqual(newEndPosition, this.getEndPosition());

            if (simpleTextSelection) {
                applyBrowserSelection(null, { preserveFocus: true, insertOperation: Utils.getBooleanOption(options, 'insertOperation', false), splitOperation: Utils.getBooleanOption(options, 'splitOperation', false), simpleTextSelection: simpleTextSelection }, { start: newStartPosition, end: newEndPosition });
            } else {
                startPoint = _.isArray(newStartPosition) ? getPointForTextPosition(newStartPosition) : null;
                endPoint = _.isArray(newEndPosition) ? getPointForTextPosition(newEndPosition) : startPoint;
                if (startPoint && endPoint) {
                    selectBackwards = Utils.getBooleanOption(options, 'backwards', false);
                    range = new DOM.Range(selectBackwards ? endPoint : startPoint, selectBackwards ? startPoint : endPoint);
                    applyBrowserSelection({ active: range, ranges: [range] }, { preserveFocus: true, forceTrigger: remoteSelectionUpdate, keepCellSelection: keepCellSelection });
                } else {
                    if (remoteSelectionUpdate && _.isArray(options.browserStartPoint)) {
                        // using the selection given by the browser, if restoring the old text selection failed
                        this.setTextSelection(options.browserStartPoint);
                    } else {
                        Utils.warn('Selection.setTextSelection(): expecting text positions, start=' + JSON.stringify(newStartPosition) + ', end=' + JSON.stringify(newEndPosition));
                    }
                }
            }

            return this;
        };

        /**
         * Updating the selection of a remote read-only editor after executing an external operation. The new
         * selection created by the browser is used to calculate and set a new logical start and end position.
         * This cannot be used, if the selection before executing the external operation was a drawing selection
         * or a cell selection.
         */
        this.updateSelectionAfterBrowserSelection = function () {

            var // the current browser selection
                browserSelection = self.getBrowserSelection();

            // checking the current browser selection before using it (fix for 32182)
            if (browserSelection && browserSelection.active && browserSelection.active.start && browserSelection.active.end) {
                applyBrowserSelection(browserSelection, { preserveFocus: true, forceTrigger: true });
            } else {
                Utils.warn('Selection.updateSelectionAfterBrowserSelection(): incomplete browser selection, ignoring remote selection.');
            }
        };

        /**
         * Returns the last logical position in the document.
         *
         * @returns {Number[]}
         *  The last logical document position.
         */
        this.getLastDocumentPosition = function () {
            return getLastPosition();
        };

        /**
         * Sets the text cursor to the first available cursor position in the
         * document. Skips leading floating drawings in the first paragraph. If
         * the first content node is a table, selects its first available
         * cell paragraph (may be located in a sub table in the first outer
         * cell).
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectTopPosition = function () {
            return this.setTextSelection(getFirstTextPosition());
        };

        /**
         * Selects the entire document.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectAll = function () {
            // start from first position (including floating images)
            return this.setTextSelection(getFirstPosition(), getLastPosition());
        };

        /**
         * If this selection selects a drawing node, changes the browser
         * selection to a range that starts directly before that drawing node,
         * and ends directly after that drawing.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectDrawingAsText = function () {

            var // the selected drawing, as plain DOM node
                drawingNode = selectedDrawing[0],
                // whether the drawing is in inline mode
                inline = DOM.isInlineDrawingNode(drawingNode),
                // previous text span of the drawing node
                prevTextSpan = inline ? drawingNode.previousSibling : null,
                // next text span of the drawing node (skip following floating drawings)
                nextTextSpan = drawingNode ? Utils.findNextSiblingNode(drawingNode, function () { return DOM.isPortionSpan(this); }) : null,
                // DOM points representing the text selection over the drawing
                startPoint = null, endPoint = null;

            if (drawingNode) {

                // remove drawing selection boxes
                DrawingFrame.clearSelection(selectedDrawing);

                // start point after the last character preceding the drawing
                if (DOM.isPortionSpan(prevTextSpan)) {
                    startPoint = new DOM.Point(prevTextSpan.firstChild, prevTextSpan.firstChild.nodeValue.length);
                }
                // end point before the first character following the drawing
                if (DOM.isPortionSpan(nextTextSpan)) {
                    endPoint = new DOM.Point(nextTextSpan.firstChild, 0);
                }

                // set browser selection (do nothing if no start and no end point
                // have been found - but that should never happen)
                if (startPoint || endPoint) {
                    self.setBrowserSelection(backwards ?
                        new DOM.Range(endPoint || startPoint, startPoint || endPoint) :
                        new DOM.Range(startPoint || endPoint, endPoint || startPoint));
                }
            }

            return this;
        };

        // iterators ----------------------------------------------------------

        /**
         * Calls the passed iterator function for each table cell, if this
         * selection is located inside a table. Processes a rectangular cell
         * selection (if supported by the browser), otherwise a row-oriented
         * text selection inside a table.
         * This function might also be called, when the selection is not inside
         * a table. In this case it leaves immediately with Utils.BREAK. This
         * happens for example during loading a document (Fix for 29021).
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every table cell node
         *  covered by this selection. Receives the following parameters:
         *      (1) {HTMLTableCellElement} the visited DOM cell element,
         *      (2) {Number[]} its logical position (the last two elements in
         *          this array represent the row and column index of the cell),
         *      (3) {Number} the row offset, relative to the first row
         *          in the rectangular cell range,
         *      (4) {Number} the column offset, relative to the first column
         *          in the rectangular cell range.
         *  The last two parameters will be undefined, if the current selection
         *  is a text range in a table. If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTableCells = function (iterator, context) {

            var // information about the cell range and containing table
                cellRangeInfo = this.getSelectedCellRange(),

                // the DOM cells
                firstCellInfo = null, lastCellInfo = null,
                // current cell, and its logical position
                cellInfo = null, cellNode = null, cellPosition = null,

                // row/column indexes for loops
                row = 0, col = 0;

            // check enclosing table, get its position
            if (!cellRangeInfo) {
                return Utils.BREAK;
            }

            // resolve position to closest table cell
            firstCellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellRangeInfo.firstCellPosition, true);
            lastCellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellRangeInfo.lastCellPosition, true);
            if (!firstCellInfo || !$(firstCellInfo.node).is('td') || !lastCellInfo || !$(lastCellInfo.node).is('td')) {
                Utils.error('Selection.iterateTableCells(): no table cells found for cell positions');
                return Utils.BREAK;
            }

            // visit all cells for rectangular cell selection mode
            if (cellRangeSelected) {

                // loop over all cells in the range
                for (row = 0; row < cellRangeInfo.height; row += 1) {
                    for (col = 0; col < cellRangeInfo.width; col += 1) {

                        // cell position relative to table
                        cellPosition = [cellRangeInfo.firstCellPosition[0] + row, cellRangeInfo.firstCellPosition[1] + col];
                        cellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellPosition);

                        // cellInfo will be undefined, if current position is covered by a merged cell
                        if (cellInfo && $(cellInfo.node).is('td')) {
                            cellPosition = cellRangeInfo.tablePosition.concat(cellPosition);
                            if (iterator.call(context, cellInfo.node, cellPosition, row, col) === Utils.BREAK) {
                                return Utils.BREAK;
                            }
                        }
                    }
                }

            // otherwise: visit all cells row-by-row (text selection mode)
            } else {

                cellNode = firstCellInfo.node;
                while (cellNode) {

                    // visit current cell
                    cellPosition = Position.getOxoPosition(cellRangeInfo.tableNode, cellNode, 0);
                    row = cellPosition[0] - cellRangeInfo.firstCellPosition[0];
                    col = cellPosition[1] - cellRangeInfo.firstCellPosition[1];
                    cellPosition = cellRangeInfo.tablePosition.concat(cellPosition);
                    if (iterator.call(context, cellNode, cellPosition, row, col) === Utils.BREAK) { return Utils.BREAK; }

                    // last cell reached
                    if (cellNode === lastCellInfo.node) { return; }

                    // find next cell node (either next sibling, or first child
                    // of next row, but skip embedded tables)
                    cellNode = Utils.findNextNode(cellRangeInfo.tableNode, cellNode, 'td', DOM.TABLE_NODE_SELECTOR);
                }

                // in a valid DOM tree, there must always be valid cell nodes until
                // the last cell has been reached, so this point should never be reached
                Utils.error('Selection.iterateTableCells(): iteration exceeded end of selection');
                return Utils.BREAK;
            }
        };

        /**
         * Calls the passed iterator function for specific content nodes
         * (tables and paragraphs) selected by this selection instance. It is
         * possible to visit all paragraphs embedded in all covered tables and
         * nested tables, or to iterate on the 'shortest path' by visiting
         * tables exactly once if they are covered completely by the selection
         * range and skipping the embedded paragraphs and sub tables. If the
         * selection range end at the very beginning of a paragraph (before the
         * first character), this paragraph is not considered to be included in
         * the selected range.
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every content node
         *  (paragraphs and tables) covered by this selection. Receives the
         *  following parameters:
         *      (1) {HTMLElement} the visited content node,
         *      (2) {Number[]} its logical position,
         *      (3) {Number|Undefined} the logical index of the first text
         *          component covered by the FIRST paragraph; undefined for
         *          all other paragraphs and tables (may point after the last
         *          existing child text component, if the selection starts at
         *          the very end of a paragraph),
         *      (4) {Number|Undefined} the logical index of the last child text
         *          component covered by the LAST paragraph (closed range);
         *          undefined for all other paragraphs and tables,
         *      (5) {Boolean} whether the selection includes the beginning of
         *          the parent container of the current content node,
         *      (6) {Boolean} whether the selection includes the end of the
         *          parent container of the current content node.
         *  If the selection represents a text cursor, the start position will
         *  exceed the end position by 1. Thus, a text cursor in an empty
         *  paragraph will be represented by the text range [0, -1]. If the
         *  iterator returns the Utils.BREAK object, the iteration process will
         *  be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @param {Object} [options]
         *  A map of options to control the iteration. Supports the following
         *  options:
         *  @param {Boolean} [options.shortestPath=false]
         *      If set to true, tables that are covered completely by this
         *      selection will be visited, but their descendant components
         *      (paragraphs and embedded tables) will be skipped in the
         *      iteration process. By default, this method visits all
         *      paragraphs embedded in all tables and their sub tables, but
         *      does not visit the table objects. Has no effect for tables that
         *      contain the end paragraph, because these tables are not fully
         *      covered by the selection. Tables that contain the start
         *      paragraph will never be visited, because they start before the
         *      selection.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateContentNodes = function (iterator, context, options) {

            var // start node and offset (pass true to NOT resolve text spans to text nodes)
                startInfo = Position.getDOMPosition(rootNode, startPosition, true),
                // end node and offset (pass true to NOT resolve text spans to text nodes)
                endInfo = Position.getDOMPosition(rootNode, endPosition, true),

                // whether to iterate on shortest path (do not traverse into completely covered tables)
                shortestPath = Utils.getBooleanOption(options, 'shortestPath', false),

                // paragraph nodes containing the passed start and end positions
                firstParagraph = null, lastParagraph = null,
                // current content node while iterating
                contentNode = null, parentCovered = false;

            // visit the passed content node (paragraph or table); or table child nodes, if not in shortest-path mode
            function visitContentNode(contentNode, parentCovered) {

                var // each call of the iterator get its own position array (iterator is allowed to modify it)
                    position = Position.getOxoPosition(rootNode, contentNode),
                    // start text offset in first paragraph
                    startOffset = (contentNode === firstParagraph) ? _.last(startPosition) : undefined,
                    // end text offset in last paragraph (convert half-open range to closed range)
                    endOffset = (contentNode === lastParagraph) ? (_.last(endPosition) - 1) : undefined;

                // If the endPosition is an empty paragraph in a cell range selection, the attribute
                // must also be assigned to this end position. But in this case is the endOffset '-1' in the
                // last cell. Concerning to task 27361 the attribute is not assigned to the empty paragraph
                // in the bottom right cell of a table. To fix this, it is in this scenario necessary to
                // increase the endOffset to the value '0'! (Example selection: [0,0,0,0,0] to [0,1,1,0,0])
                // On the other hand, it is important that a selection that ends at the beginning of a
                // paragraph, ignores the attribute assignment for the following paragraph. Example:
                // Selection [0,0] to [1,0]: Paragraph alignment 'center' must not be assigned to
                // paragraph [1].

                if ((cellRangeSelected) && (endOffset < 0)) { endOffset = 0; }  // Task 27361

                // visit the content node, but not the last paragraph, if selection
                // does not start in that paragraph and ends before its beginning
                // (otherwise, it's a cursor in an empty paragraph)
                if ((contentNode === firstParagraph) || (contentNode !== lastParagraph) || (endOffset >= Position.getFirstTextNodePositionInParagraph(contentNode))) {
                    return iterator.call(context, contentNode, position, startOffset, endOffset, parentCovered);
                }
            }

            // find the first content node in passed root node (either table or embedded paragraph depending on shortest-path option)
            function findFirstContentNode(rootNode) {

                // in shortest-path mode, use first table or paragraph in cell,
                // otherwise find first paragraph which may be embedded in a sub table)
                return Utils.findDescendantNode(rootNode, shortestPath ? DOM.CONTENT_NODE_SELECTOR : DOM.PARAGRAPH_NODE_SELECTOR);
            }

            // find the next content node in DOM tree (either table or embedded paragraph depending on shortest-path option)
            function findNextContentNode(rootNode, contentNode, lastParagraph) {

                // find next content node in DOM tree (searches in own siblings, AND in other nodes
                // following the parent node, e.g. the next table cell, or paragraphs following the
                // containing table, etc.; but skips drawing nodes that may contain their own paragraphs)
                contentNode = Utils.findNextNode(rootNode, contentNode, DOM.CONTENT_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR);

                // iterate into a table, if shortest-path option is off, or the end paragraph is inside the table
                while (DOM.isTableNode(contentNode) && (!shortestPath || (lastParagraph && Utils.containsNode(contentNode, lastParagraph)))) {
                    contentNode = Utils.findDescendantNode(contentNode, DOM.CONTENT_NODE_SELECTOR);
                }

                return contentNode;
            }

            // check validity of passed positions
            if (!startInfo || !startInfo.node || !endInfo || !endInfo.node) {
                Utils.error('Selection.iterateContentNodes(): invalid selection, cannot find first or last DOM node');
                return Utils.BREAK;
            }

            // find first and last paragraph node (also in table cell selection mode)
            firstParagraph = Utils.findClosest(rootNode, startInfo.node, DOM.PARAGRAPH_NODE_SELECTOR);
            lastParagraph = Utils.findClosest(rootNode, endInfo.node, DOM.PARAGRAPH_NODE_SELECTOR);
            if (!firstParagraph || !lastParagraph) {
                Utils.error('Selection.iterateContentNodes(): invalid selection, cannot find containing paragraph nodes');
                return Utils.BREAK;
            }

            // rectangular cell range selection
            if (cellRangeSelected) {

                // entire table selected
                if (shortestPath && tableSelected) {
                    return visitContentNode(this.getEnclosingTable(), false);
                }

                // visit all table cells, iterate all content nodes according to 'shortest-path' option
                return this.iterateTableCells(function (cell) {
                    for (contentNode = findFirstContentNode(cell); contentNode; contentNode = findNextContentNode(cell, contentNode)) {
                        if (visitContentNode(contentNode, true) === Utils.BREAK) { return Utils.BREAK; }
                    }
                }, this);
            }

            // iterate through all paragraphs and tables until the end paragraph has been reached
            contentNode = firstParagraph;
            while (contentNode) {

                // check whether the parent container is completely covered
                parentCovered = !Utils.containsNode(contentNode.parentNode, firstParagraph) && !Utils.containsNode(contentNode.parentNode, lastParagraph);

                // visit current content node
                if (visitContentNode(contentNode, parentCovered) === Utils.BREAK) { return Utils.BREAK; }

                // end of selection reached
                if (contentNode === lastParagraph) { return; }

                // find next content node in DOM tree (next sibling paragraph or
                // table, or first node in next cell, or out of last table cell...)
                contentNode = findNextContentNode(rootNode, contentNode, lastParagraph);
            }

            // in a valid DOM tree, there must always be valid content nodes until end
            // paragraph has been reached, so this point should never be reached
            Utils.error('Selection.iterateContentNodes(): iteration exceeded end of selection');
            return Utils.BREAK;
        };

        /**
         * Calls the passed iterator function for specific nodes selected by
         * this selection instance. It is possible to visit all child nodes
         * embedded in all covered paragraphs (also inside tables and nested
         * tables), or to iterate on the 'shortest path' by visiting content
         * nodes (paragraphs or tables) exactly once if they are covered
         * completely by this selection and skipping the embedded paragraphs,
         * sub tables, and text contents.
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every node covered by
         *  this selection. Receives the following parameters:
         *      (1) {HTMLElement} the visited DOM node object,
         *      (2) {Number[]} the logical start position of the visited node
         *          (if the node is a partially covered text span, this is the
         *          logical position of the first character covered by the
         *          selection, NOT the start position of the span itself),
         *      (3) {Number} the relative start offset inside the visited node
         *          (usually 0; if the visited node is a partially covered text
         *          span, this offset is relative to the first character in the
         *          span covered by the selection),
         *      (4) {Number} the logical length of the visited node (1 for
         *          content nodes, character count of the covered part of text
         *          spans).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @param {Object} [options]
         *  A map of options to control the iteration. Supports the following
         *  options:
         *  @param {Boolean} [options.shortestPath=false]
         *      If set to true, tables and paragraphs that are covered
         *      completely by this selection will be visited directly and once,
         *      and their descendant components will be skipped in the
         *      iteration process. By default, this method visits the child
         *      nodes of all paragraphs and tables embedded in tables. Has no
         *      effect for tables that contain the end paragraph, because these
         *      tables are not fully covered by the selection. Tables that
         *      contain the start paragraph will never be visited, because they
         *      start before the selection.
         *  @param {Boolean} [options.split=false]
         *      If set to true, the first and last text span not covered
         *      completely by this selection will be split before the iterator
         *      function will be called. The iterator function will always
         *      receive a text span that covers the contained text completely.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateNodes = function (iterator, context, options) {

            var // whether to iterate on shortest path (do not traverse into completely covered content nodes)
                shortestPath = Utils.getBooleanOption(options, 'shortestPath', false),
                // split partly covered text spans before visiting them
                split = Utils.getBooleanOption(options, 'split', false),

                // start node and offset
                startInfo = null;

            // special case 'simple cursor': visit the text span
            if (this.isTextCursor()) {

                // start node and offset (pass true to NOT resolve text spans to text nodes)
                startInfo = Position.getDOMPosition(rootNode, startPosition, true);
                if (!startInfo || !startInfo.node) {
                    Utils.error('Selection.iterateNodes(): invalid selection, cannot find DOM node at start position ' + JSON.stringify(startPosition));
                    return Utils.BREAK;
                }

                // if located at the beginning of a component: use end of preceding text span if available
                if ((startInfo.offset === 0) && DOM.isPortionSpan(startInfo.node.previousSibling)) {
                    startInfo.node = startInfo.node.previousSibling;
                    startInfo.offset = startInfo.node.firstChild.nodeValue.length;
                }

                // visit the text component node (clone, because iterator is allowed to change passed position)
                return iterator.call(context, startInfo.node, _.clone(startPosition), startInfo.offset, 0);
            }

            // iterate the content nodes (paragraphs and tables) covered by the selection
            return this.iterateContentNodes(function (contentNode, position, startOffset, endOffset) {

                var // single-component selection
                    singleComponent = false,
                    // text span at the very beginning or end of a paragraph
                    textSpan = null;

                // visit fully covered content node in 'shortest-path' mode
                if (shortestPath && !_.isNumber(startOffset) && !_.isNumber(endOffset)) {
                    return iterator.call(context, contentNode, position, 0, 1);
                }

                // if selection starts after the last character in a paragraph, visit the last text span
                if (_.isNumber(startOffset) && (startOffset >= Position.getLastTextNodePositionInParagraph(contentNode))) {
                    if ((textSpan = DOM.findLastPortionSpan(contentNode))) {
                        return iterator.call(context, textSpan, position.concat([startOffset]), textSpan.firstChild.nodeValue.length, 0);
                    }
                    Utils.error('Selection.iterateNodes(): cannot find last text span in paragraph at position ' + JSON.stringify(position));
                    return Utils.BREAK;
                }

                // visit covered text components in the paragraph
                singleComponent = _.isNumber(startOffset) && _.isNumber(endOffset) && (startOffset === endOffset);
                return Position.iterateParagraphChildNodes(contentNode, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {

                    // skip floating drawings (unless they are selected directly) and helper nodes
                    if (DOM.isTextSpan(node) || DOM.isInlineComponentNode(node) || (singleComponent && DOM.isFloatingDrawingNode(node))) {
                        // create local copy of position, iterator is allowed to change the array
                        return iterator.call(context, node, position.concat([nodeStart + nodeOffset]), nodeOffset, offsetLength);
                    }

                // options for Position.iterateParagraphChildNodes(): visit empty text spans
                }, this, { allNodes: true, start: startOffset, end: endOffset, split: split });

            // options for Selection.iterateContentNodes()
            }, this, { shortestPath: shortestPath });

        };

        // initialization -----------------------------------------------------

        // lazy initialization with access to the application model and view
        app.on('docs:init', function () {
            clipboardNode = app.getView().createClipboardNode();
        });

        // Bug 26283: Overwrite the focus() method of the passed editor root
        // node. When focusing the editor, the text selection needs to be
        // restored first, or the internal clipboard node will be focused,
        // depending on the current selection type. By overwriting the DOM
        // focus() method, all focus attempts will be covered, also global
        // focus traveling with the F6 key triggered by the UI core.
        rootNode[0].focus = function () {
            self.restoreBrowserSelection();
        };

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // clear clipboard node and remove it from DOM
            clearClipboardNode();
            clipboardNode.remove();
            // restore original focus() method of the root node DOM element (prevent leaks)
            rootNode[0].focus = originalFocusMethod;
        });

    } // class Selection

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: Selection });

});
