/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statusbar',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Pane, ToolBox, Controls, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button;

    // class StatusBar ========================================================

    /**
     * @constructor
     *
     * @extends Pane
     */
    function StatusBar(app) {

        var // self reference
            self = this,

            // get application model and view
            view = app.getView(),

            // the sheet selector controls
            sheetGroup = null,
            subtotalsTypeGroup = null,
            largeSheetList = null,
            smallSheetList = null,

            // the tool box for all sheet controls
            sheetToolBox = null,

            // the tool box for the zoom buttons
            zoomToolBox = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, app, 'statusbar', { position: 'bottom', classes: 'status-bar inline' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the sheet selectors according to the current
         * width of the sheet tool box.
         */
        function refreshSheetToolBox() {

            var // the root node of the tool box
                rootNode = sheetToolBox.getNode(),
                // the available width for the tool box
                availableWidth = rootNode.outerWidth();

            function toggleGroup(group, state) {
                group.getNode().closest('.group-container').toggleClass('hidden', !state);
            }

            function getCurrentWidth() {
                return Math.ceil(rootNode.outerWidth());
            }

            // calculate and set remaining space for the specified sheet selector
            function setRemainingWidth(group) {
                var remainingWidth = group.getNode().outerWidth() + availableWidth - getCurrentWidth();
                group.setWidth(remainingWidth);
            }

            // hide the sheet list selectors, show the sheet tab selector with automatic width
            rootNode.css({ width: 'auto', paddingLeft: (view.getHeaderWidth() + 13) + 'px' });

            toggleGroup(largeSheetList, false);
            toggleGroup(smallSheetList, false);
            toggleGroup(sheetGroup, true);
            sheetGroup.setFullWidth();

            // check whether ALL sheet tabs fit into the available space
            if (availableWidth < getCurrentWidth()) {
                rootNode.css({ paddingLeft: '' });
                if (availableWidth < 380) {
                    // show large sheet drop-down list for small devices
                    toggleGroup(sheetGroup, false);
                    toggleGroup(largeSheetList, true);
                    // calculate and set remaining space for the sheet drop-down button
                    largeSheetList.setWidth(availableWidth);
                    setRemainingWidth(largeSheetList);
                } else {
                    // show additional sheet selector controls
                    toggleGroup(smallSheetList, true);
                    // calculate and set remaining space for the sheet tabs
                    setRemainingWidth(sheetGroup);
                }
            }

            // reset sheet component to its current width
            rootNode.css({ width: availableWidth });
        }

        /**
         * Refreshes the layout of the status bar after its size has been
         * changed.
         */
        function refreshLayout() {

            var // the available inner width in the status bar
                availableWidth = self.getNode().width();

            // always hide the subtotal button in small displays
            subtotalsTypeGroup.setActivated(availableWidth >= 380);

            // reduce available width by the size of the zoom component
            availableWidth -= Math.ceil(zoomToolBox.getNode().outerWidth(true));

            // set new width to the sheet component and refresh all controls
            sheetToolBox.getNode().css({ width: availableWidth });
            refreshSheetToolBox();
        }

        // initialization -----------------------------------------------------

        this.addViewComponent(sheetToolBox = new ToolBox(app, 'sheets', { classes: 'sheets-toolbox' })
            .addGroup('view/sheet/active', largeSheetList = new Controls.ActiveSheetList(app, { showNames: true }))
            .addGap(13)
            .addGroup('view/sheet/active', smallSheetList = new Controls.ActiveSheetList(app))
            .addGap(13)
            .addGroup('view/sheet/active', sheetGroup = new Controls.ActiveSheetGroup(app))
            .addGap(13)
            .addGroup('sheet/insert', new Button({ icon: 'fa-plus', tooltip: gt('Insert sheet') }))
            .addPrivateGroup(new Controls.PopupMenuButton(app, 'sheetactions', { icon: 'fa-ellipsis-v', tooltip: gt('More sheet operations'), caret: false, position: 'top' })
                .addGroup('sheet/rename/dialog', new Button({ label: gt('Rename sheet') }))
                .addGroup('sheet/copy/dialog', new Button({ label: gt('Copy sheet') }))
                .addGroup('sheet/delete', new Button({ label: gt('Delete sheet') }))
                .addGroup('sheet/visible', new Button({ label: gt('Hide sheet'), value: false }))
                .addSeparator()
                .addGroup('sheet/reorder/dialog', new Button({ label: gt('Reorder sheets') }))
                .addGroup('sheet/showall', new Button({ label: gt('Show all hidden sheets') }))
            )
        );

        this.addViewComponent(zoomToolBox = new ToolBox(app, 'zoom', { classes: 'zoom-toolbox' })
            .addGroup('view/subtotals/type', subtotalsTypeGroup = new Controls.SubtotalList(app, { position: 'top' }))
            .addGap(13)
            .addGroup('view/zoom/dec', new Button(Controls.ZOOMOUT_OPTIONS))
            .addGroup('view/zoom/inc', new Button(Controls.ZOOMIN_OPTIONS))
        );

        // refresh layout after status bar changes its size
        view.on('refresh:layout', refreshLayout);
        zoomToolBox.on('component:layout', refreshLayout);

        // refresh sheet tool box after sheet tabs have been inserted, deleted, or changed
        sheetGroup.on('refresh:layout', refreshSheetToolBox);

        // destroy class members on destruction
        this.registerDestructor(function () {
            app = self = view = null;
            sheetGroup = subtotalsTypeGroup = largeSheetList = smallSheetList = null;
            sheetToolBox = zoomToolBox = null;
        });

    } // class StatusBar

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: StatusBar });

});
