/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/viewsettingsmixin', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class ViewSettingsMixin ================================================

    /**
     * A mix-in class for spreadsheet model classes that stores specific view
     * attributes. Used by the document model class to store global view
     * settings, and by the sheet model to store view settings per sheet.
     *
     * Triggers the following events:
     * - 'change:viewattributes'
     *      After at least one view attribute has been changed. Event handlers
     *      receive an incomplete attribute map containing all changed view
     *      attributes with their new values.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     *
     * @param {Object} definitions
     *  A map with definition objects for all supported view attributes,
     *  containing their default values, and other settings. Each definition
     *  object is keyed by the attribute name, and contains the following
     *  properties:
     *  - {Any} definition.def
     *      The default value for the view attribute.
     *  - {Function|RegExp} [definition.validate=_.identity]
     *      A callback function used to validate a new value for the attribute,
     *      or a regular expression that can be used to validate enumeration
     *      attributes. The callback function may return any value to be used
     *      as new attribute value, or may return Utils.BREAK to reject the new
     *      attribute value and keep the current value. Defaults to the
     *      identity function (accepts all attribute values). Will be called in
     *      the context of the object instance containing the view attributes.
     */
    function ViewSettingsMixin(app, definitions) {

        var // self reference
            self = this,

            // the current view attributes of this object
            viewAttributes = {};

        // methods ------------------------------------------------------------

        /**
         * Returns the values of all view attribute.
         *
         * @returns {Object}
         *  A cloned map containing all view attribute values.
         */
        this.getViewAttributes = function () {
            return _.copy(viewAttributes, true);
        };

        /**
         * Returns the value of a specific view attribute.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  A clone of the attribute value.
         */
        this.getViewAttribute = function (name) {
            return (name in viewAttributes) ? _.copy(viewAttributes[name], true) : null;
        };

        /**
         * Changes specific view attributes of this object, and triggers a
         * 'change:viewattributes' event.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed. The values must
         *  neither be undefined nor null.
         *
         * @returns {ViewSettingsMixin}
         *  A reference to this instance.
         */
        this.setViewAttributes = function (attributes) {

            var // all changed attributes, for listeners
                changedAttributes = {};

            _(attributes).each(function (value, name) {
                if (name in definitions) {

                    // execute validator
                    if (_.isFunction(definitions[name].validator)) {
                        value = definitions[name].validator.call(self, value);
                    } else if (_.isRegExp(definitions[name].validator)) {
                        if (!definitions[name].validator.test(value)) { value = Utils.BREAK; }
                    }

                    // change attribute value
                    if (!_.isUndefined(value) && (value !== Utils.BREAK)) {
                        if (!_.isEqual(viewAttributes[name], value)) {
                            viewAttributes[name] = _.copy(value, true);
                            changedAttributes[name] = value;
                        }
                    }
                }
            });

            if (!_.isEmpty(changedAttributes)) {
                this.trigger('change:viewattributes', changedAttributes);
            }
            return this;
        };

        /**
         * Changes a specific view attribute of this object, and triggers a
         * 'change:viewattributes' event.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {ViewSettingsMixin}
         *  A reference to this instance.
         */
        this.setViewAttribute = function (name, value) {
            return this.setViewAttributes(Utils.makeSimpleObject(name, value));
        };

        // initialization -----------------------------------------------------

        _(definitions).each(function (definition, name) {
            viewAttributes[name] = _.copy(definition.def, true);
        });

    } // class ViewSettingsMixin

    // exports ================================================================

    return ViewSettingsMixin;

});
