/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/view/hyperlinkutil',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/editframework/view/editdialogmixin',
     'gettext!io.ox/office/text'
    ], function (Utils, Dialogs, EditDialogMixin, gt) {

    'use strict';

    var PROTS = new RegExp('^(https?://|ftp://|mailto:)', 'i');

    // templates for web/ftp identifications
    var TXTMS = new RegExp('^(www\\.|ftp\\.)', 'i');

    var HyperlinkUtil = {};

    // methods ----------------------------------------------------------------

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {String} text
     *  The optional text which represents the URL
     *
     * @param {String} url
     *  An optional URL which is set for the supplied text
     *
     * @param {BaseApplication} app
     *  The calling app
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text.
     */
    HyperlinkUtil.showDialog = function (text, url, app) {
        return HyperlinkUtil.showHyperlinkDialog(app, { valueURL: url, valueText: text });
    };

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {BaseApplication} app
     *  The calling application.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.valueText='']
     *      The initial value of the text field.
     *  @param {String} [options.valueURL='']
     *      The initial value of the URL field.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button or the remove button have been activated, or rejected if the
     *  dialog has been canceled. The done handlers registered at the promise
     *  object will receive a object containing the text and URL entered by the
     *  user. The object contains null for text and the URL if remove has been
     *  clicked.
     */
    HyperlinkUtil.showHyperlinkDialog = function (app, options) {

        var // the document model
            model = app.getModel(),

            // the input field for the display text of the URL
            textValue = Utils.getStringOption(options, 'valueText', ''),
            textLabelId = _.uniqueId('text'),
            textInputId = _.uniqueId('text'),
            textInput = $('<input>', {
                value: textValue,
                tabindex: 0,
                id: textInputId,
                role: 'textbox',
                'aria-labelledby': textLabelId
            }).addClass('form-control'),

             // the input field for the URL
            urlValue = Utils.getStringOption(options, 'valueURL', ''),
            urlLabelId = _.uniqueId('url'),
            urlInputId = _.uniqueId('url'),
            urlInput = $('<input>', {
                value: urlValue,
                tabindex: 0,
                id: urlInputId,
                role: 'textbox',
                'aria-labelledby': urlLabelId
            }).addClass('form-control'),

            // the dialog instance
            title = (urlValue.length === 0) ? gt('Insert Hyperlink') : gt('Edit Hyperlink'),
            dialog = new Dialogs.ModalDialog({ title: title, okLabel: gt('Insert') }),

            // the resulting promise
            promise = null;

        // close dialog automatically after losing edit rights
        EditDialogMixin.call(dialog, app);

        // add the input controls to the dialog body
        dialog.append(
            $('<div>').addClass('form-group').append(
                $('<label>', { 'for': textInputId, id: textLabelId }).text(/*#. "Insert URL" dialog: The display text of the URL */ gt('Text')),
                textInput
            ),
            $('<div>').addClass('form-group').append(
                $('<label>', { 'for': urlInputId, id: urlLabelId }).text(/*#. "Insert URL" dialog: The URL itself */ gt('URL')),
                urlInput
            )
        );

        // only add remove button if we already have a set URL
        if (Utils.getStringOption(options, 'valueURL', '').length > 0) {
            dialog.addAlternativeButton('remove', gt('Remove'), 'remove', { tabIndex: 1 });
            dialog.getButton('remove').addClass('btn-warning');
        }

        function checkTextFields() {
            var textUrl = urlInput.val(),
                text = textInput.val();

            if (!model.getEditMode()) {
                // disable ok and delete button if we don't have edit rights
                dialog.enableOkButton(false).enableButton('remove', false);
            } else if ((text && text.length > 0) && (textUrl && textUrl.length > 0)) {
                dialog.enableOkButton(true);
            } else {
                dialog.enableOkButton(false);
            }
        }

        // register a change handlers to toggle the insert button
        urlInput.on('input', checkTextFields);
        textInput.on('input', checkTextFields);

        // disable the ok button if the dialog is already opened but we lost the edit rights.
        model.on('change:editmode', checkTextFields);

        // show the dialog
        promise = dialog.show(function () {
            var hasText = textInput.val().length > 0;
            (hasText ? urlInput : textInput).focus();
            checkTextFields();
        });

        // process the result of the dialog
        promise = promise.then(function (action) {
            switch (action) {
            case 'ok':
                var text = textInput.val();
                var url = $.trim(urlInput.val());
                url = HyperlinkUtil.removeWhiteSpaces(url);
                // add http: as default if protocol is missing
                var protIndex = url.indexOf(':');
                if (protIndex === -1) {
                    url = 'http://' + url;
                } else if (!HyperlinkUtil.isLegalUrl(url)) {
                    Utils.warn('you tried to add illegal code ' + url);
                    app.getView().yell('warning', gt('This hyperlink is invalid.'));
                    url = null;
                }
                return { text: text, url: url };
            case 'remove':
                return { text: null, url: null };
            }
            return $.Deferred().reject();
        });

        return promise.always(function () {
            model.off('change:editmode', checkTextFields);
        });
    };

    HyperlinkUtil.isLegalUrl = function (url) {
        return PROTS.test(url);
    };

    /**
     * create a jQuery Popup node with link, edit and remove function
     *
     * @param {BaseApplication} app
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     *
     * @returns {jQuery Element}
     */

    HyperlinkUtil.createPopupNode = function (app) {
        var hyperlinkPopup = $('<div>', { contenteditable: false })
            .addClass('inline-popup hyperlink f6-target').css('display', 'none')
            .append(
                $('<a>').addClass('textwrap').attr({ href: '', rel: 'noreferrer', target: '_blank', tabindex: 0 }),
                $('<br/>'),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Edit')).click(function () { app.getController().executeItem('character/hyperlink/dialog'); return false; }),
                $('<span>').text(' | '),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Remove')).click(function () { app.getController().executeItem('character/hyperlink/remove'); return false; }));

        if (Modernizr.touch) {
            // some CSS changes for touch devices
            hyperlinkPopup.addClass('touch');
            hyperlinkPopup.find('a.popupbutton').addClass('touch');
        }

        //hyperlinkPopup.bind('touchstart mousedown', function () {
        //    return false;
        //});


        return hyperlinkPopup;

    };

    HyperlinkUtil.updatePopup = function (url, popup) {
        var link = popup.find('a.textwrap');
        //var link = $('a.textwrap', hyperlinkPopup[0]);
        link.text(HyperlinkUtil.limitHyperlinkText(url));
        if (!url || url.charAt(0) === '#') {
            link.attr({
                title: ''
            });
            link.css({
                color: '',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        } else if (HyperlinkUtil.isLegalUrl(url)) {
            link.css({
                color: '',
                cursor: '',
                textDecoration: ''
            });
            link.attr({
                href: url,
                title: url
            });
        } else {
            link.attr({
                title: gt('Script code is not allowed')
            });
            link.css({
                color: 'red',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        }
    };

    /**
     * adds listener for the popupNode for changing the Editmode
     *
     * @param {SpreadsheetModel} model
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     * @param {jQuery Element} popupNode
     *  @see HyperlinkUtil.createPopupNode
     *
     * @param {Function} listener
     *  an extra listener when Editmode is changed
     */

    HyperlinkUtil.addEditModeListener = function (model, popupNode, listener) {
        var updateEditMode = function () {
            var readOnly = model.getEditMode() !== true;
            popupNode.find('br').toggle(!readOnly);
            popupNode.find('span').toggle(!readOnly);
            popupNode.find('a.popupbutton').toggle(!readOnly);
            listener();
        };
        model.on('change:editmode', updateEditMode);


        // must be called to set the initial state correctly
        updateEditMode();
    };

    /**
     * Shortens a hyperlink to a maximal number of
     * characters.
     *
     * @param text {String} Hyperlink to shorten
     */
    HyperlinkUtil.limitHyperlinkText = function (text) {

        var maxChars = 255,
            result = text;

        if (_.isString(text) && text.length > maxChars) {
            var length = text.length,
                start = text.slice(0, Math.round(maxChars / 2)),
                end = text.slice(length - Math.round(maxChars / 2));

            result = start + '...' + end;
        }

        return result;
    };


    /**
     * checks if popupnode is already created in given parent
     *
     * @param {jQuery Element} parent
     *
     */

    HyperlinkUtil.noPopup = function (parent) {
        if (parent.children('.inline-popup.hyperlink').length === 0) {
            return true;
        }
    };

    /**
     *  Checks for a text which defines a hyperlink
     *  e.g. http://www.open-xchange.com
     *
     *  @param {String} text
     *
     *  @returns {String} if an hyperlink is found
     */

    HyperlinkUtil.checkForHyperlink = function (text) {
        var result = null;

        var content = PROTS.exec(text);

        if (content) {
            var foundProtocol = content[1];
            if (foundProtocol !== 'mailto:') {
                var index = text.indexOf('//');

                // At least one character after the protocol must be there
                if ((index + 2) < text.length && HyperlinkUtil.checkURL(text)) {
                    // create result with correct Position objects
                    result = text;
                }
            } else if (foundProtocol === 'mailto:') {
                var atIndex = text.indexOf('@');
                var dotIndex = text.lastIndexOf('.');

                // mailto: needs a '@' and '.' char and at least one char after the dot
                // we also use the regexp to check for other problems
                if ((atIndex > ('mailto:'.length + 1)) &&
                    (dotIndex > atIndex) && (dotIndex < (text.length - 1)) &&
                    HyperlinkUtil.checkURL(text)) {

                    // create result with correct Position objects
                    result = text;
                }
            }
        } else {
            //some magic auto detection for text without protocol

            content = TXTMS.exec(text);

            if (content) {
                var textTemplate = content[1].toLowerCase();
                if (textTemplate === 'www.') {
                    // http://
                    if (text.length > 'www.'.length) {
                        result = 'http://' + text;
                    }
                } else if (textTemplate === 'ftp.') {
                    // ftp://
                    if (text.length > 'ftp.'.length) {
                        result = 'ftp://' + text;
                    }
                }
            }
            else {
                // some more magic for a mail address
                if (HyperlinkUtil.checkMailAddress(text)) {
                    result = 'mailto:' + text;
                }
            }
        }
        if (result) {
            result = HyperlinkUtil.removeWhiteSpaces(result);
        }
        return result;
    };

    HyperlinkUtil.removeWhiteSpaces = function (url) {
        return url.replace(/\s/g, '');
    };

    /**
     * Checks an URL based on some basic rules. This is not a fully compliant
     * URL check.
     *
     * @param {String} url
     *
     * @returns {Boolean}
     *  true if the URL seems to be correct otherwise false
     */
    HyperlinkUtil.checkURL = function (url) {
        return (/^([a-z]([a-z]|\d|\+|-|\.)*):(\/\/(((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:)*@)?((\[(|(v[\da-f]{1,}\.(([a-z]|\d|-|\.|_|~)|[!\$&'\(\)\*\+,;=]|:)+))\])|((\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5]))|(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=])*)(:\d*)?)(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*|(\/((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)?)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)){0})(\?((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|[\uE000-\uF8FF]|\/|\?)*)?(\#((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|\/|\?)*)?$/i.test(url));
    };

    /**
     * Checks an mailaddress based on some basic rules. This is not a fully compliant
     * mailaddress check.
     *
     * @param {String} mailAddress
     *
     * @returns {Boolean}
     *  true if the URL seems to be correct otherwise false
     */
    HyperlinkUtil.checkMailAddress = function (mailAddress) {
        return (/^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,4}$/i.test(mailAddress));
    };

    // exports ================================================================

    return HyperlinkUtil;

});
