/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/sidepane',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox'
    ], function (Utils, Pane, ToolBox) {

    'use strict';

    // class SidePane =========================================================

    /**
     * A view pane attached to the left or right border of the application
     * window. Contains tool boxes stacked vertically. Provides fixed areas at
     * the top and at the bottom of the side pane with tool boxes that will
     * always be visible. The inner area between the fixed areas will be
     * scrollable.
     *
     * Triggers the following events:
     * - 'refresh:layout': After the view containing this side pane has
     *      triggered a 'refresh:layout' event by itself, and this side pane
     *      has updated the position and size of the fixed and scrollable
     *      sections containing the tool boxes.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {BaseApplication} app
     *  The application containing this side pane.
     *
     * @param {String} id
     *  The identifier for this view pane. Must be unique across all view panes
     *  in the application.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the side pane. Supports
     *  all options supported by the base class Pane. The option 'position'
     *  will be restricted to the values 'left' and 'right'. The option
     *  'componentInserter' is not supported anymore. Additionally, the
     *  following options are supported:
     *  @param {String} [initOptions.position='right']
     *      The position of the side pane, either 'left' or 'right'.
     *  @param {Function} [initOptions.insertHandler]
     *      A function that will be called after this side pane has been
     *      inserted into the application window. Needed if the geometry of the
     *      pane DOM node needs to be initialized to perform further
     *      initialization tasks. Will be called in the context of this side
     *      pane instance.
     *  @param {Function} [initOptions.refreshHandler]
     *      A function that will be called when the layout of the side pane
     *      needs to be refreshed. Will be called after the application
     *      controller has sent 'change:items' events (the visibility of tool
     *      boxes may have changed), after expanding or collapsing a tool box,
     *      or when the size of the browser window has been changed.
     */
    function SidePane(app, id, initOptions) {

        var // self reference
            self = this,

            // container node for the upper fixed tool boxes in the side pane
            fixedTopNode = $('<div>').addClass('fixed-toolboxes top'),

            // scrollable container node for the tool boxes in the side panes
            scrollableNode = $('<div>').addClass('scrollable-toolboxes'),

            // container node for the upper fixed tool boxes in the side pane
            fixedBottomNode = $('<div>').addClass('fixed-toolboxes bottom');

        // base constructor ---------------------------------------------------

        Pane.call(this, app, id, Utils.extendOptions({
            // default options, can be overridden by passed options
            size: SidePane.DEFAULT_WIDTH
        }, initOptions, {
            // fixed options, will override passed options
            position: (Utils.getStringOption(initOptions, 'position') === 'left') ? 'left' : 'right',
            componentInserter: toolBoxInserter
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the side pane according to the current browser window size.
         */
        function refreshLayout() {
            if (!self.isReallyVisible()) { return; }

            // update top/bottom positions of scrollable container in case the fixed tool boxes have changed
            scrollableNode.css({ top: fixedTopNode.height() - 1, bottom: fixedBottomNode.height() - 1 });
            // toggle visibility of border lines above/below container node
            scrollableNode.toggleClass('scrolling', scrollableNode[0].clientHeight < scrollableNode[0].scrollHeight);

            // notify listeners
            self.trigger('refresh:layout');
        }

        /**
         * Inserts the passed tool box into the side pane.
         */
        function toolBoxInserter(toolBox) {
            switch (Utils.getStringOption(toolBox.getOptions(), 'fixed')) {
            case 'top':
                fixedTopNode.append(toolBox.getNode());
                break;
            case 'bottom':
                fixedBottomNode.append(toolBox.getNode());
                break;
            default:
                scrollableNode.append(toolBox.getNode());
            }
        }

        // methods ------------------------------------------------------------

        this.getFixedTopNode = function () {
            return fixedTopNode;
        };

        this.getScrollableNode = function () {
            return scrollableNode;
        };

        this.getFixedBottomNode = function () {
            return fixedBottomNode;
        };

        /**
         * Creates a tool box in this side pane, and registers it for automatic
         * visibility handling.
         *
         * @param {String} toolBoxId
         *  The identifier for the new tool box. Must be unique across all view
         *  components in the application.
         *
         * @param {Object} [options]
         *  A map of options to control the properties of the new tool box.
         *  Supports all options supported by the constructor of the class
         *  ToolBox, and the following additional options:
         *  @param {String} [options.fixed]
         *      If set to 'top' or 'bottom', the new tool box will be appended
         *      to the upper respectively lower fixed area of the side pane,
         *      and cannot be scrolled outside the visible area. Otherwise, the
         *      tool box will be appended to the scrollable area of the side
         *      pane.
         *  @param {String} [options.visible]
         *      The key of the controller item that controls the visibility of
         *      the tool box. The visibility will be bound to the 'enabled'
         *      state of the respective controller item. If omitted, the tool
         *      box will always be visible.
         *
         * @returns {ToolBox}
         *  The new tool box instance.
         */
        this.createToolBox = function (toolBoxId, options) {

            var // the new tool box instance
                toolBox = new ToolBox(app, toolBoxId, options),
                // the controller item controlling the visibility of the tool box
                visibleKey = Utils.getStringOption(options, 'visible');

            // insert tool box into the side pane, refresh layout after expanding/collapsing
            this.addViewComponent(toolBox);
            self.listenTo(toolBox, 'expand', refreshLayout);

            // update visibility according to enabled state of controller item
            if (_.isString(visibleKey)) {
                // initially hide the tool box
                toolBox.hide();
                this.listenTo(app.getController(), 'change:items', function (event, changedItems) {
                    if ((visibleKey in changedItems) && (toolBox.isVisible() !== changedItems[visibleKey].enable)) {
                        toolBox.toggle(changedItems[visibleKey].enable);
                        refreshLayout();
                    }
                });
            }

            return toolBox;
        };

        // initialization -----------------------------------------------------

        // insert the container nodes for fixed and scrollable tool boxes
        this.getNode().addClass('side-pane').append(fixedTopNode, scrollableNode, fixedBottomNode);

        // update side pane after controller updates (tool box visibility
        // may have changed), and after the view has refreshed the panes
        self.listenTo(app.getController(), 'change:items', refreshLayout);
        self.listenTo(app.getView(), 'refresh:layout', refreshLayout);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            fixedTopNode.remove();
            scrollableNode.remove();
            fixedBottomNode.remove();

            self = scrollableNode = fixedTopNode = fixedBottomNode = null;
        });

    } // class SidePane

    // constants ==============================================================

    /**
     * Default width of side panes, in pixels.
     */
    SidePane.DEFAULT_WIDTH = 249;

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: SidePane });

});
