/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.URLEncoder;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.Keymanager;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.encr.PgpKeys;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.EmailValidator;

final class Settings {
	public enum permissions {
        NONE, MAIL, FILE, BOTH
    };
    
    public permissions perm = permissions.NONE;
    public boolean extrapass = Config.ok_extra_password;  // Set to default config level
    public boolean privatepass = true;
    public boolean deleteRecovery = true;
    public boolean deleteKeyOnRevoke = Config.deleteOnRetract;
    
}

public class Auth {
	
	private static Logger logger = LoggerFactory.getLogger(Auth.class);

    public JsonObject getJSON(HttpServletRequest request) {
        StringBuffer sb = new StringBuffer();
        String line = null;
        try {
            BufferedReader reader = request.getReader();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
        } catch (Exception e) {
            logger.error("Error using JSON reader", e);
        }
        JsonParser jsonParser = new JsonParser();
        JsonObject json = (JsonObject) jsonParser.parse(sb.toString());
        return (json);
    }

    public void createKeys(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = getJSON(request);
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {
        	logger.error("Create keys but not logged into UI");
            response.setStatus(550);
            return;
        }
        logger.debug("Starting to create master key");
        Access acc = new Access();
        int userid = json.get("user_id").getAsInt();
        int cid = json.get("cid").getAsInt();
        String language = json.get("language").getAsString();
        String password = json.get("password").getAsString();
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys == null) { // make sure already don'e exist
            String email = acc.getEmail(userid, cid);
            String name = email;
            if (json.get("name") != null) {
                name = json.get("name").getAsString();
            }
            Keymanager keyman = new Keymanager();
            keys = keyman.createMasterKeys(userid, cid, email, name, password, language, true); // Flag true for user created
            if (keys == null) {
                logger.debug("Failed to create master key");
                response.getWriter().write(" {\"result\":\"fail \"} ");
            } else {
            	if (json.has("email")) {
            		acc.storeQuestion(userid, cid, "e", json.get("email").getAsString());
            	}
                logger.debug("Success creating master key");
                response.getWriter().write(" {\"result\":\"ok\"} ");
            }
        }
    }

    /**
     * Change password
     * @param request
     * @param response
     * @param cookie
     * @return
     * @throws Exception
     */
    public String changepass(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = getJSON(request);
        if (CheckBad.isBad(cookie.JSESSIONID, Config.bad_password_count)) {
        	logger.debug("Lockout for bad attempts");
            response.getWriter().write("{\"auth\":\"Lockout\"}");
            return (null);
        }
        String newpass = json.get("newpass").getAsString().trim();
        if (newpass.length() < Config.min_password_length) {
        	response.getWriter().write("{\"auth\":\"Bad new password\",\"minlength\":" + Config.min_password_length + "}");
            return (null);
        }
        int cid = json.get("cid").getAsInt();
        String question = "";
        if (json.has("question")) { // Sent with new question and answer?
            question = json.get("question").getAsString();
        }
        String answer = "";
        if (json.has("answer")) {
            answer = json.get("answer").getAsString();
        }
        if (json.has("email")) {
        	question = "e";
        	answer = json.get("email").getAsString();
        }
        // if ox member, verify actually logged in to ui
        if (cid > 0) {
            Api ap = new Api(cookie, request);
            if (ap.verifyLogin() == false) {
                response.setStatus(550);
                return (null);
            }
        }

        // OK, logged in or guest, so we can change
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        logger.debug("change password request");
        int userid = json.get("user_id").getAsInt();

        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys == null) {
        	response.getWriter().write("{\"auth\":\"Key not found\"}");
        	return (null);
        }
        String oldpass = "";
        oldpass = json.get("oldpass").getAsString().trim();
        PgpKeys keygen = new PgpKeys();
        keys = keygen.changePassword(oldpass, newpass, keys);
        if (keys == null) {
            logger.debug("Bad password for change");
            response.getWriter().write("{\"auth\":\"Bad password\"}");
            CheckBad.addBad(cookie.JSESSIONID);
            return (null);
        } else {
            Keymanager keyman = new Keymanager();
            String recovery = keyman.createRecovery(keys, newpass);
            acc.updateKeys(userid, cid, keys, recovery, false);
            if (!answer.equals("")) {// If sent with question and answer, then store
                if (question.equals(""))
                    question = "default";
                acc.storeQuestion(userid, cid, question, answer);
            }
            // String sessionID = json.get("sessionID").getAsString();
            String token = acc.getToken(cookie.JSESSIONID);
            if (token == null) token = acc.newToken(cookie.JSESSIONID, userid, cid);
            json.add("encr_password", new JsonPrimitive(newpass));
            EncrLib encr = new EncrLib();
            String returndata = " {\"auth\":\"" + encr.encryptAES(json.toString(), token) + "\"} ";
            logger.debug(returndata);
            Core.sendOK(response, returndata);
            logger.info("OK Password Change");
            return (returndata);
        }

    }
    
    /**
     * Update secondary email address after verifying OxGuard password
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void changeSecondaryEmail (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	JsonObject json = getJSON(request);
    	int cid = json.get("cid").getAsInt();
    	int userid = json.get("userid").getAsInt();
    	if (CheckBad.isBad(Integer.toString(userid) + "-" + Integer.toString(cid), Config.bad_password_count)) {
    		Core.sendFail(response, "Lockout");
    		return;
    	}
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            Core.sendFail(response, "Not logged in");
            return;
        }
        String password = json.get("password").getAsString();
        String newemail = json.get("email").getAsString();
        Access acc = new Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys.getDecodedPrivate(password) != null) {
        	if (updateEmail (userid, cid, newemail)) {
        		Core.sendOK(response, "OK");
        	} else Core.sendFail(response, "Fail");
        } else {
        	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));
        	Core.sendFail(response, "Bad password");
        	return;
        }
        
    }
    
    public void getSecondaryEmail (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            Core.sendFail(response, "Not logged in");
            return;
        }
        int cid = Integer.parseInt(request.getParameter("cid"));
        int userid = Integer.parseInt(request.getParameter("userid"));
       	String command = "SELECT answer, email FROM og_KeyTable WHERE id = ? AND CID = ?";
    	DbCommand dcom = new DbCommand(command);
    	dcom.addVariables(userid);
    	dcom.addVariables(cid);
    	DbQuery db = new DbQuery();
    	db.read(dcom, userid, cid);
    	if (db.next()) {
    		String encr_answer = db.rs.getString("answer");
    		EncrLib encr = new EncrLib();
    		String secondEmail = encr.decryptAES(encr_answer, Config.rpass, "e", Config.AESKeyLen, false);
    		if (secondEmail != null) {
    			db.close();
    			Core.sendOK(response, secondEmail);
    			return;
    		}
    	}
    	db.close();
    	Core.sendOK(response, "");
    }
    
    private boolean updateEmail (int userid, int cid, String email) {
    	Access acc = new Access();
    	return(acc.storeQuestion(userid, cid, "e", email));
    }

    public void resetPass(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	int cid = Integer.parseInt(request.getParameter("cid"));
        int userid = Integer.parseInt(request.getParameter("userid"));
        resetPass (userid, cid, request, response, cookie, true);
    }
    
    /*
     * Reset password.  Used from command line tools only
     */
    public void resetPass (String email) {
    	Access acc = new Access();
			try {
				GuardKeys keys = acc.getKeysFromEmail(email);
				if (keys == null) {
					System.out.println("NO ACCOUNT FOUND");
					return;
				}
				if (keys.contextid < 0) { // Guest
					Keymanager keyman = new Keymanager();
					String newpass = keyman.resetPassword(email);
					if (newpass.equals("NR")) {
						System.out.println("No recovery available");
						return;
					}
	    			Sender sender = new Sender();
	                com.openexchange.guard.mailcreator.Mail mf = new com.openexchange.guard.mailcreator.Mail();
	                if (!sender.send(mf.getResetEmail(email, email, newpass, Config.default_lang, 0), null)) {
	                	System.out.println("Failed to send");
	                } else {
	                	System.out.println("Reset sent");
	                }
				} else // OX member
					resetPass (keys.userid, keys.contextid, null, null, null, false);
			} catch (Exception e) {
				logger.error("Error resetting password for " + email, e);
			}
    	
    }
    
    
    public void resetPass(int userid, int cid, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, boolean web) throws Exception {
    	int templId = 0;
    	if (web) { // If from web request, make sure logged onto OX
        	Api ap = new Api(cookie, request);
            if (ap.verifyLogin() == false) {
                response.setStatus(550);
                return;
            }
            templId = OxConfiguration.getTemplateId(cid, userid);
    	}
        
        String lang = request == null ? Config.default_lang : request.getParameter("language");
    	String command = "SELECT answer, email FROM og_KeyTable WHERE id = ? AND CID = ?";
    	DbCommand dcom = new DbCommand(command);
    	dcom.addVariables(userid);
    	dcom.addVariables(cid);
    	DbQuery db = new DbQuery();
    	db.read(dcom, userid, cid);
    	if (db.next()) {
    		String encr_answer = db.rs.getString("answer");
    		String userEmail = db.rs.getString("email");
    		String defaultEmail = request.getParameter("default");
    		EncrLib encr = new EncrLib();
    		boolean secondary = true;
    		String secondEmail = encr.decryptAES(encr_answer, Config.rpass, "e", Config.AESKeyLen, false);
    		if ((secondEmail == null) || secondEmail.equals("")) {
    			secondEmail = defaultEmail;
    			secondary = false;
    		}
    		if (EmailValidator.validate(secondEmail)) {
    			Keymanager keyman = new Keymanager();
    			String newpass = keyman.resetPassword(userEmail);
    			if (newpass == null) {
    				if (web) {
                    	Core.sendOK(response, "FailChange");
                    } else System.out.println("Failed to change");
    				db.close();
    				return;
    			}
    			if (newpass.equals("NR")) {
    				if (web) {
                    	Core.sendOK(response, "NoRecovery");
                    } else System.out.println("No recovery available");
    				db.close();
    				return;
    			}
    			Sender sender = new Sender();
                com.openexchange.guard.mailcreator.Mail mf = new com.openexchange.guard.mailcreator.Mail();
                if (!sender.send(mf.getResetEmail(secondEmail, (defaultEmail == null ? secondEmail : defaultEmail), newpass, lang, templId), null)) {
                    if (web) {
                    	Core.sendOK(response, "FailNotify");
                    } else System.out.println("Failed to send nofication");
                    db.close();
                    return;
                } else {
                    if (web) {
                    	if (secondary) {
                    		Core.sendOK(response, "ok");
                    	} else {
                    		Core.sendOK(response, "primary");
                    	}
                    } else System.out.println("OK");
                }
    		} else { // Bad secondary email
    			if (web) {
                	Core.sendOK(response, "NoSecondary");
                } else System.out.println("No Secondary Email");
                db.close();
                return;
    		}
    	}
    	db.close();
    }
    /**
     * Reset password from within OX UI using OX login password --- Depreciated
     * 
     * @param request
     * @param response
     * @param cookie
     * @return
     * @throws Exception
     */
    public String changepassFromOxRecovery(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = getJSON(request);
        int cid = json.get("cid").getAsInt();
        String password = json.get("oxpassword").getAsString();
        int userid = json.get("user_id").getAsInt();
        String user = json.get("user").getAsString();
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        // if ox member, verify actually logged in to ui
        if (cid > 0) {
            Api ap = new Api(cookie, request);
            if (ap.verifyLogin() == false) {
                response.setStatus(550);
                return (null);
            }
        }
        Api ap2 = new Api(user, password);
        if (!ap2.verifyLogin()) {
            response.setStatus(550);
            return (null);
        }

        // OK, logged in , so we can change

        Keymanager keyman = new Keymanager();
        String newpass = json.get("newpass").getAsString();
        String resetpassword = keyman.resetPassword(keys.email, newpass);
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        if (resetpassword == null) {
            logger.debug("Bad password for change");
            response.getWriter().write("{\"auth\":\"Bad password\"}");
            return (null);
        } else {
            String sessionID = json.get("sessionID").getAsString();
            String token = acc.newToken(cookie.JSESSIONID, userid, cid);
            json.add("encr_password", new JsonPrimitive(newpass));
            EncrLib encr = new EncrLib();
            String returndata = " {\"auth\":\"" + encr.encryptAES(json.toString(), token) + "\"} ";
            logger.debug(returndata);
            response.getWriter().write(returndata);
            logger.info("OK Password Change");
            return (returndata);
        }

    }

    /**
     * Authorize login from guest account
     * 
     * @param request
     * @param response
     */
    public void guestlogin(HttpServletRequest request, HttpServletResponse response) {
    	JsonObject login_json = getJSON(request);
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String sessionID = EncrLib.getUUID();
        String cookieID = EncrLib.getUUID();
        Cookie cookie = new Cookie("OxReaderID", cookieID);
        cookie.setHttpOnly(true);
        response.addCookie(cookie);
        String username = login_json.get("username").getAsString();
        try {
            if (CheckBad.isBad(username, Config.bad_password_count)) {
            	response.getWriter().write("{\"auth\":\"lockout\", \"time:\": " + Config.bad_minute_lock + "}");
            	logger.debug("Guest auth lockout for " + username);
                return;
            }
            String password = login_json.get("password").getAsString();
            GuardKeys keys = acc.getKeysFromEmail(username);
            if (keys == null) {
                response.getWriter().write("{\"auth\":\"No Key\"}");
                logger.debug("Guest auth No Key for " + username);
                return;
            }
     //       if (keys.passwordNeeded == true) {
     //           response.getWriter().write(
      //              "{\"auth\":\"Password Needed\", \"userid\":\"" + keys.userid + "\",\"cid\":\"" + keys.contextid + "\",\"sessionID\" : \"" + sessionID + "\"}");
      //          return;
      //      }

            String token = acc.newToken(cookieID, keys.userid, keys.contextid);
            if (token == null) {
                response.getWriter().write("{\"auth\":\"Unable to get token\"}");
                logger.debug("Guest auth Unable to get database token for " + username);
                return;
            }

            String json = "{\"user_id\" : \"" + keys.userid + "\", \"encr_password\" : \"" + password + "\", \"sessionID\" : \"" + sessionID + "\", \"cid\" : \"" + keys.contextid + "\" , \"lastMod\" : \"" + keys.lastup + "\"}";
            EncrLib encr = new EncrLib();
            String returndata = "Bad Password";
            if (keys.getDecodedPrivate(password) != null) {
                // returndata = encr.encryptAES(json, token, cookieID, Config.AESKeyLen);
                returndata = encr.encryptAES(json, token);
                logger.debug("Guest auth good for " + username);
            } else {
            	CheckBad.addBad(username);
            	logger.debug("Guest auth bad for " + username);
            }

            response.getWriter().write(
                " {\"auth\":\"" + returndata + "\", \"userid\":\"" + keys.userid + "\",\"cid\":\"" + keys.contextid + "\",\"sessionID\" : \"" + sessionID + "\", \"lastMod\" : \"" + keys.lastup + "\"}");

        } catch (Exception e) {
            logger.error("Error during guest login  for user " + username, e);
        }
    }

    public void logout(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        acc.deleteToken(cookie.JSESSIONID);
        com.openexchange.guard.util.Core.sendOK(response, "OK");
    }

    public void login(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws IOException {
        try {

            JsonObject json = getJSON(request);
            Settings settings = getPermissions(cookie, request);
            login(json, request, response, cookie, settings);
        } catch (Exception ex) {
            logger.error("Error reading login JSON information", ex);
            response.getWriter().write("{\"auth\":\"BAD FORMAT\"} ");
            return;
        }
    }

    public void login(JsonObject json, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, Settings settings) throws IOException {
        long start = System.currentTimeMillis();
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        response.setContentType("application/json");
        // response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String lang = json.get("lang").getAsString();
        JsonObject returnJson = new JsonObject ();
        returnJson.add("lang", Languages.getJsonAvail(lang));
        int cid = 0;
        try {

            int userid = json.get("user_id").getAsInt();
            cid = json.get("cid").getAsInt();  
            
            // Check lockout of bad attempts.  If excessive, don't give encrypted auth key
            if (CheckBad.isBad(Integer.toString(userid) + "-" + Integer.toString(cid), Config.bad_password_count)) {
            	returnJson.addProperty ("auth", "Lockout");
            	returnJson.add ("settings", getSettings(null, settings));
                response.getWriter().write(returnJson.toString());
                logger.debug("Lockout auth due to bad attempts");
                return;
            }
            if (cid == 0) { // Occasionally, the UI doesn't load the CID.  Get from backend
            	logger.debug("0 cid for userid " + userid);
            	Api ap = new Api(cookie, request);
            	cid = ap.getContextId(); // Returns same 0 if fails
            	json.remove("cid"); // Replace the cid in json
            	json.addProperty("cid", cid);
            	logger.debug("Retrieved context_id of " + cid);
            }
            returnJson.addProperty("cid", cid);
            GuardKeys keys = acc.getKeys(userid, cid);
            if (keys == null) {
            	returnJson.addProperty ("auth", "No Key");
            	returnJson.add ("settings", getSettings(null, settings));
                response.getWriter().write(returnJson.toString());
                logger.debug("Auth sent with No Key for cid = " + cid + " userid = " + userid);
                return;
            }
        	returnJson.add ("settings", getSettings(keys.settings, settings));
            if (keys.passwordNeeded == true) {
            	returnJson.addProperty ("auth", "Password Needed");
                response.getWriter().write(returnJson.toString());
                logger.debug("Auth sent with Password Needed for cid = " + cid + " userid = " + userid);
                return;
            }

            String token = acc.getToken(cookie.JSESSIONID);
            if (token == null) token = acc.newToken(cookie.JSESSIONID, userid, cid);
            if (token == null) {
            	returnJson.addProperty ("auth", "Unable to get token");
                response.getWriter().write(returnJson.toString());
                logger.debug("Auth unable to get token from database for cid = " + cid + " userid = " + userid);
                return;
            }
            EncrLib encr = new EncrLib();
            try {
                String oxpassword = json.get("ox_password") == null ? "" : json.get("ox_password").getAsString();
                if (oxpassword.equals("null")) oxpassword = "";
                String encrpass = json.get("encr_password").getAsString();
                String returndata = "Bad Password";
                if (keys.getEncodedPrivate() == null) {
                    returndata = "NoPrivate";
                } else {
                    if (encrpass.length() > 1) { // If provided with encryption password, check
                        if (keys.getDecodedPrivate(encrpass) != null) {
                            returndata = encr.encryptAES(json.toString(), token);
                            CheckBad.removeBad(Integer.toString(userid) + "-" + Integer.toString(cid)); // remove bad session listing count
                            logger.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
                            // returndata = encr.encryptAES(json.toString(), token, cookie.JSESSIONID, Config.AESKeyLen);
                        } else {
                        	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));  // Increment fail password counter
                        	logger.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                        }
                    } else { // If no encr password, try the oxpassword
                        if (oxpassword.length() > 1) {
                            if (keys.getDecodedPrivate(oxpassword, false) != null) {
                                json.remove("encr_password");
                                json.addProperty("encr_password", oxpassword); // The encryption password is our ox password
                                returndata = encr.encryptAES(json.toString(), token);
                                logger.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
                                // returndata = encr.encryptAES(json.toString(), token, cookie.JSESSIONID, Config.AESKeyLen);
                            } else {
                            	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));  // Increment fail password counter
                            	logger.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                            }
                        }
                    }
                }
                if (json.has("extrapass")) { // File does extra login to encrypt passwords prior to creating links
                	String extrapass = json.get("extrapass").getAsString();
                	if (!extrapass.equals("")) {
                    	String epass = encr.encryptAES(extrapass, cookie.JSESSIONID);
                    	returnJson.addProperty("encrExtra", epass);
                	}
                }
                returnJson.addProperty("auth", returndata);
                response.getWriter().write(returnJson.toString());
                logger.info("Authorization at " + start);
            } catch (Exception e) {
                logger.error("Error during login", e);
                returnJson.addProperty("auth", "BAD FORMAT");
                response.getWriter().write(returnJson.toString());
                return;
            }

        } catch (Exception e) {
            if (e.getMessage().contains("doesn't exist")) {
                if (cid > 0) {
                    Sharding.checkTables(cid, true);
                    logger.info("Created OG tables for cid " + cid);
                }
                returnJson.addProperty("auth", "No Key");
                returnJson.add ("settings", getSettings(null, settings));
                response.getWriter().write(returnJson.toString());
            } else {
                response.getWriter().write("{\"auth\":\"BAD FORMAT\", \"lang\": " + Languages.getJsonAvail(Languages.default_code) + "} ");
                logger.error("Error during login", e);
            }
        }

    }

    /**
     * Create a json of settings, include the individuals settings, plus global
     * 
     * @param settings
     * @return
     */
    private JsonObject getSettings(String settings, Settings setting) {
        StringBuilder allSettings = new StringBuilder();
        if (settings == null)
            settings = "";
        
        // If settings string loaded with key, convert to JSON
        if (!settings.startsWith("{"))
            allSettings.append("{");
        if (settings != null) {
            allSettings.append(settings);
        }
        if (!settings.endsWith("}"))
            allSettings.append("}");
        
        // Convert to Json, then check if global defaults present and add. Individual settings override
        Gson gson = new GsonBuilder().create();
        JsonObject json = gson.fromJson(allSettings.toString(), JsonObject.class);
        if (!json.has("Private")) { // If not defined at user level database, use global settings
        	if (setting.privatepass == false) { // If config cascade value set as false, then use
        		json.add("ChangePrivate", new JsonPrimitive("false"));
        	} else json.add("ChangePrivate", new JsonPrimitive(Config.ok_private ? "true" : "false"));
        }
        if (!json.has("DeleteRecovery")) { // If not defined at user level database, use global settings
        	if (setting.deleteRecovery == false) { // If config cascade value set as false, then use
        		json.add("DeleteRecovery", new JsonPrimitive("false"));
        	} else json.add("DeleteRecovery", new JsonPrimitive(Config.delete_recovery? "true" : "false"));
        }
        if (!json.has("extrapass")) { // If not defined at user level database, use global settings
        	json.add("extrapass", new JsonPrimitive(setting.extrapass ? "true" : "false"));
            
        }

        if (!json.has("deleteOnRevoke")) {
        	json.add("deleteOnRevoke", new JsonPrimitive(setting.deleteKeyOnRevoke ? "true" : "false"));
        }

        if (!json.has("oxguard")) {
            json.add("oxguard", new JsonPrimitive(check(setting, Settings.permissions.MAIL)));
        }

        return (json);
    }

    private String toJson(String name, String value) {
        return ("\"" + name + "\":\"" + value + "\"");
    }

    public static Settings getPermissions(com.openexchange.guard.server.OxCookie cookie, HttpServletRequest request) {
    	Settings settings = new Settings(); 
        settings.perm = Settings.permissions.NONE;
        Api ap = new Api(cookie, request);
        String dat = ap.getCapabilities().toLowerCase();
        if (dat == null) {
        	logger.error("Unable to get permissions from OX backend");
        	return (settings);
        }
        if (dat.contains("guard:mail") || dat.contains("guard-mail"))
            settings.perm = Settings.permissions.MAIL;
        if (dat.contains("guard:drive") || dat.contains("guard-drive")) {
            if (settings.perm == Settings.permissions.MAIL) {
                settings.perm = Settings.permissions.BOTH;
            } else
                settings.perm = Settings.permissions.FILE;
        }
        if (dat.contains("guard-noextra")) settings.extrapass = false;
        if (dat.contains("guard-noprivate")) settings.privatepass = false;
        if (dat.contains("guard-nodeleterecovery")) settings.deleteRecovery = false;
        if (dat.contains("guard-nodeleteonrevoke")) settings.deleteKeyOnRevoke = false;
        return (settings);
    }

    public static boolean checkPermission(com.openexchange.guard.server.OxCookie cookie, HttpServletRequest request, Settings.permissions req) {
        Settings settings = getPermissions(cookie, request);
        return (check(settings, req));
    }

    private static boolean check(Settings current, Settings.permissions req) {
        if (req == current.perm)
            return (true);
        if (current.perm == Settings.permissions.BOTH)
            return (true);
        return (false);
    }
}
