/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.encr;

import java.math.BigInteger;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Date;

import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.crypto.generators.RSAKeyPairGenerator;
import org.bouncycastle.crypto.params.RSAKeyGenerationParameters;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.bc.BcPGPKeyPair;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPKeyConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.config.Config;

/**
 * Class to create PGPKeys
 * 
 * @author greg
 */
public class PgpKeys {
	private static Logger logger = LoggerFactory.getLogger(PgpKeys.class);
    public PgpKeys() {
    }

    public GuardKeys generateKeys(String name, String email, String password) throws Exception {

        // Generate key-pairs
        RSAKeyPairGenerator kpgen = new RSAKeyPairGenerator();
        // 2048 strength. Change for export?
        kpgen.init(new RSAKeyGenerationParameters(BigInteger.valueOf(0x10001), new SecureRandom(), Config.RSAKeyLen, Config.rsa_certainty));

        // Create master (signing) key -- These had to be RSA_GENERAL FOR GNU TO work with the signatures
        PGPKeyPair rsa_master = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
        // SubKey
        PGPKeyPair rsa_encrypt = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
        return (generateKeys(name, email, password, rsa_master, rsa_encrypt));
    }

    /**
     * Generate oxGuard KeyGroup
     * 
     * @param id ID of the user
     * @param password for encryption
     * @return
     * @throws Exception
     */
    public GuardKeys generateKeys(String name, String email, String password, PGPKeyPair rsa_master, PGPKeyPair rsa_encrypt) throws Exception {
        long start = System.currentTimeMillis();
        String salt = EncrLib.getsalt();
        char[] pass = EncrLib.getSHA(password, salt).toCharArray();
        if (password.equals("")) { // If password is blank (unknown), then just create with salt as password
            pass = EncrLib.getSHA(salt, salt).toCharArray();
            password = salt;
        }

        // Add self-signature
        PGPSignatureSubpacketGenerator signhashgenGenerator = new PGPSignatureSubpacketGenerator();

        // This is for master, just signing and certifying
        signhashgenGenerator.setKeyFlags(false, KeyFlags.SIGN_DATA | KeyFlags.CERTIFY_OTHER);
        signhashgenGenerator.setPreferredSymmetricAlgorithms(false, new int[] {
            SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 });

        signhashgenGenerator.setPreferredHashAlgorithms(
            false,
            new int[] {
                HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA1, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA512,
                HashAlgorithmTags.SHA224 });

        // Now for subKey (encrypting)
        PGPSignatureSubpacketGenerator enchashgen = new PGPSignatureSubpacketGenerator();

        enchashgen.setKeyFlags(false, KeyFlags.ENCRYPT_COMMS | KeyFlags.ENCRYPT_STORAGE);

        enchashgen.setPreferredSymmetricAlgorithms(false, new int[] {
            SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 });

        enchashgen.setPreferredHashAlgorithms(
            false,
            new int[] {
                HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA1, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA512,
                HashAlgorithmTags.SHA224 });

        PGPDigestCalculator sha1Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA1);

        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());

        // Encrypting the private key
        PBESecretKeyEncryptor keyEncr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(pass);

        // Generate Keyring with name and email, signed
        PGPKeyRingGenerator keyRingGen = new PGPKeyRingGenerator(
            PGPSignature.POSITIVE_CERTIFICATION,
            rsa_master,
            (name + "<" + email + ">"),
            sha1Calc,
            signhashgenGenerator.generate(),
            null,
            new BcPGPContentSignerBuilder(rsa_master.getPublicKey().getAlgorithm(), HashAlgorithmTags.SHA1),
            keyEncr);

        keyRingGen.addSubKey(rsa_encrypt, enchashgen.generate(), null);

        JcaPGPKeyConverter conv = new JcaPGPKeyConverter();

        // Get the private and public encr RSA keys
        PrivateKey privkey = conv.getPrivateKey(rsa_encrypt.getPrivateKey());
        PublicKey pubkey = conv.getPublicKey(rsa_encrypt.getPublicKey());
        // Save to keys format
        GuardKeys keys = new GuardKeys(keyRingGen, privkey, pubkey, password, salt);
        logger.info("Created PGP/RSA Keys at " + start);
        return keys;

    }

    public PGPSecretKeyRing changePasswordPGP(String oldpass, String newpass, PGPSecretKeyRing keyring) throws NoSuchProviderException, PGPException {
        Security.addProvider(new BouncyCastleProvider());
        PGPPrivateKey oldkey = keyring.getSecretKey().extractPrivateKey(oldpass.toCharArray(), "BC");
        PGPSecretKeyRing newkeyring = keyring.copyWithNewPassword(
            keyring,
            oldpass.toCharArray(),
            newpass.toCharArray(),
            getSymKeyAllgorithm(),
            new SecureRandom(),
            "BC");
        PGPPrivateKey key = newkeyring.getSecretKey().extractPrivateKey(newpass.toCharArray(), "BC");
        return (newkeyring);

    }

    public GuardKeys changePassword(String oldpass, String newpass, GuardKeys oldKey) throws NoSuchProviderException, PGPException {
        PrivateKey priv = oldKey.getDecodedPrivate(oldpass);

        if (priv == null)
            return (null);
        oldKey.setPrivate(priv, newpass);
        oldpass = EncrLib.getSHA(oldpass, oldKey.getSalt());
        newpass = EncrLib.getSHA(newpass, oldKey.getSalt());
        Security.addProvider(new BouncyCastleProvider());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        PGPSecretKeyRing newkeyring = keyring.copyWithNewPassword(
            keyring,
            oldpass.toCharArray(),
            newpass.toCharArray(),
            getSymKeyAllgorithm(),
            new SecureRandom(),
            "BC");
        if (newkeyring == null)
            return (null);
        oldKey.setPGPSecretKeyRing(newkeyring);
        return (oldKey);
    }

    public GuardKeys changePasswordWithRecovery(String oldpassHash, String newpass, GuardKeys oldKey) throws NoSuchProviderException, PGPException {

        PrivateKey priv = oldKey.getDecodedPrivateWithRecovery(oldpassHash);

        if (priv == null)
            return (null);
        oldKey.setPrivate(priv, newpass);
        newpass = EncrLib.getSHA(newpass, oldKey.getSalt());
        Security.addProvider(new BouncyCastleProvider());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        PGPSecretKeyRing newkeyring = keyring.copyWithNewPassword(
            keyring,
            oldpassHash.toCharArray(),
            newpass.toCharArray(),
            getSymKeyAllgorithm(),
            new SecureRandom(),
            "BC");
        if (newkeyring == null)
            return (null);
        oldKey.setPGPSecretKeyRing(newkeyring);
        return (oldKey);
    }

    private int getSymKeyAllgorithm() {
        if (Config.AESKeyLen == 256) {
            return (SymmetricKeyAlgorithmTags.AES_256);
        }
        return (SymmetricKeyAlgorithmTags.AES_128);
    }

    private int getPGPEncryptedData() {
        if (Config.AESKeyLen == 256) {
            return (PGPEncryptedData.AES_256);
        }
        return (PGPEncryptedData.AES_128);
    }

    private int getHashAlgorithmTags() {
        return (HashAlgorithmTags.SHA256);
    }

}
