/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.dbpool;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.config.Config;

/**
 * Handles a pool of database connections for a given database
 * 
 * @author greg
 */
public class Pool {
	
	private static Logger logger = LoggerFactory.getLogger(Pool.class);

    public ArrayList<DbConnection> pool;

    public String URL;

    int maxconn = Config.max_idle_db;

    static {
        try {
            Class.forName("com.mysql.jdbc.Driver");
        } catch (ClassNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    /**
     * Initialize pool of connections for url
     * 
     * @param url mysql url
     */
    public Pool(String url) {
        URL = url;
        pool = new ArrayList<DbConnection>();
        DbConnection con1 = new DbConnection();
        try {
            con1.conn = DriverManager.getConnection(URL);
        } catch (SQLException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        con1.avail = true;
        pool.add(con1);
    }

    /**
     * Called at shutdown to cleanly close all open connections
     */
    public void Shutdown() {
        for (int i = 0; i < pool.size(); i++) {
            try {
                pool.get(i).conn.close();
                pool.get(i).conn = null;
            } catch (SQLException e) {
                // TODO Auto-generated catch block
            }
        }
    }

    /*
     * Get next avail connection from pool
     */
    public synchronized Connection getConn() {
        for (int i = 0; i < pool.size(); i++) {
            if (pool.get(i).avail) {
                DbConnection con = pool.get(i);
                con.avail = false;
                try {
                    if (con.conn == null) {
                        con.conn = openCon();
                    }
                    if (con.conn.isClosed()) {
                        con.conn = openCon();
                    }
                    if (con.conn.isValid(1)) {
                        return (con.conn);
                    } else
                        logger.debug("******************FOUND INVALID CONNECTION, NOT CLOSED");
                } catch (Exception ex) {
                    logger.error("Failed to get connection from pool", ex);
                }
            }
        }
        if (pool.size() >= Config.max_open_db)
            return (null); // If more than allowed open, return null
        DbConnection newcon = new DbConnection();
        try {
            newcon.conn = DriverManager.getConnection(URL);
            newcon.avail = false;
            pool.add(newcon);
        } catch (SQLException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return (newcon.conn);
    }

    /**
     * Close the connection. Remove from pool if over limit
     * 
     * @param con active mysql connection
     * @return
     */
    public synchronized boolean close(Connection con) {
        for (int i = 0; i < pool.size(); i++) {
            if (pool.get(i).conn.equals(con)) {
                DbConnection c = pool.get(i);
                c.avail = true;
                if (pool.size() > maxconn) {
                    pool.remove(i);
                    try {
                        c.conn.close();
                    } catch (SQLException e) {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                    }
                }
                return (true);
            }
        }
        return (false);
    }

    private Connection openCon() {
        try {
            Connection newcon = DriverManager.getConnection(URL);
            logger.debug("Creating new DB connection");
            return (newcon);
        } catch (SQLException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return (null);
    }
}
