/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.dbpool;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.cacheitems.Cache;
import com.openexchange.guard.cacheitems.ContextToDatabase;
import com.openexchange.guard.config.Config;

/**
 * Manages database connections. Pools connections for each database Two entries, one for oxguard, other for user context based database
 * 
 * @author greg
 */
public class Manager {
	private static Logger logger = LoggerFactory.getLogger(Manager.class);

    private static ConcurrentHashMap<String, Pool> pools;

    private static ConcurrentHashMap<String, Pool> readpools;

    private static Pool oxguardpool;

    private static Pool oxguardReadPool;

    private static String mysqlIP = com.openexchange.guard.config.Config.getMysqlIP();// IP address of configdb

    private static String mysqlUsername = com.openexchange.guard.config.Config.getMysqlUsername();

    private static String mysqlPassword = com.openexchange.guard.config.Config.getMysqlPassword();

    private static String OGmysqlIP = com.openexchange.guard.config.Config.getMysqlIPOxGuard();

    private static String OGReadMysqlIP = (com.openexchange.guard.config.Config.ogMySqlReadIP == null) ? com.openexchange.guard.config.Config.getMysqlIPOxGuard() : com.openexchange.guard.config.Config.ogMySqlReadIP;

    public static void init() {
        pools = new ConcurrentHashMap<String, Pool>();
        readpools = new ConcurrentHashMap<String, Pool>();
        String OGmysqlConnectionString = "jdbc:mysql://" + OGmysqlIP + "/oxguard?user=" + mysqlUsername + "&password=" + mysqlPassword;
        oxguardpool = new Pool(OGmysqlConnectionString);
        OGmysqlConnectionString = "jdbc:mysql://" + OGReadMysqlIP + "/oxguard?user=" + // Get readonly connection string. Use write username
                                                                                       // and password if read values null
        ((Config.readmysqlUsername == null) ? mysqlUsername : Config.readmysqlUsername) + "&password=" + ((Config.readmyqlPassword == null) ? mysqlPassword : Config.readmyqlPassword);
        oxguardReadPool = new Pool(OGmysqlConnectionString);
    }
    
    /**
     * Shutdown all connections
     */
    public static void shutdown() {
        oxguardpool.Shutdown();
        oxguardReadPool.Shutdown();
        for (Pool p : pools.values()) {
            p.Shutdown();
        }
        for (Pool p : readpools.values()) {
            p.Shutdown();
        }
    }

    public static ContextDBConnection getOxGuardCon() throws SQLException {
        return (getOxGuardCon(false));
    }

    /*
     * Returns connection from oxguard database pool
     */
    public static ContextDBConnection getOxGuardCon(boolean readonly) throws SQLException {
        Connection conn = null;
        if (readonly) {
            conn = oxguardReadPool.getConn();
        } else
            conn = oxguardpool.getConn();
        if (conn == null) {
            try {
                Thread.sleep(2000);
            } catch (Exception e) {
            }
            if (readonly) {
                conn = oxguardReadPool.getConn();
            } else
                conn = oxguardpool.getConn();
            if (conn == null) {
                try {
                    Thread.sleep(5000);
                } catch (Exception e) {
                }
                if (readonly) {
                    conn = oxguardReadPool.getConn();
                } else
                    conn = oxguardpool.getConn();
            }
            if (conn == null) {
                throw (new SQLException("Exceed max number connections"));
            }
        }
        ContextDBConnection dbcon = new ContextDBConnection();
        dbcon.readonly = readonly;
        dbcon.con = conn;
        dbcon.cid = 0;
        return (dbcon);
    }

    public static synchronized void closeOG(ContextDBConnection con) {
        if (con == null)
            return;
        if (con.readonly) {
            oxguardReadPool.close(con.con);
        } else {
            oxguardpool.close(con.con);
        }
        con = null;
    }

    public static void cleanOG(ResultSet rs, Statement stmt, ContextDBConnection con) {
        if (rs != null) {
            try {
                rs.close();
            } catch (SQLException e) {
                logger.error("SQL error closing a result set", e);
            }
        }
        cleanOG(stmt, con);
    }

    public static void cleanOG(Statement stmt, ContextDBConnection con) {
        if (stmt != null) {
            try {
                stmt.close();
            } catch (SQLException e) {
                logger.error("SQL error closing a statement", e);
            }
        }
        closeOG(con);
    }

    public static ContextDBConnection getConn(int cid) throws SQLException {
        return (getConn(cid, false));
    }

    /*
     * Returns a connection from database pool appropriate for cid
     */
    public static ContextDBConnection getConn(int cid, boolean readonly) throws SQLException {
        if (cid <= 0)
            return (getOxGuardCon());
        Connection conn = getDBPool(cid, readonly).getConn();
        if (conn == null) {
            try {
                Thread.sleep(2000);
            } catch (Exception e) {
            }
            conn = getDBPool(cid, readonly).getConn();
            if (conn == null) {
                try {
                    Thread.sleep(5000);
                } catch (Exception e) {
                }
                conn = getDBPool(cid, readonly).getConn();
                if (conn == null) {
                    throw (new SQLException("Exceed max number connections"));
                }
            }
        }
        ContextDBConnection dbcon = new ContextDBConnection();
        dbcon.con = conn;
        dbcon.cid = cid;
        dbcon.readonly = readonly;
        return (dbcon);
    }

    public static synchronized void close(ContextDBConnection con) {
        if (con == null)
            return;
        if (con.cid <= 0) {
            closeOG(con);
        } else {
            getDBPool(con.cid, con.readonly).close(con.con);
        }
        con = null;
    }

    public static void clean(int cid, ResultSet rs, Statement stmt, ContextDBConnection con) {
        if (rs != null) {
            try {
                rs.close();
            } catch (SQLException e) {
                logger.error("Error closing a result set", e);
            }
        }
        clean(cid, stmt, con);
    }

    public static void clean(int cid, Statement stmt, ContextDBConnection con) {
        if (stmt != null) {
            try {
                stmt.close();
            } catch (SQLException e) {
                logger.error("Error closing a statement", e);
            }
        }
        close(con);
    }

    /**
     * Gets to database pool based on cid Keeps in cache, or looks up if not avail Pooling is based on database_name + poolid
     * 
     * @param cid
     * @return
     */
    private static Pool getDBPool(int cid, boolean readonly) {
        try {
            String ctxUsername = "";
            String ctxPassword = "";
            String db = "";
            String url = "";
            int poolid = 0;
            if (cid < 0) {
                cid = -cid;
            }
            ContextToDatabase info = Cache.getContext(cid, readonly); // Try to load from local cache data
            if (info != null) { // If we have it in cache, go ahead and use
                ctxUsername = info.ctxUsername;
                ctxPassword = info.ctxPassword;
                db = info.database;
                url = info.url;
                poolid = info.dbPool;
                if (readonly) {
                    if (readpools.containsKey(db + poolid)) {
                        return (readpools.get(db + poolid));
                    } else {
                        Pool newpool = new Pool(url);
                        readpools.put(db + poolid, newpool);
                        return (newpool);
                    }
                } else {
                    if (pools.containsKey(db + poolid)) {
                        return (pools.get(db + poolid));
                    } else {
                        Pool newpool = new Pool(url);
                        pools.put(db + poolid, newpool);
                        return (newpool);
                    }
                }

            }
            // Not found in cache, so we need to load from cache

            Connection conn = DriverManager.getConnection(Config.getConfigDbConnectionString());
            Statement stmt = conn.createStatement();
            ResultSet rs;
            if (readonly) {
                rs = stmt.executeQuery("SELECT db.db_pool_id, db.url, db.login, db.password, ctx.db_schema from db_pool AS db INNER JOIN context_server2db_pool AS ctx ON ctx.read_db_pool_id = db.db_pool_id WHERE ctx.cid = " + cid + ";");
            } else
                rs = stmt.executeQuery("SELECT db.db_pool_id, db.url, db.login, db.password, ctx.db_schema from db_pool AS db INNER JOIN context_server2db_pool AS ctx ON ctx.write_db_pool_id = db.db_pool_id WHERE ctx.cid = " + cid + ";");
            if (rs.next()) {
                poolid = rs.getInt("db_pool_id");
                ctxUsername = rs.getString("login");
                ctxPassword = rs.getString("password");
                db = rs.getString("db_schema");
                url = rs.getString("url") + "&user=" + ctxUsername + "&password=" + ctxPassword;
                url = url.replace("localhost", mysqlIP); // If database string just refers to localhost, set it to the configuration IP
                url = url.replace("127.0.0.1", mysqlIP);
                int i = url.indexOf("?");
                if (i < 0) {
                    logger.error("Unalbe to get proper url for cid " + cid);
                    rs.close();
                    stmt.close();
                    conn.close();
                    return (null);
                }
                url = url.substring(0, i - 1) + "/" + db + url.substring(i); // we need to insert database name into the url
            } else {
                rs.close();
                stmt.close();
                conn.close();
                logger.debug("Unable to get db info for cid " + cid);
                return (null);
            }
            rs.close();
            stmt.close();
            conn.close();
            logger.debug("Got " + (readonly ? "Readonly" : "") + " DB info for cid " + cid + " url is " + url);
            checkOxGuard(url, db); // Check to make sure local tables exist for this database
            Cache.saveContext(cid, url, db, ctxUsername, ctxPassword, poolid, readonly); // Save this to cache for future lookup
            Pool newpool = new Pool(url);
            if (readonly) {
                readpools.put(db + poolid, newpool);
            } else
                pools.put(db + poolid, newpool);
            return (newpool);

        } catch (SQLException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            logger.error("Error accessing the dbpool for context " + cid, e);
            return (null);
        }

    }

    /**
     * Checks to make sure oxguard databases exist for that ox database, if not create
     * 
     * @return
     */
    private static boolean checkOxGuard(String url, String db) {

        try {
            if (Cache.isInitialized(db)) { // Check from our local cache if exists
                return (true);
            }
            Connection conn = DriverManager.getConnection(url);
            Statement stmt = conn.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM information_schema.tables WHERE table_schema = '" + db + "' AND table_name = 'og_content_keys' LIMIT 1;");
            if (rs.next()) {
                rs.close();
                stmt.close();
                conn.close();
                Cache.saveInitialized(db); // Save that this database is initialized already
                return (true);
            } else {
                rs.close();
                stmt.close();
                logger.debug("Local oxguard tables not found for " + db + ", creating...");
                createTables(conn);
                Cache.saveInitialized(db);
                // Create tables
            }
            conn.close();
        } catch (Exception ex) {
            logger.error("Problem checking oxguard databases within " + db, ex);
        }

        return (false);
    }

    public static boolean createTables(Connection conn) throws SQLException {

        Statement stmt = conn.createStatement();
        String command = "CREATE TABLE `og_content_keys` ( `ItemId` varchar(38) NOT NULL,  `UserId` int(11) DEFAULT NULL," + " `ContentKey` varchar(400) DEFAULT NULL, `Status` int(11) DEFAULT NULL, `ID` int(11) NOT NULL AUTO_INCREMENT," + " `CID` int(11) DEFAULT NULL, PRIMARY KEY (`ID`), KEY `ID` (`ItemId`)) ENGINE=InnoDB AUTO_INCREMENT=284 DEFAULT CHARSET=utf8;";
        stmt.execute(command);
        command = "CREATE TABLE `og_encrypted_items` ( `Id` varchar(38) NOT NULL,  `Owner` int(11) DEFAULT NULL, " + " `Expiration` bigint(20) DEFAULT NULL, `Type` int(11) DEFAULT NULL, `XML` varchar(3072) DEFAULT NULL, " + " `Owner_cid` int(11) DEFAULT NULL, `Salt` varchar(50) DEFAULT NULL, PRIMARY KEY (`Id`(36))) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
        stmt.execute(command);
        stmt.close();
        return (true);
    }
}
