/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2013 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Julian Bäume <julian.baeume@open-xchange.com>
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 */

define(['io.ox/core/util'], function (util) {

    describe('core/util.js', function () {

        describe('unified display name', function () {

            it('returns empty string when called with null object', function () {
                expect(util.unescapeDisplayName()).to.be.empty;
            });

            it('should remove trailing white-space', function () {
                expect(util.unescapeDisplayName(' Hello World  ')).to.equal('Hello World');
            });

            it('should remove surrounding brackets', function () {
                expect(util.unescapeDisplayName('""Hello World""')).to.equal('Hello World');
            });

            it('should remove escaped brackets', function () {
                expect(util.unescapeDisplayName('\"Hello World\"')).to.equal('Hello World');
            });

            it('should not remove brackets that are not surrounding', function () {
                expect(util.unescapeDisplayName('Hello "World"')).to.equal('Hello "World"');
            });

            it('should remove escaping backslashes before brackets', function () {
                expect(util.unescapeDisplayName('"Say \"Hello\""')).to.equal('Say "Hello"');
            });
        });

        describe('isValidMailAddress()', function () {

            it('handles domain part properly', function () {
                expect(util.isValidMailAddress('name@domain.com')).to.be.true;
                expect(util.isValidMailAddress('name@host')).to.be.true;
                expect(util.isValidMailAddress('name@1337')).to.be.true;
                expect(util.isValidMailAddress('name@[1.2.3.4]')).to.be.true;
                expect(util.isValidMailAddress('name@[1.2.3.4.5]')).to.be.false;
                expect(util.isValidMailAddress('name@[1.2.3.A]')).to.be.false;
                expect(util.isValidMailAddress('name@[1.2.3.4444]')).to.be.false;
                expect(util.isValidMailAddress('name@[IPv6:2001:db8:1ff::a0b:dbd0]')).to.be.true;
                expect(util.isValidMailAddress('name@[2001:db8:1ff::a0b:dbd0]')).to.be.false;
            });

            it('handles partial addresses properly', function () {
                expect(util.isValidMailAddress('@domain.com')).to.be.false;
                expect(util.isValidMailAddress('name@')).to.be.false;
                expect(util.isValidMailAddress('@')).to.be.false;
            });

            it('handles local part properly', function () {
                expect(util.isValidMailAddress('name@abc@domain.com')).to.be.false;
                expect(util.isValidMailAddress('first.last@domain.com')).to.be.true;
                expect(util.isValidMailAddress('first,last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('first last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('first\\last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('first"last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('first..last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('.first.last@domain.com')).to.be.false;
                expect(util.isValidMailAddress('"quoted"@domain.com')).to.be.true;
                expect(util.isValidMailAddress('"another@"@domain.com')).to.be.true;
                expect(util.isValidMailAddress('"but"not"@domain.com')).to.be.false;
            });
        });

        describe('isValidPhoneNumber()', function () {

            it('handles empty strings properly', function () {
                expect(util.isValidPhoneNumber('')).to.be.true;
                expect(util.isValidPhoneNumber(' ')).to.be.true;
            });

            it('handles too short numbers properly', function () {
                expect(util.isValidPhoneNumber('+1')).to.be.false;
                expect(util.isValidPhoneNumber('+49')).to.be.false;
            });

            it('handles numbers properly', function () {
                expect(util.isValidPhoneNumber('01234567')).to.be.true;
                expect(util.isValidPhoneNumber('0123 4567')).to.be.true;
                expect(util.isValidPhoneNumber('+491234567')).to.be.true;
                expect(util.isValidPhoneNumber('0123+4567')).to.be.false;
                expect(util.isValidPhoneNumber('+49 (0) 1234567')).to.be.true;
                expect(util.isValidPhoneNumber('+49 0 1234567')).to.be.true;
                expect(util.isValidPhoneNumber('+49-0-1234567')).to.be.true;
                expect(util.isValidPhoneNumber('+49-0-1234567#1')).to.be.true;
                expect(util.isValidPhoneNumber('+49-0-1234567,1,2')).to.be.true;
                expect(util.isValidPhoneNumber('+49.0.1234567')).to.be.true;
                expect(util.isValidPhoneNumber('+49 0 / 1234567')).to.be.true;
                expect(util.isValidPhoneNumber('+49 0 / 123456 - 7')).to.be.true;
                expect(util.isValidPhoneNumber('+49 0 / 123456 - ABC')).to.be.false;
                expect(util.isValidPhoneNumber('+49 0::1234567')).to.be.false;
                expect(util.isValidPhoneNumber('+49 0 1234 [567]')).to.be.false;
                expect(util.isValidPhoneNumber('++49 0 1234567')).to.be.false;
                expect(util.isValidPhoneNumber('+49_0_1234567')).to.be.false;
                expect(util.isValidPhoneNumber('+49 0 1234567 \\ 23')).to.be.false;
            });
        });

        describe('breakableHTML()', function () {

            it('doesnt change white space', function () {
                expect(util.breakableHTML('')).to.be.empty;
                expect(util.breakableHTML(' ')).to.equal(' ');
            });

            it('doesnt change short strings', function () {
                expect(util.breakableHTML('Hello World')).to.equal('Hello World');
            });

            it('escapes HTML', function () {
                expect(util.breakableHTML('Hello<br>World')).to.equal('Hello&lt;br&gt;World');
            });

            it('breaks longs strings properly', function () {
                expect(util.breakableHTML('com.openexchange.session.contextId=1337')).to.equal('com.<wbr>openexchange.<wbr>session.<wbr>contextId=<wbr>1337<wbr>');
                expect(util.breakableHTML('com.openexchange 01234567890123456789 01234567890123456789')).to.equal('com.openexchange 01234567890123456789 01234567890123456789');
                expect(util.breakableHTML('com.openexchange.0123456789012345678901234567890123456789')).to.equal('com.<wbr>openexchange.<wbr>012345678901234567890123456789<wbr>0123456789');
            });
        });
    });
});
