/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2011 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Dickhaus <daniel.dickhaus@open-xchange.com>
 */

define('io.ox/tasks/actions',
    ['io.ox/core/extensions',
     'io.ox/tasks/util',
     'io.ox/core/extPatterns/links',
     'gettext!io.ox/tasks',
     'io.ox/core/notifications',
     'io.ox/core/print',
     'io.ox/core/extPatterns/actions',
     'io.ox/tasks/common-extensions'
    ], function (ext, util, links, gt, notifications, print, actions, extensions) {

    'use strict';

    //  actions
    var Action = links.Action;

    new Action('io.ox/tasks/actions/create', {
        requires: function (e) {
            return e.collection.has('create');// && _.device('!small');
        },
        action: function (baton) {
            ox.load(['io.ox/tasks/edit/main']).done(function (edit) {
                edit.getApp().launch({ folderid: baton.app.folder.get()});
            });
        }
    });

    new Action('io.ox/tasks/actions/edit', {
        requires: function (e) {
            return e.collection.has('one');
        },
        action: function (baton) {
            ox.load(['io.ox/tasks/edit/main']).done(function (m) {
                if (m.reuse('edit', baton.data)) return;
                m.getApp().launch({ taskData: baton.data });
            });
        }
    });

    new Action('io.ox/tasks/actions/delete', {
        requires: 'some delete',
        action: function (baton) {
            var data = baton.data,
                numberOfTasks = data.length || 1;
            ox.load(['io.ox/core/tk/dialogs']).done(function (dialogs) {
                //build popup
                var popup = new dialogs.ModalDialog({async: true})
                    .addPrimaryButton('deleteTask', gt('Delete'), 'deleteTask', {tabIndex: '1'})
                    .addButton('cancel', gt('Cancel'), 'cancel', {tabIndex: '1'});
                //Header
                popup.getBody()
                    .append($('<h4>')
                            .text(gt.ngettext('Do you really want to delete this task?',
                                              'Do you really want to delete this tasks?', numberOfTasks)));
                //go
                popup.show();
                popup.on('deleteTask', function () {
                    require(['io.ox/tasks/api'], function (api) {
                        api.remove(data)
                            .done(function () {
                                notifications.yell('success', gt.ngettext('Task has been deleted!',
                                                                          'Tasks have been deleted!', numberOfTasks));
                                popup.close();
                            }).fail(function (result) {
                                if (result.code === 'TSK-0019') { //task was already deleted somewhere else. everythings fine, just show info
                                    notifications.yell('info', gt('Task was already deleted!'));
                                    popup.close();
                                } else if (result.error) {//there is an error message from the backend
                                    popup.idle();
                                    popup.getBody().empty().append($.fail(result.error, function () {
                                        popup.trigger('deleteTask', data);
                                    })).find('h4').remove();
                                } else {//show generic error message
                                    //show retrymessage and enable buttons again
                                    popup.idle();
                                    popup.getBody().empty().append($.fail(gt.ngettext('The task could not be deleted.',
                                                                              'The tasks could not be deleted.', numberOfTasks), function () {
                                        popup.trigger('deleteTask', data);
                                    })).find('h4').remove();
                                }
                            });
                    });
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/done', {
        requires: function (e) {
            if (!e.collection.has('some')) {
                return false;
            }
            return (e.baton.data.status !== 3);
        },
        action: function (baton) {
            changeState(baton, 1);
        }
    });

    new Action('io.ox/tasks/actions/undone', {
        requires: function (e) {
            if (!e.collection.has('some')) {
                return false;
            }
            return (e.baton.data.length  !== undefined || e.baton.data.status === 3);
        },
        action: function (baton) {
            changeState(baton, 3);
        }
    });

    function changeState(baton, state) {
        var mods,
            data = baton.data;
        if (state === 3) {
            mods = {label: gt('Undone'),
                    data: {status: 1,
                           percent_completed: 0,
                           date_completed: null
                          }
                   };
        } else {
            mods = {label: gt('Done'),
                    data: {status: 3,
                           percent_completed: 100,
                           date_completed: _.now()
                          }
                   };
        }
        require(['io.ox/tasks/api'], function (api) {
            if (data.length > 1) {
                api.updateMultiple(data, mods.data)
                    .done(function () {
                        _(data).each(function (item) {
                            //update detailview
                            api.trigger('update:' + _.ecid(item));
                        });

                        notifications.yell('success', mods.label);
                    })
                    .fail(function (result) {
                        notifications.yell('error', gt.noI18n(result));
                    });
            } else {
                mods.data.id = data.id;
                mods.data.folder_id = data.folder_id || data.folder;
                api.update(mods.data)
                    .done(function () {
                        notifications.yell('success', mods.label);
                    })
                    .fail(function (result) {
                        var errorMsg = gt('A severe error occurred!');
                        if (result.code === 'TSK-0007') {//task was modified before
                            errorMsg = gt('Task was modified before, please reload');
                        }
                        notifications.yell('error', errorMsg);
                    });
            }
        });
    }

    new Action('io.ox/tasks/actions/move', {
        requires: 'some delete',
        multiple: function (list, baton) {
            var task = baton.data,
                numberOfTasks = task.length || 1,
                vGrid = baton.grid || (baton.app && baton.app.getGrid());
            ox.load(['io.ox/core/tk/dialogs', 'io.ox/core/tk/folderviews', 'io.ox/tasks/api', 'io.ox/core/api/folder'])
                    .done(function (dialogs, views, api, folderAPI) {

                function commit(target) {
                    if (vGrid) vGrid.busy();
                    api.move(list, target).then(
                        function () {
                            notifications.yell('success', gt('Tasks have been moved'));
                            folderAPI.reload(target, list);
                            if (vGrid) vGrid.idle();
                        },
                        notifications.yell
                    );
                }

                if (baton.target) {
                    commit(baton.target);
                } else {

                    //build popup
                    var popup = new dialogs.ModalDialog()
                        .header($('<h4>').text(gt('Move')))
                        .addPrimaryButton('ok', gt('Move'), 'ok', {tabIndex: '1'})
                        .addButton('cancel', gt('Cancel'), 'cancel', {tabIndex: '1'});
                    popup.getBody().css({ height: '250px' });
                    var tree = new views.FolderList(popup.getBody(), {
                            type: 'tasks',
                            tabindex: 0,
                            targetmode: true,
                            dialogmode: true
                        }),
                        id = String(task.folder || task.folder_id);

                    popup.show(function () {
                        tree.paint().done(function () {
                            tree.select(id).done(function () {
                                popup.getBody().focus();
                            });
                        });
                    })
                    .done(function (action) {
                        if (action === 'ok') {
                            var node = $('.io-ox-multi-selection');
                            node.hide();
                            node.parent().busy();
                            var target = _(tree.selection.get()).first();
                            // move only if folder differs from old folder
                            if (target && target !== id) {
                                // move action
                                api.move(task, target)
                                .done(function () {
                                    node.show();
                                    node.parent().idle();
                                    notifications.yell('success', gt.ngettext('Task moved.', 'Tasks moved.', numberOfTasks));
                                })
                                .fail(function () {
                                    node.show();
                                    node.parent().idle();
                                    notifications.yell('error', gt('A severe error occurred!'));
                                });
                            }
                        }
                        tree.destroy().done(function () {
                            tree = popup = null;
                        });
                    });
                }
            });
        }
    });

    new Action('io.ox/tasks/actions/confirm', {
        id: 'confirm',
        requires: function (args) {
            var result = false;
            if (args.baton.data.participants) {
                var userId = ox.user_id;
                _(args.baton.data.participants).each(function (participant) {
                    if (participant.id === userId) {
                        result = true;
                    }
                });
                return result;
            }
            return result;
        },
        action: function (baton) {
            var data = baton.data;
            ox.load(['io.ox/calendar/acceptdeny', 'io.ox/tasks/api']).done(function (acceptdeny, api) {
                acceptdeny(data, {
                    taskmode: true,
                    api: api,
                    callback: function () {
                        //update detailview
                        api.trigger('update:' + _.ecid({id: data.id, folder_id: data.folder_id}));
                    }
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/print', {
        requires: function (e) {
            return e.collection.has('some', 'read') && _.device('!small');
        },
        multiple: function (list) {
            print.request('io.ox/tasks/print', list);
        }
    });

    new Action('io.ox/tasks/actions/print-disabled', {
        id: 'print',
        requires: function () {
            return _.device('!small');
        },
        multiple: function (list) {
            if (list.length === 1) {
                print.open('tasks', list[0], { template: 'infostore://12496', id: list[0].id, folder: list[0].folder_id || list[0].folder }).print();
            } else if (list.length > 1) {
                ox.load(['io.ox/core/http']).done(function (http) {
                    var win = print.openURL();
                    win.document.title = gt('Print tasks');
                    http.PUT({
                        module: 'tasks',
                        params: {
                            action: 'list',
                            template: 'infostore://12500',
                            format: 'template',
                            columns: '200,201,202,203,220,300,301,302,303,305,307,308,309,312,313,314,315,221,226',
                            timezone: 'UTC'
                        },
                        data: http.simplify(list)
                    }).done(function (result) {
                        var content = $('<div>').append(result),
                            head = $('<div>').append(content.find('style')),
                            body = $('<div>').append(content.find('.print-tasklist'));
                        win.document.write(head.html() + body.html());
                        win.print();
                    });
                });

            }
        }
    });

    //attachment actions
    new links.Action('io.ox/tasks/actions/slideshow-attachment', {
        id: 'slideshow',
        requires: function (e) {
            return e.collection.has('multiple') && _(e.context).reduce(function (memo, obj) {
                return memo || (/\.(gif|bmp|tiff|jpe?g|gmp|png)$/i).test(obj.filename);
            }, false);
        },
        multiple: function (list) {
            require(['io.ox/core/api/attachment', 'io.ox/files/carousel'], function (attachmentAPI, slideshow) {
                var files = _(list).map(function (file) {
                    return {
                        url: attachmentAPI.getUrl(file, 'open'),
                        filename: file.filename
                    };
                });
                slideshow.init({
                    baton: {allIds: files},
                    attachmentMode: false,
                    selector: '.window-container.io-ox-tasks-window'
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/preview-attachment', {
        id: 'preview',
        requires: function (e) {
            return require(['io.ox/preview/main'])
                .pipe(function (p) {
                    var list = _.getArray(e.context);
                    // is at least one attachment supported?
                    return e.collection.has('some') && _(list).reduce(function (memo, obj) {
                        return memo || new p.Preview({
                            filename: obj.filename,
                            mimetype: obj.content_type
                        })
                        .supportsPreview();
                    }, false);
                });
        },
        multiple: function (list, baton) {
            ox.load(['io.ox/core/tk/dialogs',
                     'io.ox/preview/main',
                     'io.ox/core/api/attachment']).done(function (dialogs, p, attachmentAPI) {
                //build Sidepopup
                new dialogs.SidePopup({ tabTrap: true }).show(baton.e, function (popup) {
                    _(list).each(function (data) {
                        data.dataURL = attachmentAPI.getUrl(data, 'view');
                        var pre = new p.Preview(data, {
                            width: popup.parent().width(),
                            height: 'auto'
                        });
                        if (pre.supportsPreview()) {
                            popup.append(
                                $('<h4>').text(data.filename)
                            );
                            pre.appendTo(popup);
                            popup.append($('<div>').text('\u00A0'));
                        }
                    });
                    if (popup.find('h4').length === 0) {
                        popup.append($('<h4>').text(gt('No preview available')));
                    }
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/open-attachment', {
        id: 'open',
        requires: 'one',
        multiple: function (list) {
            ox.load(['io.ox/core/api/attachment']).done(function (attachmentAPI) {
                _(list).each(function (data) {
                    var url = attachmentAPI.getUrl(data, 'open');
                    window.open(url);
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/download-attachment', {
        id: 'download',
        requires: 'some',
        multiple: function (list) {
            ox.load(['io.ox/core/api/attachment', 'io.ox/core/download']).done(function (attachmentAPI, download) {
                _(list).each(function (data) {
                    var url = attachmentAPI.getUrl(data, 'download');
                    download.url(url);
                });
            });
        }
    });

    new Action('io.ox/tasks/actions/save-attachment', {
        id: 'save',
        capabilities: 'infostore',
        requires: 'some',
        multiple: function (list) {
            ox.load(['io.ox/core/api/attachment']).done(function (attachmentAPI) {
                //cannot be converted to multiple request because of backend bug (module overides params.module)
                _(list).each(function (data) {
                    attachmentAPI.save(data);
                });
                setTimeout(function () {notifications.yell('success', gt('Attachments have been saved!')); }, 300);
            });
        }
    });

    // toolbar

    new links.ActionGroup('io.ox/tasks/links/toolbar', {
        id: 'default',
        index: 100,
        icon: function () {
            return $('<i class="fa fa-plus accent-color">');
        }
    });

    new links.ActionLink('io.ox/tasks/links/toolbar/default', {
        index: 100,
        id: 'create',
        label: gt('Create new task'),
        ref: 'io.ox/tasks/actions/create'
    });

    //inline
    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'edit',
        index: 100,
        prio: 'hi',
        mobile: 'hi',
        label: gt('Edit'),
        ref: 'io.ox/tasks/actions/edit'
    }));

    //strange workaround because extend only takes new links instead of plain objects with draw method
    new Action('io.ox/tasks/actions/placeholder', {
        action: $.noop
    });

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'changeDueDate',
        index: 200,
        prio: 'hi',
        mobile: 'lo',
        ref: 'io.ox/tasks/actions/placeholder',
        draw: function (baton) {

            var link = $('<a href="#">').text(gt('Change due date'));

            this.append(
                $('<span class="io-ox-action-link">').append(link)
            );

            extensions.dueDate.call(link, baton);
        }
    }));

    // delete tasks
    ext.point('io.ox/tasks/mobileMultiSelect/toolbar').extend({
        id: 'delete',
        index: 10,
        draw: function (data) {
            var baton = new ext.Baton({data: data.data});
            $(this).append($('<div class="toolbar-button">')
                .append($('<a href="#">')
                    .append(
                        $('<i class="fa fa-trash-o">')
                            .on('click', {grid: data.grid}, function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                actions.invoke('io.ox/tasks/actions/delete', null, baton);
                                e.data.grid.selection.clear();
                            })
                    )
                )
            );
        }
    });

    // tasks done
    ext.point('io.ox/tasks/mobileMultiSelect/toolbar').extend({
        id: 'done',
        index: 20,
        draw: function (data) {
            var baton = new ext.Baton({data: data.data});
            $(this).append($('<div class="toolbar-button">')
                .append($('<a href="#">')
                    .append(
                        $('<i class="fa fa-check-square-o">')
                            .on('click', {grid: data.grid}, function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                actions.invoke('io.ox/tasks/actions/done', null, baton);
                                e.data.grid.selection.clear();
                            })
                    )
                )
            );
        }
    });

    // tasks undone
    ext.point('io.ox/tasks/mobileMultiSelect/toolbar').extend({
        id: 'unDone',
        index: 30,
        draw: function (data) {
            var baton = new ext.Baton({data: data.data});
            $(this).append($('<div class="toolbar-button">')
                .append($('<a href="#">')
                    .append(
                        $('<i class="fa fa-square-o">')
                            .on('click', {grid: data.grid}, function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                actions.invoke('io.ox/tasks/actions/undone', null, baton);
                                e.data.grid.selection.clear();
                            })
                    )
                )
            );
        }
    });

    // tasks move
    ext.point('io.ox/tasks/mobileMultiSelect/toolbar').extend({
        id: 'move',
        index: 40,
        draw: function (data) {
            var baton = new ext.Baton({data: data.data});
            $(this).append($('<div class="toolbar-button">')
                .append($('<a href="#">')
                    .append(
                        $('<i class="fa fa-sign-in">')
                            .on('click', {grid: data.grid}, function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                actions.invoke('io.ox/tasks/actions/move', null, baton);
                                e.data.grid.selection.clear();
                            })
                    )
                )
            );
        }
    });

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'done',
        index: 300,
        prio: 'hi',
        mobile: 'hi',
        icon: 'fa fa-check-square-o',
        label: gt('Mark as done'),
        ref: 'io.ox/tasks/actions/done'
    }));

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'unDone',
        index: 310,
        prio: 'hi',
        mobile: 'hi',
        label: gt('Undone'),
        ref: 'io.ox/tasks/actions/undone'
    }));

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'delete',
        index: 400,
        prio: 'hi',
        mobile: 'hi',
        icon: 'fa fa-trash-o',
        label: gt('Delete'),
        ref: 'io.ox/tasks/actions/delete'
    }));

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'move',
        index: 500,
        prio: 'lo',
        mobile: 'lo',
        label: gt('Move'),
        ref: 'io.ox/tasks/actions/move'
    }));

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'confirm',
        index: 600,
        prio: 'lo',
        mobile: 'lo',
        label: gt('Change confirmation status'),
        ref: 'io.ox/tasks/actions/confirm'
    }));

    ext.point('io.ox/tasks/links/inline').extend(new links.Link({
        id: 'print',
        index: 700,
        prio: 'lo',
        mobile: 'lo',
        label: gt('Print'),
        ref: 'io.ox/tasks/actions/print'
    }));

    // Attachments
    ext.point('io.ox/tasks/attachment/links').extend(new links.Link({
        id: 'slideshow',
        index: 100,
        label: gt('Slideshow'),
        ref: 'io.ox/tasks/actions/slideshow-attachment'
    }));

    ext.point('io.ox/tasks/attachment/links').extend(new links.Link({
        id: 'preview',
        index: 100,
        label: gt('Preview'),
        ref: 'io.ox/tasks/actions/preview-attachment'
    }));

    ext.point('io.ox/tasks/attachment/links').extend(new links.Link({
        id: 'open',
        index: 200,
        label: gt('Open in browser'),
        ref: 'io.ox/tasks/actions/open-attachment'
    }));

    ext.point('io.ox/tasks/attachment/links').extend(new links.Link({
        id: 'download',
        index: 300,
        label: gt('Download'),
        ref: 'io.ox/tasks/actions/download-attachment'
    }));

    ext.point('io.ox/tasks/attachment/links').extend(new links.Link({
        id: 'save',
        index: 400,
        label: gt('Save to Drive'),
        ref: 'io.ox/tasks/actions/save-attachment'
    }));
});
