/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.mapping.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.mapping.UIDStorage;
import com.openexchange.usm.util.Toolkit;

public class ObjectUIDMapping extends AbstractMappingDBAccess implements UIDStorage {

	public ObjectUIDMapping(Log journal, Session session) {
		super(journal, session);
	}

	public void delete(String folderID, String... longID) throws DatabaseAccessException, USMSQLException {
		deleteOrRetainInFolder("IN (", folderID, longID);
	}

	public void retain(String folderID, String... longID) throws DatabaseAccessException, USMSQLException {
		deleteOrRetainInFolder("NOT IN (", folderID, longID);
	}

	public void deleteAllInFolders(String... folderID) throws DatabaseAccessException, USMSQLException {
		deleteOrRetainAllInFolders("IN (", folderID);
	}

	public void retainAllInFolders(String... folderID) throws DatabaseAccessException, USMSQLException {
		deleteOrRetainAllInFolders("NOT IN (", folderID);
	}

	public String get(String folderID, int shortID) throws USMSQLException, DatabaseAccessException {
		return get(MappingSQLStatements.SELECT_LONG_ID_UID_MAPPING + MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER
				+ " and shortID = ? and folderID = ?", folderID, shortID);
	}

	public String get(int shortID) throws USMSQLException, DatabaseAccessException {
		return get(MappingSQLStatements.SELECT_LONG_ID_UID_MAPPING + MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER
				+ " and shortID = ?", null, shortID);
	}

	private String get(String command, String folderID, int shortID) throws USMSQLException, DatabaseAccessException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		ResultSet r = null;
		try {
			con = _session.getReadOnlyDBConnection();
			statement = con.prepareStatement(command);
			addSessionIdentifierParamsToStmt(statement);
			statement.setInt(NUMBER_OF_SESSION_ID_FIELDS + 1, shortID);
			if (folderID != null)
				statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 2, folderID);
			r = statement.executeQuery();
			String result = r.next() ? r.getString(1) : null;
			return result;
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error retrieving the UID for an object";
			_journal.error(errorMessage, e);
			throw new USMSQLException(MappingBundleErrorCodes.GET_FROM_CACHE_FAILED, errorMessage, e);
		} finally {
			Toolkit.close(r);
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	public void store(String folderID, int shortID, String longID) throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _session.getWritableDBConnection();
			statement = con.prepareStatement(MappingSQLStatements.FULL_INSERT_UID_MAPPING);
			addSessionIdentifierParamsToStmt(statement);
			statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, folderID);
			statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 2, longID);
			statement.setInt(NUMBER_OF_SESSION_ID_FIELDS + 3, shortID);
			if (statement.executeUpdate() == 0)
				_journal.warn(_session + " Storing UID did not update the DB");
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error storing the UID for an object";
			_journal.error(errorMessage, e);
			throw new USMSQLException(MappingBundleErrorCodes.STORE_FAILED, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	private void deleteOrRetainInFolder(String extraCommandText, String folderID, String[] longID)
			throws DatabaseAccessException, USMSQLException {
		if (folderID == null)
			throw new IllegalArgumentException("Missing folderID");
		deleteOrRetain("SQL error while deleting UID mappings in folder", MappingSQLStatements.DELETE_UID_MAPPIMG
				+ MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and folderID = ? and longID ", extraCommandText,
				folderID, longID);
	}

	private void deleteOrRetainAllInFolders(String extraCommandText, String[] folderID) throws DatabaseAccessException,
			USMSQLException {
		deleteOrRetain("SQL error while deleting all UID mappings in folders", MappingSQLStatements.DELETE_UID_MAPPIMG
				+ MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and folderID ", extraCommandText, null, folderID);
	}
}
