/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.ListIterator;
import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link RealtimeHelper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class ConnectionHelper {

    final static private String m_actionsKey = "actions";
    final static private String m_operationsKey = "operations";


    /**
     * Removing an item from a JSONArray at a specified position. This is necessary, because the 'remove()' function is not always available
     * to JSONArrays. This function creates a local JSONArray, that is filled with the values of the JSONArray, that is a parameter of it. A
     * second parameter is required to specify, which element shall be removed. Its value must be between 0 and the length of the JSONArray
     * - 1. The locally created JSONArray is returnd from this function.
     *
     * @param {JSONArray} arr The array, from which one element shall be removed
     * @param {int} pos The position of the item, that shall be removed from the array. This value must be between 0 and arr.length - 1
     * @return {JSONArray} If a problem occurred, the JSONArray will not be modified and returned. Otherwise a reduced version of the
     *         JSONArray is returned, that does not contain the specified item.
     */
    static private JSONArray removeItemFromJSONArray(JSONArray arr, int pos) {

        if ((arr == null) || (arr.length() - 1 < pos)) {
            return arr;
        }

        JSONArray newArr = new JSONArray();

        for (int i = 0; i < arr.length(); i++) {

            if (i != pos) {
                try {
                    newArr.put(arr.get(i));
                } catch (JSONException e) {
                    return arr;
                }
            }
        }

        return newArr;
    }

    /**
     * Comparing, if two positions in the form of JSONArrays have the same parents, grand parents, ... parentLevel = 1 means, both positions
     * have the same parent. parentLevel = 2 means, the grand parents are identical and so on.
     *
     * @param {JSONArray} position1 The first position to be compared
     * @param {JSONArray} position2 The second position to be compared
     * @param {int} parentLevel The level, until which the ancestors will be compared. parentLevel = 1 means, that the direct parents must be
     *        identical.
     * @return {boolean} If the two positions have the same ancestor until the specified level, true is returned. Otherwise false
     */
    static private boolean hasSameParentComponent(JSONArray position1, JSONArray position2, int parentLevel) {

        boolean sameParent = true;

        if ((position1 == null) || (position2 == null)) {
            return false;
        }

        int length = position1.length();

        if ((length < parentLevel) || (length != position2.length())) {
            return false;
        }

        for (int index = length - parentLevel - 1; index >= 0; index -= 1) {

            try {
                if (position1.getInt(index) != position2.getInt(index)) {
                    return false;
                }
            } catch (JSONException e) {
                return false;
            }
        }

        return sameParent;
    }

    /**
     * Trying to merge two directly following insertText operations. If the merge is possible the operation 'next' that follows the
     * operation 'last' must not be added to the operation array. Therefore the return value 'insertNextOp' will be set to false in the case
     * of successful merging. In this case the content of the operation 'last' will be modified.
     *
     * @param {JSONObject} last The preceding insertText operation. This was already added to the operations array. Therefore it will be
     *        modified within this operation. And therefore it is necessary, that the insertText operations are copied before they are added
     *        to the operations array, so that the original operations are not modified.
     * @param {JSONObject} position2 The following insertText operation. If merging this two operations is possible, then this operation will
     *        not be added to the operations array. Therefore the return value of this function is set to 'insertNextOp' false.
     * @return {boolean} If the two insertText operations can be merged, it is not necessary to add the second operation to the operations
     *         array. Therefore the return value will be false. Otherwise it is true.
     * @throws JSONException
     */
    static private boolean mergeInsertOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;

        int parentLevel = 1;
        String nextText = next.getString("text");
        JSONArray nextStart = next.getJSONArray("start");
        JSONArray lastStart = last.getJSONArray("start");

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);

            JSONObject nextAttrs = null;

            try {
                nextAttrs = next.getJSONObject("attrs");
            } catch (Exception e) {
                nextAttrs = null;
            }

            if (nextAttrs == null) {

                String lastText = last.getString("text");

                if ((lastTextPosStart + lastText.length() == nextTextPosStart)) {
                    // the new text is inserted directly behind an already existing text
                    last.put("text", lastText + nextText);
                    insertNextOp = false;
                } else if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosStart <= lastTextPosStart + lastText.length())) {
                    // the new text is inserted into an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    last.put("text", lastText.substring(0, relativeStart) + nextText + lastText.substring(relativeStart));
                    insertNextOp = false;
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Trying to merge an insertText operation with a directly following delete operation. If the merge is possible the 'delete' operation
     * 'next' that follows the insertText operation 'last' must not be added to the operation array. Therefore the return value
     * 'insertNextOp' will be set to false in the case of successful merging. In this case the content of the operation 'last' will be
     * modified.
     *
     * @param {JSONObject} last The preceding insertText operation. This was already added to the operations array. Therefore it will be
     *        modified within this operation. And therefore it is necessary, that the insertText operations are copied before they are added
     *        to the operations array, so that the original operations are not modified.
     * @param {JSONObject} position2 The following delete operation. If merging this two operations is possible, then this operation will not
     *        be added to the operations array. Therefore the return value of this function is set to 'insertNextOp' false.
     * @return {boolean} If the 'insertText' and the 'delete' operations can be merged, it is not necessary to add the 'delete' operation to
     *         the operations array. Therefore the return value will be false. Otherwise it is true.
     * @throws JSONException
     */
    static private boolean mergeInsertAndDeleteOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;
        int parentLevel = 1;
        JSONArray nextStart = next.getJSONArray("start");
        JSONArray lastStart = last.getJSONArray("start");
        // delete has an optional end parameter
        JSONArray nextEnd = null;

        try {
            nextEnd = next.getJSONArray("end");
        } catch (Exception e) {
            nextEnd = null;
        }

        if (nextEnd == null) {
            nextEnd = nextStart;
        }

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel)) && (hasSameParentComponent(
            nextStart,
            nextEnd,
            parentLevel))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);
            int nextTextPosEnd = nextEnd.getInt(nextEnd.length() - 1);
            String lastText = last.getString("text");

            if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosEnd < lastTextPosStart + lastText.length())) {
                // using 'nextTextPosEnd < lastTextPosStart + lastText.length()', not '<=' !
                // insertText at [0,3], Text 'abc' -> length is 3.
                // deleteText 'abc' is delete from [0,3] to [0,5], not till [0,6]!

                try {
                    // the deleted text is part of an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    int relativeEnd = nextTextPosEnd - lastTextPosStart + 1;
                    String newText = "";
                    if ((relativeStart > 0) && (relativeStart <= lastText.length())) {
                        newText = lastText.substring(0, relativeStart);
                    }
                    if ((relativeEnd >= 0) && (relativeEnd < lastText.length())) {
                        newText += lastText.substring(relativeEnd);
                    }

                    last.put("text", newText);
                    insertNextOp = false;

                } catch (IndexOutOfBoundsException e) {
                    //
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Optimizing the operations by trying to merge an insertText operation with a directly following insertText or delete operation.
     *
     * @param {JSONObject} jsonResult The JSONObject that contains the operations below the key "operations" as a JSONArray. The content of
     *        this JSONArray is modified. No return value is required.
     */
    static private void optimizeOperationsArray(JSONObject jsonResult) {

        if ((null != jsonResult) && (jsonResult.has(m_operationsKey))) {
            try {
                JSONArray operations = jsonResult.getJSONArray(m_operationsKey);
                JSONArray operationArr = new JSONArray();

                if (!operations.isEmpty()) {

                    int nOps = operations.length();

                    for (int i = 0; i < nOps; i++) {

                        JSONObject nextOp = operations.getJSONObject(i);
                        JSONObject next = null;

                        boolean insertNextOp = true;

                        // Optimizing the content of the JSONArray 'operationArr'.
                        // For example several insertText operations can be merged.
                        // INFO: It might be necessary to make this optimization optional,
                        // so that operations from several clients will not be merged.

                        try {

                            String nextName = nextOp.getString("name");

                            if (nextName.equals("insertText")) {
                                // Always make a copy of insertText operations, because they might be modified in
                                // optimization process. Taking care, that the original operations are not modified.
                                next = new JSONObject(nextOp, nextOp.keySet().toArray(new String[0]));
                                // next = new JSONObject(nextOrig.toString()); -> alternative way
                            } else {
                                next = nextOp; // no need to copy operation, because it will not be modified
                            }

                            JSONObject last = operationArr.getJSONObject(operationArr.length() - 1);

                            if (last != null) {

                                String lastName = last.getString("name");

                                if ((lastName.equals("insertText")) && (nextName.equals("insertText"))) {
                                    // First optimization: Merging two following insertText operations
                                    insertNextOp = mergeInsertOperations(last, next);
                                } else if ((lastName.equals("insertText")) && (nextName.equals("delete"))) {
                                    // Second optimization: Merging delete operation following insertText operation
                                    insertNextOp = mergeInsertAndDeleteOperations(last, next);

                                    if ((!insertNextOp) && (last.getString("text").length() == 0)) {
                                        // taking care of empty new texts -> removing the last operation completely
                                        operationArr = removeItemFromJSONArray(operationArr, operationArr.length() - 1);
                                    }
                                }

                                if (! insertNextOp) {
                                    if (last.has("opl") && (next.has("opl"))) {
                                        last.put("opl", last.getInt("opl") + next.getInt("opl"));
                                    }
                                }

                            }
                        } catch (JSONException e) {
                            //
                        }

                        if (insertNextOp) {
                            operationArr.put(next);
                        }
                    }

                    jsonResult.put(m_operationsKey, operationArr);
                }
            } catch (JSONException e) {
                //
            }
        }
    }

    /**
     * @param jsonResult
     * @param jsonAppender
     * @return
     */
    static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender) {
        return appendJSON(jsonResult, jsonAppender, false);
    }

    /**
     * @param jsonResult
     * @param jsonAppender
     * @param keepActions
     * @return
     */
    static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender, boolean keepActions) {
        if ((null != jsonResult) && (null != jsonAppender)) {
            try {
                final Iterator<String> keys = jsonAppender.keys();

                while (keys.hasNext()) {
                    final String curKey = keys.next();
                    final Object curObject = jsonAppender.get(curKey);

                    if (curKey.equals(m_actionsKey)) {
                        if (!keepActions) {
                            if (!jsonResult.has(m_operationsKey)) {
                                jsonResult.put(m_operationsKey, new JSONArray());
                            }

                            if (curObject instanceof JSONArray) {
                                final JSONArray operationsGroups = (JSONArray) curObject;

                                for (int i = 0, size = operationsGroups.length(); i < size; ++i) {
                                    // [{"operations":[{"text":"a","start":[0,1],"name":"insertText"}]},{"operations":[{"text":"l","start":[0,2],"name":"insertText"}]}]
                                    JSONObject operationsObject = operationsGroups.getJSONObject(i);

                                    if (operationsObject.has(m_operationsKey)) {
                                        final JSONArray newOperations = operationsObject.getJSONArray(m_operationsKey);

                                        for (int j = 0; j < newOperations.length(); j++) {
                                            jsonResult.append(m_operationsKey, newOperations.get(j));
                                        }
                                    } else {
                                        jsonResult.append(m_operationsKey, operationsGroups.get(i));
                                    }
                                }
                            } else {
                                jsonResult.append(m_operationsKey, curObject);
                            }
                        } else {
                            appendJSONElement(jsonResult, curKey, curObject);
                        }
                    } else {
                        appendJSONElement(jsonResult, curKey, curObject);
                    }
                }
            } catch (JSONException e) {
                //
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     * @param key
     * @param element
     */
    static private void appendJSONElement(JSONObject jsonResult, String key, Object element) {
        try {
            if (key.equals(m_operationsKey)) {
                if (!jsonResult.has(m_operationsKey)) {
                    jsonResult.put(m_operationsKey, new JSONArray());
                }

                if (element instanceof JSONArray) {
                    final JSONArray jsonArray = (JSONArray) element;

                    for (int i = 0; i < jsonArray.length(); ++i) {
                        jsonResult.append(m_operationsKey, jsonArray.get(i));
                    }
                } else {
                    jsonResult.append(m_operationsKey, element);
                }
            } else {
                jsonResult.put(key, element);
            }
        } catch (JSONException e) {
            //
        }
    }

    /**
     * @param jsonResult
     * @param operationChunks
     * @return
     */
    static JSONObject appendOperationChunks(JSONObject jsonResult, ArrayList<MessageChunk> operationChunks, boolean allowOptimization) {
        if ((null != jsonResult) && (null != operationChunks)) {
            final ListIterator<MessageChunk> listIter = operationChunks.listIterator();

            while (listIter.hasNext()) {
                appendJSON(jsonResult, listIter.next().toJSON());
            }

            // Check if it's safe to optimize the operations array. There are
            // situations where this is NOT safe:
            // If we have just loaded a document and set a OSN for the last operation,
            // this operation could be exchanged by a different operation missing
            // the OSN.
            if (allowOptimization) {
                // optimize operations
                optimizeOperationsArray(jsonResult);
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     * @param jsonAppender
     * @return
     */
    static JSONObject appendData(ResourceManager resourceManager, JSONObject jsonResult, InputStream contentStm, String returnType, String mimeType) {
        final JSONObject jsonObject = (null != jsonResult) ? jsonResult : new JSONObject();

        if (null != contentStm) {
            byte[] data = null;

            try {
                data = IOUtils.toByteArray(contentStm);
            } catch (IOException e) {
                //
            }

            if ((null != data) && (data.length > 0)) {
                if (returnType.equals("resourceid")) {
                    final long uid = resourceManager.addResource(data);

                    if (0 != uid) {
                        try {
                            jsonObject.put("resourceid", Resource.getManagedIdFromUID(uid));
                        } catch (JSONException e) {
                            //
                        }
                    }
                } else if (returnType.equals("dataurl")) {
                    final StringBuilder dataUrlBuilder = new StringBuilder("data:");

                    dataUrlBuilder.append(mimeType).append(";base64,").append(Base64.encode(data));

                    try {
                        jsonObject.put("dataurl", dataUrlBuilder.toString());
                    } catch (JSONException e) {
                        //
                    }
                }
            }
        }

        return jsonObject;
    }
}
