/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Row;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;

public class RowUtils {

    public static void applyRowProperties(OperationDocument operationDocument, JSONObject rowProperties, Row row, CTStylesheet stylesheet)
        throws  JSONException {

        // Row properties
        JSONObject jsonRowProperties = rowProperties.optJSONObject("row");
        if(jsonRowProperties!=null) {
            Iterator<String> keys = jsonRowProperties.keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = jsonRowProperties.get(attr);
                if(attr.equals("height")) {
                    Double ht = null;
                    if(value instanceof Number) {
                        ht = map100thmmToPoint(((Number)value).doubleValue());
                    }
                    row.setHt(ht);
                }
                else if(attr.equals("customHeight")) {
                    Boolean customHeight = null;
                    if(value instanceof Boolean) {
                        customHeight = (Boolean)value;
                    }
                    row.setCustomHeight(customHeight);
                }
                else if(attr.equals("visible")) {
                    Boolean hidden = null;
                    if(value instanceof Boolean) {
                        hidden = !(Boolean)value;
                    }
                    row.setHidden(hidden);
                }
                else if(attr.equals("customFormat")) {
                    Boolean customFormat = null;
                    if(value instanceof Boolean) {
                        customFormat = (Boolean)value;
                    }
                    row.setCustomFormat(customFormat);
                }
            }
        }

        Long newStyleIndex=null;
        if(rowProperties.has("styleId")) {
            final Object styleId = rowProperties.opt("styleId");
            if(styleId instanceof String) {
                final ApplyOperationHelper applyOperationHelper = operationDocument.getApplyOperationHelper();
                if(applyOperationHelper!=null) {
                    newStyleIndex = applyOperationHelper.getStyleIndex((String)styleId);
                }
            }
            else {
                newStyleIndex = 0L;
            }
        }

        final CTXf detachedXf = CellUtils.getDetachedXf(stylesheet, row.getS(), newStyleIndex);
        final CTXf cellStyleXf = stylesheet.getCellStyleXfsByIndex(detachedXf.getXfId());

        final JSONObject characterProperties = rowProperties.optJSONObject("character");
        if (characterProperties != null) {
            CellUtils.applyCharacterProperties(operationDocument, null, null, characterProperties, detachedXf, stylesheet, cellStyleXf);
        }
        final JSONObject cellProperties = rowProperties.optJSONObject("cell");
        if(cellProperties!=null) {
            CellUtils.applyCellProperties(operationDocument, cellProperties, detachedXf, stylesheet, cellStyleXf);
        }
        long index = stylesheet.getOrApplyCellXf(detachedXf);
        if(index>=0) {
            row.setS(index);
        }
        return;
    }

    public static double map100thmmToPoint(double rowHeight) {
        return Math.round(rowHeight*72/25.40)/100d;
    }

    public static boolean compareRowProperties(Row r1, Row r2) {
        if(r1.getS()!=r2.getS()) {
            return false;
        }
        if(r1.getHt()!=null&&r2.getHt()!=null) {
            if(!r1.getHt().equals(r2.getHt())) {
                return false;
            }
        }
        else if(r1.getHt()!=null||r2.getHt()!=null) {
            return false;
        }
        if(r1.isCustomHeight()!=r2.isCustomHeight()) {
            return false;
        }
        if(r1.isHidden()!=r2.isHidden()) {
            return false;
        }
        if(r1.isCustomFormat()!=r2.isCustomFormat()) {
            return false;
        }
        return true;
    }

    public static JSONObject createRowProperties(Row row, CTStylesheet stylesheet)
        throws JSONException {

        final JSONObject attrs = new JSONObject();
        final JSONObject rowProperties = new JSONObject();

        final Double rowHeight = row.getHt();
        if(rowHeight!=null) {
            rowProperties.put("height", mapPointTo100thmm(rowHeight));
        }
        if(row.isCustomHeight()) {
            rowProperties.put("customHeight", true);
        }
        if(row.isHidden()) {
            rowProperties.put("visible", false);
        }
        if(row.isCustomFormat()) {

            rowProperties.put("customFormat", true);
        }
        if(rowProperties.length()>0) {
            attrs.put("row", rowProperties);
        }
        long s = row.getS();
        if(s>=0) {
            attrs.put("styleId", "a" + Long.toString(s));
        }
        if(attrs.length()==0) {
            return null;
        }
        return attrs;
    }

    public static double mapPointTo100thmm(double rowHeight) {
        return Math.round(rowHeight*2540/0.72+0.5)/100d;
    }
}
