/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.annotation.XmlRootElement;
import org.apache.commons.logging.Log;
import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.Graphic;
import org.docx4j.dml.GraphicData;
import org.docx4j.dml.picture.Pic;
import org.docx4j.dml.wordprocessingDrawing.Anchor;
import org.docx4j.dml.wordprocessingDrawing.Inline;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.vml.CTImageData;
import org.docx4j.vml.CTRect;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.officedrawing.STTrueFalse;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTObject;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.P;
import org.docx4j.wml.Pict;
import org.docx4j.wml.R;
import org.docx4j.wml.R.Tab;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner;
import org.docx4j.wml.Text;
import org.docx4j.wml.Tr;
import org.jvnet.jaxb2_commons.ppp.Child;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.tools.Commons;

public class Component {

    protected static Log log = LogFactory.getLog(Component.class);

    private static HashSet<Class<?>> rootElements = new HashSet<Class<?>>();

    /*
     * simply returns o, but in case o is a JAXBElement and a RootElement annotation is available
     * then the parent content list is updated and the contentModel is returned.
     * */
    public static Object getContentModel(Object o, Object parent, int parentIndex) {
        Object ret = o;
        if (o instanceof javax.xml.bind.JAXBElement) {
            boolean hasRootElement = rootElements.contains(((javax.xml.bind.JAXBElement<?>)o).getDeclaredType());
            if(!hasRootElement) {
                final Annotation[] annotations = ((javax.xml.bind.JAXBElement<?>)o).getDeclaredType().getAnnotations();
                for(Annotation annotation : annotations) {
                    if(annotation instanceof XmlRootElement) {
                        rootElements.add(((javax.xml.bind.JAXBElement<?>)o).getDeclaredType());
                        hasRootElement = true;
                        break;
                    }
                }
            }
            if(hasRootElement) {
                ret = ((JAXBElement<?>)o).getValue();
                List<Object> l = Utils.getContent(parent);
                if (l!=null) {
                    int index = parentIndex < 0 ? l.indexOf(o) : parentIndex;
                    if (index>=0) {
                        l.set(index, ret);
                        if(ret instanceof Child) {
                            ((Child)ret).setParent(parent);
                        }
                    }
                }
            }
        }
        return ret;
    }

    protected Object object;
    protected int contextIndex;
    protected int componentNumber;

    public Component(Object _object, int _contextIndex) {
        object= _object;
        contextIndex = _contextIndex;
        componentNumber = 0;
    }
    public Object getObject() {
        return object;
    }
    public int getContextIndex() {
        return contextIndex;
    }
    public Object getContextObject() {
        return ((Child)object).getParent();
    }
    public int getComponentNumber() {
        return componentNumber;
    }
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }
    void setComponentNumber(int n) {
        componentNumber = n;
    }
    public ComponentContext createComponentContext() {
        return null;
    }

    /**
     * {@link DummyPlaceholderComponent}
     *
     * DummyPlaceholderComponent is returned by TextUtils.getInsertPosition if
     * the position equates the number of existing components + 1 (insert position)
     *
     * !!! getObject() returns null
     *
     * @author <a href="mailto:sven.jacobi@open-xchange.com">Sven Jacobi</a>
     */
    public static class DummyPlaceholderComponent extends Component {

        public DummyPlaceholderComponent(Object parentObject, int _contextIndex, int _componentNumber) {
            super(parentObject, _contextIndex);

            componentNumber = _componentNumber;
        }

        @Override
        public Object getObject() {
            return null;
        }

        @Override
        public Object getContextObject() {
            return object;
        }
    }

    /**
     * {@link ParagraphComponent}
     *
     * @author <a href="mailto:Sven.Jacobi@open-xchange.com">Sven Jacobi</a>
     */
    public static class ParagraphComponent extends Component {

        public ParagraphComponent(P _object, int _contextIndex) {
            super(_object, _contextIndex);
        }
        @Override
        public ComponentContext createComponentContext() {
            return new ParagraphContext((P)getObject());
        }
    }

    public static class TableComponent extends Component {

        public TableComponent(Tbl _object, int _contextIndex) {
            super(_object, _contextIndex);
        }
        @Override
        public ComponentContext createComponentContext() {
            return new TableContext((Tbl)getObject());
        }
    }

    public static class TrComponent extends Component {

        public TrComponent(Tr _object, int _contextIndex) {
            super(_object, _contextIndex);
        }
        @Override
        public ComponentContext createComponentContext() {
            return new TableRowContext((Tr)getObject());
        }
    }

    public static class TcComponent extends Component {

        private final int gridPosition;

        public TcComponent(Tc _object, int _contextIndex, int _gridPosition) {
            super( _object, _contextIndex);
            gridPosition = _gridPosition;
        }

        public int getGridPosition() {
            return gridPosition;
        }

        public int getNextGridPosition() {
            int gridSpan = 1;
            TcPr tcPr = ((Tc)object).getTcPr();
            if(tcPr!=null) {
                TcPrInner.GridSpan span = tcPr.getGridSpan();
                if (span!=null) {
                    gridSpan = span.getVal().intValue();
                    if (gridSpan==0) {
                        System.out.println("Error: empty gridSpan is zero... this results in problems with the component iterator");
                        gridSpan++; // increasing.. preventing an endless loop
                    }
                }
            }
            return gridPosition + gridSpan;
        }
        @Override
        public ComponentContext createComponentContext() {
            return new TableCellContext((Tc)getObject());
        }
    }

    public abstract static class TextRun_Base extends Component {

        // index into R
        private int textRunIndex = -1;

        // index into the parent of R
        private int textRunParentIndex = -1;

        protected TextRun_Base(Object _object, int _contextIndex, int _textRunIndex) {
            super(_object, _contextIndex);
            textRunIndex = _textRunIndex;
            textRunParentIndex = _contextIndex;     // default: the parent of R is an index into P
        }

        @Override
        public Object getContextObject() {
            Object o = ((Child)((Child)object).getParent()).getParent();
            if(o instanceof P.Hyperlink)
                o = ((Child)o).getParent();
            else if (o instanceof RunIns)
                o = ((Child)o).getParent();
            return o;
        }
        public R getTextRun() {
            return (R)((Child)getObject()).getParent();
        }
        public int getTextRunIndex() {
            return textRunIndex;
        }
        public boolean isSingleComponentRun() {
            return false;
        }
        public void setTextRunParentIndex(int _textRunParentIndex) {
            textRunParentIndex = _textRunParentIndex;
        }
        public int getTextRunParentIndex() {
            return textRunParentIndex;
        }
    }

    public static class TextComponent extends TextRun_Base {

        public TextComponent(Text _object, int _contextIndex, int _parentIndex) {
            super(_object, _contextIndex, _parentIndex);
        }

        @Override
        public int getNextComponentNumber() {
            int textLength = ((Text)object).getValue().length();
            if (textLength==0) {
                System.out.println("Error: empty text string is no component... this results in problems with the component iterator");
                textLength++;       // increasing.. preventing an endless loop
            }
            return componentNumber + textLength;
        }
    }

    public static class TabComponent extends TextRun_Base {

        public TabComponent(Tab _object, int _contextIndex, int _parentIndex) {
            super(_object, _contextIndex, _parentIndex);
        }
    }

    public static class HardBreakComponent extends TextRun_Base {

        public HardBreakComponent(Br _object, int _contextIndex, int _parentIndex) {
            super(_object, _contextIndex, _parentIndex);
        }
    }

    public static class FldSimpleComponent extends TextRun_Base {

        public FldSimpleComponent(CTSimpleField simpleField, int _contextIndex) {
            super(simpleField, _contextIndex, -1);
        }
        @Override
        public Object getContextObject() {
            return ((Child)object).getParent();
        }
        @Override
        public R getTextRun() {
           List<Object> content = ((CTSimpleField)object).getContent();
           if(content!=null) {
               for(int i=0; i<content.size();i++) {
                   Object o = getContentModel(content.get(i), object, i);
                   if(o instanceof R)
                       return (R)o;
               }
           }
           return new R();
        }
        @Override
        public int getTextRunIndex() {
            return 0;
        }
        @Override
        public boolean isSingleComponentRun() {
            return true;
        }
        public String getInstr() {
            return ((CTSimpleField)object).getInstr();
        }

        public String getRepresentation() {
            String representation = "";
            List<Object> content = ((CTSimpleField)object).getContent();
            if(content!=null) {
                for(int i=0; i<content.size();i++) {
                    Object o = getContentModel(content.get(i), object, i);
                    if(o instanceof R) {
                        List<Object> run = ((R)o).getContent();
                        if(run!=null) {
                            for(int j=0; j<run.size();j++) {
                                Object ro = getContentModel(run.get(j), o, j);
                                if(ro instanceof Text) {
                                    String t = ((Text)ro).getValue();
                                    if(t!=null)
                                        representation += t;
                                }
                            }
                        }
                    }
                }
            }
            return representation;
        }
    }

    public abstract static class Placeholder_Base extends TextRun_Base {

        public enum GraphicType {
            UNDEFINED,
            IMAGE,                  // undoable
            SHAPE,
            OLE,
            DIAGRAM,
            CHART,
            HORIZONTAL_LINE
        }

        protected Placeholder_Base(Object _object, int _contextIndex, int _parentIndex) {
            super(_object, _contextIndex, _parentIndex);
        }

        abstract public GraphicType getType();
    }

    // parent can either be a Pict or a CTObject

    public static class PictVMLComponent extends Placeholder_Base {
        private GraphicType graphicType = GraphicType.UNDEFINED;

        public PictVMLComponent(Object vml, int _contextIndex, int _parentIndex) {
            super(vml, _contextIndex, _parentIndex);

            List<Object> content = getContent();
            if(content != null) {
                for(Object o : content) {
                    if(o instanceof JAXBElement){
                        String type = ((JAXBElement<?>)o).getDeclaredType().getName();
                        if(type.equals("org.docx4j.vml.CTRect")) {
                            o = ((JAXBElement<?>)o).getValue();
                            STTrueFalse stTrueFalse = ((CTRect)o).getHr();
                            if(stTrueFalse == STTrueFalse.T||stTrueFalse == STTrueFalse.TRUE)
                                graphicType = GraphicType.HORIZONTAL_LINE;
                            break;
                        }
                        else if(type.equals("org.docx4j.vml.officedrawing.CTOLEObject")) {
                            graphicType = GraphicType.OLE;
                            break;
                        }
                    }
                }
            }
        }

        @Override
        public GraphicType getType() {
            return graphicType;
        }

        public List<Object> getContent() {
            Object o = getObject();
            if(o instanceof Pict) {
                return ((Pict)o).getAnyAndAny();
            }
            return ((CTObject)o).getAnyAndAny();
        }

        /**
         * @param part (the partName uri is something like: /word/document.xml)
         * @return the imageUrl in following form: word/media/xxxx.png
         */
        public String getImageUrl(Part part) {

            String rId = null;
            List<Object> content = getContent();
            if(content!=null) {
                for(Object o:content) {
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.vml.CTShape") )
                    {
                        CTShape ctShape = (CTShape)((JAXBElement<?>)o).getValue();
                        if(ctShape!=null) {
                            List<JAXBElement<?>> shapeElements = ctShape.getPathOrFormulasOrHandles();
                            for(JAXBElement<?> shapeElement : shapeElements) {
                                if(shapeElement.getDeclaredType().getName().equals("org.docx4j.vml.CTImageData")) {
                                    CTImageData ctImageData = (CTImageData)((JAXBElement<?>)shapeElement).getValue();
                                    if(ctImageData!=null) {
                                        rId = ctImageData.getId();
                                    }
                                }
                            }
                        }
                        break;
                    }
                }
            }
            return Commons.getUrl(part, rId);
        }
    }

    public static class DrawingComponent extends Placeholder_Base {

        public DrawingComponent(Drawing drawingObject, int _contextIndex, int _parentIndex) {
            super(drawingObject, _contextIndex, _parentIndex);
        }

        @Override
        public GraphicType getType() {

            GraphicType type = GraphicType.UNDEFINED;
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                String uri = graphicData.getUri();
                if(uri.equalsIgnoreCase("http://schemas.microsoft.com/office/word/2010/wordprocessingShape"))
                    type = GraphicType.SHAPE;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/picture"))
                    type = GraphicType.IMAGE;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/diagram"))
                    type = GraphicType.DIAGRAM;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/chart"))
                    type = GraphicType.CHART;
            }
            return type;
        }

        public Object getAnchorOrInline() {
            List<Object> anchorOrInlineContent = ((Drawing)object).getAnchorOrInline();
            return anchorOrInlineContent.size() > 0 ? anchorOrInlineContent.get(0) : null;
        }

        public void setAnchorOrInline(Object o) {
            List<Object> anchorOrInlineContent = ((Drawing)object).getAnchorOrInline();
            anchorOrInlineContent.clear();
            anchorOrInlineContent.add(o);
        }

        public CTNonVisualDrawingProps getDocPr() {
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                return ((Anchor)anchorOrInline).getDocPr();
            else if(anchorOrInline instanceof Inline)
                return ((Inline)anchorOrInline).getDocPr();
            else
                return null;
        }

        public Graphic getGraphic() {
            Graphic ret = null;
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                ret = ((Anchor)anchorOrInline).getGraphic();
            else if (anchorOrInline instanceof Inline)
                ret = ((Inline)anchorOrInline).getGraphic();
            return ret;
        }

        public void setGraphic(Graphic graphic) {
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                ((Anchor)anchorOrInline).setGraphic(graphic);
            else if (anchorOrInline instanceof Inline)
                ((Inline)anchorOrInline).setGraphic(graphic);
        }

        public GraphicData getGraphicData() {
            Graphic graphic = getGraphic();
            return graphic!=null ? graphic.getGraphicData() : null;
        }

        public void setGraphicData(GraphicData graphicData) {
            Graphic graphic = getGraphic();
            if(graphic!=null)
                graphic.setGraphicData(graphicData);
        }

        public Pic getPic() {
            Pic pic = null;
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                List<Object> any = graphicData.getAny();
                for (int i=0;i<any.size();i++) {
                    Object o = any.get(i);
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.dml.picture.Pic") ) {
                        pic = (Pic)((JAXBElement<?>)o).getValue();
                        any.set(i, pic);
                    }
                    else if (o instanceof Pic) {
                        pic = (Pic)o;
                    }
                    if(pic!=null)
                        break;
                }
            }
            return pic;
        }

        public void setPic(Pic pic) {
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                int i = 0;
                List<Object> any = graphicData.getAny();
                for(;i<any.size();i++) {
                    Object o = any.get(i);
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.dml.picture.Pic") )
                        break;
                    else if (o instanceof Pic)
                        break;
                }
                if (any.size()==i)
                    any.add(i, pic);
                else
                    any.set(i, pic);
            }
        }

        public CTBlipFillProperties getBlipFill() {
            Pic pic = getPic();
            return pic!=null ? pic.getBlipFill() : null;
        }

        public void setBlipFill(CTBlipFillProperties blipFillProperties) {
            Pic pic = getPic();
            if(pic!=null)
                pic.setBlipFill(blipFillProperties);
        }

        public CTShapeProperties getSpPr() {
            Pic pic = getPic();
            return pic!=null ? pic.getSpPr() : null;
        }

        public CTBlip getBlip() {
            CTBlipFillProperties blipFillProperties = getBlipFill();
            return blipFillProperties != null ? blipFillProperties.getBlip() : null;
        }

        public void setBlip(CTBlip blip) {
            CTBlipFillProperties blipFillProperties = getBlipFill();
            if(blipFillProperties!=null) {
                blipFillProperties.setBlip(blip);
                blip.setParent(blipFillProperties);
            }
        }

        public CTPositiveSize2D getExtend() {
            CTPositiveSize2D extend = null;
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                extend = ((Anchor)anchorOrInline).getExtent();
            else if (anchorOrInline instanceof Inline)
                extend = ((Inline)anchorOrInline).getExtent();
            return extend;
        }
    }

    public static class ComponentContextWrapper {

        private ComponentContext componentContext;

        public ComponentContextWrapper(ComponentContext _componentContext) {
            componentContext = _componentContext;
        }
        public ComponentContext getComponentContext() {
            return componentContext;
        }
        public void setComponentContext(ComponentContext _componentContext) {
            componentContext = _componentContext;
        }
    }

    public static abstract class ComponentContext implements Iterator<Component> {

        Object              contextObject;
        ComponentContext    childContext;
        Component           nextComponent;
        int                 index;

        public ComponentContext(Object _contextObject) {
            contextObject = _contextObject;
            index = 0;
        }
        final protected void init() {
            childContext = null;
            next();
        }
        protected abstract Component getNext(Component previousComponent);

        final public Object getContextObject() {
            return contextObject;
        }
        @Override
        final public boolean hasNext() {
            return nextComponent!=null;
        }
        @Override
        final public Component next() {
            Component ret = nextComponent;
            if(childContext!=null&&childContext.hasNext())
                nextComponent = childContext.next();
            else
                nextComponent = getNext(ret);
            if (ret!=null&&nextComponent!=null)
                nextComponent.setComponentNumber(ret.getNextComponentNumber());
            return ret;
        }
        @Override
        public void remove() {
            System.out.println("Error: ComponentIterator.remove() is not supported..");
        }
    }

    public static class MainContext extends ComponentContext {

        public MainContext(ContentAccessor _contextObject) {
            super(_contextObject);
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = Utils.getContent(contextObject);
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof P)
                    next = new ParagraphComponent((P)o, index);
                else if(o instanceof Tbl)
                    next = new TableComponent((Tbl)o, index);
                index++;
            }
            return next;
        }
    }

    //                                   Paragraph
    //          TextRun*                SimpleField*       Hyperlink*                             InsDel*
    //  Text*|Drawing*|Pict*|Tab*|Br*    TextRun(1)         TextRun*                             TextRun*
    //                                    Text(1)    Text*|Drawing*|Pict*|Tab*|Br*      Text*|Drawing*|Pict*|Tab*|Br*

    public static class ParagraphContext extends ComponentContext {

        public ParagraphContext(P _contextObject) {
            super(_contextObject);
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((P)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof R)
                    childContext = new TextRunContext((R)o, this);
                else if(o instanceof P.Hyperlink)
                    childContext = new HyperlinkContext((P.Hyperlink)o, this);
                else if(o instanceof RunIns)
                    childContext = new RunInsContext((RunIns)o, this);
                else if(o instanceof CTSimpleField)
                    next = new FldSimpleComponent((CTSimpleField)o, index);
                if(childContext!=null&&childContext.hasNext())
                    next = childContext.next();
                index++;
            }
            return next;
        }
    }

    public static class TextRunContext extends ComponentContext {

        int parentIndex;

        public TextRunContext(R _contextObject, ComponentContext _parentContext) {
            super(_contextObject);
            parentIndex = _parentContext.index;         // index into p content
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((R)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof Text) {
                    if(((Text)o).getValue().length()>0) {
                        next = new TextComponent((Text)o, parentIndex, index);
                    }
                }
                else if(o instanceof Drawing)
                    next = new DrawingComponent((Drawing)o, parentIndex, index);
                else if(o instanceof Pict||o instanceof CTObject)
                    next = new PictVMLComponent(o, parentIndex, index);
                else if(o instanceof Tab)
                    next = new TabComponent((Tab)o, parentIndex, index);
                else if(o instanceof Br)
                    next = new HardBreakComponent((Br)o, parentIndex, index);
                index++;
            }
            return next;
        }
    }

    public static class HyperlinkContext extends ComponentContext {

        int parentIndex;

        public HyperlinkContext(P.Hyperlink _contextObject, ComponentContext _parentContext) {
            super(_contextObject);
            parentIndex = _parentContext.index;         // index into p content
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((P.Hyperlink)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof R)
                    childContext = new HyperlinkRunContext((R)o, this, parentIndex);
                if(childContext!=null&&childContext.hasNext())
                    next = childContext.next();
                index++;
            }
            return next;
        }
    }

    public static class HyperlinkRunContext extends ComponentContext {

        // index into P
        int paragraphIndex;

        // index into P.Hyperlink
        int parentIndex;

        public HyperlinkRunContext(R _contextObject, HyperlinkContext _parentContext, int _paragraphIndex) {
            super(_contextObject);
            parentIndex = _parentContext.index;
            paragraphIndex = _paragraphIndex;
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            TextRun_Base next = null;
            List<Object> content = ((R)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof Text) {
                    if(((Text)o).getValue().length()>0) {
                        next = new TextComponent((Text)o, paragraphIndex, index);
                    }
                }
                else if (o instanceof Drawing)
                    next = new DrawingComponent((Drawing)o, paragraphIndex, index);
                else if (o instanceof Pict||o instanceof CTObject)
                    next = new PictVMLComponent(o, paragraphIndex, index);
                else if(o instanceof Tab)
                    next = new TabComponent((Tab)o, paragraphIndex, index);
                else if(o instanceof Br)
                    next = new HardBreakComponent((Br)o, paragraphIndex, index);
                index++;
            }
            if(next!=null) {
                next.setTextRunParentIndex(parentIndex);
            }
            return next;
        }
    }

    public static class RunInsContext extends ComponentContext {

        int parentIndex;

        public RunInsContext(RunIns _contextObject, ComponentContext _parentContext) {
            super(_contextObject);
            parentIndex = _parentContext.index;         // index into p content
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((RunIns)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof R)
                    childContext = new RunInsRunContext((R)o, this, parentIndex);
                if(childContext!=null&&childContext.hasNext())
                    next = childContext.next();
                index++;
            }
            return next;
        }
    }

    public static class RunInsRunContext extends ComponentContext {

        // index into P
        int paragraphIndex;

        // index into P.Hyperlink
        int parentIndex;

        public RunInsRunContext(R _contextObject, RunInsContext _parentContext, int _paragraphIndex) {
            super(_contextObject);
            parentIndex = _parentContext.index;
            paragraphIndex = _paragraphIndex;
            init();
        }
        @Override
        protected Component getNext(Component previousComponent) {
            TextRun_Base next = null;
            List<Object> content = ((R)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof Text) {
                    if(((Text)o).getValue().length()>0) {
                        next = new TextComponent((Text)o, paragraphIndex, index);
                    }
                }
                else if (o instanceof Drawing)
                    next = new DrawingComponent((Drawing)o, paragraphIndex, index);
                else if (o instanceof Pict||o instanceof CTObject)
                    next = new PictVMLComponent(o, paragraphIndex, index);
                else if(o instanceof Tab)
                    next = new TabComponent((Tab)o, paragraphIndex, index);
                else if(o instanceof Br)
                    next = new HardBreakComponent((Br)o, paragraphIndex, index);
                index++;
            }
            if(next!=null) {
                next.setTextRunParentIndex(parentIndex);
            }
            return next;
        }
    }

    public static class TableContext extends ComponentContext {

        public TableContext(Tbl _contextObject) {
            super(_contextObject);
            init();
        }

        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((Tbl)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof Tr)
                    next = new TrComponent((Tr)o, index);
                index++;
            }
            return next;
        }
    }

    public static class TableRowContext extends ComponentContext {

        public TableRowContext(Tr _contextObject) {
            super(_contextObject);
            init();
        }

        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((Tr)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof Tc)
                    next = new TcComponent((Tc)o, index, previousComponent!=null?((TcComponent)previousComponent).getNextGridPosition():0);
                index++;
            }
            return next;
        }
    }

    public static class TableCellContext extends ComponentContext {

        public TableCellContext(Tc _contextObject) {
            super(_contextObject);
            init();
        }

        @Override
        protected Component getNext(Component previousComponent) {
            Component next = null;
            List<Object> content = ((Tc)contextObject).getContent();
            while(next==null&&index<content.size()) {
                Object o = getContentModel(content.get(index), contextObject, index);
                if(o instanceof P)
                    next = new ParagraphComponent((P)o, index);
                else if(o instanceof Tbl)
                    next = new TableComponent((Tbl)o, index);
                index++;
            }
            return next;
        }
    }
}