/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/tooltip',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/popup/basepopup'
    ], function (Utils, BasePopup) {

    'use strict';

    // class ToolTip ==========================================================

    /**
     * Wrapper class for a DOM node used as tooltip info element, shown on top
     * of the application window, and relative to an arbitrary DOM node.
     *
     * @constructor
     *
     * @extends BasePopup
     *
     * @param {HTMLElement|jQuery} anchorNode
     *  The DOM node this tooltip element is attached to.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the tooltip element.
     *  Supports all options also supported by the base class BasePopup. The
     *  default value of the option 'anchorPadding' will be set to 3 though.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.hover=false]
     *      If set to true, the tooltip will be shown automatically when the
     *      mouse hovers the anchor node.
     */
    function ToolTip(anchorNode, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        BasePopup.call(this, anchorNode, Utils.extendOptions({
            anchorPadding: 3,
            restrictSize: 'width',
            minSize: 150,
            maxSize: 450
        }, initOptions));

        // methods ------------------------------------------------------------

        /**
         * Inserts the passed plain text into the tooltip.
         *
         * @param {String} text
         *  The text to be inserted into the tooltip.
         *
         * @return {ToolTip}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            this.getContentNode().text(text);
            this.refresh();
            return this;
        };

        // initialization -----------------------------------------------------

        // add tooltip styling
        this.getNode().addClass('popup-tooltip');

        // add hover behavior
        if (Utils.getBooleanOption(initOptions, 'hover', false)) {
            anchorNode.on({
                mouseenter: function () { self.show(); },
                mouseleave: function () { self.hide(); }
            });
        }

    } // class ToolTip

    // exports ================================================================

    // derive this class from class BasePopup
    return BasePopup.extend({ constructor: ToolTip });

});
