/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/tablecellstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/stylesheets',
     'io.ox/office/text/dom'
    ], function (Utils, Color, Border, StyleSheets, DOM) {

    'use strict';

    var // definitions for table cell attributes
        DEFINITIONS = {

            /**
             * The number of grid columns spanned by the table cell.
             */
            gridSpan: {
                def: 1,
                scope: 'element',
                format: function (element, gridSpan) {
                    element.attr('colspan', gridSpan);
                }
            },

            /**
             * Fill color of the table cell.
             */
            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                },
                preview: true // 'true' calls the own 'format' method
            },

            /**
             * Style, width and color of the left table cell border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border, { clearNone: true, doubleWidth: ((_.browser.IE === 11) && DOM.isTableCellNode(element) && (element.prev().length === 0)) }));
                },
                preview: function (element, border) {
                    element.css('border-left', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the right table cell border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border, { clearNone: true, doubleWidth: ((_.browser.IE === 11) && DOM.isTableCellNode(element) && (element.next().length === 0)) }));
                },
                preview: function (element, border) {
                    element.css('border-right', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the top table cell border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border, { clearNone: true, doubleWidth: ((_.browser.IE === 11) && DOM.isTableCellNode(element) && (element.parent().prev().length === 0)) }));
                },
                preview: function (element, border) {
                    element.css('border-top', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the bottom table cell border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border, { clearNone: true, doubleWidth: ((_.browser.IE === 11) && DOM.isTableCellNode(element) && (element.parent().next().length === 0)) }));
                },
                preview: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Inner horizontal table cell borders, used in table style sheets
             * to format inner borders of specific table areas (first/last
             * column, inner vertical bands, ...).
             */
            borderInsideHor: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Inner vertical table cell borders, used in table style sheets to
             * format inner borders of specific table areas (first/last row,
             * inner horizontal bands, ...).
             */
            borderInsideVert: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Top padding of the table cell
             */
            paddingTop: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-top', value);
                }
            },

            /**
             * Bottom padding of the table cell
             */
            paddingBottom: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px,  required for cell resizing
                    element.css('padding-bottom', value);
                }
            },

            /**
             * Left padding of the table cell
             */
            paddingLeft: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-left', value);
                }
            },

            /**
             * Right padding of the table cell
             */
            paddingRight: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-right', value);
                }
            }

        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            table: function (cell) { return cell.closest(DOM.TABLE_NODE_SELECTOR); }
        };

    // class TableCellStyles ==================================================

    /**
     * Contains the style sheets for table cell formatting attributes. The CSS
     * formatting will be read from and written to <th> and <td> elements.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function TableCellStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'cell', {
            styleSheetSupport: false,
            additionalFamilies: ['paragraph', 'character'],
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateTableCellFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table cell element whose attributes have
         * been changed. Reformats the table cell according to the passed
         * attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         */
        function updateTableCellFormatting(cell, mergedAttributes) {

            var // the paragraph style sheet container
                paragraphStyles = documentStyles.getStyleSheets('paragraph');

            // Center cell content vertically. This solution is not dynamic
            // -> changing row height or inserting text destroys center position.
            // var cellContentNode = DOM.getCellContentNode(cell);
            // cellContentNode.css('padding-top', cell.height() / 2 - cellContentNode.height() / 2);

            // update all paragraphs in the table cell
            Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(cell), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                paragraphStyles.updateElementFormatting(paragraph, { baseAttributes: mergedAttributes });
            }, undefined, { children: true });
        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('cell', DEFINITIONS);

    } // class TableCellStyles

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: TableCellStyles });

});
