/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/tokenarray',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, SheetUtils, Operations, TokenArray, gt) {

    'use strict';

    // class ChartCreator =================================================

    function getFormula(app, sheet, fx, fy, tx, ty) {
        var range = {};
        range.start = [fx, fy];
        if (_.isNumber(tx)) {
            range.end = [tx, ty];
        } else {
            range.end = range.start;
        }
        var tokenArray = new TokenArray(app, null, { silent: true });
        tokenArray.appendRange(range, { abs: true, sheet: sheet });
        var formula = tokenArray.getFormula();
        tokenArray.destroy();

        return formula;
    }

    function addDataSeries(app, generator, sheet, position, series, keyFromX, keyFromY, keyToX, keyToY, titleX, titleY, valueFromX, valueFromY, valueToX, valueToY) {
        var insert = {
            values: getFormula(app, sheet, valueFromX, valueFromY, valueToX, valueToY)
        };
        if (titleX !== null && titleY !== null) {
            insert.title = getFormula(app, sheet, titleX, titleY);
        }
        if (keyFromX !== null && keyFromY !== null) {
            insert.names = getFormula(app, sheet, keyFromX, keyFromY, keyToX, keyToY);
        }
        var options = {
            series: series,
            attrs: {
                series: insert
            }
        };
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, sheet, position, options);
    }

    function generateOperations(app, range, sheet, generator, position, leftCell, topCell) {
        var start = range.start;
        var end = range.end;
        var width = end[0] - start[0];
        var height = end[1] - start[1];

        if (width > height) {
            var keyFromX;

            if (_.isNumber(topCell.result)) {
                keyFromX = null;
                start[1] -= 1;
                height += 1;
            } else {
                if (_.isNumber(leftCell.result)) {
                    keyFromX = start[0];
                } else {
                    keyFromX = start[0] + 1;
                }
            }

            var keyFromY = start[1];
            var keyToX = end[0];
            var keyToY = start[1];

            for (var i = 0; i < height; i++) {
                var y = start[1] + 1 + i;

                var valueFromX = start[0] + 1;
                var titleX;
                if (_.isNumber(leftCell.result)) {
                    titleX = null;
                    valueFromX -= 1;
                } else {
                    titleX = start[0];
                }
                var valueToX = end[0];

                addDataSeries(app, generator, sheet, position, i, keyFromX, keyFromY, keyToX, keyToY, titleX, y, valueFromX, y, valueToX, y);
            }

        } else {
            var keyFromY = start[1] + 1;

            if (_.isNumber(leftCell.result)) {
                keyFromY = null;
                start[0] -= 1;
                width += 1;
            } else {
                if (_.isNumber(topCell.result)) {
                    keyFromY = start[1];
                } else {
                    keyFromY = start[1] + 1;
                }
            }

            var keyFromX = start[0];
            var keyToX = start[0];
            var keyToY = end[1];

            for (var i = 0; i < width; i++) {
                var x = start[0] + 1 + i;

                var valueFromY = start[1] + 1;
                var titleY;
                if (_.isNumber(topCell.result)) {
                    titleY = null;
                    valueFromY -= 1;
                } else {
                    titleY = start[1];
                }
                var valueToY = end[1];

                addDataSeries(app, generator, sheet, position, i, keyFromX, keyFromY, keyToX, keyToY, x, titleY, x, valueFromY, x, valueToY);
            }
        }
    }

    function getContentForRange(app, range, sheet, def, follow) {
        var view = app.getView();

        var start = range.start;
        var end = range.end;
        var width = end[0] - start[0];
        var height = end[1] - start[1];

        if (width > 50 || height > 50) {
            view.yell('warning', gt('It is not possible to create a chart out of more than 50 input cells.'));
            if (def) {
                def.reject();
            }
        } else {

            var leftAddress = [start[0], start[1] + 1],
                topAddress = [start[0] + 1, start[1]],
                request = app.queryRangeContents([
                    [{sheet: sheet, start: leftAddress}],
                    [{sheet: sheet, start: topAddress}]
                ]);

            request.done(function (res) {
                var leftCell = res[0][0];
                var topCell = res[1][0];

                if (!leftCell) {
                    leftCell = {result: 1};
                } else if (_.isString(leftCell.result) && !leftCell.result.length) {
                    leftCell.result = 1;
                }
                if (!topCell) {
                    topCell = {result: 1};
                } else if (_.isString(topCell.result) && !topCell.result.length) {
                    topCell.result = 1;
                }

                follow(leftCell, topCell);
            });
        }
    }

    /**
     * ChartCreator makes an insert-Chart-Op and Insert-Chart-DataSeries chosen by current Selection
     * and changes the Selection to a Multiselection by title, names and values
     */
    function ChartCreator(app) {

        var view = app.getView();
        var gridPane = view.getActiveGridPane();

        var def = $.Deferred();

        var view = app.getView();
        var sheetModel = view.getActiveSheetModel();
        var drawingCollection = sheetModel.getDrawingCollection();
        var sheet = view.getActiveSheet();
        var selection = view.getSelection();

        var range = selection.ranges[selection.activeRange];


        getContentForRange(app, range, sheet, def, function (leftCell, topCell) {
            var rect = gridPane.getVisibleRectangle();
            var clip = {
                left: Math.round(rect.left + rect.width / 3),
                top: Math.round(rect.top + rect.height / 4),
                width: Math.round(rect.width / 3),
                height: Math.round(rect.height / 2)
            };
            var attrs = drawingCollection.getAttributesForRectangle(clip);

            attrs.chart = {
                chartStyleId: 2,
                legendPos: 'right'
            };

            view.insertDrawing('chart', attrs, function (generator, sheet, position) {
                generateOperations(app, range, sheet, generator, position, leftCell, topCell);
                generator.generateOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, {axis: 'x', start: [sheet, position[0]], attrs: { axis: {type: 'value', visible: true}, line: {type: 'solid', color: {value: '000000', type: 'rgb'}}}});
                generator.generateOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, {axis: 'y', start: [sheet, position[0]], attrs: { axis: {type: 'cat', visible: true}, line: {type: 'solid', color: {value: '000000', type: 'rgb'}}}});
            });
            def.resolve();
        });


        /**
         * the deferred Object handles noting at the Moment,
         * its just for Future cases, when the situation could be changed
         * @return {Jquery|Deferred}
         */
        this.deferred = function () {
            return def;
        };
    }


    ChartCreator.generateOperations = function (app, range, chartModel) {

        var pos = chartModel.getPosition();
        var position = pos[1];
        var sheet = pos[0];

        getContentForRange(app, range, sheet, null, function (leftCell, topCell) {
            var model = app.getModel();
            var generator = model.createOperationsGenerator();

            var count = chartModel.getSeriesCount();
            _.times(count, function () {
                generator.generateOperation(Operations.DELETE_CHART_DATASERIES, {start: pos, series: 0});
            });


            generateOperations(app, range, sheet, generator, position, leftCell, topCell);
            model.applyOperations(generator.getOperations());
        });

    };

    // exports ================================================================

    return ChartCreator;

});
