/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/drawinglayer/model/drawingutils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/namecollection',
     'io.ox/office/spreadsheet/model/colrowcollection',
     'io.ox/office/spreadsheet/model/mergecollection',
     'io.ox/office/spreadsheet/model/validationcollection',
     'io.ox/office/spreadsheet/model/drawing/drawingcollection',
     'io.ox/office/spreadsheet/model/viewsettingsmixin'

    ], function (Utils, Border, AttributedModel, DrawingUtils, SheetUtils, Operations, NameCollection, ColRowCollection, MergeCollection, ValidationCollection, SheetDrawingCollection, ViewSettingsMixin) {

    'use strict';

    var // definitions for sheet view attributes
        SHEET_VIEW_PROPERTIES = {

            /**
             * Sheet selection (cells and drawing objects), as object with the
             * following properties:
             * - {Array} ranges
             *      The logical addresses of all selected cell ranges. MUST NOT
             *      be an empty array.
             * - {Number} activeRange
             *      Element index of the active range in the 'ranges' property.
             * - {Number[]} activeCell
             *      The logical address of the active cell in the active range.
             * - {Array} drawings
             *      The logical positions of all selected drawing frames.
             */
            selection: {
                def: {
                    ranges: [{ start: [0, 0], end: [0, 0] }],
                    activeRange: 0,
                    activeCell: [0, 0],
                    drawings: []
                },
                validate: function (selection) {
                    return _.isObject(selection) ? selection : Utils.BREAK;
                }
            },

            /**
             * Specifies whether the sheet has ever been displayed in the view.
             */
            visited: {
                def: false
            },

            /**
             * Zoom factor in the sheet, as floating-point number. The value 1
             * represents a zoom factor of 100%.
             */
            zoom: {
                def: 1.25,
                validate: function (zoom) {
                    return _.isNumber(zoom) ? Utils.minMax(zoom, SheetUtils.MIN_ZOOM, SheetUtils.MAX_ZOOM) : Utils.BREAK;
                }
            },

            /**
             * The type of the current view split. Has no effect, if both view
             * properties 'splitWidth' and 'splitHeight' are set to the value
             * zero (view is not split at all). Supports the following values:
             * - 'split': Dynamic split mode (movable split lines).
             * - 'frozen': Frozen split mode (a fixed number of columns and/or
             *      rows are frozen, split lines are not movable). When leaving
             *      frozen split mode, the split lines will be hidden.
             * - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
             *      mode). When leaving frozen split mode, the split lines
             *      remain visible and become movable (back to 'split' mode).
             */
            splitMode: {
                def: 'split',
                validate: /^(split|frozen|frozenSplit)$/
            },

            /**
             * The inner width of the left panes in view split mode. In dynamic
             * split mode, this value is interpreted in 1/100 of millimeters.
             * In frozen split mode, this value specifies the number of columns
             * shown in the frozen panes. Must not be negative. If this value
             * is zero, the view is not split into left and right panes (but
             * may be split into upper and lower panes).
             */
            splitWidth: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The inner height of the upper panes in view split mode. In
             * dynamic split mode, this value is interpreted in 1/100 of
             * millimeters. In frozen split mode, this value specifies the
             * number of rows shown in the frozen panes. Must not be negative.
             * If this value is zero, the view is not split into top and bottom
             * panes (but may be split into left and right panes).
             */
            splitHeight: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The identifier of the active pane in split mode. Must be one of
             * 'topLeft', 'bottomLeft', 'topRight', or 'bottomRight'.
             */
            activePane: {
                def: 'bottomRight',
                validate: /^(top|bottom)(Left|Right)$/
            },

            /**
             * The identifier of an entire active pane side. Must be one of the
             * values null, 'left', 'right', 'top', or 'bottom'. Used for
             * example while selecting entire columns or rows.
             */
            activePaneSide: {
                def: null,
                validate: function (paneSide) {
                    return (/^(left|right|top|bottom)$/).test(paneSide) ? paneSide : null;
                }
            },

            /**
             * The horizontal scroll position in the left panes. Must be an
             * object with the properties 'index' (index of the first column
             * visible in the panes), and 'ratio' (offset inside the column, as
             * ratio of the current column width).
             */
            anchorLeft: {
                def: { index: 0, ratio: 0 }
            },

            /**
             * The horizontal scroll position in the right panes. Must be an
             * object with the properties 'index' (index of the first column
             * visible in the panes), and 'ratio' (offset inside the column, as
             * ratio of the current column width).
             */
            anchorRight: {
                def: { index: 0, ratio: 0 }
            },

            /**
             * The vertical scroll position in the upper panes. Must be an
             * object with the properties 'index' (index of the first row
             * visible in the panes), and 'ratio' (offset inside the row, as
             * ratio of the current row height).
             */
            anchorTop: {
                def: { index: 0, ratio: 0 }
            },

            /**
             * The vertical scroll position in the lower panes. Must be an
             * object with the properties 'index' (index of the first row
             * visible in the panes), and 'ratio' (offset inside the row, as
             * ratio of the current row height).
             */
            anchorBottom: {
                def: { index: 0, ratio: 0 }
            },

            /**
             * Highlighted ranges beside the regular cell selection, used for
             * example in formula edit mode to visualize the ranges used in the
             * formula. Allowed values are null (no highlighting), or an array
             * of logical range addresses. Each range address object in this
             * array may contain the following additional properties:
             * - {String} [range.id]
             *      An arbitrary identifier for the range.
             * - {Boolean} [range.draggable=false]
             *      Whether the range is intended to be movable and resizable
             *      with mouse or touch gestures.
             */
            highlightRanges: {
                def: null,
                validate: function (ranges) { return _.isArray(ranges) ? ranges : null; }
            },

            /**
             * The array index of the highlighted range currently tracked.
             */
            highlightIndex: {
                def: null,
                validate: function (index) { return _.isNumber(index) ? index : null; }
            },

            /**
             * Specific cell ranges that are in a special activated state, used
             * for example for ranges that will be inserted into a formula
             * while in formula edit mode. These ranges will be rendered in a
             * special way over the regular cell selection. Allowed values are
             * null (no active ranges available), or a selection object as
             * described for the 'selection' view attribute (see above), but
             * with the following differences:
             * - The property 'drawings' must not be contained in this object.
             * - The array property 'ranges' may be an empty array.
             */
            activeSelection: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            },

            /**
             * Additional data for the current cell range while auto-fill
             * tracking is active. Allowed values are null (auto-fill tracking
             * not active), or an object with the following properties:
             * - {String} border
             *      Which border of the selected range will be modified by the
             *      auto-fill. Allowed values are 'left', 'right', 'top', or
             *      'bottom'.
             * - {Number} count
             *      The number of columns/rows to extend the selected range
             *      with (positive values), or to shrink the selected range
             *      (negative values).
             * This attribute will be ignored, if the current sheet selection
             * consists of more than one cell range.
             */
            autoFillData: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            }
        };

    // global private functions ===============================================

    /**
     * Returns a rectangle descriptor for the passed column and row entry.
     */
    function getRectangle(colPosition, rowPosition) {
        return {
            leftHmm: colPosition.offsetHmm,
            topHmm: rowPosition.offsetHmm,
            widthHmm: colPosition.sizeHmm,
            heightHmm: rowPosition.sizeHmm,
            left: colPosition.offset,
            top: rowPosition.offset,
            width: colPosition.size,
            height: rowPosition.size
        };
    }

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application instance with the document model containing this sheet.
     *
     * @param {String|SheetModel} sheetType
     *  The type identifier of this sheet. Supported values are:
     *  - 'worksheet': A regular worksheet with cells and formulas.
     *  - 'chartsheet': A sheet that consists entirely of a chart object.
     *  - 'macrosheet': A sheet with cells and formulas that contains sheet
     *      macros. This sheet type is a legacy feature of MS Excel. These
     *      sheets will not be shown in the user interface.
     *  - 'dialogsheet': A sheet that consists entirely of a user dialog. This
     *      sheet type is a legacy feature of MS Excel. These sheets will not
     *      be shown in the user interface.
     *  Alternatively, a source sheet model, if this constructor will be used
     *  as copy constructor.
     *
     * @param {Object} [initialAttrs]
     *  An attribute set with initial formatting attributes for the sheet.
     */
    function SheetModel(app, sheetType, initialAttrs) {

        var // self reference
            self = this,

            // the document model
            model = app.getModel(),

            // the cell style sheet container
            cellStyles = model.getStyleSheets('cell'),

            // the collection of defined names in this sheet
            nameCollection = null,

            // the column and row collections (size and formatting of all columns/rows)
            colCollection = null,
            rowCollection = null,

            // the collection of merged cells in this sheet
            mergeCollection = null,

            // the collection of cell ranges with validation settings
            validationCollection = null,

            // the collection of drawing objects contained in this sheet
            drawingCollection = null,

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [];

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, initialAttrs, { additionalFamilies: ['sheet', 'column', 'row'] });
        ViewSettingsMixin.call(this, app, SHEET_VIEW_PROPERTIES);

        // private methods ----------------------------------------------------

        /**
         * Registers an attribute change listener and a delete listener at the
         * drawing collection, and invokes the passed callback function. All
         * changes made at the drawing collection will be inserted as drawing
         * operations into the passed operations generator. After the callback
         * function has returned, the event handlers will be removed from the
         * drawing collection.
         *
         * @param {SpreadsheetOperationsGenerator} generator
         *  The operations generator that will contain the indirect drawing
         *  operations.
         *
         * @param {Function} callback
         *  The callback function that causes changes at the drawing collection
         *  while running.
         *
         * @returns {Any}
         *  The result of the callback function.
         */
        function generateIndirectDrawingOperations(generator, callback) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the result of the callback function
                result = null,
                // positions of deleted drawing objects
                deletedDrawings = [],
                // changed drawing anchor attributes, mapped by position
                changedAttributes = {};

            // collects deleted drawing objects (only on sheet level)
            function deleteDrawingHandler(event, drawingModel, position) {
                if (position.length === 1) {
                    deletedDrawings.push(position);
                }
            }

            // collects changed anchor attributes of moved drawing objects (only on sheet level)
            function changeDrawingHandler(event, drawingModel, position, type, newAttributes, oldAttributes) {
                if ((type === 'attributes') && (position.length === 1)) {
                    var key = JSON.stringify(position);
                    if (!(key in changedAttributes)) { changedAttributes[key] = { oldValues: oldAttributes.drawing }; }
                    changedAttributes[key].newValues = _.copy(newAttributes.drawing, true);
                }
            }

            // invoke the callback function with registered drawing event handlers
            drawingCollection.on('delete:drawing', deleteDrawingHandler);
            drawingCollection.on('change:drawing', changeDrawingHandler);
            result = callback.call(self);
            drawingCollection.off('delete:drawing', deleteDrawingHandler);
            drawingCollection.off('change:drawing', changeDrawingHandler);

            // remove entries for deleted drawings from 'changedAttributes' map
            _(deletedDrawings).each(function (position) {
                delete changedAttributes[JSON.stringify(position)];
            });

            // create 'setDrawingAttributes' operations for all changed drawings
            _(changedAttributes).each(function (attributes, key) {
                var position = JSON.parse(key);
                // delete unchanged attributes
                _(attributes.newValues).each(function (value, name) {
                    if (_.isEqual(attributes.oldValues[name], value)) { delete attributes.newValues[name]; }
                });
                // generate operations if some values really have been changed
                if (!_.isEmpty(attributes.newValues)) {
                    generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, sheet, position, { scope: 'filter', attrs: { drawing: attributes.newValues } });
                }
            });

            // create 'deleteDrawing' operations for all deleted drawings (reverse order!)
            DrawingUtils.sortDrawingPositions(deletedDrawings);
            Utils.iterateArray(deletedDrawings, function (position) {
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, sheet, position, { scope: 'filter' });
            }, { reverse: true });

            return result;
        }

        /**
         * Generates and applies all operations to insert new columns or rows
         * into the sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  Either true to insert new columns into the sheet, or false to
         *  insert new rows into the sheet.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        function insertIntervals(intervals, columns) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the name of the insert operation
                operationName = columns ? Operations.INSERT_COLUMNS : Operations.INSERT_ROWS,
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // create the operations for all intervals in reverse order (!),
            // and insert the columns/rows already (to be able to generate the
            // indirect drawing operations)
            generateIndirectDrawingOperations(generator, function () {
                Utils.iterateArray(_.getArray(intervals), function (interval) {
                    generator.generateIntervalOperation(operationName, sheet, interval);
                    collection.insertEntries(interval);
                }, { reverse: true });
            });

            // send the operations, but do not apply them
            return model.sendOperations(generator.getOperations()) ? '' : 'internal';
        }

        /**
         * Generates and applies all operations to delete columns or rows from
         * the sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  Either true to delete columns from the sheet, or false to delete
         *  rows from the sheet.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        function deleteIntervals(intervals, columns) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the name of the delete operation
                operationName = columns ? Operations.DELETE_COLUMNS : Operations.DELETE_ROWS,
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // create the operations for all intervals in reverse order (!),
            // and delete the columns/rows already (to be able to generate the
            // indirect drawing operations)
            generateIndirectDrawingOperations(generator, function () {
                Utils.iterateArray(_.getArray(intervals), function (interval) {
                    generator.generateIntervalOperation(operationName, sheet, interval);
                    collection.deleteEntries(interval);
                }, { reverse: true });
            });

            // send the operations, but do not apply them
            return model.sendOperations(generator.getOperations()) ? '' : 'internal';
        }

        /**
         * Generates and applies all operations to modify the attributes of
         * columns or rows in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Object} attributes
         *  The (incomplete) attribute set containing all column/row attributes
         *  and default cell/character attributes to be changed.
         *
         * @param {Boolean} columns
         *  Either true to modify the column attributes, or false to modify the
         *  row attributes.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        function setIntervalAttributes(intervals, attributes, columns) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the name of the attribute operation
                operationName = columns ? Operations.SET_COLUMN_ATTRIBUTES : Operations.SET_ROW_ATTRIBUTES,
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // generate the attribute operations, and apply the attributes
            // already (to be able to generate the indirect drawing operations)
            generateIndirectDrawingOperations(generator, function () {
                _.chain(intervals).getArray().each(function (interval) {
                    generator.generateIntervalOperation(operationName, sheet, interval, { attrs: attributes });
                    collection.setAttributes(interval, attributes);
                });
            });

            // send the operations, but do not apply them
            return model.sendOperations(generator.getOperations()) ? '' : 'internal';
        }

        /**
         * Generates and applies all operations to modify the size of columns
         * or rows in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Number} size
         *  The column width or row height, in 1/100 mm. If this value is less
         *  than 1, the columns/rows will be hidden, and their original size
         *  attribute will not be changed.
         *
         * @param {Boolean} columns
         *  Either true to modify the column width, or false to modify the row
         *  height.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customWidth' column attribute or
         *      'customHeight' row attribute.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        function setIntervalSize(intervals, size, columns, options) {

            var // the attributes to be inserted into the operations
                attributes = { visible: size > 0 };

            // insert the size attributes, and build the attribute set object
            if (attributes.visible) {
                attributes[columns ? 'width' : 'height'] = size;
                attributes[columns ? 'customWidth' : 'customHeight'] = Utils.getBooleanOption(options, 'custom', true);
            }
            attributes = Utils.makeSimpleObject(columns ? 'column' : 'row', attributes);

            // apply the new attributes
            return setIntervalAttributes(intervals, attributes, columns);
        }

        /**
         * Builds an attribute set containing all cell formatting attributes
         * set to the null value, but without the 'styleId' property.
         */
        function buildNullAttributes() {
            var attributes = cellStyles.buildNullAttributes();
            delete attributes.styleId;
            delete attributes.character.url;
            return attributes;
        }

        /**
         * Generates an attribute operation for the specified cell range. If
         * the range covers entire columns or entire rows, generates the
         * appropriate operations automatically.
         */
        function generateSetAttributesOperation(generator, sheet, range, attributes, operationOptions) {

            // add the attributes to the operation
            operationOptions = _({ attrs: attributes }).extend(operationOptions);

            // decide whether to generate a column, row, or cell range operation
            if (model.isColRange(range)) {
                generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, sheet, SheetUtils.getColInterval(range), operationOptions);
            } else if (model.isRowRange(range)) {
                operationOptions.attrs.row = { customFormat: true };
                generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, sheet, SheetUtils.getRowInterval(range), operationOptions);
            } else {
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, sheet, range, operationOptions);
            }
        }

        /**
         * Returns the column/row interval shown in the leading panes of the
         * split view.
         *
         * @param {Boolean} columns
         *  Whether to return the column interval of the left sheet area
         *  (true), or the row interval of the top sheet area (false).
         *
         * @returns {Object|Null}
         *  The column/row interval shown in the leading panes. If the view is
         *  not split in the specified direction, this method returns null.
         */
        function getSplitInterval(columns) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the value of the split size view attribute (number of columns/rows in frozen split)
                splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight'),
                // start and end scroll anchor in the leading sheet area
                startAnchor = null, endAnchor = null,
                // the resulting column/row interval
                interval = null;

            // no split: return null
            if (splitSize === 0) { return null; }

            // build column/row interval in frozen split mode directly
            startAnchor = self.getViewAttribute(columns ? 'anchorLeft' : 'anchorTop');
            if (self.hasFrozenSplit()) {
                return { first: startAnchor.index, last: startAnchor.index + splitSize - 1 };
            }

            // get index of last visible column/row, and build the resulting interval
            endAnchor = collection.getScrollAnchorByOffset(collection.convertScrollAnchorToHmm(startAnchor) + splitSize);
            interval = { first: startAnchor.index, last: endAnchor.index };

            // exclude last column/row, if visible less than half of its size
            if ((interval.first < interval.last) && (endAnchor.ratio < 0.5)) {
                interval.last -= 1;
            }

            // exclude first column/row, if visible less than half of its size
            if ((interval.first < interval.last) && (startAnchor.ratio > 0.5)) {
                interval.first += 1;
            }

            return interval;
        }

        /**
         * Returns the width or height of the leading area in a sheet in split
         * mode, in 1/100 of millimeters, or in pixels according to the current
         * sheet zoom factor. In frozen split mode, returns the total size of
         * the frozen columns/rows.
         *
         * @param {Boolean} columns
         *  Whether to return the width of the left sheet area (true), or the
         *  height of the top sheet area (false).
         *
         * @param {Boolean} pixel
         *  Whether to return pixels (true) or 1/100 of millimeters (false).
         *
         * @returns {Number}
         *  The width or height of the leading area if the sheet is in split
         *  mode, otherwise zero.
         */
        function getSplitSize(columns, pixel) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the column interval in frozen split mode
                interval = null,
                // the size in dynamic split mode
                splitSize = 0;

            // return total size of frozen columns/rows
            if (self.hasFrozenSplit()) {
                interval = getSplitInterval(columns);
                return interval ? collection.getIntervalPosition(interval)[pixel ? 'size' : 'sizeHmm'] : 0;
            }

            // return size of dynamic split
            splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight');
            return pixel ? self.convertHmmToPixel(splitSize) : splitSize;
        }

        /**
         * Updates the frozen split attributes after columns or rows have been
         * inserted.
         */
        function insertIntoFrozenSplit(interval, columns) {

            var // attribute names
                LEADING_ANCHOR_NAME = columns ? 'anchorLeft' : 'anchorTop',
                TRAILING_ANCHOR_NAME = columns ? 'anchorRight' : 'anchorBottom',
                SPLIT_SIZE_NAME = columns ? 'splitWidth' : 'splitHeight',
                // the number of inserted columns/rows
                insertSize = SheetUtils.getIntervalSize(interval),
                // the frozen column/row interval (will be null if not frozen)
                frozenInterval = self.hasFrozenSplit() ? getSplitInterval(columns) : null,
                // the new view attributes
                attributes = {};

            // no frozen columns/rows available
            if (!frozenInterval) { return; }

            // inserted before frozen interval: update scroll anchor
            if (interval.first < frozenInterval.first) {
                attributes[LEADING_ANCHOR_NAME] = { index: frozenInterval.first + insertSize, ratio: 0 };
                attributes[TRAILING_ANCHOR_NAME] = { index: frozenInterval.last + 1 + insertSize, ratio: 0 };
            }

            // inserted inside frozen interval: update size of frozen interval
            else if (interval.first <= frozenInterval.last) {
                attributes[SPLIT_SIZE_NAME] = SheetUtils.getIntervalSize(frozenInterval) + insertSize;
                attributes[TRAILING_ANCHOR_NAME] = { index: frozenInterval.last + 1 + insertSize, ratio: 0 };
            }

            self.setViewAttributes(attributes);
        }

        /**
         * Updates the frozen split attributes after columns or rows have been
         * inserted or deleted.
         */
        function deleteFromFrozenSplit(interval, columns) {

            var // attribute names
                LEADING_ANCHOR_NAME = columns ? 'anchorLeft' : 'anchorTop',
                TRAILING_ANCHOR_NAME = columns ? 'anchorRight' : 'anchorBottom',
                SPLIT_SIZE_NAME = columns ? 'splitWidth' : 'splitHeight',
                // the frozen column/row interval (will be null if not frozen)
                frozenInterval = self.hasFrozenSplit() ? getSplitInterval(columns) : null,
                // intersection of frozen interval and deleted interval
                intersectInterval = null,
                // the new view attributes
                attributes = {};

            // no frozen columns/rows available
            if (!frozenInterval) { return; }

            // deleted inside frozen interval: update size of frozen interval
            if ((intersectInterval = SheetUtils.getIntersectionInterval(interval, frozenInterval))) {
                attributes[SPLIT_SIZE_NAME] = SheetUtils.getIntervalSize(frozenInterval) - SheetUtils.getIntervalSize(intersectInterval);
            }

            // deleted before frozen interval: update leading scroll anchor
            if (interval.first < frozenInterval.first) {
                attributes[LEADING_ANCHOR_NAME] = { index: interval.first, ratio: 0 };
            }

            // deleted before or inside frozen interval: update trailing scroll anchor
            if (interval.first <= frozenInterval.last) {
                attributes[TRAILING_ANCHOR_NAME] = { index: interval.first, ratio: 0 };
            }

            self.setViewAttributes(attributes);
        }

        /**
         * Invokes all registered range transformation handlers, after columns
         * or rows have been inserted into or deleted from this sheet.
         */
        function invokeTransformationHandlers(interval, insert, columns) {
            _(transformationHandlers).each(function (handler) {
                handler.call(self, interval, insert, columns);
            });
        }

        /**
         * Invokes all range transformation handlers after columns have been
         * inserted into this sheet.
         */
        function insertColumnsHandler(event, interval) {
            insertIntoFrozenSplit(interval, true);
            invokeTransformationHandlers(interval, true, true);
        }

        /**
         * Invokes all range transformation handlers after columns have been
         * deleted from this sheet.
         */
        function deleteColumnsHandler(event, interval) {
            deleteFromFrozenSplit(interval, true);
            invokeTransformationHandlers(interval, false, true);
        }

        /**
         * Invokes all range transformation handlers after rows have been
         * inserted into this sheet.
         */
        function insertRowsHandler(event, interval) {
            insertIntoFrozenSplit(interval, false);
            invokeTransformationHandlers(interval, true, false);
        }

        /**
         * Invokes all range transformation handlers after rows have been
         * deleted from this sheet.
         */
        function deleteRowsHandler(event, interval) {
            deleteFromFrozenSplit(interval, false);
            invokeTransformationHandlers(interval, false, false);
        }

        /**
         * Updates the selection after the collection of merged cells has been
         * changed. All selected ranges will be expanded to the new merged
         * ranges in the merge collection.
         */
        function changeMergedHandler() {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection');

            _(selection.ranges).each(function (range, index) {
                // expand range to merged ranges, unless an entire column/row is selected
                if (!model.isColRange(range) && !model.isRowRange(range)) {
                    selection.ranges[index] = mergeCollection.expandRangeToMergedRanges(range);
                }
            });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been inserted into
         * this sheet.
         */
        function insertDrawingHandler(event, drawingModel, insertPosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = insertPosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the inserted drawing
            _(selection.drawings).each(function (position) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(insertPosition, position, length - 1) !== 0) { return; }

                // inserted drawing is a predecessor in the same parent
                if (insertPosition[length - 1] <= position[length - 1]) {
                    position[length - 1] += 1;
                }
            });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been deleted from
         * this sheet.
         */
        function deleteDrawingHandler(event, drawingModel, deletePosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = deletePosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the deleted drawing
            Utils.iterateArray(selection.drawings, function (position, index) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(deletePosition, position, length - 1) !== 0) { return; }

                // deleted drawing or any of its children has been deleted
                if (deletePosition[length - 1] === position[length - 1]) {
                    selection.drawings.splice(index, 1);
                }
                // deleted drawing is a predecessor in the same parent
                else if (deletePosition[length - 1] < position[length - 1]) {
                    position[length - 1] -= 1;
                }
            }, { reverse: true });

            self.setViewAttribute('selection', selection);
        }

        // methods ------------------------------------------------------------

        /**
         * Creates and returns a clone of this sheet model instance.
         *
         * @param {Number} targetIndex
         *  The zero-based index of the new cloned sheet.
         *
         * @returns {SheetModel}
         *  A clone of this sheet model with the same sheet type, with cloned
         *  instances of all collections this sheet model contains.
         */
        this.clone = function (targetIndex) {
            return new SheetModel(app, sheetType, this.getExplicitAttributes(), { sourceModel: this, targetIndex: targetIndex });
        };

        /**
         * Returns the type of this sheet.
         *
         * @returns {String}
         *  The type identifier of this sheet.
         */
        this.getType = function () {
            return sheetType;
        };

        /**
         * Returns the current sheet index of this sheet model.
         *
         * @returns {Number}
         *  The current zero-based sheet index of this sheet model.
         */
        this.getIndex = function () {
            return model.getSheetIndexOfModel(this);
        };

        /**
         * Returns the collection of all defined names contained in this sheet.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this sheet.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        /**
         * Returns the collection of all columns in this sheet, containing
         * their size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all columns in this sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the collection of all rows in this sheet, containing their
         * size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all rows in this sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the collection of all merged cells, containing their ranges.
         *
         * @returns {MergeCollection}
         *  The collection of all merged cells in this sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the collection of all cell ranges with validation settings.
         *
         * @returns {ValidationCollection}
         *  The collection of all cell ranges with validation settings in this
         *  sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the drawing collection of this sheet, containing the models
         * of all drawing objects contained in this sheet.
         *
         * @returns {DrawingCollection}
         *  The collection of all drawing models in this sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from this sheet.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this sheet model instance. Receives the following parameters:
         *  (1) {Object} interval
         *      The column/row interval of the insert/delete operation, with
         *      the index properties 'first' and 'last'.
         *  (2) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (3) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _(transformationHandlers).without(handler);
            return this;
        };

        // dimensions ---------------------------------------------------------

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return Utils.convertLengthToHmm(length / this.getViewAttribute('zoom'), 'px');
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return Utils.convertHmmToLength(length * this.getViewAttribute('zoom'), 'px', 1);
        };

        /**
         * Returns the absolute position of the passed cell range in the sheet,
         * in 1/100 of millimeters, as well as in pixels according to the
         * current sheet zoom factor.
         *
         * @param {Object} range
         *  The logical range address.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell range, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell range, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell range, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell range, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell range, in pixels.
         *  - {Number} top
         *      The top offset of the cell range, in pixels.
         *  - {Number} width
         *      The total width of the cell range, in pixels.
         *  - {Number} height
         *      The total height of the cell range, in pixels.
         */
        this.getRangeRectangle = function (range) {

            var // the position and size of the column interval
                colPosition = colCollection.getIntervalPosition(SheetUtils.getColInterval(range)),
                // the position and size of the row interval
                rowPosition = rowCollection.getIntervalPosition(SheetUtils.getRowInterval(range));

            return getRectangle(colPosition, rowPosition);
        };

        /**
         * Returns the absolute position of the specified cell in the sheet, in
         * 1/100 of millimeters, as well as in pixels according to the current
         * sheet zoom factor. If the cell is covered by a merged cell range,
         * returns the position of the entire merged range.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.expandMerged=false]
         *      If set to true, and the cell is part of a merged range, the
         *      position of the entire merged range will be returned, and the
         *      logical address of the merged range will be stored in the
         *      property 'mergedRange' of the result object returned by this
         *      method.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell, in pixels.
         *  - {Number} top
         *      The top offset of the cell, in pixels.
         *  - {Number} width
         *      The total width of the cell, in pixels.
         *  - {Number} height
         *      The total height of the cell, in pixels.
         *  - {Object} [mergedRange]
         *      The logical address of the entire merged range the specified
         *      cell is part of, if the option 'expandMerged' has been set, AND
         *      the passed address is part of a merged cell range.
         */
        this.getCellRectangle = function (address, options) {

            var // whether to expand to merged ranges
                expandMerged = Utils.getBooleanOption(options, 'expandMerged', false),
                // a merged range covering the cell position
                mergedRange = expandMerged ? mergeCollection.getMergedRange(address, 'all') : null;

            // try to extend to a merged range
            if (mergedRange) {
                return _(this.getRangeRectangle(mergedRange)).extend({ mergedRange: mergedRange });
            }

            // calculate position of the single cell
            return getRectangle(colCollection.getEntry(address[0]), rowCollection.getEntry(address[1]));
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the columns in
         * this sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the columns in this
         *  sheet.
         */
        this.hasColSplit = function () {
            return this.getViewAttribute('splitWidth') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for the rows in this
         * sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the rows in this sheet.
         */
        this.hasRowSplit = function () {
            return this.getViewAttribute('splitHeight') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for this sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the sheet.
         */
        this.hasSplit = function () {
            return this.hasColSplit() || this.hasRowSplit();
        };

        /**
         * Returns the current view split mode of this sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  sheet is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return this.hasSplit() ? this.getViewAttribute('splitMode') : '';
        };

        /**
         * Returns whether this sheet is currently split, and dynamic split
         * mode is activated. Returns also true, if the split is currently
         * frozen, but will thaw to dynamic split mode ('frozenSplit' mode).
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return (/^(split|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns whether this sheet is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return (/^(frozen|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidthHmm = function () {
            return getSplitSize(true, false);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeightHmm = function () {
            return getSplitSize(false, false);
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidth = function () {
            return getSplitSize(true, true);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeight = function () {
            return getSplitSize(false, true);
        };

        /**
         * Returns the column interval shown in the left area of the sheet. If
         * the first or last visible columns are visible less than half of
         * their width, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible column instead, regardless how much of its width
         * is visible.
         *
         * @returns {Object|Null}
         *  The column interval shown in the left area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitColInterval = function () {
            return getSplitInterval(true);
        };

        /**
         * Returns the row interval shown in the upper area of the sheet. If
         * the first or last visible rows are visible less than half of their
         * height, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible row instead, regardless how much of its height
         * is visible.
         *
         * @returns {Object|Null}
         *  The row interval shown in the upper area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitRowInterval = function () {
            return getSplitInterval(false);
        };

        /**
         * Sets the split size for dynamic split mode, and updates the scroll
         * anchor view attributes, if the horizontal or vertical split will be
         * enabled.
         *
         * @param {Number} splitWidth
         *  The new value for the 'splitWidth' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between left and
         *  right sheet area.
         *
         * @param {Number} splitHeight
         *  The new value for the 'splitHeight' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between upper
         *  and lower sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (splitWidth, splitHeight, attributes) {

            var // the old split size
                oldSplitWidth = this.getSplitWidthHmm(),
                oldSplitHeight = this.getSplitHeightHmm();

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            _(attributes).extend({ splitMode: 'split', splitWidth: splitWidth, splitHeight: splitHeight });

            // Update left/right scroll positions. Copy from left to right, if
            // split has been disabled, or copy from right to left, if split
            // has been enabled.
            if ((oldSplitWidth > 0) && (attributes.splitWidth === 0)) {
                attributes.anchorRight = this.getViewAttribute('anchorLeft');
            } else if ((oldSplitWidth === 0) && (attributes.splitWidth > 0)) {
                attributes.anchorLeft = this.getViewAttribute('anchorRight');
            }

            // Update top/bottom scroll positions. Copy from top to bottom, if
            // split has been disabled, or copy from bottom to top, if split
            // has been enabled.
            if ((oldSplitHeight > 0) && (attributes.splitHeight === 0)) {
                attributes.anchorBottom = this.getViewAttribute('anchorTop');
            } else if ((oldSplitHeight === 0) && (attributes.splitHeight > 0)) {
                attributes.anchorTop = this.getViewAttribute('anchorBottom');
            }

            // set the new attributes
            return this.setViewAttributes(attributes);
        };

        /**
         * Sets the split size for frozen split mode, and updates the scroll
         * anchor view attributes.
         *
         * @param {Object|Null} colInterval
         *  If set to an index interval, the new column interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  left and right sheet area.
         *
         * @param {Object|Null} rowInterval
         *  If set to an index interval, the new row interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  top and bottom sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (colInterval, rowInterval, attributes) {

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            attributes.splitMode = this.hasDynamicSplit() ? 'frozenSplit' : 'frozen';

            // set column interval if specified
            if (_.isObject(colInterval)) {
                attributes.splitWidth = SheetUtils.getIntervalSize(colInterval);
                attributes.anchorLeft = { index: colInterval.first, ratio: 0 };
                attributes.anchorRight = { index: colInterval.last + 1, ratio: 0 };
            } else {
                attributes.splitWidth = 0;
            }

            // set row interval if specified
            if (_.isObject(rowInterval)) {
                attributes.splitHeight = SheetUtils.getIntervalSize(rowInterval);
                attributes.anchorTop = { index: rowInterval.first, ratio: 0 };
                attributes.anchorBottom = { index: rowInterval.last + 1, ratio: 0 };
            } else {
                attributes.splitHeight = 0;
            }

            // set the new attributes
            return this.setViewAttributes(attributes);
        };

        /**
         * Resets all split attributes, effectively disabling any split view
         * that is currently active.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.clearSplit = function () {
            return this.setDynamicSplit(0, 0);
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether this sheet is protected. In protected sheets, it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether this sheet is protected.
         */
        this.isProtected = function () {
            return this.getMergedAttributes().sheet.locked;
        };

        // column operations --------------------------------------------------

        /**
         * Generates and applies all operations to insert new columns into the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.insertColumns = function (intervals) {
            return insertIntervals(intervals, true);
        };

        /**
         * Generates and applies all operations to delete columns from the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteColumns = function (intervals) {
            return deleteIntervals(intervals, true);
        };

        /**
         * Generates and applies all operations to modify the specified column
         * attributes in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @param {Object} attributes
         *  The (incomplete) attribute set containing all column attributes and
         *  default cell/character attributes to be changed.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.setColumnAttributes = function (intervals, attributes) {
            return setIntervalAttributes(intervals, attributes, true);
        };

        /**
         * Generates and applies all operations to modify the column width in
         * the sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @param {Number} width
         *  The column width, in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width attribute will not
         *  be changed.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customWidth' column attribute.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.setColumnWidth = function (intervals, width, options) {
            return setIntervalSize(intervals, width, true, options);
        };

        // row operations -----------------------------------------------------

        /**
         * Generates and applies all operations to insert new rows into the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.insertRows = function (intervals) {
            return insertIntervals(intervals, false);
        };

        /**
         * Generates and applies all operations to delete rows from the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteRows = function (intervals) {
            return deleteIntervals(intervals, false);
        };

        /**
         * Generates and applies all operations to modify the specified row
         * attributes in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @param {Object} attributes
         *  The (incomplete) attribute set containing all row attributes and
         *  default cell/character attributes to be changed.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.setRowAttributes = function (intervals, attributes) {
            return setIntervalAttributes(intervals, attributes, false);
        };

        /**
         * Generates and applies all operations to modify the row height in the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @param {Number} height
         *  The row height, in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height attribute will not be
         *  changed.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customHeight' row attribute.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.setRowHeight = function (intervals, height, options) {
            return setIntervalSize(intervals, height, false, options);
        };

        // cell operations ----------------------------------------------------

        /**
         * Generates and applies a 'setCellContents' operation that will insert
         * the specified cell contents (values and/or formatting) into the
         * sheet.
         *
         * @param {Number[]} start
         *  The logical address of the first cell to be modified.
         *
         * @param {Array} contents
         *  A two-dimensional array containing the values and attribute sets
         *  for the cells.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.setCellContents = function (start, contents, options) {

            var // the own sheet index
                sheet = this.getIndex(),
                // the generator for the 'setCellContents' operation
                generator = model.createOperationsGenerator(),
                // additional properties for the operation
                operationOptions = { contents: contents };

            // add the parse property
            if (Utils.getBooleanOption(options, 'parse', false)) {
                operationOptions.parse = Utils.LOCALE;
            }

            // create and apply the operation
            generator.generateCellOperation(Operations.SET_CELL_CONTENTS, sheet, start, operationOptions);
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies a 'setCellContents' operation that will insert
         * the specified contents (values and/or formatting) into a single cell
         * in the sheet.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be modified.
         *
         * @param {Any} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed. If set to null or an empty string, the
         *  cell value will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.setSingleCellContents = function (address, value, attributes, options) {

            var // the cell contents object (value and attributes)
                contents = { attrs: attributes };

            // add cell value, if passed (use null to delete the cell value)
            if (_.isString(value) && (value.length === 0)) {
                contents.value = null;
                options = Utils.extendOptions(options, { parse: false });
            } else if (!_.isUndefined(value)) {
                contents.value = value;
            } else {
                options = Utils.extendOptions(options, { parse: false });
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                contents.attrs = Utils.extendOptions(buildNullAttributes(), contents.attrs);
            }

            // do nothing, if neither value nor attributes are present
            if (!_.isObject(contents.attrs) || _.isEmpty(contents.attrs)) { delete contents.attrs; }
            if (_.isEmpty(contents)) { return ''; }

            // send the 'setCellContents' operation (operation expects two-dimensional array)
            return this.setCellContents(address, [[contents]], options);
        };

        /**
         * Generates and applies 'fillCellRange' operations that will insert
         * the specified contents (values and/or formatting) into a range of
         * cells in the sheet.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Any} [value]
         *  The new value for all cells in the passed ranges. If omitted, only
         *  the cell formatting will be changed. If set to null or an empty
         *  string, the cell values will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cells.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell ranges while applying the new attributes.
         *  @param {Number[]} [options.ref]
         *      The address of the reference cell to be inserted into the
         *      operations. If omitted, the top-left cell of the first range
         *      passed to this method will be used.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'overflow': The cell ranges contain too many cells.
         *  - 'internal': Internal error while applying the operation.
         */
        this.fillCellRanges = function (ranges, value, attributes, options) {

            var // the own sheet index
                sheet = this.getIndex(),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator(),
                // additional options for the 'fillCellRange' operations
                operationOptions = {},
                // reference cell address
                ref = Utils.getArrayOption(options, 'ref');

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // use top-left cell as reference cell, if missing
            if (!_.isArray(ref)) { ref = ranges[0].start; }

            // add value to the operation options (prepare shared formula)
            if (_.isNull(value) || (_.isString(value) && (value.length === 0))) {
                operationOptions.value = null;
            } else if (!_.isUndefined(value)) {
                // restrict the number of cells filled at the same time
                if (Utils.getSum(ranges, SheetUtils.getCellCount) > SheetUtils.MAX_FILL_CELL_COUNT) {
                    return 'overflow';
                }
                _(operationOptions).extend({ value: value, shared: 0, ref: ref });
                // add the parse property
                if (_.isString(value) && Utils.getBooleanOption(options, 'parse', false)) {
                    operationOptions.parse = Utils.LOCALE;
                }
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                attributes = Utils.extendOptions(buildNullAttributes(), attributes);
            }

            // validate attributes
            if (_.isObject(attributes) && _.isEmpty(attributes)) {
                attributes = null;
            }

            // create operation to insert a dirty cell style into the document
            if (attributes && _.isString(attributes.styleId)) {
                generator.generateMissingStyleSheetOperation('cell', attributes.styleId);
            }

            // always create 'fillCellRange' operations for cell values (and their attributes)
            if ('value' in operationOptions) {
                if (_.isObject(attributes)) {
                    operationOptions.attrs = attributes;
                }
                _(ranges).each(function (range) {
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, sheet, range, operationOptions);
                    // remove value and dependent properties (keep 'shared' property only)
                    delete operationOptions.value;
                    delete operationOptions.parse;
                    delete operationOptions.ref;
                });
            }

            // create 'fillCellRange', 'setColumnAttributes', or 'setRowAttributes' operations for attributes only
            else if (_.isObject(attributes)) {
                _(ranges).each(function (range) {
                    generateSetAttributesOperation(generator, sheet, range, attributes);
                });
            }

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the visibility of the specified borders in the cell ranges. The
         * border attributes will be applied at range level (inner and outer
         * borders) instead of per-cell level.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Object} borderAttributes
         *  An attribute map containing all border attributes to be changed in
         *  the current selection.
         *
         * @param {Object} [operationOptions]
         *  A map with additional properties for the generated 'fillCellRange'
         *  operation.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.setBorderAttributes = function (ranges, borderAttributes) {

            var // the own sheet index
                sheet = this.getIndex(),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator(),
                // additional options for the operations
                operationOptions = { attrs: { cell: borderAttributes }, rangeBorders: true };

            // generates a 'setColumnAttributes' or 'setRowAttributes' operation for the adjacent cells of a column
            function generateAdjacentIntervalOperations(index, newBorderName, columns) {
                var opName = columns ? Operations.SET_COLUMN_ATTRIBUTES : Operations.SET_ROW_ATTRIBUTES,
                    opOptions = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                // do not send borders for rows without custom format
                if (columns || rowCollection.getEntry(index).attributes.row.customFormat) {
                    generator.generateIntervalOperation(opName, sheet, { first: index, last: index }, opOptions);
                }
            }

            // generates a 'fillCellRange' operation for the adjacent cells of one border
            function generateAdjacentRangeOperations(firstCol, firstRow, lastCol, lastRow, newBorderName) {
                var range = { start: [firstCol, firstRow], end: [lastCol, lastRow] },
                    opOptions = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, sheet, range, opOptions);
            }

            // reduce all ranges to a single column/row, if only a single outer line will be changed
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (_.size(borderAttributes) === 1) {
                if ('borderLeft' in borderAttributes) {
                    _(ranges).each(function (range) { range.end[0] = range.start[0]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderRight' in borderAttributes) {
                    _(ranges).each(function (range) { range.start[0] = range.end[0]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderTop' in borderAttributes) {
                    _(ranges).each(function (range) { range.end[1] = range.start[1]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderBottom' in borderAttributes) {
                    _(ranges).each(function (range) { range.start[1] = range.end[1]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                }
            }

            // send the 'fillCellRange' operations
            _(ranges).each(function (range) {
                // generate operation for the current range
                // setColumnAttributes, setRowAttributes or fillCellRange
                if (model.isColRange(range)) {
                    generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, sheet, SheetUtils.getColInterval(range), operationOptions);
                    // Generate operation for the adjacent cells of the range
                    if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.start[0] - 1, 'borderRight', true);
                    }
                    if ((range.end[0] < model.getMaxCol()) && ('borderRight' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.end[0] + 1, 'borderLeft', true);
                    }
                } else if (model.isRowRange(range)) {
                    operationOptions.attrs.row = { customFormat: true };
                    generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, sheet, SheetUtils.getRowInterval(range), operationOptions);
                    // Generate operation for the adjacent cells of the range
                    if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.start[1] - 1, 'borderBottom', false);
                    }
                    if ((range.end[1] < model.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.end[1] + 1, 'borderTop', false);
                    }
                } else {
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, sheet, range, operationOptions);

                    // Deleting all borders at neighboring cells -> first approach: delete all neighboring borders, so that
                    // there is always only one valid border between two cells.
                    // Generate operation for the adjacent cells of the range
                    if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0] - 1, range.start[1], range.start[0] - 1, range.end[1], 'borderRight');
                    }
                    if ((range.end[0] < model.getMaxCol()) && ('borderRight' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.end[0] + 1, range.start[1], range.end[0] + 1, range.end[1], 'borderLeft');
                    }
                    if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0], range.start[1] - 1, range.end[0], range.start[1] - 1, 'borderBottom');
                    }
                    if ((range.end[1] < model.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0], range.end[1] + 1, range.end[0], range.end[1] + 1, 'borderTop');
                    }
                }
            });

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the specified border properties (line style, line color, line width)
         * of all visible cell borders in a range of cells.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the cell range.
         *  Omitted border properties will not be changed.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.changeVisibleBorders = function (ranges, border) {

            var // the own sheet index
                sheet = this.getIndex(),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator(),
                // additional options for the operations
                operationOptions = { rangeBorders: true, visibleBorders: true };

            // generates an operation for the passed range for horizontal or vertical borders
            function generateRangeOperation(range, columns) {

                var // the passed range, that will be expanded at two borders
                    expandedRange = _.copy(range, true),
                    // the array index in cell addresses
                    addressIndex = columns ? 0 : 1,
                    // the border attributes inserted into the operation
                    borderAttributes = {};

                // add inner border property
                borderAttributes[columns ? 'borderInsideVert' : 'borderInsideHor'] = border;

                // expand range at leading border, or add outer border attribute at leading sheet border
                if (expandedRange.start[addressIndex] > 0) {
                    expandedRange.start[addressIndex] -= 1;
                } else {
                    borderAttributes[columns ? 'borderLeft' : 'borderTop'] = border;
                }

                // expand range at trailing border, or add outer border attribute at trailing sheet border
                if (expandedRange.end[addressIndex] < (columns ? model.getMaxCol() : model.getMaxRow())) {
                    expandedRange.end[addressIndex] += 1;
                } else {
                    borderAttributes[columns ? 'borderRight' : 'borderBottom'] = border;
                }

                generateSetAttributesOperation(generator, sheet, expandedRange, { cell: borderAttributes }, operationOptions);
            }

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // expand the ranges and generate operations for the border attributes
            _(ranges).each(function (range) {
                generateRangeOperation(range, true);
                generateRangeOperation(range, false);
            });

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'clearCellRange' operations that will delete
         * the values and formatting from a range of cells in the sheet.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.clearCellRanges = function (ranges) {

            var // the own sheet index
                sheet = this.getIndex(),
                // the generator for the 'clearCellRange' operations
                generator = model.createOperationsGenerator();

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // the create 'clearCellRange' operations
            _(ranges).each(function (range) {
                generator.generateRangeOperation(Operations.CLEAR_CELL_RANGE, sheet, range);
            });

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'mergeCells' operations for all passed ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges the entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the ranges.
         *
         * @param {Number} usedCols
         *  The number of used columns in the document.
         *
         * @param {Number} usedRows
         *  The number of used rows in the document.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'overlap': The passed cell ranges overlap each other.
         *  - 'internal': Internal error while applying the operation.
         */
        this.mergeRanges = function (ranges, type, usedCols, usedRows) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the generator for the 'mergeCells' operations
                generator = model.createOperationsGenerator(),
                // the index inside the position that need to be adapted
                index = (type === 'horizontal') ? 1 : 0,
                // the maximum number of rows or columns in the document
                maxVal = (type === 'horizontal') ? model.getMaxRow() : model.getMaxCol(),
                // the maximum number of used rows or columns in the document
                usedVal = (type === 'horizontal') ? usedRows : usedCols;

            usedVal = usedVal || 1;

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // do not allow to merge overlapping ranges
            if ((type !== 'unmerge') && SheetUtils.anyRangesOverlap(ranges)) {
                return 'overlap';
            }

            // generate the 'mergeCells' operations
            _(ranges).each(function (range) {

                // Fix for 30662: Reducing the range for horizontal or vertical merge, if columns or rows are selected
                if (((type === 'horizontal') || (type === 'vertical')) && (range.start[index] === 0) && (range.end[index] === maxVal)) {
                    range.end[index] = Math.min(usedVal, SheetUtils.MAX_MERGED_RANGES_COUNT) - 1;
                } else if (((type === 'horizontal') || (type === 'vertical')) && ((range.end[index] - range.start[index] + 1) > SheetUtils.MAX_MERGED_RANGES_COUNT)) {
                    range.end[index] = range.start[index] + SheetUtils.MAX_MERGED_RANGES_COUNT - 1;
                }

                generator.generateRangeOperation(Operations.MERGE_CELLS, sheet, range, { type: type });
            });

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        // drawing operations -------------------------------------------------

        /**
         * Generates and applies an 'insertDrawing' operation that will insert
         * a new drawing object into this sheet.
         *
         * @param {Number[]} position
         *  The logical position of the new drawing object.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing in the sheet.
         *
         * @param {Function} [generatorCallback]
         *  A callback function that will be invoked after the initial
         *  'insertDrawing' operation has been generated. Allows to create
         *  additional operations for the drawing object, before all these
         *  operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SpreadsheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array} position
         *      The logical position of the new drawing object in the sheet.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertDrawing = function (position, type, attributes, generatorCallback) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // add missing generic anchor attributes to the passed attribute set
            drawingCollection.addGenericAnchorAttributes(attributes);

            // create the 'insertDrawing' operation, invoke callback function for more operations
            generator.generateDrawingOperation(Operations.INSERT_DRAWING, sheet, position, { type: type, attrs: attributes });
            if (_.isFunction(generatorCallback)) {
                generatorCallback.call(this, generator, sheet, position);
            }

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies one or more 'deleteDrawing' operations that
         * will delete the specified drawings from this sheet.
         *
         * @param {Array} positions
         *  An array with logical positions of the drawing objects to be
         *  deleted.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteDrawings = function (positions) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the 'deleteDrawing' operations, visit the positions backwards to prevent
            // changed positions of following drawings after deleting a previous drawing
            Utils.iterateArray(positions, function (position) {
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, sheet, position);
            }, { reverse: true });

            // apply the operations
            return model.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        // initialization -----------------------------------------------------

        // class member construction
        (function (args) {

            var // the source sheet model and target sheet index used for copy construction
                cloneData = args[SheetModel.length];

            if (_.isObject(cloneData)) {
                // copy construction
                self.setViewAttributes(cloneData.sourceModel.getViewAttributes());
                nameCollection = cloneData.sourceModel.getNameCollection().clone(self, cloneData.targetIndex);
                colCollection = cloneData.sourceModel.getColCollection().clone(self, cloneData.targetIndex);
                rowCollection = cloneData.sourceModel.getRowCollection().clone(self, cloneData.targetIndex);
                mergeCollection = cloneData.sourceModel.getMergeCollection().clone(self, cloneData.targetIndex);
                validationCollection = cloneData.sourceModel.getValidationCollection().clone(self, cloneData.targetIndex);
                drawingCollection = cloneData.sourceModel.getDrawingCollection().clone(self, cloneData.targetIndex);
            } else {
                // regular construction
                nameCollection = new NameCollection(app, self);
                colCollection = new ColRowCollection(app, self, true);
                rowCollection = new ColRowCollection(app, self, false);
                mergeCollection = new MergeCollection(app, self);
                validationCollection = new ValidationCollection(app, self);
                drawingCollection = new SheetDrawingCollection(app, self);
            }
        }(arguments));

        // handle collection changes
        colCollection.on({ 'insert:entries': insertColumnsHandler, 'delete:entries': deleteColumnsHandler });
        rowCollection.on({ 'insert:entries': insertRowsHandler, 'delete:entries': deleteRowsHandler });
        mergeCollection.on('triggered', changeMergedHandler);
        drawingCollection.on({ 'insert:drawing': insertDrawingHandler, 'delete:drawing': deleteDrawingHandler });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            drawingCollection.destroy();
            validationCollection.destroy();
            mergeCollection.destroy();
            colCollection.destroy();
            rowCollection.destroy();
            nameCollection.destroy();
            model = cellStyles = null;
            nameCollection = colCollection = rowCollection = mergeCollection = validationCollection = drawingCollection = null;
            transformationHandlers = null;
        });

    } // class SheetModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: SheetModel });

});
