/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/drawingcontroller',
    ['io.ox/office/tk/config',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/chartcreator',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/model/tokenarray',
     'gettext!io.ox/office/spreadsheet'
    ], function (Config, Operations, Controls, ChartCreator, Label, Utils, TokenArray, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        SOURCE_CANCEL_VIEW = 'change:activesheet change:visiblesheets change:layoutdata celledit:enter celledit:change celledit:leave insert:merged delete:merged change:merged',
        SOURCE_CANCEL_MODEL = 'change:sheetattributes',
        SOURCE_CANCEL_GRID = 'select:end';


    // class DrawingController =================================================

    /**
     * DrawingController (will) encapsulate(s) all Drawing-Features of the Controller and of the View (Sidepane-toolbox)
     * For now Chart are part of Drawings!
     */
    function DrawingController(app) {

        var self = this,
            model = null,
            view = null,
            charttypepicker = null,
            sourceSelector = false;

        // private methods ----------------------------------------------------

        /**
         * initToolBox initiates the view components
         * -delete Drawing
         * -change Chart Type
         * -change Chart Sub Type (clustered, stacked, percent)
         * -change chart curves (for line charts)
         * -change Chart Colors (patterns with grayscale and accents, multicolors for single series)
         * -change Chart Style (background, highlighteffect)
         */
        function initToolBox() {

            var drawingbox = view.createToolBox('drawing', { label: /*#. drawing objects: images, diagrams, ... */ gt('Drawing'), visible: 'document/editable/drawing' });
            drawingbox.addRightTab();
            drawingbox.addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_OPTIONS));

            charttypepicker = new Controls.ChartTypePicker({ width: 75 }, { width: 90 });

            if (Config.isDebug()) {
                var chartbox = view.createToolBox('chart', { label: /*#. drawing objects: images, diagrams, ... */ gt('Chart'), visible: 'drawing/chart' });


                chartbox.addGroup('drawing/charttype', charttypepicker.getCatRadioList());
                chartbox.addGap(5);
                chartbox.addGroup('drawing/chartsubtype', charttypepicker.getEntRadioList());

                chartbox.newLine();
                chartbox.addGroup('drawing/chartcolorset', new Controls.ChartColorsetPicker(app, { width: 75 }));
                chartbox.addGap(5);
                chartbox.addGroup('drawing/chartstyleset', new Controls.ChartStylesetPicker({ width: 90 }));

                chartbox.newLine();
                chartbox.addGroup('drawing/chartvarycolor', new Button(Controls.VARYCOLOR_CHART_OPTIONS));
                chartbox.addGap(5);
                chartbox.addGroup('drawing/chartcurved', new Button(Controls.CURVED_CHART_OPTIONS));
                chartbox.addGap(5);
                chartbox.addGroup('drawing/chartsource', new Button(Controls.CREATE_CHART_SERIES_OPTIONS));

            }
        }


        /**
         * changeSelection
         * if only one Drawing is selected and this is a Chart, the function highlights its source-cells via TokenArray
         */
        var changeSelection = (function () {

            var // the token arrays containing the highlighted ranges of the selected drawing object
                tokenArrays = [];

            return function (evt, selection) {
                if (tokenArrays.length > 0) {
                    view.endRangeHighlighting();
                    tokenArrays = [];
                }

                if (selection.drawings.length === 1) {

                    var drawingCollection = view.getActiveSheetModel().getDrawingCollection(),
                        chartModel = drawingCollection.findModel(selection.drawings[0], { type: 'chart' });

                    if (chartModel) {
                        chartModel.getTokenArrays(tokenArrays, Config.isDebug() && !model.getEditMode());
                    }
                }

                if (tokenArrays.length > 0) {
                    view.startRangeHighlighting(tokenArrays, { draggable: false, resolveNames: true });
                }
            };
        }());

        /**
         * @return the ID of the current selected Drawing
         */
        function getDrawingIndices() {
            var selection = view.getSelection();
            if (selection.drawings.length == 1) {
                return selection.drawings[0];
            }
        }

        /**
         * @return the current selected Drawing
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingIndices();
            if (drawingIndices) {
                var drawingCollection = view.getActiveSheetModel().getDrawingCollection();
                return drawingCollection.findModel(drawingIndices, { type: type });
            }
        }

        /**
         * setDrawingAttributes generates the Operation for changing Attributes of Drawings and Charts
         */
        function setDrawingAttributes(attrs) {
            var generator = model.createOperationsGenerator();
            generateDrawingOperation(generator, Operations.SET_DRAWING_ATTRIBUTES, attrs);
            model.applyOperations(generator.getOperations());
        }

        function generateDrawingOperation(generator, opName, attrs) {
            var position = getDrawingIndices();
            generator.generateDrawingOperation(opName, view.getActiveSheet(), position, { attrs: attrs });
        }

        /**
         * generates an insert-Chart-Op and Insert-Chart-DataSeries chosen by
         * current Selection
         *
         * @returns {jQuery.Deferred}
         *  a deferred object, which is already resolved
         */
        function insertChart() {
            return new ChartCreator(app).deferred();
        }

        function initSourceCancelListener(end) {
            view.on(SOURCE_CANCEL_VIEW, end);
            view.getActiveGridPane().on(SOURCE_CANCEL_GRID, end);
            model.on(SOURCE_CANCEL_MODEL, end);

        }

        function endSourceCancelListener(end) {
            view.off(SOURCE_CANCEL_VIEW, end);
            view.getActiveGridPane().off(SOURCE_CANCEL_GRID, end);
            model.off(SOURCE_CANCEL_MODEL, end);
        }

        function setChartSource() {
            var chartSourceToken = null;
            sourceSelector = true;

            view.registerCellSelectionHandlers(function () {

            }, function (selection) {
                if (!sourceSelector) {
                    return false;
                }
                var range = selection.ranges[0];
                var sheet = view.getActiveSheet();

                if (!chartSourceToken) {
                    chartSourceToken = new TokenArray(app, view.getActiveSheetModel());
                    view.startRangeHighlighting(chartSourceToken, { resolveNames: true });
                }

                chartSourceToken.clear().appendRange(range, { sheet: sheet, abs: true });

                //view.getActiveSheetModel().setViewAttribute('activeSelection', selection);

                return true;
            });


            function selectEnd() {
                endSourceCancelListener(selectEnd);

                view.unregisterCellSelectionHandlers();

                if (chartSourceToken) {
                    var range = chartSourceToken.extractRanges()[0].range;
                    ChartCreator.generateOperations(app, range, getDrawingModel('chart'));

                    chartSourceToken = null;
                }


                sourceSelector = false;
            }

            initSourceCancelListener(selectEnd);



            return sourceSelector;

        }

        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['image/insert/dialog'] = {
                parent: 'document/editable',
                set: function () {
                    return view.showInsertImageDialog();
                },
                // allow insertion of drawings only if sheet is not protected
                enable: function () { return !view.getActiveSheetModel().isProtected(); },
                focus: 'wait' // wait for the dialog before returning focus to application
            };

            defs['chart/insert'] = {
                parent: 'document/editable',
                set: insertChart,
                enable: function () {
                    var ranges = view.getSelection().ranges;
                    if (ranges.length != 1) { return false; }
                    if (Math.abs(ranges[0].start[0] - ranges[0].end[0]) > 0) { return true; }
                    if (Math.abs(ranges[0].start[1] - ranges[0].end[1]) > 0) { return true; }
                    return false;
                }
            };

            defs['document/editable/drawing'] = {
                parent: 'document/editable',
                enable: function () { return view.hasDrawingSelection() && !view.getActiveSheetModel().isProtected(); }
            };

            defs.drawing = {
                parent: 'document/editable/drawing',
                get: getDrawingModel
            };

            defs['drawing/delete'] = {
                parent: 'document/editable/drawing',
                set: function () { view.deleteDrawings(); }
            };

            defs['drawing/chart'] = {
                parent: 'document/editable/drawing',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getRealChartType() : null; },
                set: function (type) {
                    var stacking = charttypepicker.getCompatibleEntry(type);
                    setDrawingAttributes({ chart: {type: type + '2d', stacking: stacking.value, curved: false}});
                }
            };

            defs['drawing/chartsubtype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getChartSubType() : null; },
                set: function (stacking) { setDrawingAttributes({ chart: {stacking: stacking}}); },
                enable: function () {
                    var chartModel = getDrawingModel('chart');
                    return chartModel && chartModel.getSeriesCount() > 1 && charttypepicker.getEntryCount() > 1;
                }
            };

            defs['drawing/chartcurved'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getMergedAttributes().chart.curved : false; },
                set: function (state) { setDrawingAttributes({ chart: {curved: state}}); },
                enable: function () {
                    var chartModel = getDrawingModel('chart');
                    return chartModel && chartModel.getRealChartType() == 'line';
                }
            };

            defs['drawing/chartvarycolor'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getMergedAttributes().chart.varyColors : false; },
                set: function (state) { getDrawingModel('chart').changeVaryColors(state); },
                enable: function () {
                    var chartModel = getDrawingModel('chart');
                    return chartModel && chartModel.isVaryColorEnabled();
                }
            };

            defs['drawing/chartcolorset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeColorSet(colorset); }
            };

            defs['drawing/chartstyleset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { getDrawingModel('chart').changeStyleSet(colorset); }
            };


            defs['drawing/chartsource'] = {
                parent: 'drawing/chart',
                set: setChartSource,
                get: function () {return sourceSelector; }
            };

            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        //View and Listener initialization
        app.on('docs:init', function () {
            model = app.getModel();
            view = app.getView();

            initToolBox();
            view.on('change:selection', changeSelection);
        });

    } // class DrawingController

    // exports ================================================================

    return DrawingController;

});
