/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/basecontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/basemenu',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/unitfield',
     'io.ox/office/tk/control/combofield',
     'io.ox/office/tk/control/spinfield',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/baseframework/view/component',
     'io.ox/office/tk/control/boundradiolists',
     'gettext!io.ox/office/main'
    ], function (Utils, KeyCodes, BaseMenu, Group, Label, Button, RadioGroup, RadioList, TextField, UnitField, ComboField, SpinField, MenuMixin, Component, BoundRadioLists, gt) {

    'use strict';

    // static class BaseControls ==============================================

    /**
     * Provides different classes for GUI controls. Collects all standard
     * controls defined in the toolkit, and adds more sophisticated controls.
     */
    var BaseControls = {
            Group: Group,
            Label: Label,
            Button: Button,
            RadioGroup: RadioGroup,
            RadioList: RadioList,
            TextField: TextField,
            UnitField: UnitField,
            ComboField: ComboField,
            SpinField: SpinField,
            BoundRadioLists: BoundRadioLists
        };

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Close' button.
     *
     * @constant
     */
    BaseControls.QUIT_OPTIONS = { icon: 'icon-remove', tooltip: gt('Close document') };

    /**
     * Standard options for the 'Hide side panel' button.
     *
     * @constant
     */
    BaseControls.HIDE_SIDEPANE_OPTIONS = { icon: 'docs-hide-sidepane', tooltip: gt('Hide side panel'), value: false };

    /**
     * Standard options for the 'Show side panel' button.
     *
     * @constant
     */
    BaseControls.SHOW_SIDEPANE_OPTIONS = { icon: 'docs-show-sidepane', tooltip: gt('Show side panel'), value: true };

    /**
     * Standard options for the 'Zoom out' button.
     *
     * @constant
     */
    BaseControls.ZOOMOUT_OPTIONS = { icon: 'docs-zoom-out', tooltip: gt('Zoom out') };

    /**
     * Standard options for the 'Zoom in' button.
     *
     * @constant
     */
    BaseControls.ZOOMIN_OPTIONS = { icon: 'docs-zoom-in', tooltip: gt('Zoom in') };

    // class StatusLabel ======================================================

    /**
     * A status label with special appearance and fade-out animation.
     *
     * The method StatusLabel.setValue() accepts an options map containing the
     * same options as described for the 'options' constructor parameter,
     * allowing to override the default options of the status label.
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the default behavior of the status
     *  label. The following options are supported:
     *  @param {String} [initOptions.type='info']
     *      The label type ('success', 'warning', 'error', or 'info').
     *  @param {Boolean} [initOptions.fadeOut=false]
     *      Whether to fade out the label automatically after a short delay.
     *  @param {Number} [initOptions.delay=0]
     *      The delay time after the status label will be actually updated.
     *  @param {Number} [initOptions.changed=false]
     *      If set to true, a status label currently faded out will only be
     *      shown if its label has really changed (and not with every update
     *      call with the same label).
     */
    BaseControls.StatusLabel = Label.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the new type of the label (colors)
            type = Utils.getStringOption(initOptions, 'type', 'info'),

            // whether to fade out the label automatically
            fadeOut = Utils.getBooleanOption(initOptions, 'fadeOut', false),

            // delay time after the label will be updated
            delay = Utils.getIntegerOption(initOptions, 'delay', 0, 0),

            // whether to show the label only after the value has changed
            changed = Utils.getBooleanOption(initOptions, 'changed', false),

            // current initial delay timer before the state of the label will be changed
            initialTimer = null,

            // current animation delay timer (jQuery.delay() does not work well with jQuery.stop())
            animationTimer = null;

        // base constructor ---------------------------------------------------

        Label.call(this, { classes: 'status-label' });

        // private methods ----------------------------------------------------

        /**
         * Stops the running jQuery fade animation and removes all explicit CSS
         * attributes from the group node.
         */
        function stopNodeAnimation() {
            self.getNode().stop(true).css({ display: '', opacity: '' });
        }

        /**
         * Callback called from the Label.setValue() method.
         */
        function updateHandler(caption, options) {

            // do nothing if the label will not be updated when the text does not change
            if (Utils.getBooleanOption(options, 'changed', changed) && (self.getLabelText() === caption)) {
                return;
            }

            if (initialTimer) { initialTimer.abort(); }
            initialTimer = app.executeDelayed(function () {

                // stop running fade-out and remove CSS attributes added by jQuery
                if (animationTimer) { animationTimer.abort(); }
                stopNodeAnimation();

                // update the status label
                if (_.isString(caption) && (caption.length > 0)) {
                    self.setLabelText(caption).show().getNode().attr('data-type', Utils.getStringOption(options, 'type', type));
                    if (Utils.getBooleanOption(options, 'fadeOut', fadeOut)) {
                        animationTimer = app.executeDelayed(function () {
                            self.getNode().fadeOut(function () { self.hide(); });
                        }, { delay: 2000 });
                    }
                } else {
                    self.hide();
                }

            }, { delay: Utils.getIntegerOption(options, 'delay', delay, 0) });
        }

        // initialization -----------------------------------------------------

        this.registerUpdateHandler(updateHandler).setValue(null);
        app.on('docs:destroy', stopNodeAnimation);

    }}); // class StatusLabel

    // class PopupMenu ========================================================

    /**
     * A generic drop-down button control that shows a complete self-contained
     * view component instance with completely independent controls in its
     * drop-down menu.
     *
     * @constructor
     *
     * @extends Button
     * @extends MenuMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {String} componentId
     *  The identifier for the view component contained in the drop-down menu.
     *  Must be unique across all view components in the application.
     *
     * @param {Object} [initOptions]
     *  A map of options with properties for the drop-down button. Supports all
     *  options also supported by the base class Button, the mix-in class
     *  MenuMixin (visibility of caret icon), and the class BaseMenu (settings
     *  for the drop-down menu).
     */
    BaseControls.PopupMenu = Button.extend({ constructor: function (app, componentId, initOptions) {

        var // the view component in the drop-down menu
            component = new Component(app, componentId, { groupInserter: groupInserter, landmark: false });

        // base constructors --------------------------------------------------

        Button.call(this, initOptions);
        MenuMixin.call(this, new BaseMenu(this.getNode(), initOptions), Utils.extendOptions(initOptions, { button: this.getButtonNode() }));

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed group into the menu component.
         */
        function groupInserter(group) {

            var // the root node of the passed group
                groupNode = group.getNode(),
                // the target node for the passed group
                targetNode = component.getNode(),
                // the last child in the menu component
                lastNode = targetNode.children().last();

            if (Utils.getBooleanOption(group.getOptions(), 'appendInline', false)) {
                if (lastNode.is('.inline-container')) {
                    targetNode = lastNode;
                } else {
                    targetNode = $('<div>').addClass('inline-container').appendTo(targetNode);
                    if (lastNode.is('.group')) { targetNode.append(lastNode); }
                }
            }

            // insert the group node
            targetNode.append(groupNode);
        }

        /**
         * Handles 'keydown' events in the drop-down menu.
         */
        function menuKeyDownHandler(event) {
            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                component.moveFocus('prev');
                return false;
            case KeyCodes.DOWN_ARROW:
                component.moveFocus('next');
                return false;
            case KeyCodes.HOME:
                component.moveFocus('first');
                return false;
            case KeyCodes.END:
                component.moveFocus('last');
                return false;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Adds the passed control group as a 'private group' to the drop-down
         * menu. The 'group:change' events triggered by the group will not be
         * forwarded to the listeners of the drop-down view component.
         *
         * @param {Group} group
         *  The control group object to be inserted into the drop-down menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addPrivateGroup = function (group) {
            component.addPrivateGroup(group);
            return this;
        };

        /**
         * Adds the passed control group to the drop-down menu. The view
         * component contained in the drop-down menu listens to 'change:items'
         * events of the application controller and forwards changed values to
         * all registered control groups.
         *
         * @param {String} key
         *  The unique key of the control group.
         *
         * @param {Group} group
         *  The control group object to be inserted into the drop-down menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addGroup = function (key, group) {
            component.addGroup(key, group);
            return this;
        };

        /**
         * Adds a separator line after the last control group in the drop-down
         * menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addSeparator = function () {
            component.getNode().append($('<div>').addClass('separator-line'));
            return this;
        };

        // initialization -----------------------------------------------------

        // insert the root node of the view component into the drop-down menu
        this.getMenu().appendContentNodes(component.getNode());

        // additional keyboard shortcuts for focus navigation
        component.getNode().on('keydown', menuKeyDownHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            component.destroy();
            component = null;
        });

    }}); // class PopupMenu

    // exports ================================================================

    return BaseControls;

});
